<?
/**
 * Samurai_Loader
 * 
 * ロード処理を担当。
 * 基本的にはSamurai_Configのdirectory.samuraiに依存する。
 * 静的にのみ使用される。
 * 
 * @package    Samurai
 * @subpackage Component.Samurai
 * @copyright  Befool,Inc.
 * @author     Satoshi Kiuchi <satoshi.kiuchi@befool.co.jp>
 * @license    http://opensource.org/licenses/bsd-license.php  The modified BSD License
 */
class Samurai_Loader
{
    /**
     * コンストラクタ。
     * @access     private
     */
    private function __construct()
    {
        
    }
    
    
    
    
    
    /**
     * ロード。
     * @access     public
     * @param      string  $path   パス
     * @param      array   $dirs   候補ディレクトリ(指定しない場合はdirectory.samurai参照)
     */
    public static function load($path, $dirs=NULL)
    {
        $file = self::getPath($path, $dirs);
        if(self::isReadable($file)){
            include_once($file);
            return true;
        } else {
            return false;
        }
    }
    
    
    /**
     * クラスからロード。
     * component_dirを優先して検索した後に、library_dirを検索して、
     * 最後にinclude_pathを総ざらいします。
     * @access     public
     * @param      string  $class_name
     * @param      array   $dirs
     */
    public static function loadByClass($class_name, $dirs=NULL)
    {
        if(class_exists($class_name, false)) return true;
        $class_path = self::getPathByClass($class_name);
        //component_dirの検索
        if(preg_match('|^action/|', $class_path)){
            $component_path = Samurai_Config::get('directory.action').DS.preg_replace('|^action/|', '', $class_path);
        } else {
            $component_path = Samurai_Config::get('directory.component').DS.$class_path;
        }
        if(!self::load($component_path)){
            //library_dirの検索
            $library_path = Samurai_Config::get('directory.library').DS.$class_path;
            if(!self::load($library_path)){
                //PEARの検索
                $pear_path = str_replace('_', '/', $class_name) . '.php';
                if(!self::load($pear_path, explode(PS, get_include_path()))){
                    return self::load($class_path, explode(PS, get_include_path()));
                }
            }
        }
        return true;
    }
    
    
    /**
     * ファイルのパスを取得する。
     * 見つからない場合でも、最後の候補ディレクトリに存在した場合のパスが返される。
     * @access     public
     * @param      string  $path        パス
     * @param      boolean $firstonly   存在するかどうかに関わらず、優先度の高い場所を返却するかどうか
     * @param      array   $dirs        候補ディレクトリ(指定しない場合はdirectory.samurai参照)
     * @return     string  ファイルパス
     */
    public static function getPath($path, $firstonly=false, $dirs=NULL)
    {
        $pathes = self::getPathes($path, $dirs);
        foreach($pathes as $i => $file){
            if(self::isReadable($file)){
                return $file;
            }
            if($firstonly && !$i) return $file;
        }
        return $file;
    }
    
    
    /**
     * あるパスに対する候補ディレクトリを全て返却する。
     * @access     public
     * @return     array   候補ディレクトリ
     */
    public static function getPathes($path, $dirs=NULL)
    {
        $pathes = array();
        if(self::isAbsolutePath($path)){
            $pathes[] = $path;
        } else {
            if($dirs === NULL) $dirs = Samurai::getSamuraiDirs();
            foreach($dirs as $dir){
                $pathes[] = $dir.DS.$path;
            }
        }
        return $pathes;
    }
    
    
    /**
     * ファイルパスをクラス名から取得する。
     * ここが規約となる。
     * @access     public
     * @param      string  $class_name
     * @return     string  $class_path
     */
    public static function getPathByClass($class_name)
    {
        $names = explode('_', $class_name);
        $main_name = ucfirst(array_pop($names));
        foreach($names as &$name){
            $name = strtolower($name);
        }
        array_push($names, $main_name);
        return join(DS, $names).'.class.php';
    }
    
    
    
    
    
    /**
     * 登録されているSamurai_Dirを走査し、発見されたファイルをすべてincludeする。
     * @access     public
     * @param      string  $path
     */
    public static function includes($path)
    {
        if(Samurai_Loader::isAbsolutePath($path)){
            include($path);
        } else {
            foreach(Samurai::getSamuraiDirs() as $dir){
                $filepath = $dir . DS . $path;
                if(Samurai_Loader::isReadable($filepath)){
                    include_once($filepath);
                }
            }
        }
    }
    
    
    
    
    
    /**
     * SamuraiFW用のオートロード。
     * @access     public
     * @param      string  $class_name   クラス名
     */
    public static function autoload($class_name)
    {
        self::loadByClass($class_name);
    }
    
    
    
    
    
    /**
     * ファイルが読み込み可能かチェックする。
     * @access     public
     * @param      string  $file   ファイルの絶対パス
     * @return     boolean ファイルが存在して読み込めるかどうか
     */
    public static function isReadable($file)
    {
        return file_exists($file) && is_readable($file);
    }
    
    
    /**
     * 絶対パスかどうかを判断する。
     * @access     public
     * @param      string  $path
     * @return     boolean 絶対パスかどうか
     */
    public static function isAbsolutePath($path)
    {
        if(preg_match('/^WIN/', PHP_OS)){
            return preg_match('/^[A-Z]:\\\/i', $path);
        } else {
            return preg_match('|^/|', $path);
        }
    }
}
