<?
/**
 * Samurai_FilterChain
 * 
 * フィルターを保持、実行するクラス。
 * 
 * @package    Samurai
 * @subpackage Samurai
 * @copyright  Befool,Inc.
 * @author     Satoshi Kiuchi <satoshi.kiuchi@befool.co.jp>
 * @license    http://opensource.org/licenses/bsd-license.php  The modified BSD License
 */
class Samurai_FilterChain
{
    private
        /** @var        array   フィルターを保持 */
        $_filters = array(),
        /** @var        array   フィルターの配置を保持 */
        $_position = array(),
        /** @var        int     現在の位置を保持 */
        $_index = 0;
    
    
    /**
     * コンストラクタ。
     * @access    public
     */
    public function __construct()
    {
        
    }
    
    
    
    
    
    /**
     * FilterChainを組み立てる。
     * @access    public
     * @param     object  $ConfigStack   ConfigStackインスタンス
     */
    public function build($ConfigStack)
    {
        foreach($ConfigStack->getConfig() as $alias => $value){
            $config = $ConfigStack->getConfig($alias);
            if(!$this->add($alias, $config['filter'], $config['attributes'])){
                Samurai_Logger::error('Filter add failed... -> %s', $alias);
            }
        }
    }
    
    
    /**
     * FilterChainを実行する。
     * @access    public
     * @since     1.0.0
     */
    public function execute()
    {
        if($this->hasNext()){
            $name = $this->getCurrentFilterName();
            $Filter = $this->getFilterByName($name);
            if($Filter){
                $this->next();
                Samurai::getContainer()->injectDependency($Filter);
                $Filter->execute();
            } else {
                throw new Samurai_Exception('Filter error.');
            }
        }
    }
    
    
    
    
    
    /**
     * FilterChainの最後にFilterを追加
     * @access     public
     * @param      string  $alias        登録名
     * @param      string  $name         Filter名
     * @param      array   $attributes   属性
     */
    public function add($alias, $name, array $attributes=array())
    {
        //Filterのクラス名チェック
        if(!preg_match('/^[\w_]+$/', $name)){
            Samurai_Logger::error('Invalid Filter\'s name. -> %s', $name);
        }
        //Filterの検索
        list($class, $file) = $this->makeNames($name);
        if(!Samurai_Loader::load($file)){
            throw new Samurai_Exception('Filter unknown. -> '.$file);
        }
        //既に同名のFilterが追加されている場合は無視
        if($this->hasFilter($alias)){
            Samurai_Logger::info('This Filter is already exists. -> %s', $alias);
            return true;
        }
        //生成
        else {
            $Filter = new $class();
            if(!is_object($Filter)){
                Samurai_Logger::fatal('Filter create failed... -> %s', $file);
            }
            $Filter->setAttributes($attributes);
            //追加
            $this->_filters[$alias] = $Filter;
            $this->_position[] = $alias;
            return true;
        }
    }
    
    
    /**
     * Filterクラス名および、ファイルパスの取得。
     * @access    public
     * @param     string  $name    Filter名
     * @return    array   FIlterのクラス名とファイルパス
     */
    public function makeNames($name)
    {
        $name = Samurai_Config::get('filter.prefix') . '_' . $name;
        $name = join('_', array_map('ucfirst', explode('_', $name)));
        $path = Samurai_Config::get('directory.component') . DS . Samurai_Loader::getPathByClass($name);
        return array($name, $path);
    }
    
    
    
    
    
    /**
     * FilterChainの長さを返却。
     * @access    public
     * @return    int   FilterChainの長さ
     */
    public function getSize()
    {
        return count($this->_filters);
    }
    
    
    /**
     * 現在のFilter名を返却。
     * @access    public
     * @return    string   現在のFilterの名前
     */
    public function getCurrentFilterName()
    {
        if(isset($this->_position[$this->_index])){
            return $this->_position[$this->_index];
        }
    }
    
    
    /**
     * 指定された名前のFilterを返却。
     * @access     public
     * @param      string  $name    登録名
     * @return     object  指定されたFilterインスタンス
     */
    public function getFilterByName($name)
    {
        if($this->hasFilter($name)){
            return $this->_filters[$name];
        }
    }
    
    
    
    
    
    /**
     * 既にフィルターがあるかチェック。
     * @access     public
     * @param      string  $alias   登録名
     * @return     boolean 既に登録されているか
     */
    public function hasFilter($alias)
    {
        return isset($this->_filters[$alias]) && is_object($this->_filters[$alias]);
    }
    
    
    /**
     * 次のフィルタがあるかどうか。
     * @access     public
     * @return     boolean 次のフィルタがあるかどうか
     */
    public function hasNext()
    {
        return $this->_index < $this->getSize();
    }
    
    
    /**
     * FilterChainを次へ進める。
     * @access     public
     */
    public function next()
    {
        $this->_index++;
    }
    
    
    /**
     * FilterChainをクリア。
     * @access    public
     */
    public function clear()
    {
        $this->_filters = array();
        $this->_position = array();
        $this->_index = 0;
    }
}
