<?
/**
 * ActiveGateway_Driver_Sqlite2
 * 
 * ActiveGatewayのSqlite2用ドライバー。
 * 
 * @package    ActiveGateway
 * @subpackage Driver
 * @copyright  Befool,Inc.
 * @author     Satoshi Kiuchi <satoshi.kiuchi@befool.co.jp>
 * @license    http://opensource.org/licenses/bsd-license.php  The modified BSD License
 */
class ActiveGateway_Driver_Sqlite2 extends ActiveGateway_Driver
{
    /**
     * 接続。
     * @access     protected
     * @param      array   $dsn_info     DSN設定情報
     * @return     boolean PDOコネクション
     */
    protected function _connect($dsn_info)
    {
        $scheme = $dsn_info['scheme'];
        $path   = $dsn_info['path'];
        try {
            $connection = new PDO("$scheme:$path", '', '');
        } catch(PDOException $Exception){
            trigger_error(sprintf('[ActiveGateway_Driver_Sqlite2]:Connection failed... -> [%s]', $Exception->getMessage()), E_USER_ERROR);
        }
        return $connection;
    }
    
    
    
    
    
    /**
     * @override
     */
    public function modifyLimitQuery($sql, $offset=NULL, $limit=NULL)
    {
        if($offset!==NULL && $limit!==NULL){
            $sql = sprintf('%s LIMIT %d, %d', $sql, $offset, $limit);
        } elseif($limit!==NULL){
            $sql = sprintf('%s LIMIT %d', $sql, $limit);
        }
        return $sql;
    }
    /**
     * @override
     */
    public function modifyInsertQuery($table_name, $attributes, &$params=array())
    {
        $i = 1;
        $field_list = array();
        $value_list = array();
        foreach($attributes as $_key => $_val){
            $field_list[] = $_key;
            $value_list[] = '?';
            $params[$i] = $_val;
            $i++;
        }
        return sprintf('INSERT INTO %s ( %s ) VALUES ( %s )', $table_name, join(', ', $field_list), join(', ', $value_list));
    }
    /**
     * @override
     */
    public function modifyUpdateQuery($table_name, $sets, $wheres=array(), $orders=array())
    {
        $sql  = sprintf('UPDATE %s SET %s', $table_name, join(', ', $sets));
        $sql .= ($wheres) ? sprintf(' WHERE %s', join(' AND ', $wheres)) : '' ;
        return $sql;
    }
    /**
     * @override
     */
    public function modifyDeleteQuery($table_name, $wheres=array(), $orders=array())
    {
        $sql  = sprintf('DELETE FROM %s', $table_name);
        $sql .= ($wheres) ? sprintf(' WHERE %s', join(' AND ', $wheres)) : '' ;
        return $sql;
    }
    /**
     * @override
     */
    public function modifyUpdateLimitQuery($sql, $limit=NULL)
    {
        //this database can't use limit update!
        return $sql;
    }
    /**
     * @override
     */
    public function modifyFoundRowsQuery($sql)
    {
        //this database can't use SELECT FOUND_ROWS()!!
        return $sql;
    }
    /**
     * @override
     */
    public function modifyAttributes($table_info, &$attributes=array())
    {
        foreach($table_info as $_key => $_val){
            if($_val['default']===NULL && !$_val['null'] && !isset($attributes[$_key])){
                switch(strtoupper($_val['type'])){
                    case 'TINYINT':
                    case 'SMALLINT':
                    case 'MEDIUMINT':
                    case 'INT':
                    case 'INTENGER':
                    case 'BIGINT':
                    case 'FLOAT':
                    case 'DOUBLE':
                    case 'DECIMAL':
                        $attributes[$_key] = 0; break;
                    case 'CHAR':
                    case 'VARCHAR':
                    case 'TINYBLOB':
                    case 'TINYTEXT':
                    case 'BLOB':
                    case 'TEXT':
                    case 'MEDIUMBLOB':
                    case 'MEDIUMTEXT':
                    case 'LONGBLOB':
                    case 'LONGTEXT':
                        $attributes[$_key] = ''; break;
                    case 'DATE':
                        $attributes[$_key] = date('Y-m-d', 0); break;
                    case 'DATETIME':
                    case 'TIMESTAMP':
                        $attributes[$_key] = date('Y-m-d H:i:s', 0); break;
                    case 'TIME':
                        $attributes[$_key] = date('H:i:s', 0); break;
                    case 'YEAR':
                        $attributes[$_key] = date('Y', 0); break;
                }
            }
        }
    }
    
    
    
    
    
    /**
     * @override
     */
    public function getTotalRows($sql, $params=array())
    {
        $stmt = $this->query($sql, $params);
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        return is_array($rows) ? count($rows) : 0 ;
    }
    
    
    
    
    
    /**
     * テーブル情報の取得。
     * @access     public
     * @since      1.0.0
     * @param      string  $table_name   テーブル名
     * @return     array   テーブル情報
     */
    public function getTableInfo($table_name)
    {
        $result = array();
        
        if($table_name===NULL || $table_name=='' || !is_string($table_name)){
            return $result;
        }
        
        $sql = "PRAGMA table_info('$table_name')";
        $stmt = $this->_connection->query($sql);
        
        $table_info = $stmt->fetchAll(PDO::FETCH_ASSOC);
        foreach($table_info as $_key => $info){
            //カラム名の取得
            $name = $info['name'];
            
            //フィールド型の取得
            if(preg_match('/^(.+)\((.+)\)$/', $info['type'], $matches)){
                $type   = $matches[1];
                $length = $matches[2];
            } else {
                $type   = $info['type'];
                $length = 0;
            }
            
            //NULL値の判断
            $null = ($info['notnull']) ? false : true ;
            
            //キーの取得
            $primary_key = ($info['pk']) ? true : false ;
            
            //デフォルト値の取得
            $default = $info['dflt_value'];
            
            //その他フラグの取得
            $extras = array();
            $extras = join(' ', $extras);
            
            //値の生成
            $result[$name] = array(
                'table'       => $table_name,
                'name'        => $name,
                'type'        => $type,
                'length'      => $length,
                'null'        => $null,
                'primary_key' => $primary_key,
                'default'     => $default,
                'extras'      => $extras,
            );
        }
        
        return $result;
    }
}
