<?
/**
 * ActiveGatewayCondition
 * 
 * ActiveGatewayにおいて、検索条件を保持するクラス。
 * ActiveGatewayのfindAll系のメソッドがこれを使用する。
 * 
 * <code>
 *     $condition = ActiveGateway::getCondition();
 *     $condition->where->foo = 'bar';
 *     $condition->where->foo = $condition->isNotEqual('bar');
 *     $condition->where->bar = $condition->isGreaterThan(10, true);
 *     $condition->where->bar = $condition->isLessThan(10, false);
 * </code>
 * 
 * @package    ActiveGateway
 * @copyright  Befool,Inc.
 * @author     Satoshi Kiuchi <satoshi.kiuchi@befool.co.jp>
 * @license    http://opensource.org/licenses/bsd-license.php  The modified BSD License
 */
class ActiveGatewayCondition {
    public
        /** @var        mixed   SELECT */
        $select = NULL,
        /** @var        mixed   FROM */
        $from   = NULL,
        /** @var        mixed   WHERE */
        $where  = NULL,
        /** @var        string  WHERE(addtional) */
        $addtional_where = '',
        /** @var        mixed   ORDER */
        $order  = NULL,
        /** @var        mixed   GROUP */
        $group  = NULL,
        /** @var        int     LIMIT */
        $limit  = NULL,
        /** @var        int     OFFSET */
        $offset = NULL;
    public
        /** @var        boolean トータルローズを取得するかどうか */
        $total_rows    = true,
        /** @var        boolean 「active」フィールドを考慮するかどうか */
        $regard_active = true;
    
    
    /**
     * リミットをセット。
     * @access     public
     * @param      int     $limit   リミット
     */
    public function setLimit($limit)
    {
        $this->limit = (int)$limit;
    }
    
    
    /**
     * offset値の設定。
     * offsetというか、pageIDをセットする感じか？
     * 性質上、リミットをセットした後でないと意味を発揮しない。
     * pageID=falseの場合、裸の値がoffset値となる。
     * @access     public
     */
    public function setOffset($offset, $pageID=true)
    {
        $offset = (int)$offset;
        if($pageID){
            $offset = ($offset > 0) ? $offset - 1 : 0 ;
            $offset = (int)$this->limit * $offset;
        }
        if($offset < 0) $offset = 0;
        $this->offset = $offset;
    }
    
    
    
    
    
    /**
     * 検索の条件系
     * @access     public
     */
    public function isEqual($value)
    {
        $obj = new ActiveGatewayCondition_Value($value);
        if($value === NULL) $obj->override = 'IS NULL';
        return $obj;
    }
    public function isNotEqual($value)
    {
        $obj = new ActiveGatewayCondition_Value($value);
        $obj->operator = '!=';
        if($value === NULL) $obj->override = 'IS NOT NULL';
        return $obj;
    }
    public function isGreaterThan($value, $within=true)
    {
        $obj = new ActiveGatewayCondition_Value($value);
        $obj->operator = '>';
        if($within) $obj->operator = '>=';
        return $obj;
    }
    public function isLessThan($value, $within=true)
    {
        $obj = new ActiveGatewayCondition_Value($value);
        $obj->operator = '<';
        if($within) $obj->operator = '<=';
        return $obj;
    }
    public function isLike($value)
    {
        $obj = new ActiveGatewayCondition_Value($value);
        $obj->operator = 'LIKE';
        return $obj;
    }
}



/**
 * ActiveGatewayCondition_Value
 * 
 * ActiveGatewayの条件の値を体現するクラス
 * 
 * @package    ActiveGateway
 * @copyright  BEFOOL,Inc.
 * @author     Satoshi Kiuchi <satoshi.kiuchi@befool.co.jp>
 */
class ActiveGatewayCondition_Value
{
    public
        $value = NULL,
        $operator = '=',
        $override = '';
    
    
    public function __construct($value = NULL)
    {
        $this->value = $value;
    }
}
