<?
/**
 * ActiveGateway_Driver.abstract
 * 
 * ActiveGateway各種ドライバーの抽象クラス。
 * 各種ドライバーは必ず継承すること。
 * 
 * @package    ActiveGateway
 * @subpackage Driver
 * @copyright  Befool, Inc
 * @author     Satoshi Kiuchi <satoshi.kiuchi@befool.co.jp>
 */
abstract class ActiveGateway_Driver
{
    protected
        /** @var        resource コネクション */
        $connection,
        /** @var        resource コネクションマスタ */
        $connection_master;
    
    
    /**
     * コンストラクタ。
     * @access     public
     */
    public function __construct()
    {
        
    }
    
    
    
    
    
    /**
     * 接続。
     * @access     public
     * @param      string  $dsn         DSN文字列
     * @param      boolean $is_master   マスターかどうか
     */
    public function connect($dsn, $is_master=false)
    {
        $connection = $this->_connect(parse_url($dsn));
        if($is_master){
            $this->connection_master = $connection;
        } else {
            $this->connection = $connection;
        }
        return $connection;
    }
    
    
    /**
     * 各種ドライバー用コネクト。
     * @param      array    $dsn_info   分解されたDSN情報
     * @return     resource コネクション
     */
    abstract protected function _connect(array $dsn_info);
    
    
    
    
    
    /**
     * クエリー。
     * もっとも下位のSQL文の実行メソッド。
     * 結局すべてのクエリーメソッドは、最終的にここを通る。
     * @access     public
     * @param      string  $sql      SQL文
     * @param      array   $params   ブレースフォルダ
     * @return     object  ステートメントインスタンス
     */
    public function query($sql, $params=array())
    {
        //ステートメントの生成
        if($this->_isUpdateQuery($sql)){
            $stmt = $this->connection_master->prepare($sql);
        } else {
            $stmt = $this->connection->prepare($sql);
        }
        //ブレースフォルダの割当て
        foreach($params as $_key => $_val){
            //データタイプのディフォルトは文字列
            $param_type = PDO::PARAM_STR;
            if(is_null($_val)){
                $param_type = PDO::PARAM_NULL;
            } elseif(is_int($_val)){
                $param_type = PDO::PARAM_INT;
            } elseif(is_bool($_val)){
                $param_type = PDO::PARAM_BOOL;
            } elseif(is_resource($_val)){
                $param_type = PDO::PARAM_LOB;
            } elseif(strlen($_val)>=5120){
                $param_type = PDO::PARAM_LOB;
            }
            $stmt->bindValue($_key, $_val, $param_type);
        }
        //実行
        $execute_start = microtime(true);
        $stmt->execute();
        $execute_end   = microtime(true);
        ActiveGatewayManager::singleton()->poolQuery($stmt->queryString, $execute_end-$execute_start);
        //エラーチェック
        $this->_checkError($stmt, $params);
        return $stmt;
    }
    
    
    /**
     * リミットクエリー。
     * SQL文中にリミットを記述してもかまわないが、リミットに関しては、各DBで文法が違うため、
     * その差異を吸収するメソッドとして、このメソッドは存在する。
     * ディフォルトでうまくうごかないDBは、それぞれのドライバーに専用のメソッドを記述してオーバーライドすること。
     * @access     public
     * @param      string  $sql      SQL文
     * @param      array   $params   ブレースフォルダ
     * @param      int     $offset   開始位置
     * @param      int     $limit    取得数
     * @return     object  ステートメントインスタンス
     */
    public function limitQuery($sql, $params=array(), $offset=NULL, $limit=NULL)
    {
        if($this->_isUpdateQuery($sql)){
            $sql = $this->modifyUpdateLimitQuery($sql, $limit);
        } else {
            $sql = $this->modifyLimitQuery($sql, $offset, $limit);
        }
        return $this->query($sql, $params);
    }
    
    
    
    
    
    /**
     * リミットクエリーの整形。
     * @access     public
     * @param      string  $sql      SQL文
     * @param      int     $offset   開始位置
     * @param      int     $limit    作用制限
     * @return     string  SQL文
     */
    abstract public function modifyLimitQuery($sql, $offset=NULL, $limit=NULL);
    /**
     * インサートクエリーの生成。
     * @access     public
     * @param      string  $table_name   テーブル名
     * @param      array   $attributes   各種値
     * @param      array   &$params       ブレースフォルダ格納用
     * @return     string  SQL文
     */
    abstract public function modifyInsertQuery($table_name, $attributes, &$params=array());
    /**
     * 更新クエリーの生成。
     * @access     public
     * @param      string  $table_name   テーブル名
     * @param      array   $sets         更新値
     * @param      array   $wheres       条件値
     * @param      array   $orders       並び順
     * @return     string  SQL文
     */
    abstract public function modifyUpdateQuery($table_name, $sets, $wheres=array(), $orders=array());
    /**
     * 削除クエリーの生成。
     * @access     public
     * @param      string  $table_name   テーブル名
     * @param      array   $wheres       条件値
     * @param      array   $orders       並び順
     * @return     string  SQL文
     */
    abstract public function modifyDeleteQuery($table_name, $wheres=array(), $orders=array());
    /**
     * 更新制限クエリーの整形。
     * @access     public
     * @param      string  $sql      SQL文
     * @param      int     $offset   開始位置
     * @param      int     $limit    作用制限
     * @return     string  SQL文
     */
    abstract public function modifyUpdateLimitQuery($sql, $limit=NULL);
    
    
    /**
     * 総レコード取得用クエリー整形。
     * @access     public
     * @since      1.0.0
     * @param      string  $sql
     * @return     string  SQL文
     */
    abstract public function modifyFoundRowsQuery($sql);
    
    /**
     * インサート時に内容を調節する。
     * @access     public
     */
    public function modifyAttributes($table_info, &$attributes=array()){
        
    }
    
    
    
    
    
    
    /**
     * 直前のクエリーの総レコード数の取得。
     * @access     public
     * @param      string  $sql      SQL文
     * @param      array   $params   ブレースフォルダ
     * @return     int     直前のクエリーの総レコード数
     */
    abstract public function getTotalRows($sql, $params=array());
    
    
    
    
    
    /**
     * 更新文かどうかの判断。
     * @access     protected
     * @param      string  $sql   SQL文
     * @return     boolean 更新文かどうか
     */
    protected function _isUpdateQuery($sql)
    {
        $sql = trim($sql);
        return preg_match("/^(UPDATE|INSERT)/i", $sql);
    }
    
    
    
    
    
    /**
     * クエリー実行後のエラーチェック。
     * @access     private
     */
    protected function _checkError($stmt, $params)
    {
        @list($code, $driver_code, $message) = $stmt->errorInfo();
        if($code!="00000"){
            throw(new Exception("ActiveGateway(PDO) Error[{$code}][{$driver_code}]: {$message}"));
        }
    }
}
