<?
/**
 * ActiveGateway
 * 
 * ActiveGateway本体。
 * 
 * @package    ActiveGateway
 * @subpackage ActiveGateway
 * @copyright  Befool, Inc
 * @author     Satoshi Kiuchi <satoshi.kiuchi@befool.co.jp>
 */
class ActiveGateway
{
    private
        /** @var        string   DSN(スレーブ) */
        $_dsn_slave = "",
        /** @var        string   DSN(マスター) */
        $_dsn_master = "",
        /** @var        array    設定情報 */
        $_config = array(),
        /** @var        string   設定ファイル */
        $_config_file = "",
        /** @var        int      フェッチモード */
        $_fetch_mode = self::FETCH_OBJ,
        /** @var        array   テーブル情報 */
        $_table_info = array(),
        /** @var        object   Driver */
        $Driver,
        /** @var        resource コネクション(スレーブ) */
        $connection,
        /** @var        resource コネクション(マスター) */
        $connection_master;
    const
        FETCH_LAZY = PDO::FETCH_LAZY,
        FETCH_ASSOC = PDO::FETCH_ASSOC,
        FETCH_OBJ = PDO::FETCH_OBJ,
        FETCH_BOTH = PDO::FETCH_BOTH;
    
    
    /**
     * コンストラクタ。
     * @access     public
     */
    public function __construct()
    {
        
    }
    
    
    
    
    
    
    /**
     * コネクト。
     * @access     public
     * @return     boolean 接続の可否
     */
    public function connect()
    {
        //接続済み
        if($this->hasConnection()){
            return true;
        }
        //DSNなし
        if(!$this->hasDsn()){
            trigger_error("[ActiveGateway]:DSN is Not Found.", E_USER_ERROR);
        }
        //新規接続
        $this->connection = $this->Driver->connect($this->_dsn_slave);
        $this->connection_master = $this->Driver->connect($this->_dsn_master, true);
        return true;
    }
    
    
    
    
    
    /**
     * findシリーズ、ID検索。
     * ActiveGatewayテーブル規約に沿っていれば、プライマリキーで検索可能。
     * @access     public
     * @param      string  $alias   仮想テーブル名
     * @param      int     $id      ID
     * @return     object  ActiveGatewayレコードインスタンス
     */
    public function find($alias, $id)
    {
        $Record = $this->_buildRecord($alias);
        $primary_key = $Record->getPrimaryKey();
        
        $AGdto = ActiveGateway::getCondition();
        $AGdto->where->$primary_key = $id;
        return $this->findDetail($alias, $AGdto);
    }
    
    
    /**
     * findシリーズ、指定検索。
     * その実際はfindDetailを実行後、最初のレコードを返却している。
     * @access     public
     * @param      string  $alias    仮想テーブル名
     * @param      string  $column   カラム名
     * @param      mixed   $value    検索条件(配列も可)
     * @param      object  $AGDto    条件Dto
     * @return     object  ActiveGatewayレコードインスタンス
     */
    public function findBy($alias, $column, $value, $AGdto=NULL)
    {
        if($AGdto===NULL) $AGdto = ActiveGateway::getCondition();
        $AGdto->where->$column = $value;
        return $this->findDetail($alias, $AGdto);
    }
    
    
    /**
     * findシリーズ、詳細検索。
     * その実際はfindAllDeatiを実行後、最初のレコードを返却している。
     * @access     public
     * @param      string  $alias        仮想テーブル名
     * @param      array   $conditions   条件
     * @param      string  $order        並び順
     * @return     object  ActiveGatewayレコードインスタンス
     */
    public function findDetail($alias, ActiveGatewayCondition $AGdto)
    {
        $AGdto->total_rows = false;
        $AGdto->setLimit(1);
        $ActiveGatewayRecords = $this->findAllDetail($alias, $AGdto);
        return $ActiveGatewayRecords->getFirstRecord();
    }
    
    
    /**
     * findシリーズ、SQL検索。
     * その実際はfindAllSqlを実行後、最初のレコードを返却している。
     * @access     public
     * @param      string  $alias    仮想テーブル名
     * @param      string  $sql      SQL文
     * @param      array   $params   ブレースフォルダ
     * @return     object  ActiveGatewayレコードインスタンス
     */
    public function findSql($alias, $sql, $params=array())
    {
        $ActiveGatewayRecords = $this->findAllSql($alias, $sql, $params, 1, NULL, false);
        return $ActiveGatewayRecords->getFirstRecord();
    }
    
    
    /**
     * findAllシリーズ。
     * 内実、findAllDetailのシノニム。
     * @access     public
     * @param      string  $alias    仮想テーブル名
     * @param      object  $AGdto    条件Dto
     * @return     object  ActiveGatewayRecordsインスタンス
     */
    public function findAll($alias, ActiveGatewayCondition $AGdto)
    {
        return $this->findAllDetail($alias, $AGdto);
    }
    
    
    /**
     * findAllシリーズ、指定検索。
     * その実際はfindAllDetailを実行、返却している。
     * @access     public
     * @param      string  $alias    仮想テーブル名
     * @param      string  $column   カラム名
     * @param      mixed   $value    検索条件(配列可)
     * @param      object  $AGdto    条件dto
     * @return     object  ActiveGatewayRecordsインスタンス
     */
    public function findAllBy($alias, $column, $value, $AGdto=NULL)
    {
        if($AGdto===NULL) $AGdto = ActiveGateway::getCondition();
        $AGdto->where->$column = $value;
        return $this->findAllDetail($alias, $AGdto);
    }
    
    
    /**
     * findAllシリーズ、詳細検索。
     * その実際は、fondAllSqlを使用している。
     * @access     public
     * @param      string  $alias   仮想テーブル
     * @param      object  $AGdto   条件dto
     * @return     object  ActiveGatewayRecordsインスタンス
     */
    public function findAllDetail($alias, ActiveGatewayCondition $AGdto)
    {
        //初期化
        $Record = $this->_buildRecord($alias);
        if($AGdto->select === NULL) $AGdto->select = "*";
        if($AGdto->from === NULL)   $AGdto->from   = $Record->getTableName();
        //自動付加
        $table_info = $this->getTableInfo($alias, $Record->getTableName());
        if(isset($table_info['active']) && $AGdto->regard_active && !isset($AGdto->where->active)){
            $AGdto->where->active = "1";
        }
        
        //セレクト節の生成
        $select = (array)$AGdto->select;
        //フロム節の調節
        $from = (array)$AGdto->from;
        //条件節の生成
        $params = array();
        $wheres = $this->_chain_where($AGdto->where, $params);
        //グループ節の生成
        $groups = (array)$AGdto->group;
        //オーダー節の生成
        $orders = $this->_chain_order($AGdto->order);
        
        //SQL文の生成
        $sql  = sprintf("SELECT %s FROM %s", join(", ", $select), join(", ", $from));
        $sql .= ($wheres) ? sprintf(" WHERE %s", join(" AND ", $wheres)) : "" ;
        $sql .= ($groups) ? sprintf(" GROUP BY %s", join(", ", $groups)) : "" ;
        $sql .= ($orders) ? sprintf(" ORDER BY %s", join(", ", $orders)) : "" ;
        //SQLから検索
        $result = $this->findAllSql($alias, $sql, $params, $AGdto->limit, $AGdto->offset, $AGdto->total_rows);
        return $result;
    }
    
    
    /**
     * findAllシリーズ、SQL検索。
     * 一番下位の検索メソッド。結局のところ、すべての検索メソッドは、最終的にこいつが実行されている。
     * @access     public
     * @param      string  $alias        仮想テーブル名
     * @param      string  $sql          SQL文
     * @param      array   $params       ブレースフォルダ
     * @param      int     $limit        取得数
     * @param      int     $offset       開始位置
     * @param      boolean $total_rows   総レコード数を取得するかどうか
     * @return     object  ActiveGatewayRecordsインスタンス
     */
    public function findAllSql($alias, $sql, $params=array(), $limit=NULL, $offset=NULL, $total_rows=true)
    {
        $ActiveGatewayRecords = new ActiveGatewayRecords();
        if($total_rows) $sql = $this->Driver->modifyFoundRowsQuery($sql);
        $res = $this->executeQuery($sql, $params, $limit, $offset);
        
        while($row = $res->fetch($this->_fetch_mode)){
            $Record = $this->_buildRecord($alias, $row, false);
            $ActiveGatewayRecords->addRecord($Record);
        }
        
        $total_rows = ($total_rows) ? $this->Driver->getTotalRows($sql, $params) : 0 ;
        $ActiveGatewayRecords->setTotalRows($total_rows);
        return $ActiveGatewayRecords;
    }
    
    
    
    
    
    /**
     * レコードの挿入を行う。
     * @access     public
     * @param      string  $alias        仮想テーブル名
     * @param      array   $attributes   各種値
     * @return     boolean 挿入処理結果
     */
    public function insert($alias, $attributes=array())
    {
        $Record = $this->_buildRecord($alias, $attributes, true);
        $table_name = $Record->getTableName();
        //各種情報の付加
        $table_info = $this->getTableInfo($alias, $table_name);
        if(isset($table_info['created_at']) && !isset($attributes['created_at'])){
            $attributes['created_at'] = time();
        }
        if(isset($table_info['updated_at']) && !isset($attributes['updated_at'])){
            $attributes['updated_at'] = time();
        }
        if(isset($table_info['active']) && !isset($attributes['active'])){
            $attributes['active'] = "1";
        }
        //ディフォルト値調節
        $this->Driver->modifyAttributes($table_info, $attributes);
        //インサート
        $sql = $this->Driver->modifyInsertQuery($table_name, $attributes, $params);
        $stmt = $this->executeUpdate($sql, $params);
        $attributes[$Record->getPrimaryKey()] = $this->connection_master->lastInsertId();
        $Record = $this->_buildRecord($alias, $attributes, false);
        return $Record;
    }
    
    
    
    
    
    /**
     * updateシリーズ、レコードインスタンスの一つの情報を更新する。
     * 実際はupdateを実行する。
     * @access     public
     * @param      object  $Record   レコードインスタンス
     * @param      string  $column   カラム名
     * @param      mixed   $value    値
     * @return     boolean 更新処理結果
     */
    public function updateAttribute($Record, $column, $value)
    {
        $Record->$column = $value;
        $this->save($Record);
    }
    
    
    /**
     * updateシリーズ、レコードインスタンスの複数の情報を更新する。
     * 実際はupdateを実行する。
     * @access     public
     * @param      object  $Record       レコードインスタンス
     * @param      array   $attributes   設定値
     * @return     boolean 更新処理結果
     */
    public function updateAttributes($Record, $attributes=array())
    {
        foreach($attributes as $_key => $_val){
            if(!preg_match("/^_/")){
                $Record->$_key = $_val;
            }
        }
        $this->save($Record);
    }
    
    
    /**
     * updateシリーズ、ID更新。
     * その実際はupdateDetailを使用している。
     * @access     public
     * @param      string  $alias        仮想テーブル名
     * @param      int     $id           ID
     * @param      array   $attributes   設定値
     * @return     boolean 更新処理結果
     */
    public function update($alias, $id, $attributes=array())
    {
        $Record = $this->_buildRecord($alias);
        $primary_key = $Record->getPrimaryKey();
        
        $AGdto = ActiveGateway::getCondition();
        $AGdto->where->$primary_key = $id;
        return $this->updateDetail($alias, $attributes, $AGdto);
    }
    
    
    /**
     * updateシリーズ、指定更新。
     * その実際はupdateDetailを使用している。
     * @access     public
     * @param      string  $alias        仮想テーブル名
     * @param      string  $column       カラム名
     * @param      string  $value        条件
     * @param      array   $attributes   設定値
     * @param      mixed   $order        並び順
     * @return     boolean 更新処理結果
     */
    public function updateBy($alias, $column, $value, $attributes=array(), $AGdto=NULL)
    {
        if($AGdto===NULL) $AGdto = ActiveGateway::getCondition();
        $AGdto->where->$column = $value;
        return $this->updateDetail($alias, $attributes, $AGdto);
    }
    
    
    /**
     * updateシリーズ、詳細更新。
     * その実際はupdateAllDetailを使用している。
     * @access     public
     * @param      string  $alias        仮想テーブル名
     * @param      array   $conditions   条件値
     * @param      array   $attributes   設定値
     * @param      mixed   $order        並び順
     * @return     boolean 更新処理結果
     */
    public function updateDetail($alias, $attributes, ActiveGatewayCondition $AGdto)
    {
        $AGdto->setLimit(1);
        return $this->updateAllDetail($alias, $attributes, $AGdto);
    }
    
    
    /**
     * updateシリーズ、SQL更新。
     * その実際はupdateAllSqlを使用している。
     * @access     public
     * @since      1.0.0
     * @param      string  $sql      SQL文
     * @param      array   $params   ブレースフォルダ
     * @return     boolean 更新処理結果(成功⇒true/失敗⇒false)
     */
    public function updateSql($sql, $params){
        return $this->updateAllSql($sql, $params, 1);
    }
    /**
     * updateAllシリーズ、ID更新。
     * その実際はupdateAllDetailを使用している。
     * @access     public
     * @param      string  $alias        仮想テーブル名
     * @param      int     $id           ID
     * @param      array   $attributes   設定値
     * @return     boolean 更新処理結果
     */
    public function updateAll($alias, $id, $attributes=array())
    {
        $Record = $this->_buildRecord($alias);
        $primary_key = $Record->getPrimaryKey();
        
        $AGdto = ActiveGateway::getCondition();
        $AGdto->where->$primary_key = $id;
        return $this->updateAllDetail($alias, $attributes, $AGdto);
    }
    
    
    /**
     * updateAllシリーズ、指定更新。
     * その実際はupdateAllDetailを使用している。
     * @access     public
     * @param      string  $alias        仮想テーブル名
     * @param      string  $column       カラム名
     * @param      string  $value        条件
     * @param      array   $attributes   設定値
     * @param      mixed   $order        並び順
     * @param      int     $limit        作用制限
     * @return     boolean 更新処理結果
     */
    public function updateAllBy($alias, $column, $value, $attributes=array(), $AGdto=NULL)
    {
        if($AGdto===NULL) $AGdto = ActiveGateway::getCondition();
        $AGdto->where->$column = $value;
        return $this->updateAllDetail($alias, $attributes, $AGdto);
    }
    
    
    /**
     * updateAllシリーズ、詳細更新。
     * その実際は、updateAllSqlを使用している。
     * @access     public
     * @param      string  $alias        仮想テーブル名
     * @param      string  $attributes   設定値
     * @param      array   $conditions   条件値
     * @param      int     $limit        作用制限
     * @return     boolean 更新処理結果
     */
    public function updateAllDetail($alias, $attributes, ActiveGatewayCondition $AGdto)
    {
        //初期化
        $params = array();
        $Record = $this->_buildRecord($alias);
        //自動付加
        $attributes = (array)$attributes;
        if($Record->hasField("updated_at")){
            $attributes['updated_at'] = time();
        }
        //設定節の生成
        $sets = array();
        foreach($attributes as $_key => $_val){
            $place_holder = ":set_$_key";
            $params[$place_holder] = $_val;
            $sets[] = sprintf("`%s` = %s", $_key, $place_holder);
        }
        if(!$sets){
            trigger_error("[ActiveGateway]:No attributes update is very danger!!", E_USER_ERROR);
        }
        //条件節の生成
        $wheres = $this->_chain_where($AGdto->where, $params, "where_");
        if(!$wheres){
            trigger_error("[ActiveGateway]:No where update is very danger!!", E_USER_ERROR);
        }
        //オーダー節の生成
        $orders = $this->_chain_order($AGdto->order);
        //SQLから更新
        $sql = $this->Driver->modifyUpdateQuery($Record->getTableName(), $sets, $wheres, $orders);
        return $this->updateAllSql($sql, $params, $AGdto->limit);
    }
    
    
    /**
     * updateAllシリーズ、SQL更新。
     * updateシリーズの最下位メソッド。結局すべてのupdateシリーズはこのメソッドが実行されている。
     * @access     public
     * @param      string  $sql      SQL文
     * @param      array   $params   ブレースフォルダ
     * @param      int     $limit    更新数
     */
    public function updateAllSql($sql, $params=array(), $limit=NULL)
    {
        $stmt = $this->executeUpdate($sql, $params, $limit);
        return true;
    }
    
    
    
    
    
    /**
     * プライマリキーにおいて削除を実行する。
     * @access     public
     * @param      string  $alias   仮想テーブル名
     * @param      int     $id      ID
     */
    public function delete($alias, $id)
    {
        $Record = $this->_buildRecord($alias);
        $primary_key = $Record->getPrimaryKey();
        
        $AGdto = ActiveGateway::getCondition();
        $AGdto->where->$primary_key = $id;
        return $this->deleteDetail($alias, $AGdto);
    }
    
    
    /**
     * 詳細削除。
     * @access     public
     * @param      string  $alias   仮想テーブル名
     * @param      array   $AGdto   条件
     */
    public function deleteDetail($alias, ActiveGatewayCondition $AGdto)
    {
        $AGdto->setLimit(1);
        return $this->deleteAllDetail($alias, $AGdto);
    }
    
    
    /**
     * 詳細全削除。
     * @access     public
     * @param      string  $alias   仮想テーブル名
     * @param      array   $AGdto   条件
     */
    public function deleteAllDetail($alias, ActiveGatewayCondition $AGdto)
    {
        $Record = $this->_buildRecord($alias);
        //論理消去できるのであれば論理消去(こちらが望ましい)
        if($Record->enableDeleteByLogical()){
            $attributes['active'] = "0";
            if($Record->hasField("deleted_at")){
                $attributes['deleted_at'] = time();
            }
            return $this->updateAllDetail($alias, $attributes, $AGdto);
        }
        //物理消去
        else {
            //条件節の生成
            $params = array();
            $wheres = $this->_chain_where($AGdto->where, $params);
            if(!$wheres){
                trigger_error("[ActiveGateway]:No where delete is very danger!!", E_USER_ERROR);
            }
            //オーダー節の生成
            $orders = $this->_chain_order($AGdto->order);
            //SQLから更新
            $sql = $this->Driver->modifyDeleteQuery($Record->getTableName(), $wheres, $orders);
            return $this->updateAllSql($sql, $params, $AGdto->limit);
        }
    }
    
    
    
    
    
    
    /**
     * 新しいレコードインスタンスの生成。
     * @access     public
     * @param      string  $alias        仮想テーブル名
     * @param      array   $attributes   初期パラメータ
     * @return     object  レコードインスタンス
     */
    public function build($alias, $attributes=array())
    {
        $Record = $this->_buildRecord($alias, $attributes, true);
        return $Record;
    }
    
    
    /**
     * レコードインスタンスの生成。
     * @access     private
     * @param      string  $alias        仮想テーブル名
     * @param      mixed   $row          PDOStatement->fetchの取得結果
     * @param      boolean $new_record   新規レコードかどうかの判断値
     * @return     object  レコードインスタンス
     */
    private function _buildRecord($alias, $row=NULL, $new_record=true)
    {
        $Record = new ActiveGatewayRecord($row, $new_record, $alias);
        //設定情報の取得
        $config = array();
        if($alias !== NULL && isset($this->_config[$alias])){
            $config = $this->_config[$alias];
        }
        //テーブル名の書き換え
        if(isset($config['table_name'])){
            $Record->setTableName($config['table_name']);
        }
        //プライマリキーの書き換え
        if(isset($config['primary_key'])){
            $Record->setPrimaryKey($config['primary_key']);
        }
        //テーブル情報の取得
        $table_info = $this->getTableInfo($alias, $Record->getTableName());
        $Record->setTableInfo($table_info);
        //返却
        return $Record;
    }
    
    
    /**
     * テーブル情報の取得。
     * ドライバーの取得メソッドを使用し、取得する。
     * ドライバーの取得メソッドは、PEAR_DBのgetTableInfo()と同等であるべきである。
     * @access     public
     * @param      string  $alias        仮想テーブル名
     * @param      string  $table_name   対象となるテーブルの実名
     * @return     array   テーブル情報配列
     */
    public function getTableInfo($alias, $table_name)
    {
        //既に取得済みの場合
        if(isset($this->_table_info[$alias])){
            return $this->_table_info[$alias];
        }
        //情報の取得
        $this->connect();
        $attributes = $this->Driver->getTableInfo($table_name);
        //情報の代入
        $this->_table_info[$alias] = $attributes;
        return $this->_table_info[$alias];
    }
    
    
    
    
    
    /**
     * インスタンスを使ってデータを更新する。
     * @access     public
     * @param      object  $Record   ActiveGatewayRecord
     * @return     boolean 更新結果
     */
    public function save(ActiveGatewayRecord &$Record)
    {
        //チェック
        if(!$Record->isSavable()){
            trigger_error("[ActiveGateway]:This record can't save.", E_USER_ERROR);
        }
        //新規レコードの場合
        if($Record->isNewRecord()){
            $Record = $this->insert($Record->getAlias(), $Record->getAttributes());
            return true;
        //既存レコードの場合
        } else {
            return $this->update($Record->getAlias(), $Record->getOriginalValue("primary_key"), $Record->getAttributes());
        }
    }
    
    
    /**
     * 上記のbuildとsaveの一連の流れを一つのメソッドで完結させてしまう場合はコレ。
     * @access     public
     * @param      string  $alias        仮想テーブル名
     * @param      array   $attributes   初期パラメータ
     * @return     object  レコードインスタンス
     */
    public function create($alias, $attributes=array())
    {
        $Record = $this->build($alias, $attributes);
        $this->save($Record);
        return $Record;
    }
    
    
    /**
     * インスタンスを使用し、データを削除する。
     * @access     public
     * @param      object  $Record
     * @return     object  削除に成功したかどうか
     */
    public function destroy(ActiveGatewayRecord &$Record)
    {
        //新規レコードの場合
        if($Record->isNewRecord()){
            $Record = NULL;
            return true;
        //既存レコードの場合
        } else {
            return $this->delete($Record->getAlias(), $Record->getOriginalValue($Record->getPrimaryKey()));
        }
    }
    
    
    
    
    
    /**
     * 検索(SELECT)SQL文の実行。
     * @access     public
     * @param      string  $sql      SQL文
     * @param      array   $params   ブレースホルダ
     * @param      int     $limit    取得数
     * @param      int     $offset   開始位置
     * @return     object  ステートメントインスタンス
     */
    public function executeQuery($sql, $params=array(), $limit=NULL, $offset=NULL)
    {
        $this->connect();
        if($limit!==NULL || $offset!==NULL){
            $stmt = $this->Driver->limitQuery($sql, $params, $offset, $limit);
        } else {
            $stmt = $this->Driver->query($sql, $params);
        }
        
        return $stmt;
    }
    
    
    /**
     * 更新(INSERT|UPDATE|DELETE)SQL文の実行。
     * @access     public
     * @param      string  $sql      SQL文
     * @param      array   $params   ブレースフォルダ
     * @param      int     $limit    作用制限
     * @return     object  ステートメントインスタンス
     */
    public function executeUpdate($sql, $params=array(), $limit=NULL)
    {
        $this->connect();
        if($limit!==NULL){
            $stmt = $this->Driver->limitQuery($sql, $params, NULL, $limit);
        } else {
            $stmt = $this->Driver->query($sql, $params);
        }
        return $stmt;
    }
    
    
    
    
    
    /**
     * 凡庸クエリー。
     * すべての取得結果を持ってきたい場合。
     * PEAR_DB->getAll()と同等。
     * @param      string  $sql      SQL文
     * @param      array   $params   ブレースフォルダ
     * @param      int     $limit    取得数
     * @param      int     $offset   開始位置
     * @return     array   すべての取得結果
     */
    public function getAll($sql, $params=array(), $limit=NULL, $offset=NULL)
    {
        $stmt = $this->executeQuery($sql, $params, $limit, $offset);
        return $stmt->fetchAll($this->_fetch_mode);
    }
    
    
    /**
     * 凡庸クエリー。
     * 取得結果の一行文だけ引っ張ってきたい場合。
     * PEAR_DB->getRow()と同等。
     * @param      string  $sql      SQL文
     * @param      array   $params   ブレースフォルダ
     * @return     mixed   一行文の取得結果
     */
    public function getRow($sql, $params=array())
    {
        $stmt = $this->executeQuery($sql, $params, 1);
        $row  = $stmt->fetchAll($this->_fetch_mode);
        return $row;
    }
    
    
    /**
     * 凡庸クエリー。
     * 取得結果の縦一行文だけ引っ張ってくる。
     * PEAR_DB->getCol()と同等。
     * さらには、引数として取得したい部署の番号、またはカラム名を指定してあげることによって、
     * 特定の場所から取得することも可能である。
     * @param      string  $sql      SQL文
     * @param      array   $params   ブレースフォルダ
     * @param      mixed   $column   カラム名指定
     * @return     array   取得結果
     */
    public function getCol($sql, $params=array(), $column=NULL)
    {
        $stmt = $this->executeQuery($sql, $params);
        $rows = $stmt->fetchAll(ActiveGateway::FETCH_BOTH);
        $result = array();
        foreach($rows as $row){
            if($column!==NULL){
                if(isset($row[$column])){
                    $result[] = $row[$column];
                } else {
                    $result[] = $row[0];
                }
            } else {
                $result[] = $row[0];
            }
        }
        return $result;
    }
    
    
    /**
     * 凡庸クエリー。
     * 取得結果の一レコードの最初の値のみを引っ張ってきたいときに有用。
     * さらには、引数として取得したい部署の番号、またはカラム名を指定してあげることによって、
     * 特定の場所から取得することも可能である。
     * @param      string  $sql      SQL文
     * @param      array   $params   ブレースフォルダ
     * @param      mixed   $column   カラム名指定
     * @return     mixed   取得結果
     */
    public function getOne($sql, $params=array(), $column=NULL)
    {
        $stmt = $this->executeQuery($sql, $params, 1);
        $row = $stmt->fetch(ActiveGateway::FETCH_BOTH);
        if($column!==NULL){
            if(isset($row[$column])){
                return $row[$column];
            }
        }
        return $row[0];
    }
    
    
    
    
    
    /**
     * DSNの設定。
     * @access     public
     * @param      string  $dsn   DSN情報文字列
     */
    public function setDsn($dsn)
    {
        $this->_dsn_slave = $dsn;
        if(!$this->_dsn_master) $this->setDsnMaster($dsn);
    }
    public function setDsnMaster($dsn)
    {
        $this->_dsn_master = $dsn;
    }
    
    
    /**
     * 設定情報の取得。
     * @access     public
     * @param      string  $config_file   設定ファイル
     */
    public function import($config_file)
    {
        if($config_file){
            $this->_config = ActiveGatewayUtils::loadYaml($config_file);
            $this->_config_file = $config_file;
        }
    }
    
    
    /**
     * ドライバーの格納。
     * @access     public
     * @param      object  $Driver   ドライバー
     */
    public function setDriver($Driver)
    {
        $this->Driver = $Driver;
    }
    
    
    /**
     * ActiveGatewayの検索用DTOを返却する。
     * @access     public
     */
    public function getCondition(){
        $condition = new ActiveGatewayCondition();
        return $condition;
    }
    
    
    
    
    
    /**
     * コネクションを保持しているかどうか。
     * @access     public
     * @return     boolean コネクションを保持しているかどうか
     */
    public function hasConnection()
    {
        return is_resource($this->connection) && is_resource($this->connection_master);
    }
    
    
    /**
     * DSNを保持しているかどうか。
     * @return     boolean DSNを保持しているかどうか
     */
    public function hasDsn()
    {
        return $this->_dsn_slave && $this->_dsn_master;
    }
    
    
    
    
    
    /**
     * WHERE条件を連結する。
     * @access     private
     * @param      mixed   $where    WHERE
     * @param      array   $params   ブレースフォルダ
     */
    private function _chain_where($where, &$params, $prefix="", $original_key=NULL)
    {
        $return = array();
        if($where){
            foreach($where as $_key => $_val){
                $column_key = $original_key===NULL ? $_key : $original_key;
                switch($_key){
                    case 'range':
                        $place_holder1 = ":range_{$column_key}_1";
                        $place_holder2 = ":range_{$column_key}_2";
                        $return[] = sprintf("%s >= %s AND %s <= %s", $column_key, $place_holder1, $column_key, $place_holder2);
                        $params[$place_holder1] = array_shift($_val);
                        $params[$place_holder2] = array_shift($_val);
                        break;
                    default:
                        if(is_array($_val) || is_object($_val)){
                            $sub_wheres = $this->_chain_where($_val, $params, "{$prefix}{$_key}_", $column_key);
                            $return[] = sprintf("( %s )", join(" OR ", $sub_wheres));
                        } elseif($_val===NULL){
                            $return[] = sprintf("%s IS NULL", $column_key);
                        } else {
                            $operator = "=";
                            if(preg_match("/^[\!<>=]+/", (string)$_val, $matches)){
                                $operator = $matches[0];
                                $_val     = substr($_val, strlen($operator));
                                if($operator == "!") $operator = "!=";
                                if($_val === false) $_val = "";
                            }
                            $place_holder = ":{$prefix}{$_key}";
                            $params[$place_holder] = $_val;
                            $return[] = sprintf("%s %s %s", $column_key, $operator, $place_holder);
                        }
                        break;
                }
            }
        }
        return $return;
    }
    
    
    /**
     * ORDER条件を連結する。
     * @access     private
     * @param      mixed   ORDER
     */
    private function _chain_order($order)
    {
        $return = array();
        if($order){
            if(is_string($order)){
                $return[] = "{$order} ASC";
            } else {
                foreach($order as $_key => $_val){
                    $return[] = "{$_key} {$_val}";
                }
            }
        }
        return $return;
    }
}
