<?
/**
 * Samurai_Container
 * 
 * 様々なインスタンスを保持するためのクラス。
 * 単体ではSingletonを保障しない。
 * Samurai_Container_Factoryを使用することによってSingletonが保障されます。
 * 
 * @package    Samurai
 * @subpackage Component.Samurai
 * @copyright  Befool, Inc
 * @author     Satoshi Kiuchi <satoshi.kiuchi@befool.co.jp>
 * @see        Samurai_Container_Factory
 * @license    http://opensource.org/licenses/bsd-license.php  The modified BSD License
 */
class Samurai_Container
{
    private
        /** @var        array   コンポーネントを格納 */
        $_components = array();
    
    
    /**
     * コンストラクタ。
     * @access     public
     */
    public function __construct()
    {
        
    }
    
    
    
    
    
    /**
     * コンポーネントの登録。
     * インスタンスを直接登録することも可能だが、通常はSamurai_Container_Defを使用する。
     * @access     public
     * @param      string  $name   登録名
     * @param      object  $Def    コンポーネント宣言(実コンポーネントも可能)
     */
    public function registerComponent($name, $Def)
    {
        if(!isset($this->_components[$name])){
            $this->_components[$name] = $Def;
        }
    }
    
    
    /**
     * コンポーネントの取得。
     * @access     public
     * @param      string  $name   登録名
     */
    public function getComponent($name)
    {
        if(isset($this->_components[$name])){
            $Component = $this->_components[$name];
            if(is_object($Component) && $Component instanceof Samurai_Container_Def){
                return $this->getComponentByDef($name, $Component);
            } else {
                return $Component;
            }
        } else {
            throw new Samurai_Exception("Component is not found... -> {$name}");
        }
    }
    
    
    /**
     * 全てのコンポーネントを取得する。
     * @access     public
     * @return     array   全てのコンポーネント
     */
    public function getComponents()
    {
        return $this->_components;
    }
    
    
    /**
     * コンポーネントを宣言から取得する。
     * ここでインジェクションが解決される。
     * @access     public
     * @param      string  $name   登録名
     * @param      object  $Def    コンポーネント宣言
     * @return     object  コンポーネント
     */
    public function getComponentByDef($name, Samurai_Container_Def $Def)
    {
        //登録されていない場合は登録
        if(!$this->hasComponent($name)) $this->registerComponent($name, $Def);
        //実体化
        $Component = $this->_def2Instance($Def);
        //Singleton
        if($Def->instance == 'singleton'){
            $this->_components[$name] = $Component;
        }
        //依存性注入
        $this->injectDependency($Component, $Def);
        return $Component;
    }
    
    
    /**
     * Defからインスタンスを生成する。
     * @access     private
     * @param      object  $ComponentDef   コンポーネント宣言
     * @return     object  コンポーネント実体
     */
    private function _def2Instance(Samurai_Container_Def $Def)
    {
        //実体化(FactoryやgetInstanceなど)
        if(preg_match('/^[\w_]+::[\w_]+$/i', $Def->class)){
            list($class, $method) = explode('::', $Def->class);
            Samurai_Loader::loadByClass($class);
            $script = sprintf('%s::%s(%s)', $class, $method, $this->_array2ArgsWithInjectDependency('$Def->args', $Def->args));
        //実体化(new)
        } else {
            $class = $Def->class;
            Samurai_Loader::loadByClass($class);
            $script = sprintf('new %s(%s)', $class, $this->_array2ArgsWithInjectDependency('$Def->args', $Def->args));
        }
        //実体化
        $script = sprintf('$Component = %s;', $script);
        eval($script);
        //初期化メソッド
        if($Def->initMethod){
            $args = $Def->initMethod['args'];
            $script = sprintf('$Component->%s(%s);', $Def->initMethod['name'], $this->_array2ArgsWithInjectDependency('$args', $Def->initMethod['args']));
            eval($script);
        }
        Samurai_Logger::debug('Def to Entity success -> %s', array($class));
        return $Component;
    }
    
    
    /**
     * 依存性注入。
     * @access     public
     * @param      object  $Component   コンポーネント
     * @param      object  $Def         コンポーネント宣言
     */
    public function injectDependency($Component, $Def=NULL)
    {
        if($Def === NULL) $Def = new Samurai_Container_Def();
        //オートインジェクション
        foreach($Component as $_key => $_val){
            if((($Def->rule == 'AllowAll' && !in_array($_key, $Def->deny))
                || ($Def->rule == 'DenyAll' && in_array($_key, $Def->allow))) && $this->hasComponent($_key)){
                $Component->$_key = $this->getComponent($_key);
            }
        }
        //セッターインジェクション
        foreach($Def->setter as $_key => $_val){
            if(is_string($_val) && preg_match('/^\$([\w_]+)$/', $_val, $matches)){
                $_val = $this->getComponent($matches[1]);
            }
            $setter = 'set'.ucfirst($_key);
            if(method_exists($Component, $setter)){
                $Component->$setter($_val);
            } else {
                $Component->$_key = $_val;
            }
        }
    }
    
    
    /**
     * publicなメンバへの単純なアサインを実現。
     * @access     public
     * @param      object  $Component    コンポーネント
     * @param      array   $attributes   上書対象
     */
    public function injectAttributes($Component, array $attributes)
    {
        $vars = get_object_vars($Component);
        foreach($attributes as $_key => $_val){
            if(!preg_match('/^_/', $_key) && array_key_exists($_key, $vars)){
                $Component->$_key = $_val;
            }
        }
    }
    
    
    
    
    
    /**
     * Container設定ファイルからコンポーネント宣言情報をインポートする。
     * <code>
     *     @import:
     *         path : "foo/bar/zoo.dicon"
     * </code>
     * @access     public
     * @param      string  $dicon_file   Container設定ファイル
     */
    public function import($dicon_file)
    {
        $dicon = Samurai_Yaml::load($dicon_file);
        foreach($dicon as $name => $define){
            //importコマンド
            if(preg_match('/^@import/', $name)){
                if(isset($define['path'])){
                    $this->import($define['path']);
                }
            //登録
            } else {
                $this->registerComponent($name, new Samurai_Container_Def($define));
            }
        }
    }
    
    
    /**
     * コンポーネント宣言を取得する。
     * @access     public
     * @return     object  Samurai_Container_Def
     */
    public function getContainerDef()
    {
        return new Samurai_Container_Def();
    }
    
    
    
    
    
    /**
     * そのコンポーネント(名前)が既に登録されているかどうかを調べる。
     * @access     public
     * @param      string  $name   コンポーネント名
     */
    public function hasComponent($name)
    {
        return isset($this->_components[$name]);
    }
    
    
    
    
    
    /**
     * 配列を関数へ渡す引数形式に変換する。
     * @access     private
     * @param      string  その空間での配列名
     * @param      array   $array   配列
     * @return     string  引数を文字で表したもの($foo[0], $foo[1], $foo[2]のように)
     */
    private function _array2ArgsWithInjectDependency($name, array $array = array())
    {
        $args = array();
        foreach($array as $_key => $_val){
            if(is_string($_val) && preg_match('/^\$([\w_]+)$/', $_val, $matches)){
                $args[] = sprintf('$this->getComponent("%s")', $matches[1]);
            } else {
                $args[] = sprintf('%s[%s]', $name, $_key);
            }
        }
        return join(', ', $args);
    }
}
