<?
/**
 * Samurai_ActionChain
 * 
 * ActionChainを統括するクラス。
 * ActionChainはFilterChainをそれぞれに構築する。
 * 
 * @package    Samurai
 * @subpackage Samurai
 * @copyright  Befool, Inc
 * @author     Satoshi Kiuchi <samurai@don-quijote.jp>
 * @license    http://opensource.org/licenses/bsd-license.php  The modified BSD License
 */
Samurai_Loader::loadByClass('Samurai_Action');
class Samurai_ActionChain
{
    private
        /** @var        array   保持しているAction */
        $_actions = array(),
        /** @var        int     現在実行されているActionの位置 */
        $_index = 0,
        /** @var        array   Actionの位置を保持 */
        $_position = array(),
        /** @var        array   エラーリストを保持 */
        $_errors = array(),
        /** @var        array   実行結果を保持 */
        $_results = array();
    
    
    /**
     * コンストラクタ。
     * @access    public
     */
    public function __construct()
    {
        
    }
    
    
    
    
    
    /**
     * Actionインスタンスの追加。
     * @access    public
     * @param     string  $name   Action名
     */
    public function add($name)
    {
        //Action名の補完
        if(!$name) $name = Samurai_Config::get('action.default');
        if(!preg_match('/^[0-9a-zA-Z_]+$/', $name)){
            Samurai_Logger::info('Invalid ActionName. -> %s', $name);
            $name = Samurai_Config::get('action.default');
        }
        //Actionの存在確認
        if(!$this->isActionExists($name)){
            Samurai_Logger::debug('Action Not Found. -> %s', $name);
            $name = Samurai_Config::get('action.default');
        }
        //既に同名のActionが追加されている場合
        if($this->hasAction($name)){
            Samurai_Logger::debug('Already Exists Action. -> %s -> revolve', $name);
            $this->revolve($name);
        }
        //生成
        else {
            $Action = $this->createAction($name);
            if(!is_object($Action)){
                Samurai_Logger::fatal('Action generate failed... -> %s', $name);
            }
            //追加
            $this->_actions[$name] = $Action;
            $this->_position[] = $name;
            $this->_errors[$name] = Samurai::getContainer()->getComponent('ErrorList');
            $this->_results[$name] = NULL;
        }
        /*
        try {
            
        } catch(Samurai_Exception $E){
            
            
            list($class, $file) = $this->makeNames($name);
            Samurai_Loader::load($file);
        }
        */
        return true;
    }
    
    
    /**
     * Actionの存在確認。
     * @access     public
     * @param      string  $name   Action名
     */
    public function isActionExists($name)
    {
        $name = join('_', array_map('ucfirst', explode('_', $name)));
        $path = Samurai_Config::get('directory.action').DS.Samurai_Loader::getPathByClass($name);
        $path = Samurai_Loader::getPath($path);
        return Samurai_Loader::isReadable($path);
    }
    
    
    /**
     * Actionの生成。
     * @access     public
     * @param      string  $name   Action名
     * @return     object  Actionインスタンス
     */
    public function createAction($name)
    {
        //上位のActionをロードする(継承できるように)
        $this->_loadUpperAction($name);
        //作成
        list($class, $file) = $this->makeNames($name);
        Samurai_Loader::load($file);
        $Action = new $class();
        return $Action;
    }
    
    
    /**
     * 上位のActionをロードする。
     * @access     private
     * @param      string  $name   rootAction名
     */
    private function _loadUpperAction($name)
    {
        $names = explode('_', $name);
        $current_name = '';
        foreach($names as $name){
            $current_name = $current_name ? "{$current_name}_{$name}" : $name ;
            list($class, $file) = $this->makeNames($current_name);
            $file = Samurai_Loader::getPath($file);
            if(Samurai_Loader::isReadable($file)){
                Samurai_Loader::load($file);
            }
        }
    }
    
    
    /**
     * Actionクラス名および、ファイルパスの取得。
     * @access    public
     * @param     string  $name    Action名
     * @return    array   Actionのクラス名とファイルパス
     */
    public function makeNames($name)
    {
        $name = join('_', array_map('ucfirst', explode('_', $name)));
        $path = Samurai_Config::get('directory.action').DS.Samurai_Loader::getPathByClass($name);
        return array("Action_{$name}", $path);
    }
    
    
    
    
    
    /**
     * アクションがすでに追加されているかどうか。
     * @access     public
     * @param      string  $action_name
     * @return     boolean 同名のアクションが追加されているかどうか
     */
    public function hasAction($action_name)
    {
        return isset($this->_actions[$action_name]) && is_object($this->_actions[$action_name]);
    }
    
    
    /**
     * ActionChainを次に進めることができるかどうかを返却。
     * @access    public
     * @return    boolean   次に進めるかどうか
     */
    public function hasNext()
    {
        return $this->_index < $this->getSize();
    }
    
    
    /**
     * 最後のActionかどうかを判断。
     * @access     public
     * @return     boolean 最後のActionかどうか
     */
    public function isLast()
    {
        $current = $this->getCurrentActionName();
        $next = $this->getNextActionName();
        if(!$next || $current==$next){
            return true;
        } else {
            return false;
        }
    }
    
    
    /**
     * ActionChainを次に進める。
     * @access    public
     */
    public function next()
    {
        $this->_index++;
    }
    
    
    
    
    
    /**
     * ActionChainの長さを返却。
     * @access    public
     * @return    int   ActionChainの長さ
     */
    public function getSize()
    {
        return count($this->_actions);
    }
    
    
    /**
     * 登録されているActionすべて取得。
     * @access     public
     * @since      1.0.0
     * @return     array   $_actionsの内容
     */
    public function getActions(){
        return $this->_actions;
    }
    
    
    /**
     * 現在のAction名を返却。
     * @access    public
     * @return    string   現在のAction名
     */
    public function getCurrentActionName()
    {
        if(isset($this->_position[$this->_index])){
            return $this->_position[$this->_index];
        }
    }
    
    
    /**
     * 現在のアクションの実行結果を取得。
     * @access     public
     * @return     mixed   Actionの実行結果
     */
    public function getCurrentActionResult()
    {
        $name = $this->getCurrentActionName();
        return $this->_results[$name];
    }
    /**
     * 現在のActionの実行結果をセット。
     * @access     public
     * @param      mixed   $result   Action結果
     */
    public function setCurrentActionResult($result)
    {
        $name = $this->getCurrentActionName();
        $this->_results[$name] = $result;
    }
    
    
    /**
     * 次のAction名を返却。
     * @access     public
     * @return     string  次のAction名
     */
    public function getNextActionName()
    {
        if(isset($this->_position[($this->_index+1)])){
            return $this->_position[($this->_index+1)];
        }
    }
    
    
    /**
     * すべてのAction名を返却。
     * @access     public
     * @return     array   すべてのAction名
     */
    public function getAllActionName()
    {
        return array_values($this->_position);
    }
    
    
    /**
     * 現在のActionのインスタンスを返却。
     * @access    public
     * @return    object   現在のAction
     */
    public function getCurrentAction()
    {
        if($name = $this->getCurrentActionName()){
            return $this->getActionByName($name);
        }
    }
    
    
    /**
     * 指定された名前のActionを返却する。
     * @access    public
     * @param     string  $name   Action名
     * @return    object  Actionインスタンス
     */
    public function getActionByName($name)
    {
        if(!$this->hasAction($name)){
            Samurai_Logger::error('Action Not Registed. -> %s', array($name));
        }
        return $this->_actions[$name];
    }
    
    
    /**
     * 指定されたActionに対するErrorListインスタンスを返却。
     * @access    public
     * @param     string  $name   Action名
     * @return    object  ErrorListインスタンス
     */
    public function getErrorListByName($name)
    {
        if($this->hasAction($name)){
            return $this->_errors[$name];
        }
    }
    
    
    /**
     * 現在のActionのErrorListインスタンスを返却。
     * @access    public
     * @return    object  ErrorListインスタンス
     */
    public function getCurrentErrorList(){
        $name = $this->getCurrentActionName();
        return $this->getErrorListByName($name);
    }
}
