/* 
 * Copyright (c) 2008-2010, FUJITSU LIMITED
 * All rights reserved.
 * 
 *  Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation and/or
 *    other materials provided with the distribution.
 * 
 * 3. Redistributions with modification must carry prominent notices stating that you changed 
 *    the files and the date of any change.
 * 
 * 4. Neither the name of FUJITSU LIMITED nor the names of its contributors may be used
 *    to endorse or promote products derived from this software without specific prior
 *    written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES;LOSS OF USE,DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package jp.co.fujitsu.reffi.client.flex.model {
	import flash.events.EventDispatcher;
	
	import jp.co.fujitsu.reffi.client.flex.controller.BaseController;
	import jp.co.fujitsu.reffi.client.flex.controller.ParameterMapping;
	import jp.co.fujitsu.reffi.client.flex.events.ModelProcessEvent;
	
	/**
	 * <p>[概 要]</p>
	 * 全てのModelの基底クラスです.
	 * 
	 * <p>[詳 細]</p>
	 * この基底Modelクラスを継承したModelクラスは、以下の三つのイベントを
	 * ディスパッチする処理を実装する必要が有ります。
	 * 
	 * <p/>
	 * <b>Model処理成功イベント</b><br>
	 * このイベントがコントローラにキャッチされることで、
	 * BaseAction#successForwardがコールバックされるようになります。
	 * 
	 * <listing version="3.0">
	 * var modelSuccessEvent:ModelProcessEvent = new ModelProcessEvent(ModelProcessEvent.SUCCESS);
	 * modelProcessEvent.cause = event;
	 * dispatchModelSuccess(modelSuccessEvent);
	 * </listing>
	 * 
	 * <b>Model処理失敗イベント：</b><br>
	 * このイベントがコントローラにキャッチされることで、
	 * BaseAction#failureForwardがコールバックされるようになります。
	 * 
	 * <listing version="3.0">
	 * var modelFailureEvent:ModelProcessEvent = new ModelProcessEvent(ModelProcessEvent.FAILURE);
	 * modelProcessEvent.cause = event;
	 * dispatchModelFailure(modelProcessEvent);
	 * </listing>
	 * 
	 * <b>Model処理完了イベント</b><br>
	 * Action内での登録モデル数分のこのイベントがコントローラにキャッチされることで、
	 * BaseAction#completeがコールバックされるようになります。
	 * 
	 * <listing version="3.0">
	 * var modelFinishedEvent:ModelProcessEvent = new ModelProcessEvent(ModelProcessEvent.FINISHED);
	 * modelFinishedEvent.cause = event;
	 * dispatchModelFinished(modelFinishedEvent);
	 * </listing>
	 * 
	 * 
	 * <p>[備 考]</p>
	 * F/Wパッケージ内に存在する、.model.*.～Coreクラスにはイベントディスパッチ処理が既に実装されています。<br>
	 * これらのクラスを継承する場合は、イベントディスパッチ処理を実装する必要は有りません。
	 * 
	 * <p>Copyright (c) 2008-2009 FUJITSU Japan All rights reserved.</p>
	 * @author Project Reffi
	 */
	public class BaseModel extends EventDispatcher implements IModel {

		// アクションから譲渡されたパラメータマッピングオブジェクト
		private var _mapping:ParameterMapping;
		
		// モデル実行インデックス
		private var _executeIndex:int;
		
		// モデルスキップフラグ
		private var _skip:Boolean = false;

		// ModelProcessEvent.SUCCESSイベントを発行した回数
		private var _successCount:int = 0;
		
		// このモデルを起動したコントローラ
		private var _controller:BaseController;
		
		
		/**
		 * <p>[概 要]</p>
		 * MVC各レイヤを伝播するパラメータオブジェクトです.
		 * 
		 * <p>[詳 細]</p>
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @param パラメータマップオブジェクト
		 */	
		public function get parameterMapping():ParameterMapping{
			return this._mapping;	
		}
		public function set parameterMapping(parameterMapping:ParameterMapping):void{
			this._mapping = parameterMapping;
		}
		
		/**
		 * <p>[概 要]</p>
		 * 実行インデックス設定.
		 * 
		 * <p>[詳 細]</p>
		 * モデル実行時のインデックスを設定します。
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @param index モデル実行インデックス
		 */	
		public function get executeIndex():int {
			return this._executeIndex;
		}
		public function set executeIndex(index:int):void {
			this._executeIndex = index;
		}
		
		/**
		 * <p>[概 要] </p>
		 * このモデルを実行するか、コントローラが判断する為のフラグです.
		 * 
		 * <p>[詳 細] </p>
		 * このフラグがtrueの場合、コントローラはこのモデルを実行しません。
		 * 
		 * <p>[備 考] </p>
		 * 
		 * 下記例の場合、予約モデルは3つですが、実行されるのは2つになります。<p>
		 * @example 一回目の通信結果を判断して二回目の実行モデルを切り替える
		 * <listing version="3.0">
	       override protected function reserveModels(models:Array):void{
	           // 実行する可能性の有るモデルを予め登録
	           models.push(Class(HTTPServiceCore));
	           models.push(Class(ConsumerCore));
	           models.push(Class(ProducerCore));
	       }
	     
           override public function nextModel(index:int, prev:ModelProcessEvent, next:BaseModel):Boolean {
	           switch(index){
	               case 0:
	                   HTTPRequestCore(next).url = "webcontroller";
	                   ((HTTPRequestCore)next).addUrlParameter("model.fqcn", "demo.server.model.DataFetchModel");
	                   break;
	               case 1:
	                   // 一回目の通信結果を取得してMVCを伝播するParameterMappingに保存します
	                   var dao:DataDao = (DataDao)prev.getResult();
	                   parameterMapping.setParameter("dao", dao);
	      
	                   if(this.dao.getDivision() == 0) {
	                       ConsumerCore(next).destination = "chat1";
	                   }else {
	                       // 想定外なのでスキップ。ConsumerCoreをコントローラに実行させない
	                       ConsumerCore(next).skip = true;
	                   }
	                   break;
	               case 2:
	                   // 保存しておいた一回目の通信結果を取り出します
	                   dao = parameterMapping.getParameter("dao");
	                   if(dao.getDivision() == 1) {
	                       ProducerCore(next).destination = "chat1";
	                       ProducerCore(next).messageBody = handleName + "が入室しました。";
	                   }else{
	                       // 想定外なのでスキップ。PublishCoreをコントローラに実行させない
	                       ProducerCore(next).skip = true;
	                   }
	                   break;
	           }
	           return true;
	       }
		 * </listing>
		 */
		public function get skip():Boolean {
			return this._skip;
		}
		public function set skip(skip:Boolean):void {
			this._skip = skip;
		}
		
		/**
		 * <p>[概 要]</p>
		 * モデルインスタンス生存中に、何回モデル処理が成功したかを保持します.
		 * 
		 * <p>[詳 細] </p>
		 * 継承モデルがModelSuccessEvent.SUCCESSイベントをディスパッチした回数を返却します。
		 * 
		 * <p>[備 考] </p>
		 * Consumerのように継続して結果を受信する機能モデルは、
		 * 処理終了するまでこの値が継続的にインクリメントされます。 
		 * <p/>
		 * 
		 */	
		public function get successCount():int {
			return this._successCount;
		}
		public function set successCount(successCount:int):void {
			this._successCount = successCount;
		}
		
		/**
		 * <p>[概 要]</p>
		 * このモデルを起動したコントローラインスタンスです.
		 *  
		 * <p>[詳 細] </p>
		 * 
		 * <p>[備 考] </p>
		 * 
		 */	
		public function get controller():BaseController {
			return this._controller;
		}
		public function set controller(controller:BaseController):void {
			this._controller = controller;
		}

		/**
		 * <p>[概 要]</p>
		 * ModelProcessEvent.SUCCESS発行回数を1増加させます.
		 * 
		 * <p>[詳 細]</p>
		 * successCountプロパティをインクリメントします。
		 * 
		 * <p>[備 考]</p>
		 * 
		 * @return インクリメント後のsuccessCountプロパティ値
		 */	
		public function incrementSuccessCount():int {
			return ++this._successCount;
		}	
			
		/**
		 * <p>[概 要]</p>
		 * コントローラにコールされるモデルの主幹メソッドです.
		 * 
		 * <p>[詳 細]</p>
		 * モデル処理の基本フローを構築します。
		 * 下記のテンプレートメソッドが順にテンプレートコールされます。<p>
		 * 
		 * 	<ol>
		 * 		<li>preproc()</li>
		 * 		<li>mainproc()</li>
		 * 		<li>postproc()</li>
		 * 		<li>finalproc()</li>
		 * 	</ol>
		 * 上記メソッド内で例外が発生した場合、trapメソッドがテンプレートコールされます。
		 * 
		 * <p>[備 考] </p>
		 * 例外が発生した場合、デフォルト動作としてModel処理失敗イベントが発火されますが、
		 * trapオーバーライドメソッドでnullを返却すると、このイベントは発火されません。
		 * <p/>
		 * 
		 */	
		public function run():void {
			try {
				if (preProc()) {
					mainProc();
					postProc();
				} else {
					dispatchModelFinished(new ModelProcessEvent(ModelProcessEvent.FINISHED));
				}
			} catch(e : Error) {
				e = trap(e);
				if(e) {
					throw e;
				}
			} finally {
				finallyProc();
			}
		}

		/**
		 * <p>[概 要] </p>
		 * 前処理テンプレートメソッドです.
		 * 
		 * <p>[詳 細] </p>
		 * デフォルト処理はtrueを返却します。<p>
		 * 
		 * mainprocよりも先に呼ばれるメソッドです。<br>
		 * オーバーライドして、主処理の前に行う初期化を記述します。<br>
		 * nullを返却すると、それ以降のモデル処理は中止されます。<br>
		 * 
		 * <p>[備 考] </p>
		 * 
		 * @return 以降の処理を継続するかどうかのフラグ（デフォルト：true）
		 */	
		protected function preProc() : Boolean {
			return true;
		}

		/**
		 * <p>[概 要] </p>
		 * 主処理テンプレートメソッドです.
		 * 
		 * <p>[詳 細] </p>
		 * デフォルト処理は有りません。<br>
		 * オーバーライドしてこのモデルのメイン処理を実装します。
		 * 
		 * <p>[備 考] </p>
		 * 
		 */
		protected function mainProc() : void {
		}

		/**
		 * <p>[概 要] </p>
		 * 後処理テンプレートメソッドです.
		 * 
		 * <p>[詳 細] </p>
		 * デフォルト処理は有りません。<br>
		 * オーバーライドしてこのモデルの主処理後処理を実装します。
		 * 
		 * <p>[備 考] </p>
		 * 
		 * @throws Exception オーバーライド先で発生する可能性が有る例外
		 */
		protected function postProc() : void {
		}

		/**
		 * <p>[概 要] </p>
		 * run()内で発生した全例外をハンドリングするメソッドです.
		 *  
		 * <p>[詳 細] </p>
		 * デフォルト処理は有りません。<br>
		 * オーバーライドしてこのモデルの例外ハンドリング処理を実装します。
		 * 
		 * <p>[備 考] </p>
		 * 
		 * @param e オーバーライド先で発生する可能性が有るエラー
		 * @return 引数eか、trapオーバーライドメソッドで変換されたエラー
		 */
		protected function trap(e : Error) : Error {
			return e;
		}

		/**
		 * <p>[概 要] </p>
		 * run()が終了したタイミングでテンプレートコールされるメソッドです.
		 * 
		 * <p>[詳 細] </p>
		 * デフォルト処理は有りません。<br>
		 * オーバーライドしてこのモデルの終了処理を実装します。
		 * 
		 * <p>[備 考] </p>
		 * 
		 */
		protected function finallyProc() : void {
		}
		
		/**
		 * <p>[概 要] </p>
		 * モデル処理成功イベントを発行するメソッドです.
		 * 
		 * <p>[詳 細] </p>
		 * モデル処理成功回数を1増加させ、このモデルに登録されているモデル処理監視リスナ群
		 * に対して処理が成功したことを通知します。<br>
		 * このイベントが発行されると、コントローラはアクションにsuccessFoward通知を行います。
		 * 
		 * <p>[備 考] </p>
		 * 
		 * @example ModelProcessEventを生成してから使用します。
		 * <listing version="3.0">
		 *   var modelSuccessEvent:ModelProcessEvent = new ModelProcessEvent(ModelProcessEvent.SUCCESS);
		 *   modelProcessEvent.cause = event;
		 *   dispatchModelSuccess(modelSuccessEvent);
		 * </listing>
		 *
		 * @param evt モデル成功処理結果が入ったModelProcessEventインスタンス　
		 */
		public function dispatchModelSuccess(event:ModelProcessEvent):void {
			incrementSuccessCount();
			dispatchEvent(event);
		}
		
		/**
		 * <p>[概 要] </p>
		 * モデル処理失敗イベントを発行するメソッドです.
		 * 
		 * <p>[詳 細] </p>
		 * このモデルに登録されているモデル処理監視リスナ群
		 * に対して処理が成功したことを通知します。<br>
		 * このイベントが発行されると、コントローラはアクションにfailureFoward通知を行います。
		 * 
		 * <p>[備 考] </p>
		 * 
		 * @example ModelProcessEventを生成してから使用します。
		 * <listing version="3.0">
		 *   var modelFailureEvent:ModelProcessEvent = new ModelProcessEvent(ModelProcessEvent.FAILURE);
		 *   modelProcessEvent.cause = event;
		 *   dispatchModelFailure(modelFailureEvent);
		 * </listing>
		 *
		 * @param evt モデル失敗処理結果が入ったModelProcessEventインスタンス　
		 */
		public function dispatchModelFailure(event:ModelProcessEvent):void {
			dispatchEvent(event);
		}
		
		/**
		 * <p>[概 要] </p>
		 * モデル処理完了イベントを発行するメソッドです.
		 * 
		 * <p>[詳 細] </p>
		 * このモデルに登録されているモデル処理監視リスナ群
		 * に対して処理が完了したことを通知します。<br>
		 * コントローラが実行委譲した全てのモデルが完了すると、
		 * コントローラはアクションに対してcomplete通知を行います。
		 * 
		 * <p>[備 考] </p>
		 * 
		 * @example ModelProcessEventを生成してから使用します。
		 * <listing version="3.0">
		 *   var modelFailureEvent:ModelProcessEvent = new ModelProcessEvent(ModelProcessEvent.FAILURE);
		 *   modelProcessEvent.cause = event;
		 *   dispatchModelFailure(modelFailureEvent);
		 * </listing>
		 *
		 * @param evt モデル失敗処理結果が入ったModelProcessEventインスタンス　
		 */
		public function dispatchModelFinished(event:ModelProcessEvent):void {
			dispatchEvent(event);
		}
	}
}
