// -*-c++-*-

/*!
  \file debug_message_window.cpp
  \brief debug management window class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <qt.h>

#include "debug_message_window.h"

#include "options.h"
#include "main_data.h"
#include "debug_log_holder.h"
#include "debug_log_data.h"
#include "debug_log_dir_dialog.h"

#include <cassert>
#include <iostream>

#include "xpm/debug_view_all.xpm"
#include "xpm/debug_view_self.xpm"
#include "xpm/debug_view_ball.xpm"
#include "xpm/debug_view_teammate.xpm"
#include "xpm/debug_view_opponent.xpm"
#include "xpm/debug_view_comment.xpm"
#include "xpm/debug_view_line.xpm"
#include "xpm/debug_view_target.xpm"
#include "xpm/debug_view_message.xpm"

#include "xpm/debug_log_view.xpm"

#include "xpm/open_dir.xpm"
#include "xpm/clear.xpm"

#include "xpm/sync.xpm"
#include "xpm/logplayer_one_step_back.xpm"
#include "xpm/logplayer_one_step_forward.xpm"

#include "xpm/num_1.xpm"
#include "xpm/num_2.xpm"
#include "xpm/num_3.xpm"
#include "xpm/num_4.xpm"
#include "xpm/num_5.xpm"
#include "xpm/num_6.xpm"
#include "xpm/num_7.xpm"
#include "xpm/num_8.xpm"
#include "xpm/num_9.xpm"
#include "xpm/num_10.xpm"
#include "xpm/num_11.xpm"
#include "xpm/num_12.xpm"
#include "xpm/num_13.xpm"
#include "xpm/num_14.xpm"
#include "xpm/num_15.xpm"
#include "xpm/num_16.xpm"
#include "xpm/num_17.xpm"
#include "xpm/num_18.xpm"
#include "xpm/num_19.xpm"
#include "xpm/num_20.xpm"
#include "xpm/num_21.xpm"
#include "xpm/num_22.xpm"
#include "xpm/num_23.xpm"
#include "xpm/num_24.xpm"
#include "xpm/num_25.xpm"
#include "xpm/num_26.xpm"
#include "xpm/num_27.xpm"
#include "xpm/num_28.xpm"
#include "xpm/num_29.xpm"
#include "xpm/num_30.xpm"
#include "xpm/num_31.xpm"
#include "xpm/num_32.xpm"

/*-------------------------------------------------------------------*/
/*!

*/
DebugMessageWindow::DebugMessageWindow( QWidget * parent,
                                        MainData & main_data )
    : QMainWindow( parent )
    , M_main_data( main_data )
{
    //this->setWindowTitle( tr( "Debug Message" ) ); // qt4
    this->setCaption( tr( "Debug Message" ) ); // qt3

    //this->setUsesTextLabel( true );

    int i = -1;
    M_debug_text[++i] = tr( "System" );
    M_debug_text[++i] = tr( "Sensor" );
    M_debug_text[++i] = tr( "WorldMode" );
    M_debug_text[++i] = tr( "BasicAction" );
    M_debug_text[++i] = tr( "Intercept" );
    M_debug_text[++i] = tr( "Kick" );
    M_debug_text[++i] = tr( "Hold" );
    M_debug_text[++i] = tr( "Dribble" );
    M_debug_text[++i] = tr( "Pass" );
    M_debug_text[++i] = tr( "Cross" );
    M_debug_text[++i] = tr( "Shoot" );
    M_debug_text[++i] = tr( "Clear" );
    M_debug_text[++i] = tr( "Block" );
    M_debug_text[++i] = tr( "Mark" );
    M_debug_text[++i] = tr( "Role" );
    M_debug_text[++i] = tr( "Team" );
    M_debug_text[++i] = tr( "Analyzer" );
    M_debug_text[++i] = tr( "Chain" );

    createActions();
    createWindows();

    readSettings();

    this->resize( 640, 480 );
    //this->setWFlags( this->getWFlags() | Qt::WStyle_StaysOnTop | Qt::WX11BypassWM );
}

/*-------------------------------------------------------------------*/
/*!

*/
DebugMessageWindow::~DebugMessageWindow()
{
    //std::cerr << "delete DebugMessageWindow" << std::endl;
    saveSettings();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::readSettings()
{
    QSettings settings( QSettings::Ini );
    settings.insertSearchPath( QSettings::Unix,
                               QDir::homeDirPath() );

    settings.beginGroup( "/.soccerwindow2-qt3./DebugMessageWindow" );

    // read debug view switch statis
    {
        bool ok = false;
        bool enable = false;

        enable = settings.readBoolEntry( tr( "debug_view_all" ), true, &ok );
        if ( ok )
        {
            M_show_debug_view_all_act->setOn( enable );
            toggleShowDebugViewAll( enable );
        }

        enable = settings.readBoolEntry( tr( "debug_view_self" ), true, &ok );
        if ( ok )
        {
            M_show_debug_view_self_act->setOn( enable );
            toggleShowDebugViewSelf( enable );
        }

        enable = settings.readBoolEntry( tr( "debug_view_ball" ), true, &ok );
        if ( ok )
        {
            M_show_debug_view_ball_act->setOn( enable );
            toggleShowDebugViewBall( enable );
        }

        enable = settings.readBoolEntry( tr( "debug_view_teammates" ), true, &ok );
        if ( ok )
        {
            M_show_debug_view_teammates_act->setOn( enable );
            toggleShowDebugViewTeammates( enable );
        }

        enable = settings.readBoolEntry( tr( "debug_view_opponents" ), true, &ok );
        if ( ok )
        {
            M_show_debug_view_opponents_act->setOn( enable );
            toggleShowDebugViewOpponents( enable );
        }

        enable = settings.readBoolEntry( tr( "debug_view_comment" ), true, &ok );
        if ( ok )
        {
            M_show_debug_view_comment_act->setOn( enable );
            toggleShowDebugViewComment( enable );
        }

        enable = settings.readBoolEntry( tr( "debug_view_shape" ), true, &ok );
        if ( ok )
        {
            M_show_debug_view_shape_act->setOn( enable );
            toggleShowDebugViewShape( enable );
        }

        enable = settings.readBoolEntry( tr( "debug_view_target" ), true, &ok );
        if ( ok )
        {
            M_show_debug_view_target_act->setOn( enable );
            toggleShowDebugViewTarget( enable );
        }

        enable = settings.readBoolEntry( tr( "debug_view_message" ), true, &ok );
        if ( ok )
        {
            M_show_debug_view_message_act->setOn( enable );
            toggleShowDebugViewMessage( enable );
        }

        enable = settings.readBoolEntry( tr( "debug_log_objects" ), true, &ok );
        if ( ok )
        {
            M_show_debug_log_objects_act->setOn( enable );
            toggleShowDebugLogObjects( enable );
        }
    }

    // read debug level switch status
    for ( int i = 0; i < 32; ++i )
    {
        char key[32];
        snprintf( key, 32, "debug%02d", i + 1 );

        //QVariant val = settings.value( QString::fromAscii( key ) );
        bool ok = false;
        bool enable = settings.readBoolEntry( QString::fromAscii( key ),
                                              true,
                                              &ok );
        if ( ok )
        {
            boost::int32_t level = DebugLogData::LEVEL_01 << i;
            M_main_data.getDebugLogHolder().setLevel( level, enable );

            M_debug_level_act[i]->setOn( enable );
        }
        else
        {
            M_debug_level_act[i]->setOn( true );
        }
    }

    // read debug level text string
    for ( int i = 0; i < 32; ++i )
    {
        char key[32];
        snprintf( key, 32, "debug_text_%02d", i + 1 );

        bool ok = false;
        QString val = settings.readEntry( QString::fromAscii( key ),
                                          QString::null,
                                          &ok );
        if ( ok )
        {
            M_debug_text[i] = val;
            //QString text = tr( "Level %1 " ).arg( i + 1 ) + val;
            //M_debug_level_act[i]->setText( text );
            //M_debug_level_act[i]->setStatusTip( tr( "Debug " ) + text );
            M_debug_level_act[i]->setText( M_debug_text[i] );
            M_debug_level_act[i]->setMenuText( M_debug_text[i] );
            M_debug_level_act[i]->setStatusTip( tr( "Debug " ) + M_debug_text[i] );
        }
    }

    settings.endGroup();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::saveSettings()
{
    QSettings settings( QSettings::Ini );
    settings.insertSearchPath( QSettings::Unix,
                               QDir::homeDirPath() );

    settings.beginGroup( "/.soccerwindow2-qt3./DebugMessageWindow" );

    // save debug view switches
    settings.writeEntry( tr( "debug_view_all" ),  M_show_debug_view_all_act->isOn() );
    settings.writeEntry( tr( "debug_view_self" ), M_show_debug_view_self_act->isOn() );
    settings.writeEntry( tr( "debug_view_ball" ), M_show_debug_view_ball_act->isOn() );
    settings.writeEntry( tr( "debug_view_teammates" ), M_show_debug_view_teammates_act->isOn() );
    settings.writeEntry( tr( "debug_view_opponents" ), M_show_debug_view_opponents_act->isOn() );
    settings.writeEntry( tr( "debug_view_comment" ), M_show_debug_view_comment_act->isOn() );
    settings.writeEntry( tr( "debug_view_shape" ), M_show_debug_view_shape_act->isOn() );
    settings.writeEntry( tr( "debug_view_target" ), M_show_debug_view_target_act->isOn() );
    settings.writeEntry( tr( "debug_view_message" ), M_show_debug_view_message_act->isOn() );
    settings.writeEntry( tr( "debug_log_objects" ), M_show_debug_log_objects_act->isOn() );

    // save debug level switches
    for ( int i = 0; i < 32; ++i )
    {
        char key[32];
        snprintf( key, 32, "debug%02d", i + 1 );

        settings.writeEntry( QString::fromAscii( key ),
                             M_debug_level_act[i]->isOn() );
    }

    // save debug level text
    for ( int i = 0; i < 32; ++i )
    {
        if ( ! M_debug_text[i].isEmpty() )
        {
            char key[32];
            snprintf( key, 32, "debug_text_%02d", i + 1 );
            settings.writeEntry( QString::fromAscii( key ), M_debug_text[i] );
        }
    }

    settings.endGroup();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::createActions()
{
    // view switches

    M_show_debug_view_all_act
        = new QAction( QIconSet( QPixmap( debug_view_all_xpm ) ),
                       tr( "Toggle debug view" ),
                       QKeySequence(),
                       this );
    M_show_debug_view_all_act->setStatusTip( tr( "Show/Hide all debug view" ) );
    M_show_debug_view_all_act->setToggleAction( true );
    M_show_debug_view_all_act->setOn( true );
    connect( M_show_debug_view_all_act, SIGNAL( toggled( bool ) ),
             this, SLOT( toggleShowDebugViewAll( bool ) ) );

    M_show_debug_view_self_act
        = new QAction( QIconSet( QPixmap( debug_view_self_xpm ) ),
                       tr( "Toggle debug view self" ),
                       QKeySequence(),
                       this );
    M_show_debug_view_self_act->setStatusTip( tr( "Show/Hide debug view self" ) );
    M_show_debug_view_self_act->setToggleAction( true );
    M_show_debug_view_self_act->setOn( true );
    connect( M_show_debug_view_self_act, SIGNAL( toggled( bool ) ),
             this, SLOT( toggleShowDebugViewSelf( bool ) ) );

    M_show_debug_view_ball_act
        = new QAction( QIconSet( QPixmap( debug_view_ball_xpm ) ),
                       tr( "Toggle debug view ball" ),
                       QKeySequence(),
                       this );
    M_show_debug_view_ball_act->setStatusTip( tr( "Show/Hide debug view ball" ) );
    M_show_debug_view_ball_act->setToggleAction( true );
    M_show_debug_view_ball_act->setOn( true );
    connect( M_show_debug_view_ball_act, SIGNAL( toggled( bool ) ),
             this, SLOT( toggleShowDebugViewBall( bool ) ) );

    M_show_debug_view_teammates_act
        = new QAction( QIconSet( QPixmap( debug_view_teammate_xpm ) ),
                       tr( "Toggle debug view teammates" ),
                       QKeySequence(),
                       this );
    M_show_debug_view_teammates_act->setStatusTip( tr( "Show/Hide debug view teammates" ) );
    M_show_debug_view_teammates_act->setToggleAction( true );
    M_show_debug_view_teammates_act->setOn( true );
    connect( M_show_debug_view_teammates_act, SIGNAL( toggled( bool ) ),
             this, SLOT( toggleShowDebugViewTeammates( bool ) ) );

    M_show_debug_view_opponents_act
        = new QAction( QIconSet( QPixmap( debug_view_opponent_xpm ) ),
                       tr( "Toggle debug view opponents" ),
                       QKeySequence(),
                       this );
    M_show_debug_view_opponents_act->setStatusTip( tr( "Show/Hide debug view opponents" ) );
    M_show_debug_view_opponents_act->setToggleAction( true );
    M_show_debug_view_opponents_act->setOn( true );
    connect( M_show_debug_view_opponents_act, SIGNAL( toggled( bool ) ),
             this, SLOT( toggleShowDebugViewOpponents( bool ) ) );

    M_show_debug_view_comment_act
        = new QAction( QIconSet( QPixmap( debug_view_comment_xpm ) ),
                       tr( "Toggle debug view comment" ),
                       QKeySequence(),
                       this );
    M_show_debug_view_comment_act->setStatusTip( tr( "Show/Hide debug view comment" ) );
    M_show_debug_view_comment_act->setToggleAction( true );
    M_show_debug_view_comment_act->setOn( true );
    connect( M_show_debug_view_comment_act, SIGNAL( toggled( bool ) ),
             this, SLOT( toggleShowDebugViewComment( bool ) ) );

    M_show_debug_view_shape_act
        = new QAction( QIconSet( QPixmap( debug_view_line_xpm ) ),
                       tr( "Toggle debug view shapes" ),
                       QKeySequence(),
                       this );
    M_show_debug_view_shape_act->setStatusTip( tr( "Show/Hide debug view shapes" ) );
    M_show_debug_view_shape_act->setToggleAction( true );
    M_show_debug_view_shape_act->setOn( true );
    connect( M_show_debug_view_shape_act, SIGNAL( toggled( bool ) ),
             this, SLOT( toggleShowDebugViewShape( bool ) ) );

    M_show_debug_view_target_act
        = new QAction( QIconSet( QPixmap( debug_view_target_xpm ) ),
                       tr( "Toggle debug view target" ),
                       QKeySequence(),
                       this );
    M_show_debug_view_target_act->setStatusTip( tr( "Show/Hide debug view target" ) );
    M_show_debug_view_target_act->setToggleAction( true );
    M_show_debug_view_target_act->setOn( true );
    connect( M_show_debug_view_target_act, SIGNAL( toggled( bool ) ),
             this, SLOT( toggleShowDebugViewTarget( bool ) ) );

    M_show_debug_view_message_act
        = new QAction( QIconSet( QPixmap( debug_view_message_xpm ) ),
                       tr( "Toggle debug view message" ),
                       QKeySequence(),
                       this );
    M_show_debug_view_message_act->setStatusTip( tr( "Show/Hide debug view message" ) );
    M_show_debug_view_message_act->setToggleAction( true );
    M_show_debug_view_message_act->setOn( true );
    connect( M_show_debug_view_message_act, SIGNAL( toggled( bool ) ),
             this, SLOT( toggleShowDebugViewMessage( bool ) ) );

    //////////////////////////////////////////////
    // debug log object
    M_show_debug_log_objects_act
        = new QAction( QIconSet( QPixmap( debug_log_view_xpm ) ),
                       tr( "Toggle debug log objects" ),
                       QKeySequence(),
                       this );
    M_show_debug_log_objects_act->setStatusTip( tr( "Show/Hide debug log objects" ) );
    M_show_debug_log_objects_act->setToggleAction( true );
    M_show_debug_log_objects_act->setOn( true );
    connect( M_show_debug_log_objects_act, SIGNAL( toggled( bool ) ),
             this, SLOT( toggleShowDebugLogObjects( bool ) ) );

    //////////////////////////////////////////////
    // dir selection
    M_open_debug_log_dir_act
        = new QAction( QIconSet( QPixmap( open_dir_xpm ) ),
                       tr( "Open log dir..." ),
                       QKeySequence(),
                       this );
    M_open_debug_log_dir_act->setAccel( Qt::CTRL + Qt::Key_O );
    M_open_debug_log_dir_act->setStatusTip( tr( "Open the debug log directory" ) );
    connect( M_open_debug_log_dir_act, SIGNAL( activated() ),
             this, SLOT( showDebugLogDirDialog() ) );
    //
    M_clear_data_act = new QAction( QIconSet( QPixmap( clear_xpm ) ),
                                    tr( "Clear data" ),
                                    QKeySequence(),
                                    this );
    M_clear_data_act->setStatusTip( tr( "Clear all data" ) );
    connect( M_clear_data_act, SIGNAL( activated() ),
             this, SLOT( clearAll() ) );

    //////////////////////////////////////////////
    // cycle control
    M_sync_act
        = new QAction( QIconSet( QPixmap( sync_xpm ) ),
                       tr( "Sync" ),
                       QKeySequence(),
                       this );
    M_sync_act->setAccel( Qt::CTRL + Qt::Key_S );
    M_sync_act->setStatusTip( tr( "Synchronize with field canvas" ) );
    connect( M_sync_act, SIGNAL( activated() ),
             this, SLOT( syncCycle() ) );

    M_decrement_act
        = new QAction( QIconSet( QPixmap( logplayer_one_step_back_xpm ) ),
                       tr( "Decrement" ),
                       QKeySequence(),
                       this );
    M_decrement_act->setAccel( Qt::CTRL + Qt::Key_Left );
    M_decrement_act->setStatusTip( tr( "Decrement message cycle" ) );
    connect( M_decrement_act, SIGNAL( activated() ),
             this, SLOT( decrementCycle() ) );

    M_increment_act
        = new QAction( QIconSet( QPixmap( logplayer_one_step_forward_xpm ) ),
                       tr( "Increment" ),
                       QKeySequence(),
                       this );
    M_decrement_act->setAccel( Qt::CTRL + Qt::Key_Right );
    M_increment_act->setStatusTip( tr( "Increment message cycle" ) );
    connect( M_increment_act, SIGNAL( activated() ),
             this, SLOT( incrementCycle() ) );

    //////////////////////////////////////////////
    // debug level switches

    const char ** num_xpm[] = { num_1_xpm, num_2_xpm, num_3_xpm, num_4_xpm,
                                num_5_xpm, num_6_xpm, num_7_xpm, num_8_xpm,
                                num_9_xpm, num_10_xpm, num_11_xpm, num_12_xpm,
                                num_13_xpm, num_14_xpm, num_15_xpm, num_16_xpm,
                                num_17_xpm, num_18_xpm, num_19_xpm, num_20_xpm,
                                num_21_xpm, num_22_xpm, num_23_xpm, num_24_xpm,
                                num_25_xpm, num_26_xpm, num_27_xpm, num_28_xpm,
                                  num_29_xpm, num_30_xpm, num_31_xpm, num_32_xpm };

    QSignalMapper * debug_level_mapper = new QSignalMapper( this );
    connect( debug_level_mapper, SIGNAL( mapped( int ) ),
             this, SLOT( toggleDebugLevel( int ) ) );

    for ( int i = 0; i < 32; ++i )
    {
        QString menu_text;
        if ( M_debug_text[i].isEmpty() )
        {
            menu_text = tr( "Level %1 " ).arg( i + 1 );
            M_debug_level_act[i] = new QAction( QIconSet( QPixmap( num_xpm[i] ) ),
                                                menu_text,
                                                QKeySequence(),
                                                this ); //M_debug_level_act_group );
            M_debug_level_act[i]->setText( menu_text );
        }
        else
        {
            menu_text = M_debug_text[i];
            M_debug_level_act[i] = new QAction( QIconSet( QPixmap( num_xpm[i] ) ),
                                                menu_text,
                                                QKeySequence(),
                                                this );
            M_debug_level_act[i]->setText( M_debug_text[i] );
            M_debug_level_act[i]->setMenuText( M_debug_text[i] );
        }

        M_debug_level_act[i]->setStatusTip( tr( "Debug " ) + menu_text );
        M_debug_level_act[i]->setToggleAction( true );

        connect( M_debug_level_act[i], SIGNAL( activated() ),
                 debug_level_mapper, SLOT( map() ) );
        debug_level_mapper->setMapping( M_debug_level_act[i], i + 1 );
    }

}


/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::createWindows()
{
    createToolBar();

    M_tab_widget = new QTabWidget( this );

    connect( M_tab_widget, SIGNAL( currentChanged( QWidget * ) ),
             this, SLOT( changeCurrentTab( QWidget * ) ) );

    this->setCentralWidget( M_tab_widget );

    for ( int i = 0; i < 11; ++i )
    {
        M_message[i] = new QTextEdit( M_tab_widget );
        M_message[i]->setReadOnly( true );
        M_message[i]->setCaption( QString( "   %1 " ).arg( i + 1, 2 ) );
        M_tab_widget->addTab( M_message[i],
                              M_message[i]->caption() );
    }

}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::createToolBar()
{
    {
        QToolBar * tbar = new QToolBar( tr( "Control" ), this );

        M_open_debug_log_dir_act->addTo( tbar );

        tbar->addSeparator();

        M_sync_act->addTo( tbar );
        M_decrement_act->addTo( tbar );
        M_increment_act->addTo( tbar );

        tbar->addSeparator();

        new QLabel( tr( "Find: " ), tbar );
        M_find_box = new QLineEdit( tbar );
        connect( M_find_box, SIGNAL( returnPressed() ),
                 this, SLOT( findExistString() ) );
        connect( M_find_box, SIGNAL( textChanged( const QString & ) ),
                 this, SLOT( findString( const QString & ) ) );
        // short cut
        {
            QAccel * accel = new QAccel( this );
            accel->connectItem( accel->insertItem( Qt::CTRL + Qt::Key_F ),
                                M_find_box, SLOT( setFocus() ) );
        }
        //QToolTip::add( M_find_box, tr( "Find the matched string from the below window" ) );

        QHButtonGroup * btn_group = new QHButtonGroup( tbar );
        btn_group->setFlat( true );
        btn_group->setInsideMargin( 0 );
        btn_group->setInsideSpacing( 0 );
        M_find_forward_rb = new QRadioButton( tr( "Down" ), btn_group );
        connect( M_find_forward_rb, SIGNAL( clicked() ),
                 this, SLOT( findExistString() ) );
        M_find_backward_rb = new QRadioButton( tr( "Up" ), btn_group );
        connect( M_find_backward_rb, SIGNAL( clicked() ),
                 this, SLOT( findExistString() ) );
        M_find_forward_rb->setChecked( true );

        this->addDockWindow( tbar, Qt::DockTop );
    }

    {
        QToolBar * tbar = new QToolBar( tr( "Debug view & 1-16" ), this );

        M_show_debug_view_all_act->addTo( tbar );
        M_show_debug_view_self_act->addTo( tbar );
        M_show_debug_view_ball_act->addTo( tbar );
        M_show_debug_view_teammates_act->addTo( tbar );
        M_show_debug_view_opponents_act->addTo( tbar );
        M_show_debug_view_comment_act->addTo( tbar );
        M_show_debug_view_shape_act->addTo( tbar );
        M_show_debug_view_target_act->addTo( tbar );
        M_show_debug_view_message_act->addTo( tbar );

        tbar->addSeparator();

        M_show_debug_log_objects_act->addTo( tbar );

        this->addDockWindow( tbar, Qt::DockTop );
    }

    const int max_visible_level = 18;
    {
        QToolBar * tbar = new QToolBar( tr( "Debug Level" ), this );

         for ( int i = 0; i < max_visible_level; ++i )
         {
             M_debug_level_act[i]->addTo( tbar );
         }

        this->addDockWindow( tbar, Qt::DockLeft );
    }

    {
        QToolBar * tbar = new QToolBar( tr( "Debug Level Hidden" ), this );

         for ( int i = max_visible_level; i < 32; ++i )
         {
             M_debug_level_act[i]->addTo( tbar );
         }

         this->addDockWindow( tbar, Qt::DockLeft, true );
         tbar->hide();
    }

}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::openDebugLogDir( const rcsc::SideID side,
                                     const std::string & dir_path )
{
    MonitorViewConstPtr ptr = M_main_data.getCurrentViewData();
    if ( ! ptr )
    {
        std::cerr << "No valid data!" << std::endl;
        return;
    }

    if ( ptr->leftTeam().name().empty() )
    {
        std::cerr << "Empty team name!" << std::endl;
        return;
    }

    const int unum = M_tab_widget->currentPageIndex() + 1;

    if ( side == rcsc::LEFT )
    {
        if ( M_main_data.getDebugLogHolder().setDir( ptr->leftTeam().name(),
                                                     side,
                                                     unum,
                                                     dir_path ) )
        {
            syncCycle();
        }
    }
    else if ( side == rcsc::RIGHT )
    {
        if ( M_main_data.getDebugLogHolder().setDir( ptr->rightTeam().name(),
                                                     side,
                                                     unum,
                                                     dir_path ) )
        {
            syncCycle();
        }
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::changeCurrentTab( QWidget * )
{
    //std::cerr << "DebugMessageWindow  changed tab index to ["
    //          << widget->caption().ascii() << "]"
    //          << std::endl;
    syncCycle();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::toggleShowDebugViewAll( bool checked )
{
    if ( Options::instance().showDebugView() != checked )
    {
        Options::instance().toggleShowDebugView();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::toggleShowDebugViewSelf( bool checked )
{
    if ( Options::instance().showDebugViewSelf() != checked )
    {
        Options::instance().toggleShowDebugViewSelf();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::toggleShowDebugViewBall( bool checked )
{
    if ( Options::instance().showDebugViewBall() != checked )
    {
        Options::instance().toggleShowDebugViewBall();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::toggleShowDebugViewTeammates( bool checked )
{
    if ( Options::instance().showDebugViewTeammates() != checked )
    {
        Options::instance().toggleShowDebugViewTeammates();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::toggleShowDebugViewOpponents( bool checked )
{
    if ( Options::instance().showDebugViewOpponents() != checked )
    {
        Options::instance().toggleShowDebugViewOpponents();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::toggleShowDebugViewComment( bool checked )
{
    if ( Options::instance().showDebugViewComment() != checked )
    {
        Options::instance().toggleShowDebugViewComment();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::toggleShowDebugViewShape( bool checked )
{
    if ( Options::instance().showDebugViewShape() != checked )
    {
        Options::instance().toggleShowDebugViewShape();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::toggleShowDebugViewTarget( bool checked )
{
    if ( Options::instance().showDebugViewTarget() != checked )
    {
        Options::instance().toggleShowDebugViewTarget();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::toggleShowDebugViewMessage( bool checked )
{
    if ( Options::instance().showDebugViewMessage() != checked )
    {
        Options::instance().toggleShowDebugViewMessage();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::toggleShowDebugLogObjects( bool checked )
{
    if ( Options::instance().showDebugLogObjects() != checked )
    {
        Options::instance().toggleShowDebugLogObjects();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::showDebugLogDirDialog()
{
    DebugLogDirDialog dlg( this );

    if ( ! dlg.exec() )
    {
        //std::cerr << "DebugLogDirDialog is not ok." << std::endl;
        return;
    }

    rcsc::SideID side = dlg.side();
    QString dir_path = dlg.dirPath();

    std::string dir_path_std = dlg.dirPath().ascii();
    std::cerr << "Select side = "
              << ( side == rcsc::LEFT ? "left" : "right" )
              << "  debug log dir = [" << dir_path_std << "]"
              << std::endl;

    Options::instance().setDebugLogDir( dir_path_std );

    openDebugLogDir( side, dir_path_std );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::syncCycle()
{
    MonitorViewConstPtr ptr = M_main_data.getCurrentViewData();
    if ( ! ptr )
    {
        //std::cerr << "sync debug cycle. no view data." << std::endl;
        return;
    }

    //std::cerr << "sync debug data cycle" << std::endl;
    // seek current cycle data
    if ( ! M_main_data.getDebugLogHolder().seekData( M_tab_widget->currentPageIndex() + 1,
                                     ptr->cycle() ) )
    {
        std::cerr << "No data! number = "
                  << M_tab_widget->currentPageIndex() + 1
                  << std::endl;
        return;
    }

    updateMessage();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::decrementCycle()
{
    if ( ! M_main_data.getDebugLogHolder().decDataCycle( M_tab_widget->currentPageIndex() + 1 ) )
    {
        return;
    }

    updateMessage();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::incrementCycle()
{
    if ( ! M_main_data.getDebugLogHolder().incDataCycle( M_tab_widget->currentPageIndex() + 1 ) )
    {
        //std::cerr << "Failed increment debug log" << std::endl;
        return;
    }

    updateMessage();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::findExistString()
{
    QString expr = M_find_box->text();
    if ( ! expr.isEmpty() )
    {
        findString( expr );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::findString( const QString & expr )
{
    QTextEdit * edit = M_message[ M_tab_widget->currentPageIndex() ];
    bool forward = M_find_forward_rb->isOn();

    if ( ! edit->find( expr,
                       false, // case insentive
                       false, // no whole word
                       forward ) )
    {
        int para = 0;
        int index = 0;
        if ( ! M_find_forward_rb->isOn() )
        {
            para = edit->paragraphs();
            index = std::max( 0, edit->paragraphLength( para - 1 ) - 1 );
        }

        edit->find( expr, false, false, forward, &para, &index );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::toggleDebugLevel( int level )
{
    //M_level ^= DebugLogData::LEVEL_1; // reverse only 1 bit

    if ( level < 1 || 32 < level )
    {
        std::cerr << "Invalid level value " << level << std::endl;
        return;
    }

    boost::int32_t level_flag = 1 << ( level - 1 );

    M_main_data.getDebugLogHolder().setLevel( level_flag,
                                              M_debug_level_act[level-1]->isOn() );

    updateMessage();
    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::clearAll()
{
    M_main_data.getDebugLogHolder().clear();

    for ( int i = 0; i < 11; ++i )
    {
        M_message[i ]->clear();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::updateMessage()
{
    //std::cerr << "DebuMessageWindow::updateMessage()" << std::endl;

    const DebugLogHolder::DataList * data_list
        = M_main_data.getDebugLogHolder().getDataList( M_tab_widget->currentPageIndex() + 1 );
    if ( ! data_list )
    {
        std::cerr << "DebuMessageWindow::updateMessage() no data list" << std::endl;
        return;
    }

    const boost::int32_t level = M_main_data.getDebugLogHolder().level();
    if ( level == 0 )
    {
        std::cerr << "DebuMessageWindow::updateMessage() no debug level" << std::endl;
        return;
    }

    const int cur_cycle = M_main_data.getDebugLogHolder().getCurrentCycle();

    QTextEdit * text_edit = M_message[ M_tab_widget->currentPageIndex() ];

    // clear current data
    text_edit->clear();

    QString main_buffer;
    main_buffer.reserve( 1000 * 256 );
    int n_line = 0;

    int show_para = 0;
    const DebugLogHolder::DataList::const_iterator data_end = data_list->end();
    for ( DebugLogHolder::DataList::const_iterator data_it = data_list->begin();
          data_it != data_end;
          ++data_it )
    {
        if ( ! (*data_it) )
        {
            std::cerr << __FILE__ << ":" << __LINE__
                      << " Null data found!" << std::endl;
            continue;
        }

        if ( (*data_it)->cycle() <= cur_cycle )
        {
            //text_edit->getCursorPosition( &show_para, &show_index );
            show_para = text_edit->paragraphs() - 1;
        }

        QString cycle_line( "CYCLE " );
        cycle_line += QString::number( (*data_it)->cycle() );
        cycle_line += tr( " ------------------------------\n" );
        text_edit->append( cycle_line );

        const DebugLogData::TextCont::const_iterator text_end = (*data_it)->textCont().end();
        for ( DebugLogData::TextCont::const_iterator text_it = (*data_it)->textCont().begin();
              text_it != text_end;
              ++text_it )
        {
            // level check
            if ( level & text_it->level_ )
            {
                //main_buffer += QString::fromStdString( text_it->msg_ );
                //main_buffer += QString::fromAscii( text_it->msg_.c_str() );
                main_buffer += text_it->msg_;
                if ( ++n_line > 1000 )
                {
                    // append message to text control
                    text_edit->append( main_buffer );
                    main_buffer.remove( 0, main_buffer.length() );
                    n_line = 0;
                }
            }
        }

        if ( ! main_buffer.isEmpty() )
        {
            text_edit->append( main_buffer );
            main_buffer.remove( 0, main_buffer.length() );
        }
    }

    text_edit->setCursorPosition( show_para, 0 );
}
