// -*-c++-*-

/*!
  \file fedit_dialog.cpp
  \brief Formation editor operation Dialog class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <qt.h>

#include "fedit_dialog.h"

#include "fedit_data.h"

#include <iostream>

/*-------------------------------------------------------------------*/
/*!

*/
FEditDialog::FEditDialog( QWidget * parent )
    : QDialog( parent, "FEditDialog", false,
               WStyle_Customize | WStyle_Title
               | WStyle_DialogBorder | WStyle_SysMenu )
{
    this->setCaption( tr( "Edit Box" ) );
    this->setSizeGripEnabled( false );

    createWidgets();
    setDefaultValues();
}

/*-------------------------------------------------------------------*/
/*!

*/
FEditDialog::~FEditDialog()
{
    //std::cerr << "delete FEditDialog" << std::endl;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditDialog::createWidgets()
{
    QVBoxLayout * top_vbox = new QVBoxLayout( this );
    top_vbox->setMargin( 4 );

    // method name
    {
        QHBox * box = new QHBox( this );

        new QLabel( tr( "Training Method Name: " ), box );

        M_method_name = new QLineEdit( tr( "---" ), box );
        M_method_name->setReadOnly( true ); // no editable
        M_method_name->setBackgroundMode(  Qt::PaletteDark ); // Qt::PaletteMidlight

        top_vbox->addWidget( box );

    }

    // ball info
    {
        QHBox * box = new QHBox( this );
        const int alignment = ( Qt::AlignLeft | Qt::AlignVCenter );
        {
            QLabel * label = new QLabel( tr( "Ball" ), box );
            label->setAlignment( alignment );
            label->setMaximumSize( 40, this->fontMetrics().height() + 12 );
        }
        {
            QLabel * label = new QLabel( tr( " X:" ), box );
            label->setAlignment( alignment );
            label->setMaximumSize( 24, this->fontMetrics().height() + 12 );
        }
        {
            M_ball_pos_x = new QLineEdit( tr( "0" ), box );
            M_ball_pos_x->setMinimumSize( 48, 24 );
            M_ball_pos_x->setMaximumSize( 64, 24 );
            M_ball_pos_x->setValidator( new QDoubleValidator( -54.5, 54.5, 2, M_ball_pos_x ) );
        }
        {
            QLabel * label = new QLabel( tr( " Y:" ), box );
            label->setAlignment( alignment );
            label->setMaximumSize( 24, this->fontMetrics().height() + 12 );
        }
        {
            M_ball_pos_y = new QLineEdit( tr( "0" ), box );
            M_ball_pos_y->setMinimumSize( 48, 24 );
            M_ball_pos_y->setMaximumSize( 64, 24 );
            M_ball_pos_y->setValidator( new QDoubleValidator( -34.0, 34.0, 2, M_ball_pos_y ) );
        }
//         {
//             QRect rect = this->fontMetrics().boundingRect( tr( "ReverseY" ) );
//             QPushButton * btn = new QPushButton( tr( "ReverseY" ), box );
//             //btn->setMinimumSize( rect.width() + 12, rect.height() + 12 );
//             connect( btn, SIGNAL( clicked() ),
//                      this, SLOT( reverseY() ) );
//         }

        top_vbox->addWidget( box );
    }

    const int unum_col_width = this->fontMetrics().width( tr( "Unum" ) ) + 4;
    const int symmetry_col_width = this->fontMetrics().width( tr( "Symmetry" ) ) + 4;

    // label info
    {
        QHBox * box = new QHBox( this );
        {
            QLabel * label = new QLabel( tr( "Unum" ), box );
            label->setAlignment( Qt::AlignCenter );
            label->setMinimumSize( unum_col_width, 24 );
            label->setMaximumSize( unum_col_width, 24 );
        }
        {
            QLabel * label = new QLabel( tr( "Symmetry" ), box );
            label->setAlignment( Qt::AlignCenter );
            label->setMinimumSize( symmetry_col_width, 24 );
            label->setMaximumSize( symmetry_col_width, 24 );
        }
        {
            QLabel * label = new QLabel( tr( "Role" ), box );
            label->setMaximumSize( this->fontMetrics().width( tr( "CenterForward" ) ) + 4,
                                   24 );
            label->setAlignment( Qt::AlignCenter );
        }
        {
            QLabel * label = new QLabel( tr( "X" ), box );
            label->setMaximumSize( this->fontMetrics().width( tr( "-00.0000" ) ) + 4,
                                   24 );
            label->setAlignment( Qt::AlignCenter );
        }
        {
            QLabel * label = new QLabel( tr( "Y" ), box );
            label->setMaximumSize( this->fontMetrics().width( tr( "-00.0000" ) ) + 4,
                                   24 );
            label->setAlignment( Qt::AlignCenter );
        }

        top_vbox->addWidget( box );
    }

    for ( int i = 0; i < 11; ++i )
    {
        QHBox * box = new QHBox( this );
        {
            QLabel * label = new QLabel( QString().sprintf( "%2d", i + 1 ),
                                         box );
            label->setAlignment( Qt::AlignCenter | Qt::AlignVCenter );
            label->setMinimumSize( unum_col_width, 24 );
            label->setMaximumSize( unum_col_width + 8, 24 );
        }
        {
            M_symmetry_unum[i] = new QLineEdit( tr( "0" ), box );
            M_symmetry_unum[i]->setMaximumSize( symmetry_col_width, 24 );
            M_symmetry_unum[i]->setMinimumSize( symmetry_col_width, 24 );
        }
        {
            M_role_name[i] = new QLineEdit( tr( "Role" ), box );
            M_role_name[i]->setMaximumSize( this->fontMetrics().width( tr( "CenterForward" ) ) + 4,
                                            24 );
            //M_role_name[i]->setMaximumSize( 80, 24 );
        }
        {
            M_pos_x[i] = new QLineEdit( tr( "0.0" ), box );
            M_pos_x[i]->setMaximumSize( this->fontMetrics().width( tr( "-00.0000" ) ) + 4,
                                        24 );
            M_pos_x[i]->setValidator( new QDoubleValidator( -54.5, 54.5, 2, M_pos_x[i] ) );
        }
        {
            M_pos_y[i] = new QLineEdit( tr( "0.0" ), box );
            M_pos_y[i]->setMaximumSize( this->fontMetrics().width( tr( "-00.0000" ) ) + 4,
                                        24 );
            M_pos_y[i]->setValidator( new QDoubleValidator( -36.0, 36.0, 2, M_pos_y[i] ) );
        }

        top_vbox->addWidget( box );
    }

    {
        QHBoxLayout * layout = new QHBoxLayout( top_vbox );

        QPushButton * btn = new QPushButton( tr( "Apply" ), this );
        connect( btn, SIGNAL( clicked() ),
                 this, SLOT( applyToField() ) );

        layout->addWidget( btn, 0, Qt::AlignLeft );
    }

}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditDialog::setDefaultValues()
{
    for ( int i = 0; i < 11; ++i )
    {
        M_symmetry_unum[i]->setText( tr( "0" ) );
        M_role_name[i]->setText( tr( "Role" ) );
        M_pos_x[i]->setText( tr( "0.0" ) );
        M_pos_y[i]->setText( tr( "0.0" ) );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditDialog::closeEvent( QCloseEvent * event )
{
    event->accept();
    emit shown( false );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditDialog::updateData()
{
    boost::shared_ptr< FEditData > data_ptr = M_data_ptr.lock();
    if ( ! data_ptr )
    {
        std::cerr << "FEditDialog::update  no data ptr" << std::endl;
        return;
    }

    const boost::shared_ptr< const rcsc::Formation > f = data_ptr->formation();
    if ( ! f )
    {
        std::cerr << "FEditDialog::update  no formation data" << std::endl;
        return;
    }

    M_method_name->setText( QString::fromAscii( f->methodName().c_str() ) );

    char buf[32];

    // ball info
    snprintf( buf, 32, "%.2f", data_ptr->ball().x );
    M_ball_pos_x->setText( QString::fromAscii( buf ) );

    snprintf( buf, 32, "%.2f", data_ptr->ball().y );
    M_ball_pos_y->setText( QString::fromAscii( buf ) );

    // player info
    const std::vector< rcsc::Vector2D > & positions = data_ptr->ourPlayers();
    const int MAX = positions.size();
    for ( int unum = 1; unum <= MAX; ++unum )
    {
        //const bool symmetry = f->isSymmetryType( unum );

        snprintf( buf, 32, "%2d", f->getSymmetryNumber( unum ) );
        M_symmetry_unum[unum - 1]->setText( QString::fromAscii( buf ) );

        M_role_name[unum - 1]->setText( QString::fromAscii( f->getRoleName( unum ).c_str() ) );

        snprintf( buf, 32, "%.2f", positions[unum - 1].x );
        M_pos_x[unum - 1]->setText( QString::fromAscii( buf ) );

        snprintf( buf, 32, "%.2f", positions[unum - 1].y );
        M_pos_y[unum - 1]->setText( QString::fromAscii( buf ) );

        //const Qt::BackgroundMode mode = ( symmetry
        //                                  ? Qt::PaletteDark
        //                                  : Qt::PaletteBackground );
        //M_role_name[unum - 1]->setReadOnly( symmetry );
        //M_role_name[unum - 1]->setBackgroundMode( mode );
        //M_pos_x[unum - 1]->setBackgroundMode( mode );
        //M_pos_y[unum - 1]->setBackgroundMode( mode );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditDialog::reverseY()
{
    boost::shared_ptr< FEditData > data_ptr = M_data_ptr.lock();
    if ( ! data_ptr )
    {
        std::cerr << "FEditDialog::handleSymmetryBall  no data ptr" << std::endl;
        return;
    }

    data_ptr->reverseY();

    emit viewUpdated();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditDialog::applyToField()
{
    //std::cerr << "FEditDialog::applyToField" << std::endl;

    boost::shared_ptr< FEditData > data_ptr = M_data_ptr.lock();
    if ( ! data_ptr )
    {
        std::cerr << "FEditDialog::handleApplyToField  no data ptr" << std::endl;
        return;
    }

    // ball
    {
        bool ok_x = false;
        bool ok_y = false;
        double x = M_ball_pos_x->text().toDouble( &ok_x );
        double y = M_ball_pos_y->text().toDouble( &ok_y );
        if ( ok_x && ok_y )
        {
            data_ptr->setBallPosition( x, y );
        }
    }

    // players
    for ( int unum = 1; unum <= 11; ++unum )
    {
        bool ok = false;
        int symmetry_unum = M_symmetry_unum[unum-1]->text().toInt( &ok );
        if ( ! ok )
        {
            std::cerr << __FILE__ << ':' << __LINE__
                      << " *** ERROR *** Invalid symmetry number."
                      << std::endl;
            continue;
        }

        std::string role_name = M_role_name[unum-1]->text().ascii();

        if ( role_name.empty() )
        {
            std::cerr << __FILE__ << ':' << __LINE__
                      << " *** ERROR *** Empty role name."
                      << std::endl;
            continue;
        }

        data_ptr->updateRoleData( unum, symmetry_unum, role_name );

        bool ok_x = false;
        bool ok_y = false;
        double x = M_pos_x[unum-1]->text().toDouble( &ok_x );
        double y = M_pos_y[unum-1]->text().toDouble( &ok_y );
        if ( ok_x && ok_y )
        {
            data_ptr->setPlayerPosition( unum, x, y );
        }
    }

    updateData();

    emit viewUpdated();
}
