// -*-c++-*-

/*!
  \file field_painter.cpp
  \brief field painter class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <qt.h>

#include "field_painter.h"

#include "draw_config.h"
#include "options.h"

#include <rcsc/common/server_param.h>

#include <iostream>

/*-------------------------------------------------------------------*/
/*!

 */
FieldPainter::FieldPainter()
{

}

/*-------------------------------------------------------------------*/
/*!

 */
void
FieldPainter::draw( QPainter & painter )
{
    drawBackGround( painter );
    drawLines( painter );
    drawPenaltyAreaLines( painter );
    drawGoalAreaLines( painter );
    drawGoals( painter );

    if ( Options::instance().showFlags() )
    {
        drawFlags( painter );
    }

    if ( Options::instance().gridStep() > 0 )
    {
        drawGrid( painter );
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
FieldPainter::drawBackGround( QPainter & painter ) const
{
    // fill the whole region
    painter.fillRect( painter.window(),
                      DrawConfig::instance().fieldBrush() );

    ////////////////////////////////////////////////////////////////
    switch ( Options::instance().fieldGrassType() ) {
    case Options::GRASS_LINES:
        drawLinesGrass( painter );
        break;
    case Options::GRASS_CHECKER:
        drawCheckerGrass( painter );
        break;
    default:
        break;
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
FieldPainter::drawLinesGrass( QPainter & painter ) const
{
    const rcsc::ServerParam & SP = rcsc::ServerParam::i();
    const Options & opt = Options::instance();
    const DrawConfig & dconf = DrawConfig::instance();

    const int left_x = opt.fieldCenterInt().x - opt.scaleInt( SP.pitchHalfLength() );
    const int top_y = opt.fieldCenterInt().y - opt.scaleInt( SP.pitchHalfWidth() );
    const int right_x = opt.fieldCenterInt().x + opt.scaleInt( SP.pitchHalfLength() );
    const int bottom_y = opt.fieldCenterInt().y + opt.scaleInt( SP.pitchHalfWidth() );

    const int grid = opt.scaleInt( 10.0 ); // grid length

    painter.setPen( dconf.transparentPen() );
    //painter.setBrush( dconf.fieldDarkBrush() );

    int i = opt.fieldCenterInt().x - grid/2;
    int cnt = 0;
    while ( i > left_x )
    {
        if ( ! ( cnt % 2 ) )
        {
            int left = i - grid;
            painter.fillRect( left, top_y - grid/2,
                              i - left, bottom_y - top_y + grid,
                              dconf.fieldDarkBrush() );
        }
        i -= grid;
        ++cnt;
    }

    i = opt.fieldCenterInt().x + grid/2;
    cnt = 0;
    while ( i < right_x )
    {
        if ( ! ( cnt % 2 ) )
        {
            int right = i + grid;
            painter.fillRect( i, top_y - grid/2,
                              right - i, bottom_y - top_y + grid,
                              dconf.fieldDarkBrush() );
        }
        i += grid;
        ++cnt;
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
FieldPainter::drawCheckerGrass( QPainter & painter )const
{
    const rcsc::ServerParam & SP = rcsc::ServerParam::i();
    const Options & opt = Options::instance();
    const DrawConfig & dconf = DrawConfig::instance();

    painter.setPen( dconf.transparentPen() );
    painter.setBrush( dconf.fieldDarkBrush() );

    const int left_x = opt.fieldCenterInt().x - opt.scaleInt( SP.pitchHalfLength() );
    const int top_y = opt.fieldCenterInt().y - opt.scaleInt( SP.pitchHalfWidth() );
    const int right_x = opt.fieldCenterInt().x + opt.scaleInt( SP.pitchHalfLength() );
    const int bottom_y = opt.fieldCenterInt().y + opt.scaleInt( SP.pitchHalfWidth() );

    const int grid = opt.scaleInt( 10.0 ); // grid length

    int flag = 0;
    for ( int i = opt.fieldCenterInt().x; i > left_x; i -= grid )
    {
        int cnt = flag;
        int left = i - grid;
        for ( int j = opt.fieldCenterInt().y; j > top_y; j -= grid )
        {
            if ( cnt % 2 )
            {
                int top = j - grid;
                painter.fillRect( left, top, i - left, j - top,
                                  dconf.fieldDarkBrush() );
            }
            ++cnt;
        }
        ++flag;
    }

    flag = 1;
    for ( int i = opt.fieldCenterInt().x;  i > left_x; i -= grid )
    {
        int cnt = flag;
        int left = i - grid;
        for ( int j = opt.fieldCenterInt().y; j < bottom_y; j += grid )
        {
            if ( cnt % 2 )
            {
                int bottom = j + grid;
                painter.fillRect( left, j, i - left, bottom - j,
                                  dconf.fieldDarkBrush() );
            }
            ++cnt;
        }
        ++flag;
    }
    flag = 0;
    for ( int i = opt.fieldCenterInt().x;  i < right_x; i += grid )
    {
        int cnt = flag;
        int right = i + grid;
        for ( int j = opt.fieldCenterInt().y; j < bottom_y; j += grid )
        {
            if ( cnt % 2 )
            {
                int bottom = j + grid;
                painter.fillRect( i, j, right - i, bottom - j,
                                  dconf.fieldDarkBrush() );
            }
            ++cnt;
        }
        ++flag;
    }
    flag = 1;
    for ( int i = opt.fieldCenterInt().x;  i < right_x; i += grid )
    {
        int cnt = flag;
        int right = i + grid;
        for ( int j = opt.fieldCenterInt().y; j > top_y; j -= grid )
        {
            if ( cnt % 2 )
            {
                int top = j - grid;
                painter.fillRect( i, top, right - i, j - top,
                                  dconf.fieldDarkBrush() );
            }
            ++cnt;
        }
        ++flag;
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
FieldPainter::drawLines( QPainter & painter ) const
{
    const rcsc::ServerParam & SP = rcsc::ServerParam::i();
    const Options & opt = Options::instance();
    const DrawConfig & dconf = DrawConfig::instance();

    // set gdi objects
    painter.setPen( dconf.linePen() );
    painter.setBrush( dconf.transparentBrush() );

    // set screen coordinates of field
    int left_x   = opt.absScreenXInt( - SP.pitchHalfLength() );
    int right_x  = opt.absScreenXInt( + SP.pitchHalfLength() );
    int top_y    = opt.absScreenYInt( - SP.pitchHalfWidth() );
    int bottom_y = opt.absScreenYInt( + SP.pitchHalfWidth() );

    // side lines & goal lines
    painter.drawLine( left_x, top_y, right_x, top_y );
    painter.drawLine( right_x, top_y, right_x, bottom_y );
    painter.drawLine( right_x, bottom_y, left_x, bottom_y );
    painter.drawLine( left_x, bottom_y, left_x, top_y );

    if ( opt.keepawayMode()
         || SP.keepawayMode() )
    {
        // keepaway area
        int ka_left = opt.absScreenXInt( - SP.keepawayLength() * 0.5 );
        int ka_top = opt.absScreenYInt( - SP.keepawayWidth() * 0.5 );
        int ka_length = opt.scaleInt( SP.keepawayLength() );
        int ka_width = opt.scaleInt( SP.keepawayWidth() );

        painter.drawRect( ka_left, ka_top, ka_length, ka_width );
    }
    else
    {
        // center line
        painter.drawLine( opt.fieldCenterInt().x, top_y,
                          opt.fieldCenterInt().x, bottom_y );

        // center circle
        int center_radius = opt.scaleInt( SP.centerCircleR() );
        painter.drawEllipse( opt.fieldCenterInt().x - center_radius,
                             opt.fieldCenterInt().y - center_radius,
                             center_radius * 2,
                             center_radius * 2 );
    }

    // corner arc
    {
        int r = opt.scaleInt( SP.cornerArcR() );

        painter.drawArc( left_x - r, top_y - r, r * 2, r * 2,
                         -90*16, 90*16 );
        painter.drawArc( left_x - r, bottom_y - r, r * 2, r * 2,
                         0*16, 90*16 );
        painter.drawArc( right_x - r, bottom_y - r, r * 2, r * 2,
                         90*16, 90*16 );
        painter.drawArc( right_x - r, top_y - r, r * 2, r * 2,
                         180*16, 90*16 );
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
FieldPainter::drawPenaltyAreaLines( QPainter & painter ) const
{
    const rcsc::ServerParam & SP = rcsc::ServerParam::i();
    const Options & opt = Options::instance();
    const DrawConfig & dconf = DrawConfig::instance();

    // set gdi objects
    painter.setPen( dconf.linePen() );
    painter.setBrush( dconf.transparentBrush() );

    // set absScreen coordinates of field
    int left_x   = opt.absScreenXInt( - SP.pitchHalfLength() );
    int right_x  = opt.absScreenXInt( + SP.pitchHalfLength() );

    // set penalty area params
    int pen_top_y    = opt.absScreenYInt( - SP.penaltyAreaHalfWidth() );
    int pen_bottom_y = opt.absScreenYInt( + SP.penaltyAreaHalfWidth() );
    double pen_circle_y_degree_abs
        = std::acos( ( SP.penaltyAreaLength()
                       - rcsc::ServerParam::DEFAULT_PENALTY_SPOT_DIST )
                     / rcsc::ServerParam::DEFAULT_PENALTY_CIRCLE_R )
        * ( 180.0 / M_PI );
    int span_angle = qRound( pen_circle_y_degree_abs * 2.0 * 16 );
    int pen_circle_r = opt.scaleInt( rcsc::ServerParam::DEFAULT_PENALTY_CIRCLE_R );
    int pen_circle_size = opt.scaleInt( rcsc::ServerParam::DEFAULT_PENALTY_CIRCLE_R * 2.0 );

    // left penalty area X
    int pen_x = opt.absScreenXInt( -( SP.pitchHalfLength()
                                   - SP.penaltyAreaLength() ) );
    // left arc
    int pen_spot_x = opt.absScreenXInt( -( SP.pitchHalfLength()
                                        - rcsc::ServerParam::DEFAULT_PENALTY_SPOT_DIST ) );
    painter.drawArc( pen_spot_x - pen_circle_r + 1,
                     opt.fieldCenterInt().y - pen_circle_r,
                     pen_circle_size,
                     pen_circle_size,
                     qRound( -pen_circle_y_degree_abs * 16 ),
                     span_angle );
    // left rectangle
    painter.drawLine( left_x, pen_top_y, pen_x, pen_top_y );
    painter.drawLine( pen_x, pen_top_y, pen_x, pen_bottom_y );
    painter.drawLine( pen_x, pen_bottom_y, left_x, pen_bottom_y );
    // left spot
    painter.drawPoint( pen_spot_x, opt.fieldCenterInt().y );

    // right penalty area X
    pen_x = opt.absScreenXInt( +( SP.pitchHalfLength()
                               - SP.penaltyAreaLength() ) );
    // right arc
    pen_spot_x = opt.absScreenXInt( +( SP.pitchHalfLength()
                                    - rcsc::ServerParam::DEFAULT_PENALTY_SPOT_DIST ) );
    painter.drawArc( pen_spot_x - pen_circle_r,
                     opt.fieldCenterInt().y - pen_circle_r,
                     pen_circle_size, pen_circle_size,
                     qRound( ( 180.0 - pen_circle_y_degree_abs + 0.5 ) * 16 ),
                     span_angle );
    // right rectangle
    painter.drawLine( right_x, pen_top_y, pen_x, pen_top_y );
    painter.drawLine( pen_x, pen_top_y, pen_x, pen_bottom_y );
    painter.drawLine( pen_x, pen_bottom_y, right_x, pen_bottom_y );
    // right spot
    painter.drawPoint( pen_spot_x, opt.fieldCenterInt().y );
}

/*-------------------------------------------------------------------*/
/*!

 */
void
FieldPainter::drawGoalAreaLines( QPainter & painter ) const
{
    const rcsc::ServerParam & SP = rcsc::ServerParam::i();
    const Options & opt = Options::instance();
    const DrawConfig & dconf = DrawConfig::instance();

    // set gdi objects
    painter.setPen( dconf.linePen() );
    painter.setBrush( dconf.transparentBrush() );

    // set absScreen coordinates of field
    int left_x   = opt.absScreenXInt( - SP.pitchHalfLength() );
    int right_x  = opt.absScreenXInt( + SP.pitchHalfLength() );

    // set coordinates opts
    int goal_area_y_abs = opt.scaleInt( SP.goalAreaHalfWidth() );
    int goal_area_top_y = opt.fieldCenterInt().y - goal_area_y_abs;
    int goal_area_bottom_y = opt.fieldCenterInt().y + goal_area_y_abs;

    // left goal area
    int goal_area_x = opt.absScreenXInt( - SP.pitchHalfLength()
                                      + SP.goalAreaLength() );
    painter.drawLine( left_x, goal_area_top_y, goal_area_x, goal_area_top_y );
    painter.drawLine( goal_area_x, goal_area_top_y, goal_area_x, goal_area_bottom_y );
    painter.drawLine( goal_area_x, goal_area_bottom_y, left_x, goal_area_bottom_y );

    // right goal area
    goal_area_x = opt.absScreenXInt( SP.pitchHalfLength()
                                  - SP.goalAreaLength() );
    painter.drawLine( right_x, goal_area_top_y, goal_area_x, goal_area_top_y );
    painter.drawLine( goal_area_x, goal_area_top_y, goal_area_x, goal_area_bottom_y );
    painter.drawLine( goal_area_x, goal_area_bottom_y, right_x, goal_area_bottom_y );
}

/*-------------------------------------------------------------------*/
/*!

 */
void
FieldPainter::drawGoals( QPainter & painter ) const
{
    const rcsc::ServerParam & SP = rcsc::ServerParam::i();
    const Options & opt = Options::instance();

    // set gdi objects
    painter.setPen( Qt::black );
    painter.setBrush( Qt::black );

    // set coordinates param
    int goal_top_y = opt.absScreenYInt( - SP.goalHalfWidth() );
    int goal_size_x = opt.scaleInt( SP.goalDepth() );
    int goal_size_y = opt.scaleInt( SP.goalWidth() );

    int post_top_y = opt.absScreenYInt( - SP.goalHalfWidth()
                                     - SP.goalPostRadius() * 2.0 );
    int post_bottom_y = opt.absScreenYInt( + SP.goalHalfWidth() );
    int post_diameter = opt.scaleInt( SP.goalPostRadius() * 2.0 );

    // left goal
    painter.drawRect( opt.absScreenXInt( - SP.pitchHalfLength()
                                      - SP.goalDepth() ) - 1,
                      goal_top_y,
                      goal_size_x,
                      goal_size_y );
    if ( post_diameter >= 1 )
    {
        int post_x = opt.absScreenXInt( - SP.pitchHalfLength() );
        painter.drawEllipse( post_x,
                             post_top_y,
                             post_diameter,
                             post_diameter );
        painter.drawEllipse( post_x,
                             post_bottom_y,
                             post_diameter,
                             post_diameter );
    }
    // right goal
    painter.drawRect( opt.absScreenXInt( SP.pitchHalfLength() ) + 1,
                      goal_top_y,
                      goal_size_x,
                      goal_size_y );
    if ( post_diameter >= 1 )
    {
        int post_x = opt.absScreenXInt( SP.pitchHalfLength()
                                     - SP.goalPostRadius() * 2.0 );
        painter.drawEllipse( post_x,
                             post_top_y,
                             post_diameter,
                             post_diameter );
        painter.drawEllipse( post_x,
                             post_bottom_y,
                             post_diameter,
                             post_diameter );
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
FieldPainter::drawFlags( QPainter & painter ) const
{
    const rcsc::ServerParam & SP = rcsc::ServerParam::i();
    const Options & opt = Options::instance();

    // set gdi objects
    painter.setPen( DrawConfig::instance().linePen() );
    painter.setBrush( DrawConfig::instance().transparentBrush() );

    // set size or coordinates params
    int flag_radius = opt.scaleInt( 0.5 );
    if ( flag_radius < 2 ) flag_radius = 2;
    if ( flag_radius > 5 ) flag_radius = 5;
    int flag_diameter = flag_radius * 2;

    int x, y;
    int pitch_half_length = opt.scaleInt( SP.pitchHalfLength() );
    int pitch_half_width = opt.scaleInt( SP.pitchHalfWidth() );
    int pitch_margin_x = opt.scaleInt( SP.pitchHalfLength()
                                    + rcsc::ServerParam::DEFAULT_PITCH_MARGIN );
    int pitch_margin_y = opt.scaleInt( SP.pitchHalfWidth()
                                    + rcsc::ServerParam::DEFAULT_PITCH_MARGIN );
    int penalty_x = opt.scaleInt( SP.pitchHalfLength()
                               - SP.penaltyAreaLength() );
    int penalty_y = opt.scaleInt( SP.penaltyAreaHalfWidth() );
    int goal_y = opt.scaleInt( SP.goalHalfWidth() );
    int scale10 = opt.scaleInt( 10.0 );
    int scale20 = opt.scaleInt( 20.0 );
    int scale30 = opt.scaleInt( 30.0 );
    int scale40 = opt.scaleInt( 40.0 );
    int scale50 = opt.scaleInt( 50.0 );

    // goal left
    x = opt.fieldCenterInt().x - pitch_half_length - flag_radius;
    y = opt.fieldCenterInt().y - flag_radius;
    painter.drawEllipse( x, y, flag_diameter, flag_diameter );
    // goal right
    x = opt.fieldCenterInt().x + pitch_half_length - flag_radius;
    y = opt.fieldCenterInt().y - flag_radius;
    painter.drawEllipse( x, y, flag_diameter, flag_diameter );
    // flag c
    x  = opt.fieldCenterInt().x - flag_radius;
    y = opt.fieldCenterInt().y - flag_radius;
    painter.drawEllipse( x, y, flag_diameter, flag_diameter );
    // flag c t
    x = opt.fieldCenterInt().x - flag_radius;
    y = opt.fieldCenterInt().y - pitch_half_width - flag_radius;
    painter.drawEllipse( x, y, flag_diameter, flag_diameter );

    // flag c b
    x = opt.fieldCenterInt().x - flag_radius;
    y = opt.fieldCenterInt().y + pitch_half_width - flag_radius;
    painter.drawEllipse( x, y, flag_diameter, flag_diameter );
    // flag l t
    x = opt.fieldCenterInt().x - pitch_half_length - flag_radius;
    y = opt.fieldCenterInt().y - pitch_half_width - flag_radius;
    painter.drawEllipse( x, y, flag_diameter, flag_diameter );
    // flag l b
    x = opt.fieldCenterInt().x - pitch_half_length - flag_radius;
    y = opt.fieldCenterInt().y + pitch_half_width - flag_radius;
    painter.drawEllipse( x, y, flag_diameter, flag_diameter );
    // flag r t
    x = opt.fieldCenterInt().x + pitch_half_length - flag_radius;
    y = opt.fieldCenterInt().y - pitch_half_width - flag_radius;
    painter.drawEllipse( x, y, flag_diameter, flag_diameter );
    // flag r b
    x = opt.fieldCenterInt().x + pitch_half_length - flag_radius;
    y = opt.fieldCenterInt().y + pitch_half_width - flag_radius;
    painter.drawEllipse( x, y, flag_diameter, flag_diameter );
    // flag p l t
    x = opt.fieldCenterInt().x - penalty_x - flag_radius;
    y = opt.fieldCenterInt().y - penalty_y - flag_radius;
    painter.drawEllipse( x, y, flag_diameter, flag_diameter );
    // flag p l c
    x = opt.fieldCenterInt().x - penalty_x - flag_radius;
    y = opt.fieldCenterInt().y - flag_radius;
    painter.drawEllipse( x, y, flag_diameter, flag_diameter );
    // flag p l b
    x = opt.fieldCenterInt().x - penalty_x - flag_radius;
    y = opt.fieldCenterInt().y + penalty_y - flag_radius;
    painter.drawEllipse( x, y, flag_diameter, flag_diameter );
    // flag p r t
    x = opt.fieldCenterInt().x + penalty_x - flag_radius;
    y = opt.fieldCenterInt().y - penalty_y - flag_radius;
    painter.drawEllipse( x, y, flag_diameter, flag_diameter );
    // flag p r c
    x = opt.fieldCenterInt().x + penalty_x - flag_radius;
    y = opt.fieldCenterInt().y - flag_radius;
    painter.drawEllipse( x, y, flag_diameter, flag_diameter );
    // flag p r b
    x = opt.fieldCenterInt().x + penalty_x - flag_radius;
    y = opt.fieldCenterInt().y + penalty_y - flag_radius;
    painter.drawEllipse( x, y, flag_diameter, flag_diameter );

    // flag g l t
    x = opt.fieldCenterInt().x - pitch_half_length - flag_radius;
    y = opt.fieldCenterInt().y - goal_y - flag_radius;
    painter.drawEllipse( x, y, flag_diameter, flag_diameter );
    // flag g l b
    x = opt.fieldCenterInt().x - pitch_half_length - flag_radius;
    y = opt.fieldCenterInt().y + goal_y - flag_radius;
    painter.drawEllipse( x, y, flag_diameter, flag_diameter );
    // flag g r t
    x = opt.fieldCenterInt().x + pitch_half_length - flag_radius;
    y = opt.fieldCenterInt().y - goal_y - flag_radius;
    painter.drawEllipse( x, y, flag_diameter, flag_diameter );
    // flag g r b
    x = opt.fieldCenterInt().x + pitch_half_length - flag_radius;
    y = opt.fieldCenterInt().y + goal_y - flag_radius;
    painter.drawEllipse( x, y, flag_diameter, flag_diameter );

    // flag t ...

    y = opt.fieldCenterInt().y - pitch_margin_y - flag_radius;
    // flag t l 50
    x = opt.fieldCenterInt().x - scale50 - flag_radius;
    painter.drawEllipse( x, y, flag_diameter, flag_diameter );
    // flag t l 40
    x = opt.fieldCenterInt().x - scale40 - flag_radius;
    painter.drawEllipse( x, y, flag_diameter, flag_diameter );
    // flag t l 30
    x = opt.fieldCenterInt().x - scale30 - flag_radius;
    painter.drawEllipse( x, y, flag_diameter, flag_diameter );
    // flag t l 20
    x = opt.fieldCenterInt().x - scale20 - flag_radius;
    painter.drawEllipse( x, y, flag_diameter, flag_diameter );
    // flag t l 10
    x = opt.fieldCenterInt().x - scale10 - flag_radius;
    painter.drawEllipse( x, y, flag_diameter, flag_diameter );
    // flag t 0
    x = opt.fieldCenterInt().x - flag_radius;
    painter.drawEllipse( x, y, flag_diameter, flag_diameter );
    // flag t r 10
    x = opt.fieldCenterInt().x + scale10 - flag_radius;
    painter.drawEllipse( x, y, flag_diameter, flag_diameter );
    // flag t r 20
    x = opt.fieldCenterInt().x + scale20 - flag_radius;
    painter.drawEllipse( x, y, flag_diameter, flag_diameter );
    // flag t r 30
    x = opt.fieldCenterInt().x + scale30 - flag_radius;
    painter.drawEllipse( x, y, flag_diameter, flag_diameter );
    // flag t r 40
    x = opt.fieldCenterInt().x + scale40 - flag_radius;
    painter.drawEllipse( x, y, flag_diameter, flag_diameter );
    // flag t r 50
    x = opt.fieldCenterInt().x + scale50 - flag_radius;
    painter.drawEllipse( x, y, flag_diameter, flag_diameter );

    // flag b ...

    y = opt.fieldCenterInt().y + pitch_margin_y - flag_radius;
    // flag b l 50
    x = opt.fieldCenterInt().x - scale50 - flag_radius;
    painter.drawEllipse( x, y, flag_diameter, flag_diameter );
    // flag b l 40
    x = opt.fieldCenterInt().x - scale40 - flag_radius;
    painter.drawEllipse( x, y, flag_diameter, flag_diameter );
    // flag b l 30
    x = opt.fieldCenterInt().x - scale30 - flag_radius;
    painter.drawEllipse( x, y, flag_diameter, flag_diameter );
    // flag b l 20
    x = opt.fieldCenterInt().x - scale20 - flag_radius;
    painter.drawEllipse( x, y, flag_diameter, flag_diameter );
    // flag b l 10
    x = opt.fieldCenterInt().x - scale10 - flag_radius;
    painter.drawEllipse( x, y, flag_diameter, flag_diameter );
    // flag b 0
    x = opt.fieldCenterInt().x - flag_radius;
    painter.drawEllipse( x, y, flag_diameter, flag_diameter );
    // flag b r 10
    x = opt.fieldCenterInt().x + scale10 - flag_radius;
    painter.drawEllipse( x, y, flag_diameter, flag_diameter );
    // flag b r 20
    x = opt.fieldCenterInt().x + scale20 - flag_radius;
    painter.drawEllipse( x, y, flag_diameter, flag_diameter );
    // flag b r 30
    x = opt.fieldCenterInt().x + scale30 - flag_radius;
    painter.drawEllipse( x, y, flag_diameter, flag_diameter );
    // flag b r 40
    x = opt.fieldCenterInt().x + scale40 - flag_radius;
    painter.drawEllipse( x, y, flag_diameter, flag_diameter );
    // flag b r 50
    x = opt.fieldCenterInt().x + scale50 - flag_radius;
    painter.drawEllipse( x, y, flag_diameter, flag_diameter );

    // flag l ...

    x = opt.fieldCenterInt().x - pitch_margin_x - flag_radius;
    // flag l t 30
    y = opt.fieldCenterInt().y - scale30 - flag_radius;
    painter.drawEllipse( x, y, flag_diameter, flag_diameter );
    // flag l t 20
    y = opt.fieldCenterInt().y - scale20 - flag_radius;
    painter.drawEllipse( x, y, flag_diameter, flag_diameter );
    // flag l t 10
    y = opt.fieldCenterInt().y - scale10 - flag_radius;
    painter.drawEllipse( x, y, flag_diameter, flag_diameter );
    // flag l 0
    y = opt.fieldCenterInt().y - flag_radius;
    painter.drawEllipse( x, y, flag_diameter, flag_diameter );
    // flag l b 10
    y = opt.fieldCenterInt().y + scale10 - flag_radius;
    painter.drawEllipse( x, y, flag_diameter, flag_diameter );
    // flag l b 20
    y = opt.fieldCenterInt().y + scale20 - flag_radius;
    painter.drawEllipse( x, y, flag_diameter, flag_diameter );
    // flag l b 30
    y = opt.fieldCenterInt().y + scale30 - flag_radius;
    painter.drawEllipse( x, y, flag_diameter, flag_diameter );

    // flag r ...

    x = opt.fieldCenterInt().x + pitch_margin_x - flag_radius;
    // flag r t 30
    y = opt.fieldCenterInt().y - scale30 - flag_radius;
    painter.drawEllipse( x, y, flag_diameter, flag_diameter );
    // flag r t 20
    y = opt.fieldCenterInt().y - scale20 - flag_radius;
    painter.drawEllipse( x, y, flag_diameter, flag_diameter );
    // flag r t 10
    y = opt.fieldCenterInt().y - scale10 - flag_radius;
    painter.drawEllipse( x, y, flag_diameter, flag_diameter );
    // flag r 0
    y = opt.fieldCenterInt().y - flag_radius;
    painter.drawEllipse( x, y, flag_diameter, flag_diameter );
    // flag r b 10
    y = opt.fieldCenterInt().y + scale10 - flag_radius;
    painter.drawEllipse( x, y, flag_diameter, flag_diameter );
    // flag r b 20
    y = opt.fieldCenterInt().y + scale20 - flag_radius;
    painter.drawEllipse( x, y, flag_diameter, flag_diameter );
    // flag r b 30
    y = opt.fieldCenterInt().y + scale30 - flag_radius;
    painter.drawEllipse( x, y, flag_diameter, flag_diameter );
}

/*-------------------------------------------------------------------*/
/*!

 */
void
FieldPainter::drawGrid( QPainter & painter ) const
{
    const Options & opt = Options::instance();

    const double grid_step = 0.1 * opt.gridStep();
    const int istep = opt.scaleInt( grid_step );

    if ( istep <= 4 )
    {
        return;
    }

    const QFontMetrics metrics = painter.fontMetrics();
    const int text_step_x = ( opt.showGridCoord()
                              ? metrics.width( QObject::tr( "-00.0" ) )
                              : 100000 );
    const int text_step_y = ( opt.showGridCoord()
                              ? metrics.ascent()
                              : 100000 );

    const QRect win = painter.window();

    const int max_ix = win.right();
    const int min_ix = win.left();
    const int max_iy = win.bottom();
    const int min_iy = win.top();
    const double max_x = opt.fieldX( max_ix );
    const double min_x = opt.fieldX( min_ix );
    const double max_y = opt.fieldY( max_iy );
    const double min_y = opt.fieldY( min_iy );

    //     std::cerr << "drawGrid  min_x = " < min_x
    //               << "  max_x = " << max_x
    //               << "  min_y = " << min_y
    //               << "  max_y = " << max_y
    //               << std::endl;

    painter.setPen( DrawConfig::instance().linePen() );
    painter.setBrush( DrawConfig::instance().transparentBrush() );

    double x = 0.0;
    while ( x < max_x )
    {
        int ix = opt.absScreenXInt( x );
        if ( istep > text_step_x )
        {
            painter.drawText( ix, max_iy, QString::number( x ) );
        }
        painter.drawLine( ix, max_iy, ix, min_iy );
        x += grid_step;
    }

    x = -grid_step;
    while ( min_x < x )
    {
        int ix = opt.absScreenXInt( x );
        if ( istep > text_step_x )
        {
            painter.drawText( ix, max_iy, QString::number( x ) );
        }
        painter.drawLine( ix, max_iy, ix, min_iy );
        x -= grid_step;
    }


    double y = 0.0;
    while ( y < max_y )
    {
        int iy = opt.absScreenYInt( y );
        if ( istep > text_step_y )
        {
            painter.drawText( min_ix, iy, QString::number( y ) );
        }
        painter.drawLine( max_ix, iy, min_ix, iy );
        y += grid_step;
    }

    y = -grid_step;
    while ( min_y < y )
    {
        int iy = opt.absScreenYInt( y );
        if ( istep > text_step_y )
        {
            painter.drawText( min_ix, iy, QString::number( y ) );
        }
        painter.drawLine( max_ix, iy, min_ix, iy );
        y -= grid_step;
    }
}
