// -*-c++-*-

/*!
  \file ball_painter.cpp
  \brief ball painter class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <qt.h>

#include "ball_painter.h"

#include "draw_config.h"

#include "options.h"
#include "main_data.h"
#include "monitor_view_data.h"

#include <rcsc/common/server_param.h>
#include <rcsc/geom/vector_2d.h>

#include <vector>

/*-------------------------------------------------------------------*/
/*!

*/
void
BallPainter::draw( QPainter & painter )
{
    const Options & opt = Options::instance();

    if ( ! opt.showBall() )
    {
        return;
    }

    MonitorViewConstPtr view = M_main_data.getCurrentViewData();
    if ( ! view )
    {
        return;
    }

    const DrawConfig & dconf = DrawConfig::instance();

    // decide radius
    const int ball_radius
        = std::max( 1,
                    ( opt.enlargeMode()
                      ? opt.scaleInt( opt.ballSize() )
                      : opt.scaleInt( rcsc::ServerParam::i().ballSize() ) )
                    );
    const int ball_diameter = ball_radius * 2;
    const int ix = opt.screenXInt( view->ball().x() );
    const int iy = opt.screenYInt( view->ball().y() );


    // set GDI objects
    painter.setPen( dconf.transparentPen() );
    painter.setBrush( dconf.ballBrush() );

    // draw ball body
    painter.drawEllipse( ix - ball_radius,
                         iy - ball_radius,
                         ball_diameter,
                         ball_diameter );

    // draw setplay ball owner team color circle
    if ( view->playmode().isLeftSetPlay() )
    {
        painter.setBrush( dconf.leftTeamBrush() );
        int color_radius = std::max( 1, ball_radius - 2 );
        painter.drawEllipse( ix - color_radius,
                             iy - color_radius,
                             color_radius * 2,
                             color_radius * 2 );
    }
    else if ( view->playmode().isRightSetPlay() )
    {
        painter.setBrush( dconf.rightTeamBrush() );
        int color_radius = std::max( 1, ball_radius - 2 );
        painter.drawEllipse( ix - color_radius,
                             iy - color_radius,
                             color_radius * 2,
                             color_radius * 2 );
    }

    // draw additional circle
    if ( ! opt.enlargeMode()
         || ball_radius <= 1  )
    {
        painter.setPen( dconf.ballPen() );
        painter.setBrush( dconf.transparentBrush() );

        int kickable_radius
            = opt.scaleInt( rcsc::ServerParam::i().defaultKickableMargin() );
        painter.drawEllipse( ix - kickable_radius,
                             iy - kickable_radius,
                             kickable_radius * 2,
                             kickable_radius * 2 );
    }

    // draw future status
    if ( opt.ballFutureCycle() > 0
         && view->ball().hasDelta() )
    {
        drawFutureState( painter );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
BallPainter::drawFutureState( QPainter & painter ) const
{
    MonitorViewConstPtr view = M_main_data.getCurrentViewData();

    const Options & opt = Options::instance();
    const DrawConfig & dconf = DrawConfig::instance();
    const double ball_decay = rcsc::ServerParam::i().ballDecay();

    rcsc::Vector2D bpos( view->ball().x(),
                         view->ball().y() );
    rcsc::Vector2D bvel( view->ball().deltaX(),
                         view->ball().deltaY() );

    bpos *= opt.reverseValue();
    bvel *= opt.reverseValue();

    QPoint first_point( opt.absScreenXInt( bpos.x ),
                        opt.absScreenYInt( bpos.y ) );

    painter.save();
    painter.translate( first_point.x(), first_point.y() );
    painter.rotate( bvel.th().degree() );

    painter.setPen( dconf.ballPen() );
    painter.setBrush( dconf.transparentBrush() );

    double travel = 0.0;
    double speed = bvel.r();
    int last_x = 0;
    const int max_loop = opt.ballFutureCycle();
    for ( int i = 0; i < max_loop; ++i )
    {
        travel += speed;
        speed *= ball_decay;

        int x = opt.scaleInt( travel );
        if ( std::abs( last_x - x ) < 1 )
        {
            break;
        }
        last_x = x;
        painter.drawEllipse( x - 1, -1, 3, 3 );
    }

    painter.drawLine( 0, 0, last_x, 0 );

    painter.restore();
}
