// -*-c++-*-

/*!
  \file fedit_config.cpp
  \brief application configuration class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "fedit_config.h"

#include <rcsc/param/param_map.h>
#include <rcsc/param/cmd_line_parser.h>

#include <iostream>
#include <cstdio>
#include <cmath>

const double FEditConfig::PITCH_LENGTH = 105.0;
const double FEditConfig::PITCH_WIDTH = 68.0;
const double FEditConfig::PITCH_MARGIN = 5.0;
const double FEditConfig::CENTER_CIRCLE_R = 9.15;
const double FEditConfig::PENALTY_AREA_LENGTH = 16.5;
const double FEditConfig::PENALTY_AREA_WIDTH = 40.32;
const double FEditConfig::PENALTY_CIRCLE_R = 9.15; // 7.1
const double FEditConfig::PENALTY_SPOT_DIST = 11.0;
const double FEditConfig::GOAL_AREA_LENGTH = 5.5;
const double FEditConfig::GOAL_AREA_WIDTH = 18.2;
const double FEditConfig::GOAL_DEPTH = 2.5;
const double FEditConfig::GOAL_WIDTH = 14.02;
const double FEditConfig::GOAL_POST_R = 0.06;

/*-------------------------------------------------------------------*/
/*!

*/
FEditConfig &
FEditConfig::instance()
{
    static FEditConfig s_instance;
    return s_instance;
}

/*-------------------------------------------------------------------*/
/*!

*/
FEditConfig::FEditConfig()
{
    init();
}

/*-------------------------------------------------------------------*/
/*!

*/
FEditConfig::~FEditConfig()
{

}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditConfig::init()
{
    M_initial_ball_x = 0.0;
    M_initial_ball_y = 0.0;

    M_maximize = false;
    M_full_screen = false;

    M_pos_x = -1;
    M_pos_y = -1;
    M_width = -1;
    M_height = -1;

    M_auto_backup = true;
    M_logging = false;
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
FEditConfig::parseCmdLine( int argc,
                           char ** argv )
{
    rcsc::ParamMap system_options( "System Options" );
    rcsc::ParamMap editor_options( "Editor Options" );
    rcsc::ParamMap view_options( "View Preference Options" );

    bool help = false;
    bool version = false;
    std::string geometry;
    std::string ball_pos;

    system_options.add()
        ( "help", "h",
          rcsc::BoolSwitch( &help ),
          "print this message." )
        ( "version", "v",
          rcsc::BoolSwitch( &version ),
          "print version." )
        ;

    editor_options.add()
        ( "conf", "c",
          &M_conf_file,
          "specifies the .conf file to be opened." )
        ( "data", "d",
          &M_data_file,
          "specifies the .dat file to be opened." )
        ( "background-conf", "C",
          &M_background_conf_file,
          "specifies the .conf file as a background data." )
        ( "initial-ball-pos", "",
          &ball_pos,
          "specfies the initial ball position \"X:Y\". e.g. --initial-ball-pos=10.0:-8.3" )
        ( "auto-backup", "",
          &M_auto_backup,
          "make backup files automatically" )
        //( "logging", "",
        //  rcsc::BoolSwitch( &M_logging ),
        //  "logging formation editor operation." )
        ;

    view_options.add()
        ( "maximize", "",
          rcsc::BoolSwitch( &M_maximize ),
          "start with a maximized window." )
        ( "full-screen", "",
          rcsc::BoolSwitch( &M_full_screen ),
          "start with a full screen window." )
        ( "geometry", "",
          &geometry,
          "specifies the window geometry \"[WxH][+X+Y]\". e.g. --geometry=1024x768+0+0" )
        ;

    rcsc::CmdLineParser parser( argc, argv );

    parser.parse( system_options );
    parser.parse( editor_options );
    parser.parse( view_options );

    //
    // analyze input files
    //
    for ( std::vector< std::string >::const_iterator it
              = parser.positionalOptions().begin();
          it != parser.positionalOptions().end();
          ++it )
    {
        // ".conf"
        if ( it->length() > 5
             && it->compare( it->length() - 5, 5, ".conf" ) == 0 )
        {
            if ( M_conf_file.empty() )
            {
                M_conf_file = *it;
            }
            else if ( M_background_conf_file.empty() )
            {
                M_background_conf_file = *it;
            }
            else // if ( ! M_conf_file.empty() && ! M_bg_conf_file.empty() )
            {
                help = true;
                break;
            }
        }
        else
        {
            help = true;
            break;
        }
    }

    //
    // help message
    //
    if ( help
         || parser.failed()
         || parser.positionalOptions().size() > 2 )
    {
        std::cout << "Usage: " << "fedit"
                  << " [options ... ] [conf file] [[BG conf file]]"
                  << std::endl;
        system_options.printHelp( std::cout, false );
        editor_options.printHelp( std::cout );
        view_options.printHelp( std::cout );
        return false;
    }

    //
    // version message
    //
    if ( version )
    {
        std::cout << "fedit" << " Version " << VERSION
                  << std::endl;
        return false;
    }

    //
    // initial ball position
    //
    if ( ! ball_pos.empty() )
    {
        char delim;
        double x, y;
        if ( std::sscanf( ball_pos.c_str(),
                          " %lf %c %lf " ,
                          &x, &delim, &y ) == 3 )
        {
            if ( std::fabs( x ) < PITCH_LENGTH * 0.5 + 5.0
                 && std::fabs( y ) < PITCH_WIDTH * 0.5 + 5.0 )
            {
                if ( delim == '-' && y >= 0.0 )
                {
                    y = -y;
                }

                M_initial_ball_x = x;
                M_initial_ball_y = y;
            }
            else
            {
                std::cerr << "Initial ball position is over the range ["
                          << ball_pos
                          << "]" << std::endl;
            }
        }
        else
        {
            std::cerr << "Illegal initial ball position [" << ball_pos
                      << "]" << std::endl;
        }
    }

    //
    // window size and position
    //
    if ( ! geometry.empty() )
    {
        int w = -1, h = -1;
        int x = -1, y = -1;

        int num = std::sscanf( geometry.c_str(),
                               " %d x %d %d %d " ,
                               &w, &h, &x, &y );
        if ( num == 4 || num == 2 )
        {
            if ( w <= 0 || h <= 0 )
            {
                std::cerr << "Illegal window size [" << geometry
                          << "]" << std::endl;
            }
            else
            {
                M_width = w;
                M_height = h;
                M_pos_x = x;
                M_pos_y = y;
            }
        }
        else if ( std::sscanf( geometry.c_str(),
                               " %d %d " ,
                               &x, &y ) == 2 )
        {
            M_pos_x = x;
            M_pos_y = y;
        }
        else
        {
            std::cerr << "Illegal geometry format [" << geometry
                      << "]" << std::endl;
        }
    }

    return true;
}
