#include "simple_pass_checker.h"

#include <rcsc/common/server_param.h>
#include <rcsc/common/logger.h>

static const double GOALIE_PASS_EVAL_THRESHOLD = 17.5;
static const double BACK_PASS_EVAL_THRESHOLD = 17.5;
static const double PASS_EVAL_THRESHOLD = 14.5;
static const double CHANCE_PASS_EVAL_THRESHOLD = 14.5;
static const double PASS_RECEIVER_PREDICT_STEP = 2.5;

namespace rcsc {

bool
SimplePassChecker::operator() ( const rcsc::PredictState & state,
                                const rcsc::AbstractPlayerObject * from,
                                const rcsc::AbstractPlayerObject * to,
                                const rcsc::Vector2D & receive_point ) const
{
    if ( ! from || ! to )
    {
        return false;
    }


    //
    // inhibit self pass on some game modes
    //
    if ( from->unum() == to->unum() )
    {
        return false;
    }


    const long VALID_TEAMMATE_ACCURACY = 8;
    const long VALID_OPPONENT_ACCURACY = 20;

    if ( from->isGhost()
         || to->isGhost()
         || from->posCount() > VALID_TEAMMATE_ACCURACY
         || to->posCount() > VALID_TEAMMATE_ACCURACY )
    {
        return false;
    }

    if ( ( receive_point - from->pos() ).r() <= 4.0 )
    {
#ifdef DEBUG_PRINT
        dlog.addText( rcsc::Logger::PASS,
                      "checking pass %d to %d: too near",
                      from->unum(), to->unum() );
#endif
        return false;
    }

    if ( ( receive_point - from->pos() ).r() >= 35.0 )
    {
#ifdef DEBUG_PRINT
        dlog.addText( rcsc::Logger::PASS,
                      "checking pass %d to %d: too far",
                      from->unum(), to->unum() );
#endif
        return false;
    }

    if ( to->pos().x >= state.offsideLineX() )
    {
#ifdef DEBUG_PRINT
        dlog.addText( rcsc::Logger::PASS,
                      "checking pass %d to %d: offside",
                      from->unum(), to->unum() );
#endif
        return false;
    }

    if ( receive_point.x <= rcsc::ServerParam::i().ourPenaltyAreaLineX() + 3.0
         && receive_point.absY() <= rcsc::ServerParam::i().penaltyAreaHalfWidth() + 3.0 )
    {
#ifdef DEBUG_PRINT
        dlog.addText( rcsc::Logger::PASS,
                      "checking pass %d to %d: penalty area",
                      from->unum(), to->unum() );
#endif
        return false;
    }

    if ( receive_point.absX() >= rcsc::ServerParam::i().pitchHalfLength()
         || receive_point.absY() >= rcsc::ServerParam::i().pitchHalfWidth() )
    {
#ifdef DEBUG_PRINT
        dlog.addText( rcsc::Logger::PASS,
                      "checking pass %d to %d: out of field",
                      from->unum(), to->unum() );
#endif
        return false;
    }

    if ( to->goalie()
         && receive_point.x < rcsc::ServerParam::i().ourPenaltyAreaLineX() + 1.0
         && receive_point.absY() < rcsc::ServerParam::i().penaltyAreaHalfWidth() + 1.0 )
    {
#ifdef DEBUG_PRINT
        dlog.addText( rcsc::Logger::PASS,
                      "checking pass %d to %d: goalie can catch",
                      from->unum(), to->unum() );
#endif
        return false;
    }

    if ( to->goalie()
         && receive_point.x > state.defenseLineX() - 3.0 )
    {
#ifdef DEBUG_PRINT
        dlog.addText( rcsc::Logger::PASS,
                      "checking pass %d to %d: goalie over defense line",
                      from->unum(), to->unum() );
#endif
        return false;
    }


    rcsc::Vector2D from_pos = from->pos();

    if ( from->isSelf() )
    {
        from_pos = state.ballPos();
    }

    rcsc::AngleDeg test_dir = ( receive_point - from_pos ).th();

    double pass_course_cone = + 360.0;


#if 0
    const double OVER_TEAMMATE_IGNORE_DISTANCE = ( receive_point.x >= +25.0
                                                   ? 2.0
                                                   : 5.0 );
#else
    const double OVER_TEAMMATE_IGNORE_DISTANCE = ( receive_point.x >= +25.0
                                                   ? 2.0
                                                   : 6.0 );
#endif

    const rcsc::PlayerCont::const_iterator o_end = state.opponents().end();
    for ( rcsc::PlayerCont::const_iterator opp = state.opponents().begin();
          opp != o_end;
          ++opp )
    {
        if ( (*opp).posCount() > VALID_OPPONENT_ACCURACY )
        {
            continue;
        }

#if 1
        if ( ( (*opp).pos() - receive_point ).r() < 5.0 )
        {
            return false;
        }
#endif

        rcsc::Vector2D opp_pos = (*opp).pos();
        if ( (*opp).velValid() )
        {
            opp_pos += (*opp).vel() * 2.0;
        }

        if ( ( opp_pos - from_pos ).r()
             > ( receive_point - from_pos ).r() + OVER_TEAMMATE_IGNORE_DISTANCE )
        {
            continue;
        }

        double angle_diff = ( ( opp_pos - from_pos ).th() - test_dir ).abs();

        if ( from->isSelf() )
        {
            double kickable_dist = to->playerTypePtr()->playerSize()
                + to->playerTypePtr()->kickableMargin();
            double hide_radian = std::asin( std::min( kickable_dist / ( opp_pos - from_pos ).r(),
                                                      1.0 ) );
            angle_diff = std::max( angle_diff - rcsc::AngleDeg::rad2deg( hide_radian ), 0.0 );
        }

        if ( pass_course_cone > angle_diff )
        {
            pass_course_cone = angle_diff;
        }
    }


    double eval_threshold = PASS_EVAL_THRESHOLD;

    if ( to->pos().x - 2.0 <= from->pos().x )
    {
        eval_threshold = BACK_PASS_EVAL_THRESHOLD;
    }

    if ( from->pos().x >= +25.0
         && to->pos().x >= +25.0 )
    {
        eval_threshold = CHANCE_PASS_EVAL_THRESHOLD;
    }

    if ( from->goalie() )
    {
        eval_threshold = GOALIE_PASS_EVAL_THRESHOLD;
    }

    if ( from->isSelf()
         && from->distFromBall() <= 3.0 )
    {
#if 1
        if ( pass_course_cone > eval_threshold )
        {
            return true;
        }
        else
        {
# ifdef DEBUG_PRINT
            dlog.addText( rcsc::Logger::PASS,
                          "pass course too narrow: %.3f",
                          pass_course_cone );
# endif
            return false;
        }
#else
        long self_access_step;
        long teammate_access_step;
        long opponent_access_step;
        long unknown_access_step;
        int first_access_teammate;
        int first_access_opponent;

        Simulator sim;

        rcsc::Vector2D vec( to->pos() - state.self().pos() );
        vec.setLength( 2.5 );

        sim.simulate( wm,
                      wm.getPlayerCont
                      ( new rcsc::CoordinateAccuratePlayerPredicate( 20 ) ),
                      wm.ball().pos(),
                      vec,
                      2 /* ball_velocity_change_step */,
                      3 /* self_penalty_step */,
                      3 /* self_penalty_step */,
                      0 /* opponent_penalty_step */,
                      &self_access_step,
                      &teammate_access_step,
                      &opponent_access_step,
                      &unknown_access_step,
                      &first_access_teammate,
                      &first_access_opponent );

        long our_team_access_step = teammate_access_step;
        long opp_team_access_step = opponent_access_step;

        if ( self_access_step < our_team_access_step )
        {
            our_team_access_step = self_access_step;
        }

        if ( unknown_access_step < opp_team_access_step )
        {
            opp_team_access_step = unknown_access_step;
        }

        if ( our_team_access_step < opp_team_access_step
             || pass_course_cone > eval_threshold )
        {
            return true;
        }
        else
        {
# ifdef DEBUG_PRINT
            dlog.addText( rcsc::Logger::PASS,
                          "pass course too narrow: %.3f",
                          pass_course_cone );
# endif
            return false;
        }
#endif
    }
    else
    {
        if ( pass_course_cone > eval_threshold )
        {
#ifdef DEBUG_PRINT
            rcsc::dlog.addText( rcsc::Logger::PASS,
                                "ok %.3f",
                                pass_course_cone );
#endif
            return true;
        }
        else
        {
#ifdef DEBUG_PRINT
            dlog.addText( rcsc::Logger::PASS,
                          "pass course too narrow: %.3f",
                          pass_course_cone );
#endif
            return false;
        }
    }
}

#if 0
static
double
s_get_ball_speed_for_pass( const double & distance )
{
    if ( distance >= 20.0 )
    {
        return 3.0;
    }
    else if ( distance >= 8.0 )
    {
        return 2.5;
    }
    else if ( distance >= 5.0 )
    {
        return 1.8;
    }
    else
    {
        return 1.5;
    }
}
#endif

}
