// -*-c++-*-

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "role_442_center_half.h"

#include "strategy.h"

#include "bhv_basic_defensive_kick.h"
#include "bhv_basic_offensive_kick.h"
#include "bhv_cross.h"
#include "bhv_keep_shoot_chance.h"

#include "bhv_basic_move.h"
#include "bhv_defender_basic_block_move.h"
#include "bhv_defender_get_ball.h"
#include "bhv_press.h"
#include "bhv_block_dribble.h"
#include "bhv_offensive_half_offensive_move.h"

#include "bhv_basic_tackle.h"

#include "body_advance_ball_test.h"
#include "body_clear_ball2008.h"
#include <rcsc/action/body_dribble2008.h>
#include <rcsc/action/body_intercept2008.h>
#include <rcsc/action/body_hold_ball2008.h>
#include "bhv_pass_test.h"

#include <rcsc/action/basic_actions.h>
#include <rcsc/action/body_go_to_point.h>
#include <rcsc/action/neck_turn_to_ball_or_scan.h>
#include <rcsc/action/neck_turn_to_low_conf_teammate.h>
#include <rcsc/action/neck_scan_field.h>

#include <rcsc/formation/formation.h>

#include <rcsc/player/player_agent.h>
#include <rcsc/player/intercept_table.h>
#include <rcsc/player/debug_client.h>

#include <rcsc/common/logger.h>
#include <rcsc/common/server_param.h>
#include <rcsc/math_util.h>
#include <rcsc/soccer_math.h>

#include "bhv_center_half_cross_block_move.h"
#include "bhv_center_half_defensive_move.h"
#include "bhv_defensive_half_danger_move.h"
#include "neck_default_intercept_neck.h"
#include "neck_offensive_intercept_neck.h"

//#define USE_BHV_DEFENSIVE_HALF_DANGER_MOVE

const std::string Role442CenterHalf::NAME( "442CenterHalf" );

/*-------------------------------------------------------------------*/
/*!

*/
bool
Role442CenterHalf::execute( rcsc::PlayerAgent * agent )
{
    //////////////////////////////////////////////////////////////
    // play_on play
    bool kickable = agent->world().self().isKickable();
    if ( agent->world().existKickableTeammate()
         && agent->world().teammatesFromBall().front()->distFromBall()
         < agent->world().ball().distFromSelf() )
    {
        kickable = false;
    }

    if ( kickable )
    {
        // kick
        doKick( agent );
    }
    else
    {
        // positioning
        doMove( agent );
    }

    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
Role442CenterHalf::doKick( rcsc::PlayerAgent* agent )
{
    const double nearest_opp_dist
        = agent->world().getDistOpponentNearestToSelf( 5 );

    switch ( Strategy::get_ball_area(agent->world().ball().pos()) ) {
    case Strategy::BA_CrossBlock:
        if ( nearest_opp_dist < 3.0 )
        {
            rcsc::dlog.addText( rcsc::Logger::ROLE,
                                __FILE__": cross block area. danger" );
            if ( ! rcsc::Bhv_PassTest().execute( agent ) )
            {
                Body_AdvanceBallTest().execute( agent );

                if ( agent->effector().queuedNextBallKickable() )
                {
                    agent->setNeckAction( new rcsc::Neck_ScanField() );
                }
                else
                {
                    agent->setNeckAction( new rcsc::Neck_TurnToBall() );
                }
            }
        }
        else
        {
            Bhv_BasicDefensiveKick().execute( agent );
        }
        break;
    case Strategy::BA_Stopper:
    case Strategy::BA_Danger:
        if ( nearest_opp_dist < 3.0 )
        {
            rcsc::dlog.addText( rcsc::Logger::ROLE,
                                __FILE__": stopper or danger" );
            if ( ! rcsc::Bhv_PassTest().execute( agent ) )
            {
                Body_ClearBall2008().execute( agent );

                if ( agent->effector().queuedNextBallKickable() )
                {
                    agent->setNeckAction( new rcsc::Neck_ScanField() );
                }
                else
                {
                    agent->setNeckAction( new rcsc::Neck_TurnToBall() );
                }
            }
        }
        else
        {
            Bhv_BasicDefensiveKick().execute( agent );
        }
        break;
    case Strategy::BA_DribbleBlock:
    case Strategy::BA_DefMidField:
        Bhv_BasicDefensiveKick().execute( agent );
        break;
    case Strategy::BA_DribbleAttack:
    case Strategy::BA_OffMidField:
        Bhv_BasicOffensiveKick().execute( agent );
        break;
    case Strategy::BA_Cross:
        Bhv_Cross().execute( agent );
        break;
    case Strategy::BA_ShootChance:
        if ( Bhv_Cross::get_best_point( agent, NULL ) )
        {
            rcsc::dlog.addText( rcsc::Logger::ROLE,
                                __FILE__": shoot chance area" );
            Bhv_Cross().execute( agent );
        }
        else
        {
            rcsc::dlog.addText( rcsc::Logger::ROLE,
                                __FILE__": keep shoot chance" );
            Bhv_KeepShootChance().execute( agent );
        }
        break;
    default:
        rcsc::dlog.addText( rcsc::Logger::ROLE,
                            __FILE__": unknown ball area" );
        rcsc::Body_HoldBall2008().execute( agent );
        break;
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
Role442CenterHalf::doMove( rcsc::PlayerAgent * agent )
{
    rcsc::Vector2D home_pos = Strategy::i().getPosition( agent->world().self().unum() );
    if ( ! home_pos.valid() ) home_pos.assign( 0.0, 0.0 );

    switch ( Strategy::get_ball_area( agent->world() ) ) {
    case Strategy::BA_CrossBlock:
        //doCrossBlockMove( agent, home_pos );
        Bhv_CenterHalfCrossBlockMove().execute( agent );
        break;
    case Strategy::BA_Stopper:
        doCrossBlockMove( agent, home_pos );
        break;
    case Strategy::BA_Danger:
#ifdef USE_BHV_DEFENSIVE_HALF_DANGER_MOVE
        Bhv_DefensiveHalfDangerMove().execute( agent );
#else
        if ( agent->world().existKickableTeammate()
             && ! agent->world().existKickableOpponent() )
        {
            rcsc::Vector2D new_target = home_pos;
            new_target.x += 5.0;
            Bhv_BasicMove( new_target ).execute( agent );
        }
        else
        {
            Bhv_DefenderBasicBlockMove( home_pos ).execute( agent );
        }
#endif
        break;
    case Strategy::BA_DribbleBlock:
        doDribbleBlockMove( agent, home_pos );
        break;
    case Strategy::BA_DefMidField:
        Bhv_CenterHalfDefensiveMove().execute( agent );
        break;
    case Strategy::BA_DribbleAttack:
        Bhv_BasicMove( home_pos ).execute( agent );
        break;
    case Strategy::BA_OffMidField:
        doOffensiveMove( agent, home_pos );
        break;
    case Strategy::BA_Cross:
        doCrossAreaMove( agent, home_pos );
        break;
    case Strategy::BA_ShootChance:
        Bhv_BasicMove( home_pos ).execute( agent );
        break;
    default:
        rcsc::dlog.addText( rcsc::Logger::ROLE,
                            __FILE__": unknown ball area" );
        Bhv_BasicMove( home_pos ).execute( agent );
        agent->setNeckAction( new rcsc::Neck_TurnToBallOrScan() );
        break;
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
Role442CenterHalf::doCrossBlockMove( rcsc::PlayerAgent* agent,
                                     const rcsc::Vector2D& home_pos )
{
    rcsc::dlog.addText( rcsc::Logger::ROLE,
                        __FILE__": doCrossBlockMove" );

    //-----------------------------------------------
    // tackle
    if ( Bhv_BasicTackle( 0.8, 80.0 ).execute( agent ) )
    {
        return;
    }

    const rcsc::WorldModel & wm = agent->world();

    const int self_min = wm.interceptTable()->selfReachCycle();
    const int mate_min = wm.interceptTable()->teammateReachCycle();
    const int opp_min = wm.interceptTable()->opponentReachCycle();

    //if ( wm.interceptTable()->isSelfFastestPlayer() )
    if ( ! wm.existKickableTeammate()
         && ! wm.existKickableOpponent()
         && self_min < mate_min
         && self_min < opp_min )
    {
        rcsc::Body_Intercept2008().execute( agent );

        if ( opp_min >= self_min + 3 )
        {
            agent->setNeckAction( new Neck_OffensiveInterceptNeck() );
        }
        else
        {
            agent->setNeckAction( new Neck_DefaultInterceptNeck
                                  ( new rcsc::Neck_TurnToBallOrScan() ) );
        }

        return;
    }
    double dash_power = 0.0;

    if ( wm.ball().distFromSelf() > 30.0 )
    {
        dash_power = wm.self().playerType().staminaIncMax() * 0.9;
        rcsc::dlog.addText( rcsc::Logger::ROLE,
                            __FILE__": ball is too far. dash_power=%.1f",
                            dash_power );
    }
    else if ( wm.ball().distFromSelf() > 20.0 )
    {
        dash_power = rcsc::ServerParam::i().maxDashPower() * 0.5;
        rcsc::dlog.addText( rcsc::Logger::ROLE,
                            __FILE__": ball far. dash_power=%.1f",
                            dash_power );
    }
    else
    {
        dash_power = rcsc::ServerParam::i().maxDashPower() * 0.9;
        rcsc::dlog.addText( rcsc::Logger::ROLE,
                            __FILE__": near. dash_power = %.1f",
                            dash_power );
    }

    if ( wm.existKickableTeammate() )
    {
        dash_power = std::min( rcsc::ServerParam::i().maxDashPower() * 0.5,
                               dash_power );
    }

    dash_power = wm.self().getSafetyDashPower( dash_power );

    double dist_thr = wm.ball().distFromSelf() * 0.1;
    if ( dist_thr < 0.5 ) dist_thr = 0.5;

    agent->debugClient().addMessage( "CrossBlock" );
    agent->debugClient().setTarget( home_pos );
    agent->debugClient().addCircle( home_pos, dist_thr );

    if ( ! rcsc::Body_GoToPoint( home_pos, dist_thr, dash_power ).execute( agent ) )
    {
        rcsc::AngleDeg body_angle( wm.ball().angleFromSelf().abs() < 80.0
                                   ? 0.0
                                   : 180.0 );
        rcsc::Body_TurnToAngle( body_angle ).execute( agent );
    }

    agent->setNeckAction( new rcsc::Neck_TurnToBall() );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
Role442CenterHalf::doOffensiveMove( rcsc::PlayerAgent * agent,
                                    const rcsc::Vector2D & home_pos )
{
#if 1
    (void)home_pos;
    Bhv_OffensiveHalfOffensiveMove().execute( agent );
#else
    static bool s_recover_mode = false;

    //-----------------------------------------------
    // tackle
    if ( Bhv_BasicTackle( 0.8, 80.0 ).execute( agent ) )
    {
        return;
    }

    const rcsc::WorldModel & wm = agent->world();
    /*--------------------------------------------------------*/
    // chase ball
    int self_min = wm.interceptTable()->selfReachCycle();
    int mate_min = wm.interceptTable()->teammateReachCycle();
    int opp_min = wm.interceptTable()->opponentReachCycle();

    if ( ! wm.existKickableTeammate()
         && self_min <= mate_min
         && ( self_min <= opp_min + 1
              || self_min <= 3 )
         )
    {
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            __FILE__": doOffensiveMove() intercept" );
        rcsc::Body_Intercept2008().execute( agent );
        agent->setNeckAction( new Neck_OffensiveInterceptNeck() );
        return;
    }

    if ( wm.self().stamina() < rcsc::ServerParam::i().staminaMax() * 0.5 )
    {
        s_recover_mode = true;
    }
    else if ( wm.self().stamina() > rcsc::ServerParam::i().staminaMax() * 0.7 )
    {
        s_recover_mode = false;
    }


    double dash_power = 100.0;
    const double my_inc
        = wm.self().playerType().staminaIncMax()
        * wm.self().recovery();

    if ( wm.defenseLineX() > wm.self().pos().x
         && wm.ball().pos().x < wm.defenseLineX() + 20.0 )
    {
        dash_power = rcsc::ServerParam::i().maxDashPower();
    }
    else if ( s_recover_mode )
    {
        dash_power = my_inc - 25.0; // preffered recover value
        if ( dash_power < 0.0 ) dash_power = 0.0;
    }
    else if ( wm.existKickableTeammate()
              && wm.ball().distFromSelf() < 20.0 )
    {
        dash_power = std::min( my_inc * 1.1,
                               rcsc::ServerParam::i().maxDashPower() );
    }
    // in offside area
    else if ( wm.self().pos().x > wm.offsideLineX() )
    {
        dash_power = rcsc::ServerParam::i().maxDashPower();
    }
    else if ( home_pos.x < wm.self().pos().x
              && wm.self().stamina() > rcsc::ServerParam::i().staminaMax() * 0.7 )
    {
        dash_power = rcsc::ServerParam::i().maxDashPower();
    }
    // normal
    else
    {
        dash_power = std::min( my_inc * 1.7,
                               rcsc::ServerParam::i().maxDashPower() );
    }


    double dist_thr = wm.ball().distFromSelf() * 0.1;
    if ( dist_thr < 1.5 ) dist_thr = 1.5;

    agent->debugClient().addMessage( "OffMove%.0f", dash_power );
    agent->debugClient().setTarget( home_pos );
    agent->debugClient().addCircle( home_pos, dist_thr );

    if ( ! rcsc::Body_GoToPoint( home_pos, dist_thr, dash_power
                                 ).execute( agent ) )
    {
        rcsc::Vector2D my_next = wm.self().pos() + wm.self().vel();
        rcsc::Vector2D ball_next = ( wm.existKickableOpponent()
                                     ? wm.ball().pos()
                                     : wm.ball().pos() + wm.ball().vel() );
        rcsc::AngleDeg body_angle = ( ball_next - my_next ).th() + 90.0;
        if ( body_angle.abs() < 90.0 ) body_angle -= 180.0;

        rcsc::Body_TurnToAngle( body_angle ).execute( agent );
    }

    if ( wm.existKickableOpponent()
         && wm.ball().distFromSelf() < 18.0 )
    {
        agent->setNeckAction( new rcsc::Neck_TurnToBall() );
    }
    else
    {
        agent->setNeckAction( new rcsc::Neck_TurnToBallOrScan() );
    }
#endif
}

/*-------------------------------------------------------------------*/
/*!

*/
void
Role442CenterHalf::doDribbleBlockMove( rcsc::PlayerAgent* agent,
                                       const rcsc::Vector2D& home_pos )
{
    rcsc::dlog.addText( rcsc::Logger::ROLE,
                        __FILE__": doDribbleBlockMove()" );

    const rcsc::WorldModel & wm = agent->world();

    ///////////////////////////////////////////////////
    // tackle
    if ( Bhv_BasicTackle( 0.8, 80.0 ).execute( agent ) )
    {
        return;
    }

    const int self_min = wm.interceptTable()->selfReachCycle();
    const int mate_min = wm.interceptTable()->teammateReachCycle();
    const int opp_min = wm.interceptTable()->opponentReachCycle();

    //if ( wm.interceptTable()->isSelfFastestPlayer() )
    if ( ! wm.existKickableTeammate()
         && ! wm.existKickableOpponent()
         && self_min < mate_min
         && self_min < opp_min )
    {
        rcsc::dlog.addText( rcsc::Logger::ROLE,
                            __FILE__": doDribbleBlockMove() I am fastest. intercept" );
        rcsc::Body_Intercept2008().execute( agent );

        if ( opp_min >= self_min + 3 )
        {
            agent->setNeckAction( new Neck_OffensiveInterceptNeck() );
        }
        else
        {
            agent->setNeckAction( new Neck_DefaultInterceptNeck
                                  ( new rcsc::Neck_TurnToBallOrScan() ) );
        }

        return;
    }

    ///////////////////////////////////////////////////
    double dash_power;

    if ( wm.self().pos().x + 5.0 < wm.ball().pos().x )
    {
        if ( wm.self().stamina() < rcsc::ServerParam::i().staminaMax() * 0.7 )
        {
            dash_power
                = rcsc::ServerParam::i().maxDashPower() * 0.7
                * ( wm.self().stamina() / rcsc::ServerParam::i().staminaMax() );
            rcsc::dlog.addText( rcsc::Logger::ROLE,
                                __FILE__": doDribbleBlockMove() dash_power=%.1f",
                                dash_power );
        }
        else
        {
            dash_power
                = rcsc::ServerParam::i().maxDashPower() * 0.75
                - std::min( 30.0, wm.ball().distFromSelf() );
            rcsc::dlog.addText( rcsc::Logger::ROLE,
                                __FILE__": doDribbleBlockMove() dash_power=%.1f",
                                dash_power );
        }
    }
    else
    {
        dash_power
            = rcsc::ServerParam::i().maxDashPower()
            + 10.0
            - wm.ball().distFromSelf();
        dash_power = rcsc::min_max( 0.0, dash_power, rcsc::ServerParam::i().maxDashPower() );
        rcsc::dlog.addText( rcsc::Logger::ROLE,
                            __FILE__": doDribbleBlockMove() dash_power=%.1f",
                            dash_power );
    }



    if ( wm.existKickableTeammate() )
    {
        dash_power = std::min( rcsc::ServerParam::i().maxDashPower() * 0.5,
                               dash_power );
    }

    dash_power = wm.self().getSafetyDashPower( dash_power );

    double dist_thr = wm.ball().distFromSelf() * 0.1;
    if ( dist_thr < 0.5 ) dist_thr = 0.5;

    agent->debugClient().addMessage( "DribBlockMove" );
    agent->debugClient().setTarget( home_pos );
    agent->debugClient().addCircle( home_pos, dist_thr );

    if ( ! rcsc::Body_GoToPoint( home_pos, dist_thr, dash_power ).execute( agent ) )
    {
        // face to front or side
        rcsc::AngleDeg body_angle = 0.0;
        if ( wm.ball().angleFromSelf().abs() > 90.0 )
        {
            body_angle = ( wm.ball().pos().y > wm.self().pos().y
                           ? 90.0
                           : -90.0 );
        }
        rcsc::Body_TurnToAngle( body_angle ).execute( agent );
    }
    agent->setNeckAction( new rcsc::Neck_TurnToBallOrScan() );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
Role442CenterHalf::doCrossAreaMove( rcsc::PlayerAgent * agent,
                                    const rcsc::Vector2D & home_pos )
{
    rcsc::dlog.addText( rcsc::Logger::TEAM,
                        __FILE__": doCrossAreaMove. normal move" );
    Bhv_BasicMove( home_pos ).execute( agent );
}

/*-------------------------------------------------------------------*/
/*!

 */
namespace {
rcss::RegHolder role = SoccerRole::creators().autoReg( &Role442CenterHalf::create,
                                                       Role442CenterHalf::NAME );
}
