// -*-c++-*-

/*
 *Copyright:

 Copyright (C) Hiroki SHIMORA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

#ifndef RCSC_RECURSIVE_ACTION_CHAIN_GRAPH_H
#define RCSC_RECURSIVE_ACTION_CHAIN_GRAPH_H

/////////////////////////////////////////////////////////////////////

#include "predict_player_object.h"

#include <rcsc/player/world_model.h>
#include <rcsc/player/player_agent.h>
#include <rcsc/player/abstract_player_object.h>
#include <rcsc/common/server_param.h>

#include <boost/shared_ptr.hpp>
#include <vector>
#include <algorithm>
#include <iostream>

namespace rcsc {

class PredictState /* : public AbstractWorldModel */ {

public:
    typedef std::vector< PredictPlayerObject > PlayerCont;

private:
    static const int VALID_PLAYER_THRESHOLD = 8;

private:
    const WorldModel * M_world;
    unsigned long M_spend_time;

    int M_ball_holder_unum;
    rcsc::Vector2D M_ball_pos;

    int M_self_unum;

    PlayerCont M_all_teammates;

protected:
    void initializeWithWorldModel( const WorldModel & wm )
    {
        //
        // initialize world
        //
        M_world = &wm;


        //
        // initialize spend_time
        //
        M_spend_time = 0;


        //
        // initialize ball pos
        //
        M_ball_pos = wm.ball().pos();


        //
        // initialize self_unum
        //
        M_self_unum = wm.self().unum();


        //
        // initialize ball holder
        //
        const AbstractPlayerObject * h = wm.getTeammateNearestToBall
                                            ( VALID_PLAYER_THRESHOLD );

        if ( h && ( h->pos() - wm.ball().pos() ).r()
                  < ( wm.self().pos() - wm.ball().pos() ).r() )
        {
            M_ball_holder_unum = h->unum();
        }
        else
        {
            M_ball_holder_unum = wm.self().unum();
        }

        //
        // initialize all teammates
        //
        for ( int n = 1; n <= ServerParam::i().maxPlayer(); ++n )
        {
            if ( n == M_self_unum )
            {
                M_all_teammates.push_back( PredictPlayerObject( wm.self() ) );
            }
            else
            {
                const AbstractPlayerObject * t = wm.teammate( n );

                if ( t )
                {
                    M_all_teammates.push_back( PredictPlayerObject( *t ) );
                }
                else
                {
                    M_all_teammates.push_back( PredictPlayerObject() );
                }
            }
        }
    }


public:
    PredictState( const PredictState & rhs,
                  unsigned long append_spend_time = 0 )
        : M_world( rhs.M_world )
        , M_spend_time( rhs.spendTime() + append_spend_time )
        , M_ball_holder_unum( rhs.ballHolderUnum() )
        , M_ball_pos( rhs.ballPos() )
        , M_self_unum( rhs.M_self_unum )
        , M_all_teammates( rhs.M_all_teammates )
      { }

    PredictState( const WorldModel & wm,
                  unsigned long append_spend_time = 0 )
        : M_world( &wm )
        , M_spend_time( append_spend_time )
        , M_ball_holder_unum( -1 )
        , M_ball_pos()
        , M_self_unum( -1 )
        , M_all_teammates()
      {
          initializeWithWorldModel( wm );
      }

    PredictState( const PredictState & rhs,
                  unsigned long append_spend_time,
                  int ball_holder_unum,
                  const rcsc::Vector2D & ball_pos )
        : M_world( rhs.M_world )
        , M_spend_time( rhs.M_spend_time + append_spend_time )
        , M_ball_holder_unum( ball_holder_unum )
        , M_ball_pos( ball_pos )
        , M_self_unum( rhs.M_self_unum )
        , M_all_teammates( rhs.M_all_teammates )
      {
      }

    PredictState( const WorldModel & wm,
                  unsigned long append_spend_time,
                  int ball_holder_unum,
                  const rcsc::Vector2D & ball_pos )
        : M_world( &wm )
        , M_spend_time( 0 )
        , M_ball_holder_unum( -1 )
        , M_ball_pos()
        , M_self_unum( -1 )
        , M_all_teammates()
      {
          initializeWithWorldModel( wm );

          M_spend_time = append_spend_time;
          M_ball_holder_unum = ball_holder_unum;
          M_ball_pos = ball_pos;
      }


    unsigned long spendTime() const
      {
          return M_spend_time;
      }

    int ballHolderUnum() const
      {
          return M_ball_holder_unum;
      }

    const AbstractPlayerObject * ballHolder() const
      {
          return teammate( ballHolderUnum() );
      }

    const Vector2D & ballPos() const
      {
          return M_ball_pos;
      }

    const
    AbstractPlayerObject &
    self() const
      {
          if ( ! ( 1 <= M_self_unum
                   && M_self_unum <= ServerParam::i().maxPlayer() ) )
          {
              std::cerr << "internal error: "
                        << __FILE__ << ":" << __LINE__
                        << "invalid self unum " << M_self_unum << std::endl;
          }

          return M_all_teammates[ M_self_unum - 1 ];
      }

    const
    AbstractPlayerObject * teammate( const int unum ) const
      {
          if ( ! ( 1 <= unum && unum <= ServerParam::i().maxPlayer() ) )
          {
              std::cerr << "internal error: "
                        << __FILE__ << ":" << __LINE__
                        << "invalid unum " << unum << std::endl;
          }

          return &( M_all_teammates[ unum - 1 ] );
      }

    const
    AbstractPlayerObject * opponent( const int unum ) const
      {
          return M_world->opponent( unum );
      }

    const
    PlayerCont &
    allTeammates() const
      {
          return M_all_teammates;
      }

    const rcsc::PlayerCont & opponents() const
      {
          return M_world->opponents();
      }

    const
    rcsc::GameMode & gameMode() const
      {
        return M_world->gameMode();
      }

    const
    rcsc::GameTime & time() const
      {
        return M_world->time();
      }


    rcsc::SideID ourSide() const
      {
          return M_world->ourSide();
      }

    double offsideLineX() const
      {
          return std::max( M_world->offsideLineX(), M_ball_pos.x );
      }

    double defenseLineX() const
      {
          // XXX: tentative implementation, should consider M_all_teammates
          return std::min( M_world->defenseLineX(), M_ball_pos.x );
      }
};


class CooperativeAction
{
public:
    struct ActionType {
        enum Type{
            // ball holder action
            Hold,
            Dribble,
            Pass,
            Shoot
#if 0
            // off the ball action
            NormalPositioning,
            FinalPointBasePositioning,
            GotoTargetPoint,
#endif
        };
    };

    class SupportPlayerMove {
    private:
        int M_unum;
        rcsc::Vector2D M_target_point;

    public:
        SupportPlayerMove( int unum, const rcsc::Vector2D & target_point )
            : M_unum( unum ), M_target_point( target_point )
          { }

        int unum() const
          {
              return M_unum;
          }

        const rcsc::Vector2D & getTargetPoint() const
          {
              return M_target_point;
          }
    };

    typedef std::vector<SupportPlayerMove> SupportPlayerMovesCont;

private:
    const char * M_tag;
    ActionType::Type M_type;
    rcsc::Vector2D M_target_point;
    int M_target_player_unum;
    double M_ball_speed;

    SupportPlayerMovesCont M_support_player_moves;

public:
    CooperativeAction( const ActionType::Type & type,
                       const char * tag = static_cast<char *>(0) )
        : M_tag( tag )
        , M_type( type )
        , M_target_point()
        , M_target_player_unum( -1 )
        , M_ball_speed( 0.0 )
        , M_support_player_moves()
      { }

    CooperativeAction( const ActionType::Type & type,
                       const rcsc::Vector2D & target_point,
                       int target_player_unum,
                       const char * tag = static_cast<char *>(0) )
        : M_tag( tag )
        , M_type( type )
        , M_target_point( target_point )
        , M_target_player_unum( target_player_unum )
        , M_ball_speed( 0.0 )
        , M_support_player_moves()
      { }

    CooperativeAction( const ActionType::Type & type,
                       const rcsc::Vector2D & target_point,
                       int target_player_unum,
                       double ball_speed,
                       const char * tag = static_cast<char *>(0) )
        : M_tag( tag )
        , M_type( type )
        , M_target_point( target_point )
        , M_target_player_unum( target_player_unum )
        , M_ball_speed( ball_speed )
        , M_support_player_moves()
      { }

    void addSupportPlayerMove( const SupportPlayerMove & move )
      {
          M_support_player_moves.push_back( move );
      }

    const char * tag() const
      {
          return M_tag;
      }

    const ActionType::Type & type() const
      {
          return M_type;
      }

    const rcsc::Vector2D & getTargetPoint() const
      {
          return M_target_point;
      }

    int getTargetPlayerUnum() const
      {
          return M_target_player_unum;
      }

    double getBallSpeed() const
      {
          return M_ball_speed;
      }

    const SupportPlayerMovesCont & getSupportPlayerMoves() const
      {
          return M_support_player_moves;
      }
};


/*!
  \class ActionStatePair
  \brief a pair of a cooperative action and predicted state when this action performed
*/
class ActionStatePair {

public:
    CooperativeAction action;
    PredictState state;

public:
    ActionStatePair( const CooperativeAction & action,
                     const PredictState & state )
        : action( action ), state( state )
      { }

    ActionStatePair( const ActionStatePair & rhs )
        : action( rhs.action ), state( rhs.state )
      { }

    ActionStatePair & operator=( const ActionStatePair & rhs )
      {
          this->action = rhs.action;
          this->state = rhs.state;

          return *this;
      }
};


/*!
  \class FieldEvaluator
  \brief abstract field evaluator function object class
*/
class FieldEvaluator {

protected:
    /*!
      \brief protected constructor to inhibit instantiation of this class
     */
    FieldEvaluator()
      { }

public:
    /*!
      \brief virtual destructor
     */
    virtual
    ~FieldEvaluator()
      { }

    /*!
      \brief evaluation function
      \param wm world model to evaluate
      \return evaluation value of world model
     */
    virtual
    double operator() ( const PredictState & state,
                        const rcsc::WorldModel & wm,
                        const std::vector< ActionStatePair > & path ) const = 0;
};

/*!
  \class CooperativeActionCandidateGenerator
  \brief a class to abstruct algorithms which generating action state pairs
*/
class CooperativeActionCandidateGenerator {

protected:
    /*!
      \brief protected constructor to inhibit instantiation of this class
     */
    CooperativeActionCandidateGenerator()
      { }

public:
    /*!
      \brief virtual destructor
     */
    virtual
    ~CooperativeActionCandidateGenerator()
      { }

    /*!
      \brief genarate candidates
      \param state assumption of state to generate action and new state
      \param path sequence of action-state pairs from beginning to `state'
      \param current_wm world model of initial state
      \return generated action-state pairs
     */
    virtual
    void addCandidates( std::vector< ActionStatePair > * result,
                        const PredictState & state,
                        const rcsc::WorldModel & current_wm,
                        const std::vector< ActionStatePair > & path ) const = 0;
};


class CompositeCooperativeActionCandidateGenerator
    : public CooperativeActionCandidateGenerator {

private:
    std::vector< boost::shared_ptr< const CooperativeActionCandidateGenerator > > M_generators;

public:
    void addGenerator( const CooperativeActionCandidateGenerator * g )
      {
          if ( ! g )
          {
              return;
          }

          M_generators.push_back
              ( boost::shared_ptr<const CooperativeActionCandidateGenerator>
                ( g ) );
      }


public:
    /*!
      \brief constructor
      \param g1 generator 1
      \param g2 generator 2
      \param g3 generator 3
      \param g4 generator 4
      \param g5 generator 5
      \param g6 generator 6
      \param g7 generator 7
      \param g8 generator 8
     */
    CompositeCooperativeActionCandidateGenerator
      ( const CooperativeActionCandidateGenerator * g1 = 0,
        const CooperativeActionCandidateGenerator * g2 = 0,
        const CooperativeActionCandidateGenerator * g3 = 0,
        const CooperativeActionCandidateGenerator * g4 = 0,
        const CooperativeActionCandidateGenerator * g5 = 0,
        const CooperativeActionCandidateGenerator * g6 = 0,
        const CooperativeActionCandidateGenerator * g7 = 0,
        const CooperativeActionCandidateGenerator * g8 = 0 )
        : M_generators()
      {
          addGenerator( g1 );
          addGenerator( g2 );
          addGenerator( g3 );
          addGenerator( g4 );
          addGenerator( g5 );
          addGenerator( g6 );
          addGenerator( g7 );
          addGenerator( g8 );
      }

    /*!
      \brief destructor
     */
    ~CompositeCooperativeActionCandidateGenerator()
      { }

    /*!
      \brief genarate candidates
      \param state assumption of world
      \return candidates of action
     */
    void addCandidates( std::vector< ActionStatePair > * result,
                        const PredictState & state,
                        const rcsc::WorldModel & current_wm,
                        const std::vector< ActionStatePair > & path ) const
    {
        for ( size_t i = 0; i < M_generators.size(); ++i )
        {
            M_generators[i] -> addCandidates( result, state, current_wm, path );
        }
    }
};


class RecursiveActionChainGraph {

private:
    static const size_t MAX_RECURSIVE_LEVEL;

private:
    boost::shared_ptr< const FieldEvaluator > M_evaluator;
    boost::shared_ptr< const CooperativeActionCandidateGenerator > M_action_generator;

private:
    std::vector< ActionStatePair > M_result;

    void calculateResult( const WorldModel & wm );
    bool doRecursiveSearch( const WorldModel & wm,
                            const PredictState & state,
                            const std::vector< ActionStatePair > & path,
                            std::vector< ActionStatePair > * result,
                            double * result_evaluation,
                            unsigned long * n_evaluated );

public:
    /*!
      \brief constructor
      \param evaluator evaluator of each state
      \param generator action and state generator
      \param agent pointer to player agent
     */
    RecursiveActionChainGraph
      ( const boost::shared_ptr< const FieldEvaluator > & evaluator,
        const boost::shared_ptr< const CooperativeActionCandidateGenerator > & generator,
        const WorldModel & wm )
        : M_evaluator( evaluator )
        , M_action_generator( generator )
      {
          calculateResult( wm );
      }

    const std::vector< ActionStatePair > & getAllChain() const
      {
          return M_result;
      };

    const CooperativeAction & getFirstAction() const
      {
          return (*(M_result.begin())).action;
      };

    const PredictState & getFirstState() const
      {
          return (*(M_result.begin())).state;
      };

    const PredictState & getFinalResult() const
      {
          return (*(M_result.rbegin())).state;
      };

public:
    void setDebugPassRoute( PlayerAgent * agent ) const;

    void writeChainInfoLog( const std::string & pre_log_message,
                            const std::vector< ActionStatePair > & path,
                            const PredictState & state,
                            double eval ) const;

    void writeChainInfoLog( const std::vector< ActionStatePair > & path,
                            const PredictState & state,
                            double eval ) const;
};

}

#endif
