// -*-c++-*-

/*
 *Copyright:

 Copyright (C) Hiroki SHIMORA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "recursive_action_chain_graph.h"
#include <rcsc/player/player_agent.h>
#include <rcsc/common/server_param.h>
#include <rcsc/common/logger.h>
#include <rcsc/time/timer.h>
#include <cfloat>
#include <string>
#include <sstream>

#if 0
# define RECURSIVE_ACTION_CHAIN_DEBUG
#endif

#ifdef RECURSIVE_ACTION_CHAIN_DEBUG
# include <iostream>
#endif

namespace rcsc {

const size_t RecursiveActionChainGraph::MAX_RECURSIVE_LEVEL = 3;

void
RecursiveActionChainGraph::calculateResult( const WorldModel & wm )
{
    M_result.clear();

    PredictState current_state( wm );

    std::vector< ActionStatePair > empty_path;
    double evaluation;
    unsigned long n_evaluated = 0;

#ifdef RECURSIVE_ACTION_CHAIN_DEBUG
    rcsc::Timer stopwatch;
#endif
    doRecursiveSearch( wm, current_state, empty_path,
                       &M_result, &evaluation, &n_evaluated );

#ifdef RECURSIVE_ACTION_CHAIN_DEBUG
    const double msec = stopwatch.elapsedReal();
    std::cerr << "# recursive search: " << msec << " msec"
              << ", n = " << n_evaluated;
    if ( n_evaluated != 0 )
    {
        std::cerr << ", average = " << msec / n_evaluated;
    }
    std::cerr << std::endl;
    std::cerr << std::flush;
#endif

    if ( M_result.empty() )
    {
        M_result.push_back( ActionStatePair
                            ( CooperativeAction
                              ( CooperativeAction::ActionType::Hold ),
                              PredictState( current_state, 1 ) ) );
    }

    writeChainInfoLog( "best chain: ", M_result, current_state, evaluation );
}


bool
RecursiveActionChainGraph::doRecursiveSearch( const WorldModel & wm,
                                              const PredictState & state,
                                              const std::vector< ActionStatePair > & path,
                                              std::vector< ActionStatePair > * result,
                                              double * result_evaluation,
                                              unsigned long * n_evaluated )
{
    if ( path.size() >= MAX_RECURSIVE_LEVEL )
    {
        return false;
    }


    //
    // generate candidates
    //
    std::vector< ActionStatePair > candidates;
    M_action_generator->addCandidates( &candidates, state, wm, path );


    //
    // check exist candidate
    //
    if ( candidates.empty() )
    {
        return false;
    }


    //
    // test each candidate
    //
    std::vector< ActionStatePair > best_path = path;
    double max_ev = (*M_evaluator)( state, wm, path );
    ++ (*n_evaluated);

    for ( std::vector< ActionStatePair >::const_iterator it = candidates.begin();
          it != candidates.end(); ++it )
    {
        std::vector< ActionStatePair > new_path = path;
        std::vector< ActionStatePair > candidate_result;

        new_path.push_back( *it );

        double ev;
        const bool exist = doRecursiveSearch( wm,
                                              (*it).state,
                                              new_path,
                                              &candidate_result,
                                              &ev,
                                              n_evaluated );
        if ( ! exist )
        {
            continue;
        }

        writeChainInfoLog( candidate_result, state, ev );

        if ( ev > max_ev )
        {
            max_ev = ev;
            best_path = candidate_result;
        }
    }

    *result = best_path;
    *result_evaluation = max_ev;

    return true;
}


void
RecursiveActionChainGraph::setDebugPassRoute( PlayerAgent * agent ) const
{
    const double DIRECT_PASS_DIST = 3.0;

    const std::vector< ActionStatePair > & path = M_result;
    const PredictState current_state( agent->world() );

    for ( size_t i = 0; i < path.size(); ++i )
    {
        std::string action_string = "?";
        const CooperativeAction & action = path[i].action;
        const PredictState * s0;
        const PredictState * s1;

        if ( i == 0 )
        {
            s0 = &current_state;
            s1 = &(path[0].state);
        }
        else
        {
            s0 = &(path[i - 1].state);
            s1 = &(path[i].state);
        }


        switch( action.type() )
        {
        case rcsc::CooperativeAction::ActionType::Hold:
            {
                action_string = "hold";
                break;
            }

        case rcsc::CooperativeAction::ActionType::Dribble:
            {
                action_string = "dribble";

                agent->debugClient().addLine( s0->ballPos(), s1->ballPos() );
                break;
            }

        case rcsc::CooperativeAction::ActionType::Pass:
            {
                action_string = "pass";

                if ( i == 0 )
                {
                    agent->debugClient().setTarget( action.getTargetPlayerUnum() );

                    if ( (s1->ballPos() - s0->teammate( action.getTargetPlayerUnum() )->pos() ).r() > DIRECT_PASS_DIST )
                    {
                        agent->debugClient().addLine( s0->ballPos(), s1->ballPos() );
                    }
                }
                else
                {
                    agent->debugClient().addLine( s0->ballPos(), s1->ballPos() );
                }

                break;
            }

        case rcsc::CooperativeAction::ActionType::Shoot:
            {
                action_string = "shoot";

                agent->debugClient().addLine
                    ( s0->ballPos(),
                      rcsc::Vector2D( rcsc::ServerParam::i().pitchHalfLength(),
                                      0.0 ) );

                break;
            }

        default:
            {
                action_string = "?";

                break;
            }
        }

        const CooperativeAction::SupportPlayerMovesCont::const_iterator end = action.getSupportPlayerMoves().end();
        for ( CooperativeAction::SupportPlayerMovesCont::const_iterator it = action.getSupportPlayerMoves().begin();
              it != end;
              ++it)
        {
            agent->debugClient().addLine
                ( current_state.teammate( (*it).unum() )->pos(),
                  (*it).getTargetPoint() );

            agent->debugClient().addLine
                ( current_state.teammate( (*it).unum() )->pos() + rcsc::Vector2D( 0.0, + 2.0 ),
                  (*it).getTargetPoint() );
            agent->debugClient().addLine
                ( current_state.teammate( (*it).unum() )->pos() + rcsc::Vector2D( 0.0, - 2.0 ),
                  (*it).getTargetPoint() );
        }

        if ( action.tag() )
        {
            action_string += '|';
            action_string += action.tag();
        }

        agent->debugClient().addMessage( "%s", action_string.c_str() );

        rcsc::dlog.addText( rcsc::Logger::ACTION,
                            "chain %d %s, time = %lu",
                            i, action_string.c_str(), s1->spendTime() );
    }
}


void
RecursiveActionChainGraph::writeChainInfoLog( const std::vector< ActionStatePair > & path,
                                              const PredictState & state,
                                              double eval ) const
{
    writeChainInfoLog( "", path, state, eval );
}

void
RecursiveActionChainGraph::writeChainInfoLog( const std::string & pre_log_message,
                                              const std::vector< ActionStatePair > & path,
                                              const PredictState & state,
                                              double eval ) const
{
    std::stringstream buf;

    buf << pre_log_message;

    for ( size_t i = 0; i < path.size(); ++i )
    {
        buf << '(' << i << ": " ;

        std::string action_string = "?";
        const CooperativeAction & a = path[i].action;
        const PredictState * s0;
        const PredictState * s1;

        if ( i == 0 )
        {
            s0 = &state;
            s1 = &(path[0].state);
        }
        else
        {
            s0 = &(path[i - 1].state);
            s1 = &(path[i].state);
        }


        switch( a.type() )
        {
        case rcsc::CooperativeAction::ActionType::Hold:
            {
                buf << "hold";
                break;
            }

        case rcsc::CooperativeAction::ActionType::Dribble:
            {
                buf << "dribble unum " << s0->ballHolderUnum()
                    << " target = [" << s1->ballPos() << "]";

                break;
            }

        case rcsc::CooperativeAction::ActionType::Pass:
            {
                buf << "pass from " << s0->ballHolderUnum()
                    << " to " << s1->ballHolderUnum()
                    << " " << a.getTargetPoint();
                break;
            }

        case rcsc::CooperativeAction::ActionType::Shoot:
            {
                buf << "shoot";

                break;
            }

        default:
            {
                buf << "?";

                break;
            }
        }

        if ( a.tag() )
        {
            buf << " " << a.tag();
        }

        buf << " t=" << s1->spendTime();

        buf << ")";

        rcsc::dlog.addText( rcsc::Logger::ACTION,
                            __FILE__": %s: %f", buf.str().c_str(), eval );
    }
}

}
