// -*-c++-*-

/*!
  \file body_pass.h
  \brief advanced pass planning & behavior.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifndef RCSC_ACTION_BODY_PASS_H
#define RCSC_ACTION_BODY_PASS_H

#include <functional>

#include <rcsc/geom/vector_2d.h>
#include <rcsc/player/player_agent.h>
#include <rcsc/player/soccer_action.h>

namespace rcsc {

class WorldModel;
class PlayerObject;

/*!
  \class Body_Pass
  \brief advanced pass planning & behavior.
 */
class Body_Pass
    : public BodyAction {
public:
    enum PassType {
        DIRECT  = 1,
        LEAD    = 2,
        THROUGH = 3
    };

    /*!
      \struct PassRoute
      \brief pass route information object, that contains type,
      receiver info, receive point and ball first speed.
     */
    struct PassRoute {
        PassType type_;
        const PlayerObject * receiver_;
        Vector2D receive_point_;
        double first_speed_;
        bool one_step_kick_;
        double score_;

        PassRoute( PassType type,
                   const PlayerObject * receiver,
                   const Vector2D & point,
                   const double & speed,
                   const bool one_step_kick )
            : type_( type )
            , receiver_( receiver )
            , receive_point_( point )
            , first_speed_( speed )
            , one_step_kick_( one_step_kick )
            , score_( 0.0 )
          { }
    };

    class PassRouteScoreComp
        : public std::binary_function< PassRoute, PassRoute, bool > {
    public:
        result_type operator()( const first_argument_type & lhs,
                                const second_argument_type & rhs ) const
          {
              return lhs.score_ < rhs.score_;
          }
    };

private:

    //! cached calculated pass data
    static std::vector< PassRoute > S_cached_pass_route;


public:
    /*!
      \brief accessible from global.
    */
    Body_Pass()
      { }

    /*!
      \brief execute action
      \param agent pointer to the agent itself
      \return true if action is performed
    */
    bool execute( PlayerAgent * agent );

    /*!
      \brief calculate best pass route
      \param world consr rerefence to the WorldModel
      \param target_point receive target point is stored to this
      \param first_speed ball first speed is stored to this
      \param receiver receive number is stored to this
      \return true if pass route is found.
    */
    static
    bool get_best_pass( const WorldModel & world,
                        Vector2D * target_point,
                        double * first_speed,
                        int * receiver );

private:
    static
    void create_routes( const WorldModel & world );

    static
    void create_direct_pass( const WorldModel & world,
                             const PlayerObject * teammates );
    static
    void create_lead_pass( const WorldModel & world,
                           const PlayerObject * teammates );
    static
    void create_through_pass( const WorldModel & world,
                              const PlayerObject * teammates );

    static
    bool verify_direct_pass( const WorldModel & world,
                             const PlayerObject * receiver,
                             const Vector2D & target_point,
                             const double & target_dist,
                             const AngleDeg & target_angle,
                             const double & first_speed );
    static
    bool verify_through_pass( const WorldModel & world,
                              const PlayerObject * receiver,
                              const Vector2D & receiver_pos,
                              const Vector2D & target_point,
                              const double & target_dist,
                              const AngleDeg & target_angle,
                              const double & first_speed,
                              const double & reach_step );

    static
    void evaluate_routes( const WorldModel & world );

    static
    bool can_kick_by_one_step( const WorldModel & world,
                               const double & first_speed,
                               const AngleDeg & target_angle );
};

}

#endif
