package jp.ac.nagoyau.is.ss.kishii.suntori.message.information;

import java.util.EnumMap;
import java.util.List;

import jp.ac.nagoyau.is.ss.kishii.suntori.message.BaseMessageType;
import jp.ac.nagoyau.is.ss.kishii.suntori.message.data.DataType;
import jp.ac.nagoyau.is.ss.kishii.suntori.message.data.EntityIDData;
import rescuecore2.worldmodel.EntityID;

/**
 * 通行不可を表すクラスです<br>
 * The UnpassableInformation show the information that some area is not
 * available to go to the neighbor area.
 * 
 * @author takefumi
 * 
 */
public class UnpassableInformation extends WorldInformation {
	/**
	 * コンストラクタ<br>
	 * 移動不可能情報を生成します．<br>
	 * 含まれる情報は以下のものです．
	 * <ul>
	 * <li>time:情報生成時に設定した(情報が作成されたと考えられる)時間</li>
	 * <li>platoonID:救助隊のEntityID</li>
	 * <li>from:移動元のエリアID</li>
	 * <li>to:移動先のエリアID</li>
	 * <li>blockade:移動不可能の原因となっている閉塞のEntityID</li>
	 * </ul>
	 * <h2>Constructor</h2> Create the unpassable information.<br>
	 * Included data are follow.
	 * <ul>
	 * <li>time:the time that the message is created.(int)</li>
	 * <li>platoonID:EntityID of the rescue agent(pf,at,fb).</li>
	 * <li>from:EntityID of area that the agent is standing</li>
	 * <li>to:EntityID of destination area</li>
	 * <li>blockade:EntityID of blockade that is a cause of the unpassable
	 * situation</li>
	 * </ul>
	 * 
	 * @param time
	 *            ステップ数<br>
	 *            step num
	 * @param platoonID
	 *            救助隊のEntityID <br>
	 *            EntityID of the rescue agent
	 * @param from
	 *            移動元エリアのEntityID<br>
	 *            EntityID of area that the agent is standing.
	 * @param to
	 *            移動先エリアのEntityID列<br>
	 *            EntityID of destination area
	 * @param blockade
	 *            移動不可能の原因となっている閉塞のEntityID<br>
	 *            EntityID of blockade that is a cause of the unppasable
	 *            situation.
	 */
	public UnpassableInformation(int time, EntityID platoonID, EntityID from,
			EntityID to, EntityID blockade) {
		super(BaseMessageType.UNPASSABLE, time);
		super.setData(new EntityIDData(DataType.PLATOON_AGENT, platoonID));
		super.setData(new EntityIDData(DataType.AREA, from), 0);
		super.setData(new EntityIDData(DataType.AREA, to), 1);
		super.setData(new EntityIDData(DataType.BLOCKADE, blockade));
	}

	/**
	 * コンバート時にシステムが使用するコンストラクタ <br>
	 * The method that the library use to convert the message.
	 * 
	 * @param bitList
	 * @param offset
	 * @param bitSizeMap
	 */
	public UnpassableInformation(List<Integer> bitList, int offset,
			EnumMap<DataType, Integer> bitSizeMap) {
		this(BaseMessageType.UNPASSABLE, bitList, offset, bitSizeMap);
	}

	protected UnpassableInformation(BaseMessageType type,
			List<Integer> bitList, int offset,
			EnumMap<DataType, Integer> bitSizeMap) {
		super(type, bitList, offset, bitSizeMap);
	}

	/**
	 * 救助隊のEntityIDを取得します．<br>
	 * Return EntityID of the rescue agent.
	 * 
	 * @return 救助隊のEntityID
	 */
	public EntityID getAgentID() {
		return super.getID(DataType.PLATOON_AGENT, 0);
	}

	@Override
	public EntityID getEntityID() {
		return this.getAgentID();
	}

	/**
	 * 移動元エリアのEntityIDを取得します．<br>
	 * Return EntityID of area that the agent is standing
	 * 
	 * @return エリアID<br>
	 *         EntityID of area
	 */
	public EntityID getFromAreaID() {
		return super.getID(DataType.AREA, 0);
	}

	/**
	 * 移動先エリアのEntityID列<br>
	 * EntityID of destination area
	 * 
	 * @return エリアID<br>
	 *         EntityID of area
	 */
	public EntityID getToAreaID() {
		return super.getID(DataType.AREA, 1);
	}

	/**
	 * 移動不可能の原因となっている閉塞のEntityID<br>
	 * EntityID of blockade that is a cause of the unppasable situation.
	 * 
	 * @return EntityID of blockade
	 */
	public EntityID getBLockadeID() {
		return super.getID(DataType.BLOCKADE, 0);
	}
}
