/*************************************************************************************************
 * Implementation of common features
 *                                                      Copyright (C) 2003-2005 Mikio Hirabayashi
 * This file is part of RBBS, a web-based bulletin board system.
 * RBBS is free software; you can redistribute it and/or modify it under the terms of the GNU
 * General Public License as published by the Free Software Foundation; either version 2 of the
 * License, or any later version.
 * RBBS is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License along with RBBS;
 * if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA.
 *************************************************************************************************/


#include "rbbscommon.h"


/* private function prototypes */
static int validxmlent(const char *xml);
static char *uniqueid(const char *prefix);
static int validid(const char *id, const char *prefix);
static char *datestr(time_t t);
static int validdate(const char *date);
static const char *wdayname(int wday);
static const char *monname(int mon);
static int strisprintable(const char *str);
static const char *skiplabel(const char *str);
static void wikicat(CBDATUM *buf, const char *text, int trim);
static char *headtohtml(ARTICLE *art);
static char *bodytohtml(ARTICLE *art, const char *systemuri, int num);
static char *tailtohtml(ARTICLE *art, int num);
const char *artsubclass(const char *flags);
const char *ressubclass(const char *flags);
static const char *hashclass(const char *prefix, const char *str);
static const char *latexverbstr(void);



/*************************************************************************************************
 * public objects
 *************************************************************************************************/


/* Get a handle connected to the databases. */
RBBS *rbbsopen(const char *name, int writer){
  RBBS *rbbs;
  DEPOT *artdb, *fileidx;
  VILLA *cdidx, *mdidx;
  CURIA *filedb;
  time_t t;
  char path[PATHBUFSIZ], *date;
  assert(name);
  artdb = NULL;
  cdidx = NULL;
  mdidx = NULL;
  if(writer) mkdir(name, 00755);
  sprintf(path, "%s/%s", name, ARTDBNAME);
  artdb = dpopen(path, writer ? (DP_OWRITER | DP_OCREAT) : DP_OREADER, -1);
  sprintf(path, "%s/%s", name, CDIDXNAME);
  cdidx = vlopen(path, writer ? (VL_OWRITER | VL_OCREAT) : VL_OREADER, VL_CMPLEX);
  sprintf(path, "%s/%s", name, MDIDXNAME);
  mdidx = vlopen(path, writer ? (VL_OWRITER | VL_OCREAT) : VL_OREADER, VL_CMPLEX);
  sprintf(path, "%s/%s", name, FILEDBNAME);
  filedb = cropen(path, writer ? (CR_OWRITER | CR_OCREAT) : CR_OREADER, -1, -1);
  sprintf(path, "%s/%s", name, FILEIDXNAME);
  fileidx = dpopen(path, writer ? (DP_OWRITER | DP_OCREAT) : DP_OREADER, -1);
  if(!artdb || !cdidx || !mdidx || !filedb){
    if(fileidx) dpclose(fileidx);
    if(filedb) crclose(filedb);
    if(mdidx) vlclose(mdidx);
    if(cdidx) vlclose(cdidx);
    if(artdb) dpclose(artdb);
    return NULL;
  }
  t = time(NULL);
  if(writer){
    date = datestr(t);
    if(dprnum(artdb) < 1) dpput(artdb, CDATEKEY, -1, date, -1, DP_DOVER);
    dpput(artdb, MDATEKEY, -1, date, -1, DP_DOVER);
    free(date);
  }
  dpsetalign(artdb, -2);
  rbbs = cbmalloc(sizeof(RBBS));
  rbbs->artdb = artdb;
  rbbs->cdidx = cdidx;
  rbbs->mdidx = mdidx;
  rbbs->idx = NULL;
  rbbs->next = NULL;
  rbbs->filedb = filedb;
  rbbs->fileidx = fileidx;
  if(!(rbbs->cdate = dpget(artdb, CDATEKEY, -1, 0, -1, NULL))) rbbs->cdate = datestr(t);
  if(!(rbbs->mdate = dpget(artdb, MDATEKEY, -1, 0, -1, NULL))) rbbs->mdate = datestr(t);
  rbbs->errmsg = NULL;
  return rbbs;
}


/* Release a database handle. */
int rbbsclose(RBBS *rbbs){
  int err;
  assert(rbbs);
  err = FALSE;
  free(rbbs->errmsg);
  free(rbbs->mdate);
  free(rbbs->cdate);
  if(!dpclose(rbbs->fileidx)) err = TRUE;
  if(!crclose(rbbs->filedb)) err = TRUE;
  if(!vlclose(rbbs->mdidx)) err = TRUE;
  if(!vlclose(rbbs->cdidx)) err = TRUE;
  if(!dpclose(rbbs->artdb)) err = TRUE;
  free(rbbs);
  return err ? FALSE : TRUE;
}


/* Remove a database directory. */
int rbbsremove(const char *name){
  char path[PATHBUFSIZ];
  assert(name);
  sprintf(path, "%s/%s", name, ARTDBNAME);
  dpremove(path);
  sprintf(path, "%s/%s", name, CDIDXNAME);
  vlremove(path);
  sprintf(path, "%s/%s", name, MDIDXNAME);
  vlremove(path);
  sprintf(path, "%s/%s", name, FILEDBNAME);
  crremove(path);
  sprintf(path, "%s/%s", name, FILEIDXNAME);
  dpremove(path);
  if(rmdir(name) == -1) return FALSE;
  return TRUE;
}


/* Get the error message of a database. */
const char *rbbsemsg(RBBS *rbbs){
  assert(rbbs);
  if(!rbbs->errmsg) return "unknown error";
  return rbbs->errmsg;
}


/* Get the creation date of a database. */
const char *rbbscdate(RBBS *rbbs){
  assert(rbbs);
  return rbbs->cdate;
}


/* Get the modification date of a database. */
const char *rbbsmdate(RBBS *rbbs){
  assert(rbbs);
  return rbbs->mdate;
}


/* Get the number of articles in a database. */
int rbbsanum(RBBS *rbbs){
  assert(rbbs);
  return dprnum(rbbs->artdb) - 2;
}


/* Optimize a database. */
int rbbsoptimize(RBBS *rbbs){
  assert(rbbs);
  if(!dpoptimize(rbbs->artdb, -1)){
    if(!rbbs->errmsg)
      rbbs->errmsg = cbsprintf("DB-Error: the database for articles is broken or not writable");
    return FALSE;
  }
  if(!vloptimize(rbbs->cdidx)){
    if(!rbbs->errmsg)
      rbbs->errmsg = cbsprintf("DB-Error: the database for creation date is broken");
    return FALSE;
  }
  if(!vloptimize(rbbs->mdidx)){
    if(!rbbs->errmsg)
      rbbs->errmsg = cbsprintf("DB-Error: the database for modification date is broken");
    return FALSE;
  }
  if(!croptimize(rbbs->filedb, -1)){
    if(!rbbs->errmsg)
      rbbs->errmsg = cbsprintf("DB-Error: the database for file entities is broken");
    return FALSE;
  }
  if(!dpoptimize(rbbs->fileidx, -1)){
    if(!rbbs->errmsg)
      rbbs->errmsg = cbsprintf("DB-Error: the database for file attributes is broken");
    return FALSE;
  }
  return TRUE;
}


/* Set the sorting order of a database. */
void setsortorder(RBBS* rbbs, int sort){
  assert(rbbs);
  switch(sort){
  case SORTCDATEA:
    rbbs->idx = rbbs->cdidx;
    vlcurfirst(rbbs->idx);
    rbbs->next = vlcurnext;
    break;
  case SORTCDATED:
    rbbs->idx = rbbs->cdidx;
    vlcurlast(rbbs->idx);
    rbbs->next = vlcurprev;
    break;
  case SORTMDATEA:
    rbbs->idx = rbbs->mdidx;
    vlcurfirst(rbbs->idx);
    rbbs->next = vlcurnext;
    break;
  case SORTMDATED:
    rbbs->idx = rbbs->mdidx;
    vlcurlast(rbbs->idx);
    rbbs->next = vlcurprev;
    break;
  }
}


/* Get the ID string of the next article. */
char *getnextid(RBBS *rbbs){
  char *val;
  assert(rbbs);
  if(!rbbs->idx || !rbbs->next) return NULL;
  val = vlcurval(rbbs->idx, NULL);
  rbbs->next(rbbs->idx);
  return val;
}


/* Get the list of the ID strings of the articles created in a day. */
CBLIST *getcdatelist(RBBS *rbbs, int year, int month, int day){
  char date[NUMBUFSIZ], *key, *val;
  CBLIST *list;
  assert(rbbs);
  list = cblistopen();
  sprintf(date, "%04d-%02d-%02d", year, month, day);
  vlcurjump(rbbs->cdidx, date, -1, VL_JFORWARD);
  while((key = vlcurkey(rbbs->cdidx, NULL)) != NULL){
    if(!cbstrfwmatch(key, date)){
      free(key);
      break;
    }
    if((val = vlcurval(rbbs->cdidx, NULL)) != NULL){
      cblistpush(list, val, -1);
      free(val);
    }
    free(key);
    vlcurnext(rbbs->cdidx);
  }
  return list;
}


/* Retrieve article from a database. */
ARTICLE *getarticle(RBBS *rbbs, const char *id){
  ARTICLE *art;
  char *xml;
  assert(rbbs && id);
  if(!(xml = dpget(rbbs->artdb, id, -1, 0, -1, NULL))){
    if(!rbbs->errmsg) rbbs->errmsg = cbsprintf("DB-Error: there is no article for \"%s\"", id);
    return NULL;
  }
  art = makearticle(xml);
  free(art->id);
  art->id = cbmemdup(id, -1);
  free(xml);
  return art;
}


/* Post an article and store it into a database. */
int postarticle(RBBS *rbbs, ARTICLE *art, const char *id){
  const char *oid;
  char *xml;
  int err;
  assert(rbbs && art);
  if(id && !deletearticle(rbbs, id)) return FALSE;
  if(!id && dpvsiz(rbbs->artdb, art->id, -1) != -1){
    if(!rbbs->errmsg)
      rbbs->errmsg = cbsprintf("DB-Error: the ID of the article is duplicated");
    return FALSE;
  }
  oid = id ? id : art->id;
  xml = arttoxml(art);
  err = FALSE;
  if(!dpput(rbbs->artdb, oid, -1, xml, -1, DP_DOVER)){
    if(!rbbs->errmsg)
      rbbs->errmsg = cbsprintf("DB-Error: the database for articles is broken or not writable");
    err = TRUE;
  } else if(!vlput(rbbs->cdidx, art->cdate, -1, oid, -1, VL_DDUP)){
    if(!rbbs->errmsg)
      rbbs->errmsg = cbsprintf("DB-Error: the database for creation date is broken");
    err = TRUE;
  } else if(!vlput(rbbs->mdidx, art->mdate, -1, oid, -1, VL_DDUP)){
    if(!rbbs->errmsg)
      rbbs->errmsg = cbsprintf("DB-Error: the database for modification date is broken");
    err = TRUE;
  }
  free(xml);
  return err ? FALSE : TRUE;
}


/* Delete an article existing in a database. */
int deletearticle(RBBS *rbbs, const char *id){
  ARTICLE *oart;
  CBLIST *olist, *nlist;
  const char *val;
  int i, err;
  assert(rbbs && id);
  if(!(oart = getarticle(rbbs, id))) return FALSE;
  err = FALSE;
  if(!dpout(rbbs->artdb, id, -1)){
    err = TRUE;
    if(!rbbs->errmsg)
      rbbs->errmsg = cbsprintf("DB-Error: the database for articles is broken or not writable");
  }
  if(!(olist = vlgetlist(rbbs->cdidx, oart->cdate, -1))) olist = cblistopen();
  nlist = cblistopen();
  for(i = 0; i < cblistnum(olist); i++){
    val = cblistval(olist, i, NULL);
    if(strcmp(val, id)) cblistpush(nlist, val, -1);
  }
  if(cblistnum(olist) == cblistnum(nlist)) err = TRUE;
  if(!vloutlist(rbbs->cdidx, oart->cdate, -1)) err = TRUE;
  if(cblistnum(nlist) > 0 && !vlputlist(rbbs->cdidx, oart->cdate, -1, nlist)) err = TRUE;
  cblistclose(nlist);
  cblistclose(olist);
  if(err && !rbbs->errmsg)
    rbbs->errmsg = cbsprintf("DB-Error: the database for creation date is broken");
  if(!(olist = vlgetlist(rbbs->mdidx, oart->mdate, -1))) olist = cblistopen();
  nlist = cblistopen();
  for(i = 0; i < cblistnum(olist); i++){
    val = cblistval(olist, i, NULL);
    if(strcmp(val, id)) cblistpush(nlist, val, -1);
  }
  if(cblistnum(olist) == cblistnum(nlist)) err = TRUE;
  if(!vloutlist(rbbs->mdidx, oart->mdate, -1)) err = TRUE;
  if(cblistnum(nlist) > 0 && !vlputlist(rbbs->mdidx, oart->mdate, -1, nlist)) err = TRUE;
  cblistclose(nlist);
  cblistclose(olist);
  if(err && !rbbs->errmsg)
    rbbs->errmsg = cbsprintf("DB-Error: the database for creation date is broken");
  freearticle(oart);
  return err ? FALSE : TRUE;
}


/* Get an attribute of a database. */
char *getattribute(RBBS *rbbs, const char *name){
  assert(rbbs && name);
  if(cbstrfwmatch(name, AIDPREFIX)) return NULL;
  if(cbstrfwmatch(name, FIDPREFIX)) return NULL;
  return dpget(rbbs->artdb, name, -1, 0, -1, NULL);
}


/* Set an attribute of a database. */
int setattribute(RBBS *rbbs, const char *name, const char *value){
  assert(rbbs && name);
  if(cbstrfwmatch(name, AIDPREFIX)) return FALSE;
  if(cbstrfwmatch(name, FIDPREFIX)) return FALSE;
  if(value){
    return dpput(rbbs->artdb, name, -1, value, -1, DP_DOVER);
  } else {
    return dpout(rbbs->artdb, name, -1);
  }
}


/* Save a file into a database. */
int savefile(RBBS *rbbs, const char *id, const char *password, const char *subject,
             const char *author, const char *cdate, const char *ptr, int size){
  CBMAP *attrs;
  char *myid, *mbuf, numbuf[NUMBUFSIZ];
  int err, mbsiz;
  assert(rbbs && password && subject && author && cdate && ptr && size >= 0);
  myid = id ? cbmemdup(id, -1) : uniqueid(FIDPREFIX);
  err = FALSE;
  attrs = cbmapopenex(1);
  cbmapput(attrs, "password", -1, makecrypt(password), -1, TRUE);
  cbmapput(attrs, "subject", -1, subject, -1, TRUE);
  cbmapput(attrs, "author", -1, author, -1, TRUE);
  cbmapput(attrs, "cdate", -1, cdate, -1, TRUE);
  sprintf(numbuf, "%d", size);
  cbmapput(attrs, "size", -1, numbuf, -1, TRUE);
  mbuf = cbmapdump(attrs, &mbsiz);
  if(!dpput(rbbs->fileidx, myid, -1, mbuf, mbsiz, DP_DOVER)){
    if(!rbbs->errmsg)
      rbbs->errmsg = cbsprintf("DB-Error: the database for files is broken or not writable");
    err = TRUE;
  }
  if(!crput(rbbs->filedb, myid, -1, ptr, size, DP_DOVER)){
    if(!rbbs->errmsg)
      rbbs->errmsg = cbsprintf("DB-Error: the database for file attriubtes"
                               " is broken or not writable");
    err = TRUE;
  }
  free(mbuf);
  cbmapclose(attrs);
  free(myid);
  return err ? FALSE : TRUE;
}


/* Load a file in a database. */
char *loadfile(RBBS *rbbs, const char *id, int *sp, CBMAP *attrs){
  CBMAP *map;
  const char *tmp;
  char *mbuf;
  int mbsiz;
  assert(rbbs && id);
  if(!(mbuf = dpget(rbbs->fileidx, id, -1, 0, -1, &mbsiz))){
    if(!rbbs->errmsg) rbbs->errmsg = cbsprintf("DB-Error: there is no file for \"%s\"", id);
    return NULL;
  }
  if(attrs){
    map = cbmapload(mbuf, mbsiz);
    if(!(tmp = cbmapget(map, "password", -1, NULL))) tmp = "";
    cbmapput(attrs, "password", -1, tmp, -1, TRUE);
    if(!(tmp = cbmapget(map, "subject", -1, NULL))) tmp = "";
    cbmapput(attrs, "subject", -1, tmp, -1, TRUE);
    if(!(tmp = cbmapget(map, "author", -1, NULL))) tmp = "";
    cbmapput(attrs, "author", -1, tmp, -1, TRUE);
    if(!(tmp = cbmapget(map, "cdate", -1, NULL))) tmp = "";
    cbmapput(attrs, "cdate", -1, tmp, -1, TRUE);
    if(!(tmp = cbmapget(map, "size", -1, NULL))) tmp = "";
    cbmapput(attrs, "size", -1, tmp, -1, TRUE);
    cbmapclose(map);
  }
  free(mbuf);
  if(!sp) return NULL;
  return crget(rbbs->filedb, id, -1, 0, -1, sp);
}


/* Save a file into a database. */
int deletefile(RBBS *rbbs, const char *id){
  int err;
  assert(rbbs && id);
  err = FALSE;
  if(!dpout(rbbs->fileidx, id, -1)){
    if(!rbbs->errmsg) rbbs->errmsg = cbsprintf("DB-Error: there is no file for \"%s\"", id);
    err = TRUE;
  }
  if(!crout(rbbs->filedb, id, -1)) err = TRUE;
  return err ? FALSE : TRUE;
}


/* Get a list of all files in a database. */
CBLIST *listfiles(RBBS *rbbs){
  CBLIST *list;
  char *buf;
  assert(rbbs);
  list = cblistopen();
  criterinit(rbbs->filedb);
  while((buf = criternext(rbbs->filedb, NULL)) != NULL){
    cblistpush(list, buf, -1);
    free(buf);
  }
  cblistsort(list);
  return list;
}


/* Get the number of files in a database. */
int numfiles(RBBS *rbbs){
  assert(rbbs);
  return crrnum(rbbs->filedb);
}


/* Make an object of an article from an XML string. */
ARTICLE *makearticle(const char *xml){
  ARTICLE *art;
  CBLIST *elems, *stack;
  CBMAP *attrs;
  CBDATUM *cont;
  const char *elem, *name, *pname, *ntext, *celem, *val;
  char *rtext;
  int i, j, isart, ishead, isbody, istail, bstart, rstart;
  assert(xml);
  art = cbmalloc(sizeof(ARTICLE));
  art->id = NULL;
  art->password = NULL;
  art->flags = NULL;
  art->language = NULL;
  art->category = NULL;
  art->subject = NULL;
  art->author = NULL;
  art->cdate = NULL;
  art->mdate = NULL;
  art->body = cblistopen();
  art->tail = cblistopen();
  art->errmsg = NULL;
  art->hash = cbsprintf("%08X", dpouterhash(xml, -1));
  rtext = strnormalize(xml, FALSE, FALSE);
  elems = cbxmlbreak(rtext, TRUE);
  free(rtext);
  stack = cblistopen();
  isart = FALSE;
  ishead = FALSE;
  isbody = FALSE;
  istail = FALSE;
  bstart = -1;
  rstart = -1;
  cont = cbdatumopen("", 0);
  for(i = 0; i < cblistnum(elems); i++){
    elem = cblistval(elems, i, NULL);
    ntext = cblistval(elems, i + 1, NULL);
    pname = NULL;
    if(cblistnum(stack) > 0) pname = cblistval(stack, cblistnum(stack) - 1, NULL);
    if(!ntext || ntext[0] == '<') ntext = "";
    if(elem[0] == '<'){
      if(elem[1] == '?' || elem[1] == '!') continue;
      attrs = cbxmlattrs(elem);
      name = cbmapget(attrs, "", 0, NULL);
      if(elem[1] == '/'){
        if(!pname || strcmp(name, pname)){
          art->errmsg = cbsprintf("Invalid-XML: an end tag \"%s\" is mismatching", name);
        } else {
          if(bstart >= 0 &&
             (!strcmp(pname, "topic") || !strcmp(pname, "subtopic") || !strcmp(pname, "para") ||
              !strcmp(pname, "asis") || !strcmp(pname, "list"))){
            for(j = bstart; j <= i; j++){
              celem = cblistval(elems, j, NULL);
              cbdatumcat(cont, celem, -1);
            }
            cblistpush(art->body, cbdatumptr(cont), -1);
            bstart = -1;
            cbdatumsetsize(cont, 0);
          } else if(rstart >= 0 && !strcmp(pname, "response")){
            for(j = rstart; j <= i; j++){
              celem = cblistval(elems, j, NULL);
              cbdatumcat(cont, celem, -1);
            }
            rtext = strnormalize(cbdatumptr(cont), TRUE, TRUE);
            cblistpush(art->tail, rtext, -1);
            free(rtext);
            rstart = -1;
            cbdatumsetsize(cont, 0);
          }
          free(cblistpop(stack, NULL));
        }
      } else {
        if(!strcmp(name, "article")){
          if(pname){
            art->errmsg = cbsprintf("Invalid-XML: \"article\" must be root");
          } else if(isart){
            art->errmsg = cbsprintf("Invalid-XML: \"article\" must be root");
          } else if(cbstrbwmatch(elem, "/>")){
            art->errmsg = cbsprintf("Invalid-XML: \"article\" must not be empty");
          } else {
            isart = TRUE;
            if(!art->id && (val = cbmapget(attrs, "id", -1, NULL)) != NULL)
              art->id = cbmemdup(val, -1);
            if(!art->password && (val = cbmapget(attrs, "password", -1, NULL)) != NULL)
              art->password = cbmemdup(val, -1);
          }
        } else if(!strcmp(name, "head")){
          if(!pname || strcmp(pname, "article")){
            art->errmsg = cbsprintf("Invalid-XML: \"head\" must be a child of \"article\"");
          } else if(ishead){
            art->errmsg = cbsprintf("Invalid-XML: \"head\" must not be duplicated");
          } else if(cbstrbwmatch(elem, "/>")){
            art->errmsg = cbsprintf("Invalid-XML: \"head\" must not be empty");
          } else {
            ishead = TRUE;
          }
        } else if(!strcmp(name, "flags")){
          if(!pname || strcmp(pname, "head")){
            art->errmsg = cbsprintf("Invalid-XML: \"flags\" must be a child of \"head\"");
          } else if(art->flags){
            art->errmsg = cbsprintf("Invalid-XML: \"flags\" must not be duplicated");
          } else if(cbstrbwmatch(elem, "/>")){
            art->errmsg = cbsprintf("Invalid-XML: \"flags\" must not be empty");
          } else {
            rtext = cbxmlunescape(ntext);
            art->flags = strnormalize(rtext, TRUE, TRUE);
            free(rtext);
          }
        } else if(!strcmp(name, "language")){
          if(!pname || strcmp(pname, "head")){
            art->errmsg = cbsprintf("Invalid-XML: \"language\" must be a child of \"head\"");
          } else if(art->language){
            art->errmsg = cbsprintf("Invalid-XML: \"language\" must not be duplicated");
          } else if(cbstrbwmatch(elem, "/>")){
            art->errmsg = cbsprintf("Invalid-XML: \"language\" must not be empty");
          } else {
            rtext = cbxmlunescape(ntext);
            art->language = strnormalize(rtext, TRUE, TRUE);
            free(rtext);
          }
        } else if(!strcmp(name, "category")){
          if(!pname || strcmp(pname, "head")){
            art->errmsg = cbsprintf("Invalid-XML: \"category\" must be a child of \"head\"");
          } else if(art->category){
            art->errmsg = cbsprintf("Invalid-XML: \"category\" must not be duplicated");
          } else if(cbstrbwmatch(elem, "/>")){
            art->errmsg = cbsprintf("Invalid-XML: \"category\" must not be empty");
          } else {
            rtext = cbxmlunescape(ntext);
            art->category = strnormalize(rtext, TRUE, TRUE);
            free(rtext);
          }
        } else if(!strcmp(name, "subject")){
          if(!pname || strcmp(pname, "head")){
            art->errmsg = cbsprintf("Invalid-XML: \"subject\" must be a child of \"head\"");
          } else if(art->subject){
            art->errmsg = cbsprintf("Invalid-XML: \"subject\" must not be duplicated");
          } else if(cbstrbwmatch(elem, "/>")){
            art->errmsg = cbsprintf("Invalid-XML: \"subject\" must not be empty");
          } else {
            rtext = cbxmlunescape(ntext);
            art->subject = strnormalize(rtext, TRUE, TRUE);
            free(rtext);
          }
        } else if(!strcmp(name, "author")){
          if(!pname || strcmp(pname, "head")){
            art->errmsg = cbsprintf("Invalid-XML: \"author\" must be a child of \"head\"");
          } else if(art->author){
            art->errmsg = cbsprintf("Invalid-XML: \"author\" must not be duplicated");
          } else if(cbstrbwmatch(elem, "/>")){
            art->errmsg = cbsprintf("Invalid-XML: \"author\" must not be empty");
          } else {
            rtext = cbxmlunescape(ntext);
            art->author = strnormalize(rtext, TRUE, TRUE);
            free(rtext);
          }
        } else if(!strcmp(name, "cdate")){
          if(!pname || strcmp(pname, "head")){
            art->errmsg = cbsprintf("Invalid-XML: \"cdate\" must be a child of \"head\"");
          } else if(art->cdate){
            art->errmsg = cbsprintf("Invalid-XML: \"cdate\" must not be duplicated");
          } else if(cbstrbwmatch(elem, "/>")){
            art->errmsg = cbsprintf("Invalid-XML: \"cdate\" must not be empty");
          } else {
            rtext = cbxmlunescape(ntext);
            art->cdate = strnormalize(rtext, TRUE, TRUE);
            free(rtext);
          }
        } else if(!strcmp(name, "mdate")){
          if(!pname || strcmp(pname, "head")){
            art->errmsg = cbsprintf("Invalid-XML: \"mdate\" must be a child of \"head\"");
          } else if(art->mdate){
            art->errmsg = cbsprintf("Invalid-XML: \"mdate\" must not be duplicated");
          } else if(cbstrbwmatch(elem, "/>")){
            art->errmsg = cbsprintf("Invalid-XML: \"mdate\" must not be empty");
          } else {
            rtext = cbxmlunescape(ntext);
            art->mdate = strnormalize(rtext, TRUE, TRUE);
            free(rtext);
          }
        } else if(!strcmp(name, "body")){
          if(!pname || strcmp(pname, "article")){
            art->errmsg = cbsprintf("Invalid-XML: \"body\" must be a child of \"article\"");
          } else if(isbody){
            art->errmsg = cbsprintf("Invalid-XML: \"body\" must not be duplicated");
          } else if(cbstrbwmatch(elem, "/>")){
            art->errmsg = cbsprintf("Invalid-XML: \"body\" must not be empty");
          } else {
            isbody = TRUE;
          }
        } else if(!strcmp(name, "topic")){
          if(!pname || strcmp(pname, "body")){
            art->errmsg = cbsprintf("Invalid-XML: \"topic\" must be a child of \"body\"");
          } else if(cbstrbwmatch(elem, "/>")){
            art->errmsg = cbsprintf("Invalid-XML: \"topic\" must not be empty");
          } else {
            bstart = i;
          }
        } else if(!strcmp(name, "subtopic")){
          if(!pname || strcmp(pname, "body")){
            art->errmsg = cbsprintf("Invalid-XML: \"subtopic\" must be a child of \"body\"");
          } else if(cbstrbwmatch(elem, "/>")){
            art->errmsg = cbsprintf("Invalid-XML: \"subtopic\" must not be empty");
          } else {
            bstart = i;
          }
        } else if(!strcmp(name, "para")){
          if(!pname || strcmp(pname, "body")){
            art->errmsg = cbsprintf("Invalid-XML: \"para\" must be a child of \"body\"");
          } else if(cbstrbwmatch(elem, "/>")){
            art->errmsg = cbsprintf("Invalid-XML: \"para\" must not be empty");
          } else {
            bstart = i;
          }
        } else if(!strcmp(name, "asis")){
          if(!pname || strcmp(pname, "body")){
            art->errmsg = cbsprintf("Invalid-XML: \"asis\" must be a child of \"body\"");
          } else if(cbstrbwmatch(elem, "/>")){
            art->errmsg = cbsprintf("Invalid-XML: \"asis\" must not be empty");
          } else {
            bstart = i;
          }
        } else if(!strcmp(name, "list")){
          if(!pname || strcmp(pname, "body")){
            art->errmsg = cbsprintf("Invalid-XML: \"list\" must be a child of \"body\"");
          } else if(cbstrbwmatch(elem, "/>")){
            art->errmsg = cbsprintf("Invalid-XML: \"list\" must not be empty");
          } else {
            bstart = i;
          }
        } else if(!strcmp(name, "item")){
          if(!pname || strcmp(pname, "list")){
            art->errmsg = cbsprintf("Invalid-XML: \"item\" must be a child of \"list\"");
          } else if(cbstrbwmatch(elem, "/>")){
            art->errmsg = cbsprintf("Invalid-XML: \"item\" must not be empty");
          }
        } else if(!strcmp(name, "graph")){
          if(!pname || strcmp(pname, "body")){
            art->errmsg = cbsprintf("Invalid-XML: \"graph\" must be a child of \"body\"");
          } else if(!cbstrbwmatch(elem, "/>")){
            art->errmsg = cbsprintf("Invalid-XML: \"graph\" must be empty");
          } else if(!cbmapget(attrs, "data", -1, NULL)){
            art->errmsg = cbsprintf("Invalid-XML: \"graph\" must have the attribute \"data\"");
          } else {
            cblistpush(art->body, elem, -1);
          }
        } else if(!strcmp(name, "break")){
          if(!pname || strcmp(pname, "body")){
            art->errmsg = cbsprintf("Invalid-XML: \"break\" must be a child of \"body\"");
          } else if(!cbstrbwmatch(elem, "/>")){
            art->errmsg = cbsprintf("Invalid-XML: \"break\" must be empty");
          } else {
            cblistpush(art->body, elem, -1);
          }
        } else if(!strcmp(name, "link")){
          if(!pname || bstart < 0){
            art->errmsg = cbsprintf("Invalid-XML: \"link\" must be in a block");
          } else if(cbstrbwmatch(elem, "/>")){
            art->errmsg = cbsprintf("Invalid-XML: \"link\" must not be empty");
          } else if(!cbmapget(attrs, "to", -1, NULL)){
            art->errmsg = cbsprintf("Invalid-XML: \"link\" must have the attribute \"to\"");
          }
        } else if(!strcmp(name, "emp")){
          if(!pname || bstart < 0){
            art->errmsg = cbsprintf("Invalid-XML: \"emp\" must be in a block");
          } else if(cbstrbwmatch(elem, "/>")){
            art->errmsg = cbsprintf("Invalid-XML: \"emp\" must not be empty");
          }
        } else if(!strcmp(name, "cite")){
          if(!pname || bstart < 0){
            art->errmsg = cbsprintf("Invalid-XML: \"cite\" must be in a block");
          } else if(cbstrbwmatch(elem, "/>")){
            art->errmsg = cbsprintf("Invalid-XML: \"cite\" must not be empty");
          }
        } else if(!strcmp(name, "ins")){
          if(!pname || bstart < 0){
            art->errmsg = cbsprintf("Invalid-XML: \"ins\" must be in a block");
          } else if(cbstrbwmatch(elem, "/>")){
            art->errmsg = cbsprintf("Invalid-XML: \"ins\" must not be empty");
          }
        } else if(!strcmp(name, "del")){
          if(!pname || bstart < 0){
            art->errmsg = cbsprintf("Invalid-XML: \"del\" must be in a block");
          } else if(cbstrbwmatch(elem, "/>")){
            art->errmsg = cbsprintf("Invalid-XML: \"del\" must not be empty");
          }
        } else if(!strcmp(name, "tail")){
          if(!pname || strcmp(pname, "article")){
            art->errmsg = cbsprintf("Invalid-XML: \"tail\" must be a child of \"article\"");
          } else if(istail){
            art->errmsg = cbsprintf("Invalid-XML: \"tail\" must not be duplicated");
          }
          istail = TRUE;
        } else if(!strcmp(name, "response")){
          if(!pname || strcmp(pname, "tail")){
            art->errmsg = cbsprintf("Invalid-XML: \"response\" must be in a block");
          } else if(cbstrbwmatch(elem, "/>")){
            art->errmsg = cbsprintf("Invalid-XML: \"response\" must not be empty");
          } else if(!cbmapget(attrs, "flags", -1, NULL)){
            art->errmsg =
              cbsprintf("Invalid-XML: \"response\" must have the attribute \"flags\"");
          } else if(!cbmapget(attrs, "author", -1, NULL)){
            art->errmsg =
              cbsprintf("Invalid-XML: \"response\" must have the attribute \"author\"");
          } else if(!cbmapget(attrs, "cdate", -1, NULL)){
            art->errmsg =
              cbsprintf("Invalid-XML: \"response\" must have the attribute \"cdate\"");
          } else {
            rstart = i;
          }
        } else {
          art->errmsg = cbsprintf("Invalid-XML: a tag \"%s\" is unknown", name);
        }
        if(!cbstrbwmatch(elem, "/>")) cblistpush(stack, name, -1);
      }
      cbmapclose(attrs);
    } else {
      if(strisprintable(elem)){
        if(!pname) pname = "";
        if(strcmp(pname, "flags") && strcmp(pname, "language") && strcmp(pname, "category") &&
           strcmp(pname, "subject") && strcmp(pname, "author") && strcmp(pname, "cdate") &&
           strcmp(pname, "mdate") && strcmp(pname, "topic") && strcmp(pname, "subtopic") &&
           strcmp(pname, "para") && strcmp(pname, "asis") && strcmp(pname, "item") &&
           strcmp(pname, "link") && strcmp(pname, "emp") && strcmp(pname, "cite") &&
           strcmp(pname, "ins") && strcmp(pname, "del") && strcmp(pname, "response")){
          art->errmsg =
            cbsprintf("Invalid-XML: a printable character must not be included in a block");
        }
      }
    }
    if(art->errmsg) break;
  }
  if(!art->id){
    art->id = uniqueid(AIDPREFIX);
  } else if(!validid(art->id, AIDPREFIX)){
    if(!art->errmsg) art->errmsg = cbsprintf("Invalid-XML: the ID is invalid");
  }
  if(!art->password) art->password = cbmemdup("", 0);
  if(!art->flags) art->flags = cbmemdup("", 0);
  if(!art->language || art->language[0] == '\0'){
    if(!art->errmsg) art->errmsg = cbsprintf("Invalid-XML: the language is undefined");
    if(!art->language) art->language = cbmemdup("", 0);
  }
  if(!art->category || art->category[0] == '\0'){
    if(!art->errmsg) art->errmsg = cbsprintf("Invalid-XML: the category is undefined");
    if(!art->category) art->category = cbmemdup("", 0);
  }
  if(!art->subject || art->subject[0] == '\0'){
    if(!art->errmsg) art->errmsg = cbsprintf("Invalid-XML: the subject is undefined");
    if(!art->subject) art->subject = cbmemdup("", 0);
  }
  if(!art->author || art->author[0] == '\0'){
    if(!art->errmsg) art->errmsg = cbsprintf("Invalid-XML: the author is undefined");
    if(!art->author) art->author = cbmemdup("", 0);
  }
  if(!art->cdate || art->cdate[0] == '\0' || !validdate(art->cdate)){
    if(!art->errmsg)
      art->errmsg = cbsprintf("Invalid-XML: the creation date is undefined or malformed");
    if(!art->cdate) art->cdate = cbmemdup("1970-01-01 00:00:00", 0);
  }
  if(!art->mdate || art->mdate[0] == '\0' || !validdate(art->mdate)){
    if(!art->errmsg)
      art->errmsg = cbsprintf("Invalid-XML: the modification date is undefined or malformed");
    if(!art->mdate) art->mdate = cbmemdup("1970-01-01 00:00:00", 0);
  }
  cbdatumclose(cont);
  cblistclose(stack);
  cblistclose(elems);
  if(!art->errmsg && !validxmlent(xml))
    art->errmsg = cbsprintf("Invalid-XML: there is an unknown entity reference");
  return art;
}


/* Make an object of an article from a wiki-style string. */
ARTICLE *makeartfromwiki(const char *wiki){
  const char *delimsrc = ">>>>>>>>>>>>>>>>";
  const char *topicstr = "!";
  const char *subtpcstr = "!!";
  const char *itemstr = "*";
  const char *asisstr = "|";
  const char *graphstr = "@graph:";
  const char *breakstr = "----";
  ARTICLE *art;
  CBDATUM *buf;
  CBLIST *lines;
  const char *id, *password, *flags, *language, *category, *subject, *author, *cdate, *mdate;
  const char *delim, *cline, *pline, *nline;
  char *rtext;
  int i, bs, be;
  assert(wiki);
  rtext = strnormalize(wiki, FALSE, FALSE);
  buf = cbdatumopen("", 0);
  lines = cbsplit(rtext, -1, "\n");
  id = "";
  password = "";
  flags = "";
  language = "";
  category = "";
  subject = "";
  author = "";
  cdate = "";
  mdate = "";
  delim = delimsrc;
  bs = 0;
  for(i = 0; i < cblistnum(lines); i++){
    cline = cblistval(lines, i, NULL);
    if(cbstrfwmatch(cline, "id:")){
      id = skiplabel(cline);
    } else if(cbstrfwmatch(cline, "password:")){
      password = skiplabel(cline);
    } else if(cbstrfwmatch(cline, "flags:")){
      flags = skiplabel(cline);
    } else if(cbstrfwmatch(cline, "language:")){
      language = skiplabel(cline);
    } else if(cbstrfwmatch(cline, "category:")){
      category = skiplabel(cline);
    } else if(cbstrfwmatch(cline, "subject:")){
      subject = skiplabel(cline);
    } else if(cbstrfwmatch(cline, "author:")){
      author = skiplabel(cline);
    } else if(cbstrfwmatch(cline, "cdate:")){
      cdate = skiplabel(cline);
    } else if(cbstrfwmatch(cline, "mdate:")){
      mdate = skiplabel(cline);
    } else if(cbstrfwmatch(cline, delim)){
      bs = i + 1;
      delim = cline;
      break;
    }
  }
  xsprintf(buf, "<article");
  if(id[0] != '\0') xsprintf(buf, " id=\"%@\"", id);
  if(password[0] != '\0') xsprintf(buf, " password=\"%@\"", password);
  xsprintf(buf, ">\n");
  xsprintf(buf, "<head>\n");
  xsprintf(buf, "<flags>%@</flags>\n", flags);
  xsprintf(buf, "<language>%@</language>\n", language);
  xsprintf(buf, "<category>%@</category>\n", category);
  xsprintf(buf, "<subject>%@</subject>\n", subject);
  xsprintf(buf, "<author>%@</author>\n", author);
  xsprintf(buf, "<cdate>%@</cdate>\n", cdate);
  xsprintf(buf, "<mdate>%@</mdate>\n", mdate);
  xsprintf(buf, "</head>\n");
  xsprintf(buf, "<body>\n");
  be = cblistnum(lines);
  for(i = bs; i < cblistnum(lines); i++){
    if(!strcmp(cblistval(lines, i, NULL), delim)){
      be = i;
      break;
    }
  }
  for(i = bs; i < be; i++){
    pline = i > bs ? cblistval(lines, i - 1, NULL) : "";
    cline = cblistval(lines, i, NULL);
    nline = i < be - 1 ? cblistval(lines, i + 1, NULL) : "";
    if(cline[0] != '\0'){
      if(cbstrfwmatch(cline, subtpcstr)){
        xsprintf(buf, "<subtopic>");
        wikicat(buf, cline + strlen(subtpcstr), TRUE);
        xsprintf(buf, "</subtopic>\n");
      } else if(cbstrfwmatch(cline, topicstr)){
        xsprintf(buf, "<topic>");
        wikicat(buf, cline + strlen(topicstr), TRUE);
        xsprintf(buf, "</topic>\n");
      } else if(cbstrfwmatch(cline, itemstr)){
        if(!cbstrfwmatch(pline, itemstr)) xsprintf(buf, "<list>\n");
        xsprintf(buf, "<item>");
        wikicat(buf, cline + strlen(itemstr), TRUE);
        xsprintf(buf, "</item>\n");
        if(!cbstrfwmatch(nline, itemstr)) xsprintf(buf, "</list>\n");
      } else if(cbstrfwmatch(cline, asisstr)){
        if(!cbstrfwmatch(pline, asisstr)) xsprintf(buf, "<asis>");
        xsprintf(buf, "%s\n", cline + strlen(asisstr));
        if(!cbstrfwmatch(nline, asisstr)) xsprintf(buf, "</asis>\n");
      } else if(cbstrfwmatch(cline, graphstr)){
        cline += strlen(graphstr);
        while(*cline == ' '){
          cline++;
        }
        xsprintf(buf, "<graph data=\"%@\"/>\n", cline);
      } else if(!strcmp(cline, breakstr)){
        xsprintf(buf, "<break/>\n");
      } else {
        xsprintf(buf, "<para>");
        wikicat(buf, cline, TRUE);
        xsprintf(buf, "</para>\n");
      }
    }
  }
  xsprintf(buf, "</body>\n");
  xsprintf(buf, "<tail>\n");
  for(i = be + 1; i < cblistnum(lines) - 3; i += 4){
    flags = cblistval(lines, i, NULL);
    author = cblistval(lines, i + 1, NULL);
    cdate = cblistval(lines, i + 2, NULL);
    cline = cblistval(lines, i + 3, NULL);
    if(author[0] != '\0' && cdate[0] != '\0' && validdate(cdate) && cline != '\0')
      xsprintf(buf, "<response flags=\"%@\" author=\"%@\" cdate=\"%@\">%@</response>\n",
               flags, author, cdate, cline);
  }
  xsprintf(buf, "</tail>\n");
  xsprintf(buf, "</article>\n");
  art = makearticle(cbdatumptr(buf));
  cblistclose(lines);
  cbdatumclose(buf);
  free(rtext);
  return art;
}


/* Release resources of an article object. */
void freearticle(ARTICLE *art){
  assert(art);
  free(art->hash);
  free(art->errmsg);
  cblistclose(art->tail);
  cblistclose(art->body);
  free(art->mdate);
  free(art->cdate);
  free(art->author);
  free(art->subject);
  free(art->category);
  free(art->language);
  free(art->flags);
  free(art->password);
  free(art->id);
  free(art);
}


/* Check the article is OK. */
int articleisok(ARTICLE *art){
  assert(art);
  return art->errmsg ? FALSE : TRUE;
}


/* Get the error message of an article. */
const char *artemsg(ARTICLE *art){
  assert(art);
  if(!art->errmsg) return "unknown error";
  return art->errmsg;
}


/* Get the hash code of an article. */
const char *arthash(ARTICLE *art){
  assert(art);
  return art->hash;
}


/* Get the ID string of an article. */
const char *artid(ARTICLE *art){
  assert(art);
  return art->id;
}


/* Get the password of an article. */
const char *artpassword(ARTICLE *art){
  assert(art);
  return art->password;
}


/* Get the flags of an article. */
const char *artflags(ARTICLE *art){
  assert(art);
  return art->flags;
}


/* Get the language of an article. */
const char *artlanguage(ARTICLE *art){
  assert(art);
  return art->language;
}


/* Get the category of an article. */
const char *artcategory(ARTICLE *art){
  assert(art);
  return art->category;
}


/* Get the subject of an article. */
const char *artsubject(ARTICLE *art){
  assert(art);
  return art->subject;
}


/* Get the author of an article. */
const char *artauthor(ARTICLE *art){
  assert(art);
  return art->author;
}


/* Get the creation date of an article. */
const char *artcdate(ARTICLE *art){
  assert(art);
  return art->cdate;
}


/* Get the modification date of an article. */
const char *artmdate(ARTICLE *art){
  assert(art);
  return art->mdate;
}


/* Get the number of responses of an article. */
int artresnum(ARTICLE *art){
  assert(art);
  return cblistnum(art->tail);
}


/* Add a response to an article. */
void artaddres(ARTICLE *art, const char *flags, const char *author, const char *cdate,
               const char *text){
  CBDATUM *buf;
  assert(art && flags && author && cdate && text);
  if(strchr(art->flags, LONECHAR)) return;
  buf = cbdatumopen("", 0);
  xsprintf(buf, "<response flags=\"%@\" author=\"%@\" cdate=\"%@\">%@</response>",
           flags, author, cdate, text);
  cblistpush(art->tail, cbdatumptr(buf), -1);
  cbdatumclose(buf);
}


/* Set the creation date of an article. */
void artsetcdate(ARTICLE *art, const char *date){
  assert(art && date);
  free(art->cdate);
  art->cdate = cbmemdup(date, -1);
}


/* Set the modification date of an article. */
void artsetmdate(ARTICLE *art, const char *date){
  assert(art && date);
  free(art->mdate);
  art->mdate = cbmemdup(date, -1);
}


/* Serialize an article into a XML string. */
char *arttoxml(ARTICLE *art){
  CBDATUM *buf;
  CBLIST *elems;
  CBMAP *attrs;
  const char *blk, *elem, *name, *val, *flags, *author, *cdate;
  char *rtext;
  int i, j, inlist, initem;
  assert(art);
  buf = cbdatumopen("", 0);
  xsprintf(buf, "<article id=\"%@\" password=\"%@\">\n", art->id, art->password);
  xsprintf(buf, "<head>\n");
  xsprintf(buf, "<flags>%@</flags>\n", art->flags);
  xsprintf(buf, "<language>%@</language>\n", art->language);
  xsprintf(buf, "<category>%@</category>\n", art->category);
  xsprintf(buf, "<subject>%@</subject>\n", art->subject);
  xsprintf(buf, "<author>%@</author>\n", art->author);
  xsprintf(buf, "<cdate>%@</cdate>\n", art->cdate);
  xsprintf(buf, "<mdate>%@</mdate>\n", art->mdate);
  xsprintf(buf, "</head>\n");
  xsprintf(buf, "<body>\n");
  inlist = FALSE;
  initem = FALSE;
  for(i = 0; i < cblistnum(art->body); i++){
    blk = cblistval(art->body, i, NULL);
    elems = cbxmlbreak(blk, FALSE);
    for(j = 0; j < cblistnum(elems); j++){
      elem = cblistval(elems, j, NULL);
      if(elem[0] == '<'){
        attrs = cbxmlattrs(elem);
        name = cbmapget(attrs, "", -1, NULL);
        if(elem[1] == '/'){
          xsprintf(buf, "</%@>", name);
          if(!strcmp(name, "topic") || !strcmp(name, "subtopic") || !strcmp(name, "para") ||
             !strcmp(name, "asis") || !strcmp(name, "list") || !strcmp(name, "item"))
            xsprintf(buf, "\n");
          if(!strcmp(name, "list")){
            inlist = FALSE;
          } else if(!strcmp(name, "item")){
            initem = FALSE;
          }
        } else {
          if(!strcmp(name, "graph")){
            if(!(val = cbmapget(attrs, "data", -1, NULL))) val = "";
            xsprintf(buf, "<graph data=\"%@\"/>\n", val);
          } else if(!strcmp(name, "break")){
            xsprintf(buf, "<break/>\n");
          } else if(!strcmp(name, "link")){
            if(!(val = cbmapget(attrs, "to", -1, NULL))) val = "";
            xsprintf(buf, "<link to=\"%@\">", val);
          } else {
            xsprintf(buf, "<%@>", name);
            if(!strcmp(name, "list")){
              xsprintf(buf, "\n");
              inlist = TRUE;
            } else if(!strcmp(name, "item")){
              initem = TRUE;
            }
          }
        }
        cbmapclose(attrs);
      } else {
        if(!inlist || initem) xsprintf(buf, "%s", elem);
      }
    }
    cblistclose(elems);
  }
  xsprintf(buf, "</body>\n");
  xsprintf(buf, "<tail>\n");
  for(i = 0; i < cblistnum(art->tail); i++){
    blk = cblistval(art->tail, i, NULL);
    elems = cbxmlbreak(blk, FALSE);
    for(j = 0; j < cblistnum(elems); j++){
      elem = cblistval(elems, j, NULL);
      if(elem[0] == '<'){
        attrs = cbxmlattrs(elem);
        name = cbmapget(attrs, "", -1, NULL);
        if(!strcmp(name, "response")){
          if(elem[1] == '/'){
            xsprintf(buf, "</%@>\n", name);
          } else {
            if(!(flags = cbmapget(attrs, "flags", -1, NULL))) flags = "";
            if(!(author = cbmapget(attrs, "author", -1, NULL))) author = "";
            if(!(cdate = cbmapget(attrs, "cdate", -1, NULL))) cdate = "";
            xsprintf(buf, "<response flags=\"%@\" author=\"%@\" cdate=\"%@\">",
                     flags, author, cdate);
          }
        }
        cbmapclose(attrs);
      } else {
        if(strisprintable(elem)){
          rtext = strnormalize(elem, TRUE, TRUE);
          xsprintf(buf, "%s", rtext);
          free(rtext);
        }
      }
    }
    cblistclose(elems);
  }
  xsprintf(buf, "</tail>\n");
  xsprintf(buf, "</article>\n");
  return cbdatumtomalloc(buf, NULL);
}


/* Serialize an article into a string of plain text. */
char *arttotext(ARTICLE *art){
  CBDATUM *buf;
  CBLIST *elems;
  CBMAP *attrs;
  const char *blk, *elem, *name, *tmp;
  char *flags, *author, *cdate, *text, *rtext;
  int i, j;
  assert(art);
  buf = cbdatumopen("", 0);
  xsprintf(buf, "%s\n", art->flags);
  xsprintf(buf, "%s\n", art->language);
  xsprintf(buf, "%s\n", art->category);
  xsprintf(buf, "%s\n", art->subject);
  xsprintf(buf, "%s\n", art->author);
  xsprintf(buf, "%s\n", art->cdate);
  xsprintf(buf, "%s\n", art->mdate);
  xsprintf(buf, "\n");
  for(i = 0; i < cblistnum(art->body); i++){
    blk = cblistval(art->body, i, NULL);
    elems = cbxmlbreak(blk, FALSE);
    for(j = 0; j < cblistnum(elems); j++){
      elem = cblistval(elems, j, NULL);
      if(elem[0] == '<'){
        attrs = cbxmlattrs(elem);
        name = cbmapget(attrs, "", -1, NULL);
        if(!strcmp(name, "graph")){
          xsprintf(buf, "[graph]");
        } else if(!strcmp(name, "break")){
          xsprintf(buf, "------");
        }
        cbmapclose(attrs);
      } else {
        rtext = cbxmlunescape(elem);
        xsprintf(buf, "%s", rtext);
        free(rtext);
      }
    }
    xsprintf(buf, "\n");
    cblistclose(elems);
  }
  xsprintf(buf, "\n");
  for(i = 0; i < cblistnum(art->tail); i++){
    flags = NULL;
    author = NULL;
    cdate = NULL;
    text = NULL;
    blk = cblistval(art->tail, i, NULL);
    elems = cbxmlbreak(blk, FALSE);
    for(j = 0; j < cblistnum(elems); j++){
      elem = cblistval(elems, j, NULL);
      if(elem[0] == '<'){
        attrs = cbxmlattrs(elem);
        name = cbmapget(attrs, "", -1, NULL);
        if(!strcmp(name, "response") && elem[1] != '/'){
          if(!flags && (tmp = cbmapget(attrs, "flags", -1, NULL)) != NULL)
            flags = cbmemdup(tmp, -1);
          if(!author && (tmp = cbmapget(attrs, "author", -1, NULL)) != NULL)
            author = cbmemdup(tmp, -1);
          if(!cdate && (tmp = cbmapget(attrs, "cdate", -1, NULL)) != NULL)
            cdate = dateformen(tmp);
        }
        cbmapclose(attrs);
      } else {
        if(!text && strisprintable(elem)){
          rtext = strnormalize(elem, TRUE, TRUE);
          text = cbxmlunescape(rtext);
          free(rtext);
        }
      }
    }
    cblistclose(elems);
    if(flags && author && cdate && text)
      xsprintf(buf, "%s\t%s\t%s\t%s\n", flags, author, cdate, text);
    free(text);
    free(cdate);
    free(author);
    free(flags);
  }
  return cbdatumtomalloc(buf, NULL);
}


/* Serialize an article into a string of HTML. */
char *arttohtml(ARTICLE *art, const char *systemuri, int bnum, int rnum){
  CBDATUM *buf;
  char *tmp;
  assert(art && systemuri);
  buf = cbdatumopen("", 0);
  xsprintf(buf, "<div id=\"%@\" lang=\"%@\" xml:lang=\"%@\" class=\"article%s\">\n",
           art->id, art->language, art->language, artsubclass(art->flags));
  tmp = headtohtml(art);
  xsprintf(buf, "%s", tmp);
  free(tmp);
  tmp = bodytohtml(art, systemuri, bnum);
  xsprintf(buf, "%s", tmp);
  free(tmp);
  tmp = tailtohtml(art, rnum);
  xsprintf(buf, "%s", tmp);
  free(tmp);
  xsprintf(buf, "</div>\n");
  return cbdatumtomalloc(buf, NULL);
}


/* Serialize an article into a string of Atom. */
char *arttoatom(ARTICLE *art, const char *systemuri, int bnum, int rnum){
  CBDATUM *buf;
  char *tmp;
  assert(art && systemuri);
  buf = cbdatumopen("", 0);
  xsprintf(buf, "<entry>\n");
  xsprintf(buf, "<id>%@?id=%@</id>\n", systemuri, art->id);
  xsprintf(buf, "<link rel=\"alternate\" type=\"text/html\" href=\"%@?id=%@\"/>\n",
           systemuri, art->id);
  xsprintf(buf, "<title>%@</title>\n", art->subject);
  xsprintf(buf, "<author>\n");
  xsprintf(buf, "<name>%@</name>\n", art->author);
  xsprintf(buf, "</author>\n");
  tmp = dateforwww(art->cdate, TRUE);
  xsprintf(buf, "<issued>%@</issued>\n", tmp);
  free(tmp);
  tmp = dateforwww(art->cdate, FALSE);
  xsprintf(buf, "<created>%@</created>\n", tmp);
  free(tmp);
  tmp = dateforwww(art->mdate, FALSE);
  xsprintf(buf, "<modified>%@</modified>\n", tmp);
  free(tmp);
  xsprintf(buf, "<content type=\"text/html\" mode=\"escaped\""
           " xml:base=\"%@?id=%s\" xml:lang=\"%@\">\n", systemuri, art->id, art->language);
  tmp = bodytohtml(art, systemuri, bnum);
  xsprintf(buf, "%@", tmp);
  free(tmp);
  tmp = tailtohtml(art, rnum);
  xsprintf(buf, "%@", tmp);
  free(tmp);
  xsprintf(buf, "</content>\n");
  xsprintf(buf, "</entry>\n");
  return cbdatumtomalloc(buf, NULL);
}


/* Serialize an article into a string for LaTeX. */
char *arttolatex(ARTICLE *art, int header){
  CBDATUM *buf;
  CBLIST *elems;
  CBMAP *attrs;
  const char *vstr, *blk, *elem, *name, *attr, *tmp;
  char *rtext, *ntext, *ptext, *url, *flags, *author, *cdate, *text, *etext;
  int i, j, asis, del, inlist, initem, tlf, len;
  assert(art);
  buf = cbdatumopen("", 0);
  if(header){
    rtext = strtolatex(artsubject(art));
    xsprintf(buf, "\\chapter{%s}\n\n", rtext);
    free(rtext);
    xsprintf(buf, "\\begin{flushright}\n");
    xsprintf(buf, "\\vspace{-3.8em}\n");
    rtext = strtolatex(artauthor(art));
    xsprintf(buf, "\\noindent --- {\\it %s}\\\\[0.4em]\n", rtext);
    free(rtext);
    rtext = strtolatex(artcdate(art));
    xsprintf(buf, "{\\footnotesize %s}\\\\\n", rtext);
    free(rtext);
    rtext = strtolatex(artmdate(art));
    xsprintf(buf, "{\\footnotesize %s}\n", rtext);
    free(rtext);
    xsprintf(buf, "\\end{flushright}\n");
    xsprintf(buf, "\\vspace{0.2em}\n\n");
  }
  vstr = NULL;
  for(i = 0; i < cblistnum(art->body); i++){
    if(strstr(cblistval(art->body, i, NULL), "\\end{verbatim}")){
      vstr = latexverbstr();
      break;
    }
  }
  if(!vstr) vstr = "verbatim";
  asis = FALSE;
  del = FALSE;
  inlist = FALSE;
  initem = FALSE;
  tlf = FALSE;
  url = NULL;
  for(i = 0; i < cblistnum(art->body); i++){
    blk = cblistval(art->body, i, NULL);
    elems = cbxmlbreak(blk, FALSE);
    for(j = 0; j < cblistnum(elems); j++){
      elem = cblistval(elems, j, NULL);
      if(elem[0] == '<'){
        attrs = cbxmlattrs(elem);
        name = cbmapget(attrs, "", -1, NULL);
        if(!strcmp(name, "topic")){
          if(elem[1] == '/'){
            xsprintf(buf, "}\n\n");
          } else {
            xsprintf(buf, "\\section{");
          }
        } else if(!strcmp(name, "subtopic")){
          if(elem[1] == '/'){
            xsprintf(buf, "}\n\n");
          } else {
            xsprintf(buf, "\\subsection{");
          }
        } else if(!strcmp(name, "para")){
          if(elem[1] == '/'){
            xsprintf(buf, "\n\n");
          }
        } else if(!strcmp(name, "asis")){
          if(elem[1] == '/'){
            if(!tlf) xsprintf(buf, "\n");
            xsprintf(buf, "\\end{%s}\n", vstr);
            xsprintf(buf, "\\end{minipage}\n");
            xsprintf(buf, "\\vspace{0.5em}\n\n");
            asis = FALSE;
            tlf = FALSE;
          } else {
            xsprintf(buf, "\\vspace{0.5em}\n");
            xsprintf(buf, "\\noindent \\hspace{0.8em}\n");
            xsprintf(buf, "\\begin{minipage}[t]{\\textwidth}\n");
            xsprintf(buf, "\\begin{%s}\n", vstr);
            asis = TRUE;
            tlf = TRUE;
          }
        } else if(!strcmp(name, "list")){
          if(elem[1] == '/'){
            xsprintf(buf, "\\end{itemize}\n");
            xsprintf(buf, "\\vspace{-1.0em}\n\n");
            inlist = FALSE;
          } else {
            xsprintf(buf, "\\vspace{-1.0em}\n");
            xsprintf(buf, "\\begin{itemize}\n");
            xsprintf(buf, "\\itemsep=-0.25em\n");
            inlist = TRUE;
          }
        } else if(!strcmp(name, "item")){
          if(elem[1] == '/'){
            xsprintf(buf, "\n");
            initem = FALSE;
          } else {
            xsprintf(buf, "\\item ");
            initem = TRUE;
          }
        } else if(!strcmp(name, "graph")){
          xsprintf(buf, "\\begin{figure}[!h]\n");
          xsprintf(buf, "\\begin{center}\n");
          if(!(attr = cbmapget(attrs, "data", -1, NULL))) attr = "";
          if((tmp = strrchr(attr, '/')) != NULL) attr = tmp + 1;
          xsprintf(buf, "\\includegraphics[width=8cm]{%s}\n", attr);
          xsprintf(buf, "\\end{center}\n");
          xsprintf(buf, "\\end{figure}\n\n");
        } else if(!strcmp(name, "break")){
          xsprintf(buf, "\\vspace{1.5em}\n");
          xsprintf(buf, "\\hrule width 0.8\\textwidth\n");
          xsprintf(buf, "\\vspace{1.3em}\n\n");
        } else if(!asis && !strcmp(name, "link")){
          if(elem[1] == '/'){
            xsprintf(buf, "}");
            if(url){
              ptext = strtolatex(url);
              xsprintf(buf, "\\footnote{%s}", ptext);
              free(ptext);
            }
          } else {
            free(url);
            if(!(attr = cbmapget(attrs, "to", -1, NULL))) attr = "";
            url = cbmemdup(attr, -1);
            xsprintf(buf, "\\textcolor{link}{");
          }
        } else if(!asis && !strcmp(name, "emp")){
          if(elem[1] == '/'){
            xsprintf(buf, "}");
          } else {
            xsprintf(buf, "{\\bf ");
          }
        } else if(!asis && !strcmp(name, "cite")){
          if(elem[1] == '/'){
            xsprintf(buf, "}");
          } else {
            xsprintf(buf, "{\\it ");
          }
        } else if(!asis && !strcmp(name, "ins")){
          if(elem[1] == '/'){
            xsprintf(buf, "}");
          } else {
            xsprintf(buf, "\\textcolor{ins}{");
          }
        } else if(!strcmp(name, "del")){
          if(elem[1] == '/'){
            if(asis){
              del = FALSE;
            } else {
              xsprintf(buf, "}");
            }
          } else {
            if(asis){
              del = TRUE;
            } else {
              xsprintf(buf, "\\textcolor{del}{");
            }
          }
        }
        cbmapclose(attrs);
      } else if(!del && !(inlist && !initem)){
        rtext = cbxmlunescape(elem);
        if(asis){
          xsprintf(buf, "%s", rtext);
          len = strlen(rtext);
          tlf = len > 0 && rtext[len-1] == '\n';
        } else {
          ntext = strnormalize(rtext, TRUE, FALSE);
          ptext = strtolatex(ntext);
          xsprintf(buf, "%s", ptext);
          free(ptext);
          free(ntext);
        }
        free(rtext);
      }
    }
    cblistclose(elems);
  }
  free(url);
  if(cblistnum(art->tail) > 0){
    xsprintf(buf, "\\vspace{2.2em}\n");
    xsprintf(buf, "\\hrule width 0.95\\textwidth\n");
    xsprintf(buf, "\\vspace{0.2em}\n\n");
    xsprintf(buf, "\\begin{itemize}\n");
    xsprintf(buf, "\\itemsep=-0.25em\n");
    xsprintf(buf, "\\itemindent=-1.5em\n");
    for(i = 0; i < cblistnum(art->tail); i++){
      flags = NULL;
      author = NULL;
      cdate = NULL;
      text = NULL;
      blk = cblistval(art->tail, i, NULL);
      elems = cbxmlbreak(blk, FALSE);
      for(j = 0; j < cblistnum(elems); j++){
        elem = cblistval(elems, j, NULL);
        if(elem[0] == '<'){
          attrs = cbxmlattrs(elem);
          name = cbmapget(attrs, "", -1, NULL);
          if(!strcmp(name, "response") && elem[1] != '/'){
            if(!flags && (tmp = cbmapget(attrs, "flags", -1, NULL)) != NULL)
              flags = strtolatex(tmp);
            if(!author && (tmp = cbmapget(attrs, "author", -1, NULL)) != NULL)
              author = strtolatex(tmp);
            if(!cdate && (tmp = cbmapget(attrs, "cdate", -1, NULL)) != NULL)
              cdate = strtolatex(tmp);
          }
          cbmapclose(attrs);
        } else {
          if(!text && strisprintable(elem)){
            rtext = strnormalize(elem, TRUE, TRUE);
            etext = cbxmlunescape(rtext);
            text = strtolatex(etext);
            free(etext);
            free(rtext);
          }
        }
      }
      cblistclose(elems);
      if(flags && author && cdate && text){
        xsprintf(buf, "\\item[] {\\small {\\it %s} --- %s }{\\scriptsize --- %s}\n",
                 author, text, cdate);
      }
      free(text);
      free(cdate);
      free(author);
      free(flags);
    }
    xsprintf(buf, "\\end{itemize}\n\n");
  }
  return cbdatumtomalloc(buf, NULL);
}


/* Serialize an article into a Wiki-style string. */
char *arttowiki(ARTICLE *art){
  CBDATUM *buf;
  CBLIST *elems;
  CBMAP *attrs;
  const char *blk, *elem, *name, *attr, *tmp;
  char delim[NUMBUFSIZ*2], *rtext, *url, *flags, *author, *cdate, *text;
  int i, j, k, asis, inlist, initem, tlf;
  assert(art);
  sprintf(delim, ">>>>>>>>>>>>>>>>%s%08X", art->hash,
          (dpouterhash(art->subject, -1) * dpouterhash(art->author, -1)) & 0xffffffff);
  buf = cbdatumopen("", 0);
  xsprintf(buf, "id: %s\n", art->id);
  xsprintf(buf, "password: %s\n", art->password);
  xsprintf(buf, "flags: %s\n", art->flags);
  xsprintf(buf, "language: %s\n", art->language);
  xsprintf(buf, "category: %s\n", art->category);
  xsprintf(buf, "subject: %s\n", art->subject);
  xsprintf(buf, "author: %s\n", art->author);
  xsprintf(buf, "cdate: %s\n", art->cdate);
  xsprintf(buf, "mdate: %s\n", art->mdate);
  xsprintf(buf, "%s\n\n", delim);
  asis = FALSE;
  inlist = FALSE;
  initem = FALSE;
  tlf = FALSE;
  url = NULL;
  for(i = 0; i < cblistnum(art->body); i++){
    blk = cblistval(art->body, i, NULL);
    elems = cbxmlbreak(blk, FALSE);
    for(j = 0; j < cblistnum(elems); j++){
      elem = cblistval(elems, j, NULL);
      if(elem[0] == '<'){
        attrs = cbxmlattrs(elem);
        name = cbmapget(attrs, "", -1, NULL);
        if(!strcmp(name, "topic")){
          if(elem[1] == '/'){
            xsprintf(buf, "\n\n");
          } else {
            xsprintf(buf, "! ");
          }
        } else if(!strcmp(name, "subtopic")){
          if(elem[1] == '/'){
            xsprintf(buf, "\n\n");
          } else {
            xsprintf(buf, "!! ");
          }
        } else if(!strcmp(name, "para")){
          if(elem[1] == '/'){
            xsprintf(buf, "\n\n");
          } else {
            xsprintf(buf, "  ");
          }
        } else if(!strcmp(name, "asis")){
          if(elem[1] == '/'){
            if(!tlf) xsprintf(buf, "\n");
            xsprintf(buf, "\n");
            asis = FALSE;
            tlf = FALSE;
          } else {
            asis = TRUE;
            tlf = TRUE;
          }
        } else if(!strcmp(name, "list")){
          if(elem[1] == '/'){
            xsprintf(buf, "\n");
            inlist = FALSE;
          } else {
            inlist = TRUE;
          }
        } else if(!strcmp(name, "item")){
          if(elem[1] == '/'){
            xsprintf(buf, "\n");
            initem = FALSE;
          } else {
            xsprintf(buf, "* ");
            initem = TRUE;
          }
        } else if(!strcmp(name, "graph")){
          if(!(attr = cbmapget(attrs, "data", -1, NULL))) attr = "";
          xsprintf(buf, "@graph: %s\n\n", attr);
        } else if(!strcmp(name, "break")){
          xsprintf(buf, "----\n\n");
        } else if(!asis && !strcmp(name, "link")){
          if(elem[1] == '/'){
            xsprintf(buf, "|%s]]", url ? url : "");
          } else {
            free(url);
            if(!(attr = cbmapget(attrs, "to", -1, NULL))) attr = "";
            url = cbmemdup(attr, -1);
            xsprintf(buf, "[[");
          }
        } else if(!asis && !strcmp(name, "emp")){
          if(elem[1] == '/'){
            xsprintf(buf, "]]");
          } else {
            xsprintf(buf, "[[^");
          }
        } else if(!asis && !strcmp(name, "cite")){
          if(elem[1] == '/'){
            xsprintf(buf, "]]");
          } else {
            xsprintf(buf, "[[~");
          }
        } else if(!asis && !strcmp(name, "ins")){
          if(elem[1] == '/'){
            xsprintf(buf, "]]");
          } else {
            xsprintf(buf, "[[+");
          }
        } else if(!asis && !strcmp(name, "del")){
          if(elem[1] == '/'){
            xsprintf(buf, "]]");
          } else {
            xsprintf(buf, "[[-");
          }
        }
        cbmapclose(attrs);
      } else if(!(inlist && !initem)){
        rtext = cbxmlunescape(elem);
        if(tlf) cbdatumcat(buf, "|", 1);
        for(k = 0; rtext[k] != '\0'; k++){
          cbdatumcat(buf, rtext + k, 1);
          if(asis && rtext[k] == '\n' && rtext[k+1] != '\0') cbdatumcat(buf, "|", 1);
        }
        tlf = asis && k > 0 && rtext[k-1] == '\n';
        free(rtext);
      }
    }
    cblistclose(elems);
  }
  free(url);
  xsprintf(buf, "%s\n", delim);
  for(i = 0; i < cblistnum(art->tail); i++){
    flags = NULL;
    author = NULL;
    cdate = NULL;
    text = NULL;
    blk = cblistval(art->tail, i, NULL);
    elems = cbxmlbreak(blk, FALSE);
    for(j = 0; j < cblistnum(elems); j++){
      elem = cblistval(elems, j, NULL);
      if(elem[0] == '<'){
        attrs = cbxmlattrs(elem);
        name = cbmapget(attrs, "", -1, NULL);
        if(!strcmp(name, "response") && elem[1] != '/'){
          if(!flags && (tmp = cbmapget(attrs, "flags", -1, NULL)) != NULL)
            flags = cbmemdup(tmp, -1);
          if(!author && (tmp = cbmapget(attrs, "author", -1, NULL)) != NULL)
            author = cbmemdup(tmp, -1);
          if(!cdate && (tmp = cbmapget(attrs, "cdate", -1, NULL)) != NULL)
            cdate = cbmemdup(tmp, -1);
        }
        cbmapclose(attrs);
      } else {
        if(!text && strisprintable(elem)){
          rtext = strnormalize(elem, TRUE, TRUE);
          text = cbxmlunescape(rtext);
          free(rtext);
        }
      }
    }
    cblistclose(elems);
    if(flags && author && cdate && text)
      xsprintf(buf, "%s\n%s\n%s\n%s\n", flags, author, cdate, text);
    free(text);
    free(cdate);
    free(author);
    free(flags);
  }
  return cbdatumtomalloc(buf, NULL);
}


/* Get the jet lag by minute. */
int jetlag(int min){
  static int lag = DEFJETLAG;
  struct tm *tp;
  time_t t, gt, lt;
  if(min < DEFJETLAG) lag = min;
  if(lag < DEFJETLAG) return lag;
  if((t = time(NULL)) < 0) return 0;
  if(!(tp = gmtime(&t))) return 0;
  gt = mktime(tp);
  if(!(tp = localtime(&t))) return 0;
  lt = mktime(tp);
  lag = (lt - gt) / 60;
  return lag;
}


/* Create the string for a current date in internal form. */
char *datecurrent(void){
  return datestr(time(NULL));
}


/* Create the string for a date in W3DTF. */
char *dateforwww(const char *date, int local){
  struct tm ts, *tp;
  time_t t;
  int jl;
  CBLIST *elems;
  char rdate[NUMBUFSIZ], tzone[NUMBUFSIZ];
  assert(date);
  if(!validdate(date)) return cbmemdup("1901-01-01T00:00:00Z", -1);
  elems = cbsplit(date, -1, "- :");
  memset(&ts, 0, sizeof(struct tm));
  ts.tm_year = atoi(cblistval(elems, 0, NULL)) - 1900;
  ts.tm_mon = atoi(cblistval(elems, 1, NULL)) - 1;
  ts.tm_mday = atoi(cblistval(elems, 2, NULL));
  ts.tm_hour = atoi(cblistval(elems, 3, NULL));
  ts.tm_min = atoi(cblistval(elems, 4, NULL));
  ts.tm_sec = atoi(cblistval(elems, 5, NULL));
  sprintf(tzone, "Z");
  if(local){
    jl = jetlag(DEFJETLAG);
    ts.tm_min += jl;
    t = mktime(&ts);
    if((tp = localtime(&t)) != NULL) ts = *tp;
    if(jl >= 0){
      tzone[0] = '+';
    } else {
      jl *= -1;
      tzone[0] = '-';
    }
    sprintf(tzone + 1, "%02d:%02d", jl / 60, jl % 60);
  }
  sprintf(rdate, "%04d-%02d-%02dT%02d:%02d:%02d%s",
          ts.tm_year + 1900, ts.tm_mon + 1, ts.tm_mday, ts.tm_hour, ts.tm_min, ts.tm_sec, tzone);
  cblistclose(elems);
  return cbmemdup(rdate, -1);
}


/* Create the string for a date in RFC1123. */
char *dateforhttp(const char *date, int local){
  struct tm ts, *tp;
  time_t t;
  int jl;
  CBLIST *elems;
  char rdate[NUMBUFSIZ], tzone[NUMBUFSIZ];
  assert(date);
  if(!validdate(date)) return cbmemdup("Thu, 01 Jan 1970 00:00:00 GMT", -1);
  elems = cbsplit(date, -1, "- :");
  memset(&ts, 0, sizeof(struct tm));
  ts.tm_year = atoi(cblistval(elems, 0, NULL)) - 1900;
  ts.tm_mon = atoi(cblistval(elems, 1, NULL)) - 1;
  ts.tm_mday = atoi(cblistval(elems, 2, NULL));
  ts.tm_hour = atoi(cblistval(elems, 3, NULL));
  ts.tm_min = atoi(cblistval(elems, 4, NULL));
  ts.tm_sec = atoi(cblistval(elems, 5, NULL));
  sprintf(tzone, "GMT");
  if(local){
    jl = jetlag(DEFJETLAG);
    ts.tm_min += jl;
    t = mktime(&ts);
    if((tp = localtime(&t)) != NULL) ts = *tp;
    if(jl >= 0){
      tzone[0] = '+';
    } else {
      jl *= -1;
      tzone[0] = '-';
    }
    sprintf(tzone + 1, "%02d%02d", jl / 60, jl % 60);
  }
  sprintf(rdate, "%s, %02d %s %04d %02d:%02d:%02d %s",
          wdayname(ts.tm_wday), ts.tm_mday, monname(ts.tm_mon), ts.tm_year + 1900,
          ts.tm_hour, ts.tm_min, ts.tm_sec, tzone);
  cblistclose(elems);
  return cbmemdup(rdate, -1);
}


/* Create the string for a date which is human readable. */
char *dateformen(const char *date){
  char *buf;
  buf = dateforwww(date, TRUE);
  buf[4] = '/';
  buf[7] = '/';
  buf[10] = ' ';
  buf[16] = '\0';
  return buf;
}


/* Normalize a string.
   `str' specifies a string.
   `nolf' specifies whether to convert a line feed to a space.
   `trim' specifies whether to cut the heading spaces and tailing spaces.
   The return value is a converted string. */
char *strnormalize(const char *str, int nolf, int trim){
  char *buf, *rp, *wp;
  int miss;
  assert(str);
  if((buf = cbiconv(str, -1, "UTF-8", "UTF-8", NULL, &miss)) != NULL && miss > 0){
    free(buf);
    buf = cbiconv(str, -1, "UTF-8", "US-ASCII", NULL, NULL);
  }
  if(!buf) buf = cbmemdup(str, -1);
  rp = buf;
  wp = buf;
  while(*rp != '\0'){
    if(*rp == '\n'){
      *(wp++) = nolf ? ' ' : '\n';
    } else if(*rp == '\t' || *rp == '\v' || *rp == '\f'){
      *(wp++) = ' ';
    } else if(*rp < 0 || (*rp >= ' ' && *rp <= '~')){
      *(wp++) = *rp;
    }
    rp++;
  }
  *wp = '\0';
  if(trim) cbstrtrim(buf);
  return buf;
}


/* Perform formatted conversion and write the result into a buffer. */
void xsprintf(CBDATUM *buf, const char *format, ...){
  va_list ap;
  char *tmp, numbuf[NUMBUFSIZ];
  unsigned char c;
  assert(buf && format);
  va_start(ap, format);
  while(*format != '\0'){
    if(*format == '%'){
      format++;
      switch(*format){
      case 'c':
        c = va_arg(ap, int);
        cbdatumcat(buf, (char *)&c, 1);
        break;
      case 's':
        tmp = va_arg(ap, char *);
        if(!tmp) tmp = "(null)";
        cbdatumcat(buf, tmp, -1);
        break;
      case 'd':
        sprintf(numbuf, "%d", va_arg(ap, int));
        cbdatumcat(buf, numbuf, -1);
        break;
      case '@':
        tmp = va_arg(ap, char *);
        if(!tmp) tmp = "(null)";
        while(*tmp){
          switch(*tmp){
          case '&': cbdatumcat(buf, "&amp;", -1); break;
          case '<': cbdatumcat(buf, "&lt;", -1); break;
          case '>': cbdatumcat(buf, "&gt;", -1); break;
          case '"': cbdatumcat(buf, "&quot;", -1); break;
          default:
            if(!((*tmp >= 0 && *tmp <= 0x8) || (*tmp >= 0x0e && *tmp <= 0x1f)))
              cbdatumcat(buf, tmp, 1);
            break;
          }
          tmp++;
        }
        break;
      case '?':
        tmp = va_arg(ap, char *);
        if(!tmp) tmp = "(null)";
        while(*tmp){
          c = *(unsigned char *)tmp;
          if((c >= 'A' && c <= 'Z') || (c >= 'a' && c <= 'z') ||
             (c >= '0' && c <= '9') || (c != '\0' && strchr("_-.", c))){
            cbdatumcat(buf, (char *)&c, 1);
          } else {
            sprintf(numbuf, "%%%02X", c);
            cbdatumcat(buf, numbuf, -1);
          }
          tmp++;
        }
        break;
      case '%':
        cbdatumcat(buf, "%", 1);
        break;
      }
    } else {
      cbdatumcat(buf, format, 1);
    }
    format++;
  }
  va_end(ap);
}


/* Get a DES hash string of a key string. */
const char *makecrypt(const char *key){
  const char tbl[] = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789./";
  const char *rp;
  char salt[3];
  int pid;
  assert(key);
  pid = getpid();
  salt[0] = tbl[((pid%65536)/256)%(sizeof(tbl)-1)];
  salt[1] = tbl[(pid%256)%(sizeof(tbl)-1)];
  salt[2] = '\0';
  if(!(rp = crypt(key, salt))) rp = "";
  return rp;
}


/* Check whether a key matches a DES hash string. */
int matchcrypt(const char *key, const char *code){
  const char *rp;
  char salt[3];
  assert(key && code);
  if(strlen(code) < 2) return FALSE;
  salt[0] = code[0];
  salt[1] = code[1];
  salt[2] = '\0';
  if(!(rp = crypt(key, salt))) return FALSE;
  return strcmp(rp, code) ? FALSE : TRUE;
}


/* Get the preamble for a LaTeX document. */
const char *latexpreamble(const char *dclass, const char *title,
                          const char *author, const char *date){
  static char buf[CONSTBUFSIZ];
  char *wp, *ptext;
  wp = buf;
  wp += sprintf(wp, "\\documentclass[a4papar]{%s}\n", dclass);
  wp += sprintf(wp, "\\usepackage{graphicx}\n");
  wp += sprintf(wp, "\\usepackage{verbatim}\n");
  wp += sprintf(wp, "\\usepackage{color}\n");
  wp += sprintf(wp, "\\usepackage{indentfirst}\n");
  wp += sprintf(wp, "\\pagestyle{plain}\n");
  wp += sprintf(wp, "\\pagenumbering{arabic}\n");
  wp += sprintf(wp, "\\topmargin=-5mm\n");
  wp += sprintf(wp, "\\oddsidemargin=-5mm\n");
  wp += sprintf(wp, "\\evensidemargin=-5mm\n");
  wp += sprintf(wp, "\\textwidth=170mm\n");
  wp += sprintf(wp, "\\textheight=240mm\n");
  wp += sprintf(wp, "\\parindent=1.0em\n");
  wp += sprintf(wp, "\\parskip=0.7em\n");
  wp += sprintf(wp, "\\renewcommand{\\baselinestretch}{1.0}\n");
  wp += sprintf(wp, "\\let\\%s=\\verbatim\n", latexverbstr());
  wp += sprintf(wp, "\\let\\end%s=\\endverbatim\n", latexverbstr());
  wp += sprintf(wp, "\\definecolor{link}{rgb}{0.0,0.0,0.4}\n");
  wp += sprintf(wp, "\\definecolor{ins}{rgb}{0.3,0.0,0.0}\n");
  wp += sprintf(wp, "\\definecolor{del}{rgb}{0.3,0.3,0.3}\n");
  ptext = strtolatex(title);
  wp += sprintf(wp, "\\title{%s}\n", ptext);
  free(ptext);
  ptext = strtolatex(author);
  wp += sprintf(wp, "\\author{%s}\n", ptext);
  free(ptext);
  ptext = strtolatex(date);
  wp += sprintf(wp, "\\date{%s}\n", ptext);
  free(ptext);
  return buf;
}


/* Get a plain text in LaTeX format. */
char *strtolatex(const char *str){
  CBDATUM *buf;
  char esc[NUMBUFSIZ];
  int i;
  assert(str);
  buf = cbdatumopen("", 0);
  for(i = 0; str[i] != '\0'; i++){
    if(str[i] == '\\'){
      cbdatumcat(buf, "{\\textbackslash}", -1);
    } else if(str[i] == '<'){
      cbdatumcat(buf, "{\\textless}", -1);
    } else if(str[i] == '>'){
      cbdatumcat(buf, "{\\textgreater}", -1);
    } else if(str[i] == '|'){
      cbdatumcat(buf, "{\\textbar}", -1);
    } else if(strchr("^~`'-", str[i])){
      sprintf(esc, "{\\char'%03o}", str[i]);
      cbdatumcat(buf, esc, -1);
    } else if(strchr("$&%#_{}", str[i])){
      sprintf(esc, "\\%c", str[i]);
      cbdatumcat(buf, esc, -1);
    } else {
      cbdatumcat(buf, str + i, 1);
    }
  }
  return cbdatumtomalloc(buf, NULL);
}



/*************************************************************************************************
 * private objects
 *************************************************************************************************/


/* Check whether an XML text does include no invalid entity reference.
   `xml' specifies a string.
   The return value is true if no invalid entity reference, else it is false */
static int validxmlent(const char *xml){
  assert(xml);
  while(*xml != '\0'){
    if((*xml == '&') && !cbstrfwmatch(xml, "&amp;") &&
       !cbstrfwmatch(xml, "&lt;") && !cbstrfwmatch(xml, "&gt;") &&
       !cbstrfwmatch(xml, "&quot;") && !cbstrfwmatch(xml, "&apos;")) return FALSE;
    xml++;
  }
  return TRUE;
}


/* Create a unique ID.
   The return value is the string of a unique ID. */
static char *uniqueid(const char *prefix){
  FILE *ifp;
  char str[PATHBUFSIZ];
  int i, rnd;
  rnd = 0;
  if((ifp = fopen("/dev/urandom", "r")) != NULL){
    for(i = 0; i < 4; i++){
      rnd = (rnd << 8) + fgetc(ifp);
    }
    fclose(ifp);
  }
  sprintf(str, "%s%010d%05d%05d",
          prefix, (int)time(NULL), (int)getpid() % 100000, (unsigned int)rnd % 100000);
  return cbmemdup(str, -1);
}


/* Check whether an ID string is valid.
   The return value is true if an ID string is valid, else, it is not. */
static int validid(const char *id, const char *prefix){
  int i;
  assert(id);
  if(!cbstrfwmatch(id, prefix)) return FALSE;
  for(i = strlen(prefix); id[i] != '\0'; i++){
    if(id[i] < '0' || id[i] > '9') return FALSE;
  }
  return TRUE;
}


/* Create the string for a date.
   `t' specifies the UNIX time.
   The return value is the string of the date in the internal form (YYYY-MM-DD hh:mm:ss). */
static char *datestr(time_t t){
  struct tm *tp;
  char date[NUMBUFSIZ];
  if(t < 0 || !(tp = gmtime(&t))) return "1970-01-01 00:00:00";
  sprintf(date, "%04d-%02d-%02d %02d:%02d:%02d",
          tp->tm_year + 1900, tp->tm_mon + 1, tp->tm_mday, tp->tm_hour, tp->tm_min, tp->tm_sec);
  return cbmemdup(date, -1);
}


/* Check whether a date string is valid or not.
   `date' specifies a string for a date.
   The return value is true if it is valid, else it is false. */
static int validdate(const char *date){
  int i;
  assert(date);
  for(i = 0; i <= 3; i++){
    if(date[i] < '0' || date[i] > '9') return FALSE;
  }
  if(date[4] != '-') return FALSE;
  for(i = 5; i <= 6; i++){
    if(date[i] < '0' || date[i] > '9') return FALSE;
  }
  if(atoi(date + 5) > 12) return FALSE;
  if(date[7] != '-') return FALSE;
  for(i = 8; i <= 9; i++){
    if(date[i] < '0' || date[i] > '9') return FALSE;
  }
  if(atoi(date + 8) > 31) return FALSE;
  if(date[10] != ' ') return FALSE;
  for(i = 11; i <= 12; i++){
    if(date[i] < '0' || date[i] > '9') return FALSE;
  }
  if(atoi(date + 11) > 24) return FALSE;
  if(date[13] != ':') return FALSE;
  for(i = 14; i <= 15; i++){
    if(date[i] < '0' || date[i] > '9') return FALSE;
  }
  if(atoi(date + 14) > 60) return FALSE;
  if(date[16] != ':') return FALSE;
  for(i = 17; i <= 18; i++){
    if(date[i] < '0' || date[i] > '9') return FALSE;
  }
  if(atoi(date + 17) > 60) return FALSE;
  if(date[19] != '\0') return FALSE;
  return TRUE;
}


/* Get the name of a week day.
   `wday' specifies the number of a week day (0-6).
   The return value is the strint of the name of the week day. */
static const char *wdayname(int wday){
  switch(wday){
  case 1: return "Mon";
  case 2: return "Tue";
  case 3: return "Wed";
  case 4: return "Thu";
  case 5: return "Fri";
  case 6: return "Sat";
  }
  return "Sun";
}


/* Get the name of a month.
   `mon' specifies the number of a month (0-11).
   The return value is the strint of the name of the month. */
static const char *monname(int mon){
  switch(mon){
  case 1: return "Feb";
  case 2: return "Mar";
  case 3: return "Apr";
  case 4: return "May";
  case 5: return "Jun";
  case 6: return "Jul";
  case 7: return "Aug";
  case 8: return "Sep";
  case 9: return "Oct";
  case 10: return "Nov";
  case 11: return "Dec";
  }
  return "Jan";
}


/* Check whether a string has one or more printable characters.
   `str' specifies a string.
   The return value is true if a string has one or more printable characters, else it is false. */
static int strisprintable(const char *str){
  assert(str);
  while(*str != '\0'){
    if(!strchr(" \t\r\n\v\f\a\b", *str)) return TRUE;
    str++;
  }
  return FALSE;
}


/* Skip the label of a line.
   `str' specifies a string with a label.
   The return value is the pointer to the beginning of the label. */
static const char *skiplabel(const char *str){
  if(!(str = strchr(str, ':'))) return str;
  str++;
  while(*str != '\0' && (*str == ' ' || *str == '\t')){
    str++;
  }
  return str;
}


/* Concatenate a Wiki-style text to an XML buffer.
   `buf' specifies an XML buffer.
   `text' specifies a text.
   `trim' specifies whether to trim the text. */
static void wikicat(CBDATUM *buf, const char *text, int trim){
  const char *prefs[] = {
    "http://", "https://", "ftp://", "gopher://", "ldap://", "file://", NULL
  };
  const char *pref, *pv, *rp;
  char *rtext, *tmp, *mp;
  int i, j;
  rtext = cbmemdup(text, -1);
  if(trim) cbstrtrim(rtext);
  for(i = 0; rtext[i] != '\0'; i++){
    if(cbstrfwmatch(rtext + i, "[[") && (pv = strstr(rtext + i, "]]")) != NULL){
      tmp = cbmemdup(rtext + i + 2, pv - rtext - i - 2);
      switch(tmp[0]){
      case '^':
        xsprintf(buf, "<emp>%@</emp>", tmp + 1);
        break;
      case '~':
        xsprintf(buf, "<cite>%@</cite>", tmp + 1);
        break;
      case '+':
        xsprintf(buf, "<ins>%@</ins>", tmp + 1);
        break;
      case '-':
        xsprintf(buf, "<del>%@</del>", tmp + 1);
        break;
      default:
        if((mp = strchr(tmp, '|')) != NULL){
          *mp = '\0';
          xsprintf(buf, "<link to=\"%@\">%@</link>", mp + 1, tmp);
        } else {
          xsprintf(buf, "%@", tmp);
        }
        break;
      }
      free(tmp);
      i = pv - rtext + 1;
    } else {
      pref = NULL;
      for(j = 0; prefs[j]; j++){
        if(cbstrfwimatch(rtext + i, prefs[j])){
          pref = prefs[j];
        }
      }
      if(pref){
        rp = rtext + i;
        while(*rp != '\0' && ((*rp >= '0' && *rp <= '9') || (*rp >= 'A' && *rp <= 'Z') ||
                              (*rp >= 'a' && *rp <= 'z') || strchr("-_.!~*'();/?:@&=+$,%#", *rp))){
          rp++;
        }
        tmp = cbmemdup(rtext + i, rp - rtext - i);
        xsprintf(buf, "<link to=\"%@\">%@</link>", tmp, tmp);
        free(tmp);
        i = rp - rtext - 1;
      } else {
        switch(rtext[i]){
        case '&': xsprintf(buf, "&amp;"); break;
        case '<': xsprintf(buf, "&lt;"); break;
        case '>': xsprintf(buf, "&gt;"); break;
        default: xsprintf(buf, "%c", rtext[i]); break;
        }
      }
    }
  }
  free(rtext);
}


/* Extract HTML from the head of an article.
   `art' specifies a handle of a article object.
   The return value is a string of HTML. */
static char *headtohtml(ARTICLE *art){
  CBDATUM *buf;
  char *tmp;
  assert(art);
  buf = cbdatumopen("", 0);
  xsprintf(buf, "<div id=\"%@H\" class=\"head\">\n", art->id);
  xsprintf(buf, "<div class=\"line\">\n");
  xsprintf(buf, "<h2 class=\"subject %@\">%@</h2>\n",
           hashclass("subject", art->subject), art->subject);
  xsprintf(buf, "<div class=\"category\">%@</div>\n", art->category);
  xsprintf(buf, "<div class=\"resnum\">%d</div>\n", cblistnum(art->tail));
  xsprintf(buf, "</div>\n");
  xsprintf(buf, "<div class=\"line\">\n");
  xsprintf(buf, "<div class=\"author %@\">%@</div>\n",
           hashclass("author", art->author), art->author);
  tmp = dateformen(art->cdate);
  xsprintf(buf, "<div class=\"cdate\">%@</div>\n", tmp);
  free(tmp);
  tmp = dateformen(art->mdate);
  xsprintf(buf, "<div class=\"mdate\">%@</div>\n", tmp);
  free(tmp);
  xsprintf(buf, "</div>\n");
  xsprintf(buf, "</div>\n");
  return cbdatumtomalloc(buf, NULL);
}


/* Extract HTML from the body of an article.
   `art' specifies a handle of a article object.
   `systemuri' specifies the URI of the generator of the HTML.
   `num' specifies the number of shown blocks in the body.  If it is zero, all responses are
   shown.  If it is negative, the absolute value is calculated and the lower blocks are shown.
   The return value is a string of HTML. */
static char *bodytohtml(ARTICLE *art, const char *systemuri, int num){
  CBDATUM *buf;
  CBLIST *elems;
  CBMAP *attrs;
  const char *blk, *elem, *name, *oname, *val;
  char *rtext;
  int i, j, start, end, inlist, initem;
  assert(art);
  buf = cbdatumopen("", 0);
  xsprintf(buf, "<div id=\"%@B\" class=\"body\">\n", art->id);
  start = 0;
  end = cblistnum(art->body);
  if(num > 0 && num < end){
    end = num;
  } else if(num < 0 && -num < end){
    start += end + num;
  }
  if(start > 0)
    xsprintf(buf, "<div class=\"abbrev\"><code>......</code></div>\n");
  inlist = FALSE;
  initem = FALSE;
  for(i = start; i < end; i++){
    blk = cblistval(art->body, i, NULL);
    elems = cbxmlbreak(blk, FALSE);
    for(j = 0; j < cblistnum(elems); j++){
      elem = cblistval(elems, j, NULL);
      if(elem[0] == '<'){
        attrs = cbxmlattrs(elem);
        name = cbmapget(attrs, "", -1, NULL);
        oname = name;
        if(!strcmp(name, "topic")){
          name = "h3";
        } else if(!strcmp(name, "subtopic")){
          name = "h4";
        } else if(!strcmp(name, "para")){
          name = "p";
        } else if(!strcmp(name, "asis")){
          name = "pre";
        } else if(!strcmp(name, "list")){
          name = "ul";
        } else if(!strcmp(name, "item")){
          name = "li";
        } else if(!strcmp(name, "graph")){
          name = "img";
        } else if(!strcmp(name, "break")){
          name = "hr";
        } else if(!strcmp(name, "link")){
          name = "a";
        } else if(!strcmp(name, "emp")){
          name = "strong";
        } else if(!strcmp(name, "cite")){
          name = "cite";
        } else if(!strcmp(name, "ins")){
          name = "ins";
        } else if(!strcmp(name, "del")){
          name = "del";
        }
        if(elem[1] == '/'){
          xsprintf(buf, "</%@>", name);
          if(!strcmp(name, "h3") || !strcmp(name, "h4") || !strcmp(name, "p") ||
             !strcmp(name, "pre") || !strcmp(name, "ul") || !strcmp(name, "li"))
            xsprintf(buf, "\n");
          if(!strcmp(name, "ul")){
            inlist = FALSE;
          } else if(!strcmp(name, "li")){
            initem = FALSE;
          }
        } else {
          if(!strcmp(name, "img")){
            if(!(val = cbmapget(attrs, "data", -1, NULL))) val = "";
            xsprintf(buf, "<div class=\"%@\"><img src=\"%@\" alt=\"[graph]\" /></div>\n",
                     oname, val);
          } else if(!strcmp(name, "hr")){
            xsprintf(buf, "<div class=\"%@\"><code>------</code></div>\n", oname);
          } else if(!strcmp(name, "a")){
            if(!(val = cbmapget(attrs, "to", -1, NULL))) val = "";
            xsprintf(buf, "<a href=\"%@\" rel=\"nofollow\" class=\"%@\">", val, oname);
          } else {
            xsprintf(buf, "<%@ class=\"%@\">", name, oname);
            if(!strcmp(name, "ul")){
              xsprintf(buf, "\n");
              inlist = TRUE;
            } else if(!strcmp(name, "li")){
              initem = TRUE;
            }
          }
        }
        cbmapclose(attrs);
      } else {
        if(!inlist || initem){
          rtext = cbxmlunescape(elem);
          xsprintf(buf, "%@", rtext);
          free(rtext);
        }
      }
    }
    cblistclose(elems);
  }
  if(end < cblistnum(art->body))
    xsprintf(buf, "<div class=\"abbrev\"><code>......</code></div>\n");
  xsprintf(buf, "</div>\n");
  return cbdatumtomalloc(buf, NULL);
}


/* Extract HTML from the tail of an article.
   `art' specifies a handle of a article object.
   `num' specifies the number of shown responses.  If it is zero, all responses are shown.
   If it is negative, the absolute value is calculated and the lower responses are shown.
   The return value is a string of HTML. */
static char *tailtohtml(ARTICLE *art, int num){
  CBDATUM *buf;
  CBLIST *elems;
  CBMAP *attrs;
  const char *blk, *elem, *name, *tmp;
  char *flags, *author, *cdate, *text, *rtext;
  int i, j, start, end;
  assert(art);
  if(cblistnum(art->tail) < 1) return cbmemdup("", 0);
  buf = cbdatumopen("", 0);
  xsprintf(buf, "<div id=\"%@T\" class=\"tail\">\n", art->id);
  start = 0;
  end = cblistnum(art->tail);
  if(num > 0 && num < end){
    end = num;
  } else if(num < 0 && -num < end){
    start += end + num;
  }
  if(start > 0)
    xsprintf(buf, "<div class=\"abbrev\"><code>......</code></div>\n");
  for(i = start; i < end; i++){
    flags = NULL;
    author = NULL;
    cdate = NULL;
    text = NULL;
    blk = cblistval(art->tail, i, NULL);
    elems = cbxmlbreak(blk, FALSE);
    for(j = 0; j < cblistnum(elems); j++){
      elem = cblistval(elems, j, NULL);
      if(elem[0] == '<'){
        attrs = cbxmlattrs(elem);
        name = cbmapget(attrs, "", -1, NULL);
        if(!strcmp(name, "response") && elem[1] != '/'){
          if(!flags && (tmp = cbmapget(attrs, "flags", -1, NULL)) != NULL)
            flags = cbmemdup(tmp, -1);
          if(!author && (tmp = cbmapget(attrs, "author", -1, NULL)) != NULL)
            author = cbmemdup(tmp, -1);
          if(!cdate && (tmp = cbmapget(attrs, "cdate", -1, NULL)) != NULL)
            cdate = dateformen(tmp);
        }
        cbmapclose(attrs);
      } else {
        if(!text && strisprintable(elem)){
          rtext = strnormalize(elem, TRUE, TRUE);
          text = cbxmlunescape(rtext);
          free(rtext);
        }
      }
    }
    cblistclose(elems);
    if(flags && author && cdate && text){
      xsprintf(buf, "<div class=\"response%s\">\n", ressubclass(flags));
      xsprintf(buf, "<span class=\"author %@\">%@</span>\n", hashclass("author", author), author);
      xsprintf(buf, "<span class=\"text\">%@</span>\n", text);
      xsprintf(buf, "<span class=\"cdate\">%@</span>\n", cdate);
      xsprintf(buf, "</div>\n");
    }
    free(text);
    free(cdate);
    free(author);
    free(flags);
  }
  if(end < cblistnum(art->tail))
    xsprintf(buf, "<div class=\"abbrev\"><code>......</code></div>\n");
  xsprintf(buf, "</div>\n");
  return cbdatumtomalloc(buf, NULL);
}


/* Get the sub class string of an article.
   `flags' specifies the flags of an article.
   The return value is the sub class string of the article. */
const char *artsubclass(const char *flags){
  static char name[PATHBUFSIZ];
  char *wp;
  assert(flags);
  wp = name;
  if(strchr(flags, LONECHAR)) wp += sprintf(wp, " article-%s", LONESTR);
  if(strchr(flags, WIKICHAR)) wp += sprintf(wp, " article-%s", WIKISTR);
  *wp = '\0';
  return name;
}


/* Get the sub class string of a response.
   `flags' specifies the flags of a response.
   The return value is the sub class string of the response. */
const char *ressubclass(const char *flags){
  static char name[PATHBUFSIZ];
  char *wp;
  assert(flags);
  wp = name;
  if(strchr(flags, SINKCHAR)) wp += sprintf(wp, " response-%s", SINKSTR);
  *wp = '\0';
  return name;
}


/* Get the class name of a string.
   `str' specifies a string.
   The return value is the class name of the string. */
static const char *hashclass(const char *prefix, const char *str){
  static char name[PATHBUFSIZ];
  assert(str);
  sprintf(name, "%s-%d", prefix, ((unsigned int)(dpouterhash(str, -1) * str[0])) % HASHNUM + 1);
  return name;
}


/* Get the delimit string for vervatim sections of LaTeX.
   The return value is the delimit string for vervatim sections of LaTeX. */
static const char *latexverbstr(void){
  static char buf[NUMBUFSIZ];
  static int first = TRUE;
  char *wp;
  int i;
  if(!first) return buf;
  wp = buf;
  srand(time(NULL));
  wp += sprintf(wp, "verbatim");
  for(i = 0; i < 8; i++){
    *(wp++) = 'a' + rand() % ('z' - 'a');
  }
  *wp = '\0';
  first = FALSE;
  return buf;
}



/* END OF FILE */
