/*************************************************************************************************
 * Implementation of common features
 *                                                      Copyright (C) 2003-2004 Mikio Hirabayashi
 * This file is part of RBBS, a web-based bulletin board system.
 * RBBS is free software; you can redistribute it and/or modify it under the terms of the GNU
 * General Public License as published by the Free Software Foundation; either version 2 of the
 * License, or any later version.
 * RBBS is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License along with RBBS;
 * if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA.
 *************************************************************************************************/


#include "rbbscommon.h"


/* private function prototypes */
static int validxmlent(const char *xml);
static char *uniqueid(void);
static int validid(const char *id);
static char *datestr(time_t t);
static int validdate(const char *date);
static const char *wdayname(int wday);
static const char *monname(int mon);
static int strisprintable(const char *str);
static char *headtohtml(ARTICLE *art);
static char *bodytohtml(ARTICLE *art, int num);
static char *tailtohtml(ARTICLE *art, int num);
const char *artsubclass(const char *flags);
const char *ressubclass(const char *flags);
static const char *hashclass(const char *prefix, const char *str);



/*************************************************************************************************
 * public objects
 *************************************************************************************************/


/* Get a handle connected to the databases. */
RBBS *rbbsopen(const char *name, int writer){
  RBBS *rbbs;
  DEPOT *artdb;
  VILLA *cdidx, *mdidx;
  time_t t;
  char path[PATHBUFSIZ], *date;
  assert(name);
  artdb = NULL;
  cdidx = NULL;
  mdidx = NULL;
  if(writer) mkdir(name, 00755);
  sprintf(path, "%s/%s", name, ARTDBNAME);
  artdb = dpopen(path, writer ? (DP_OWRITER | DP_OCREAT) : DP_OREADER, -1);
  sprintf(path, "%s/%s", name, CDIDXNAME);
  cdidx = vlopen(path, writer ? (VL_OWRITER | VL_OCREAT) : VL_OREADER, VL_CMPLEX);
  sprintf(path, "%s/%s", name, MDIDXNAME);
  mdidx = vlopen(path, writer ? (VL_OWRITER | VL_OCREAT) : VL_OREADER, VL_CMPLEX);
  if(!artdb || !cdidx || !mdidx){
    if(mdidx) vlclose(mdidx);
    if(cdidx) vlclose(cdidx);
    if(artdb) dpclose(artdb);
    return NULL;
  }
  t = time(NULL);
  if(writer){
    date = datestr(t);
    if(dprnum(artdb) < 1) dpput(artdb, CDATEKEY, -1, date, -1, DP_DOVER);
    dpput(artdb, MDATEKEY, -1, date, -1, DP_DOVER);
    free(date);
  }
  dpsetalign(artdb, -2);
  rbbs = cbmalloc(sizeof(RBBS));
  rbbs->artdb = artdb;
  rbbs->cdidx = cdidx;
  rbbs->mdidx = mdidx;
  rbbs->idx = NULL;
  rbbs->next = NULL;
  if(!(rbbs->cdate = dpget(artdb, CDATEKEY, -1, 0, -1, NULL))) rbbs->cdate = datestr(t);
  if(!(rbbs->mdate = dpget(artdb, MDATEKEY, -1, 0, -1, NULL))) rbbs->mdate = datestr(t);
  rbbs->errmsg = NULL;
  return rbbs;
}


/* Release a database handle. */
int rbbsclose(RBBS *rbbs){
  int err;
  assert(rbbs);
  err = FALSE;
  free(rbbs->errmsg);
  free(rbbs->mdate);
  free(rbbs->cdate);
  if(!vlclose(rbbs->mdidx)) err = TRUE;
  if(!vlclose(rbbs->cdidx)) err = TRUE;
  if(!dpclose(rbbs->artdb)) err = TRUE;
  free(rbbs);
  return err ? FALSE : TRUE;
}


/* Remove a database directory. */
int rbbsremove(const char *name){
  char path[PATHBUFSIZ];
  assert(name);
  sprintf(path, "%s/%s", name, ARTDBNAME);
  dpremove(path);
  sprintf(path, "%s/%s", name, CDIDXNAME);
  vlremove(path);
  sprintf(path, "%s/%s", name, MDIDXNAME);
  vlremove(path);
  if(rmdir(name) == -1) return FALSE;
  return TRUE;
}


/* Get the error message of a database. */
const char *rbbsemsg(RBBS *rbbs){
  assert(rbbs);
  if(!rbbs->errmsg) return "unknown error";
  return rbbs->errmsg;
}


/* Get the creation date of a database. */
const char *rbbscdate(RBBS *rbbs){
  assert(rbbs);
  return rbbs->cdate;
}


/* Get the modification date of a database. */
const char *rbbsmdate(RBBS *rbbs){
  assert(rbbs);
  return rbbs->mdate;
}


/* Get the number of articles in a database. */
int rbbsanum(RBBS *rbbs){
  assert(rbbs);
  return dprnum(rbbs->artdb) - 2;
}


/* Optimize a database. */
int rbbsoptimize(RBBS *rbbs){
  assert(rbbs);
  if(!dpoptimize(rbbs->artdb, -1)){
    if(!rbbs->errmsg)
      rbbs->errmsg = cbsprintf("DB-Error: the database for articles is broken or not writable");
    return FALSE;
  }
  if(!vloptimize(rbbs->cdidx)){
    if(!rbbs->errmsg)
      rbbs->errmsg = cbsprintf("DB-Error: the database for creation date is broken");
    return FALSE;
  }
  if(!vloptimize(rbbs->mdidx)){
    if(!rbbs->errmsg)
      rbbs->errmsg = cbsprintf("DB-Error: the database for modification date is broken");
    return FALSE;
  }
  return TRUE;
}


/* Set the sorting order of a database. */
void setsortorder(RBBS* rbbs, int sort){
  assert(rbbs);
  switch(sort){
  case SORTCDATEA:
    rbbs->idx = rbbs->cdidx;
    vlcurfirst(rbbs->idx);
    rbbs->next = vlcurnext;
    break;
  case SORTCDATED:
    rbbs->idx = rbbs->cdidx;
    vlcurlast(rbbs->idx);
    rbbs->next = vlcurprev;
    break;
  case SORTMDATEA:
    rbbs->idx = rbbs->mdidx;
    vlcurfirst(rbbs->idx);
    rbbs->next = vlcurnext;
    break;
  case SORTMDATED:
    rbbs->idx = rbbs->mdidx;
    vlcurlast(rbbs->idx);
    rbbs->next = vlcurprev;
    break;
  }
}


/* Get the ID string of the next article. */
char *getnextid(RBBS *rbbs){
  char *val;
  assert(rbbs);
  if(!rbbs->idx || !rbbs->next) return NULL;
  val = vlcurval(rbbs->idx, NULL);
  rbbs->next(rbbs->idx);
  return val;
}


/* Retrieve article from a database. */
ARTICLE *getarticle(RBBS *rbbs, const char *id){
  ARTICLE *art;
  char *xml;
  assert(rbbs && id);
  if(!(xml = dpget(rbbs->artdb, id, -1, 0, -1, NULL))){
    if(!rbbs->errmsg) rbbs->errmsg = cbsprintf("DB-Error: there is no article for \"%s\"", id);
    return NULL;
  }
  art = makearticle(xml);
  free(art->id);
  art->id = cbmemdup(id, -1);
  free(xml);
  return art;
}


/* Post an article and store it into a database. */
int postarticle(RBBS *rbbs, ARTICLE *art, const char *id){
  const char *oid;
  char *xml;
  int err;
  assert(rbbs && art);
  if(id && !deletearticle(rbbs, id)) return FALSE;
  oid = id ? id : art->id;
  xml = arttoxml(art);
  err = FALSE;
  if(!dpput(rbbs->artdb, oid, -1, xml, -1, DP_DOVER)){
    if(!rbbs->errmsg)
      rbbs->errmsg = cbsprintf("DB-Error: the database for articles is broken or not writable");
    err = TRUE;
  } else if(!vlput(rbbs->cdidx, art->cdate, -1, oid, -1, VL_DDUP)){
    if(!rbbs->errmsg)
      rbbs->errmsg = cbsprintf("DB-Error: the database for creation date is broken");
    err = TRUE;
  } else if(!vlput(rbbs->mdidx, art->mdate, -1, oid, -1, VL_DDUP)){
    if(!rbbs->errmsg)
      rbbs->errmsg = cbsprintf("DB-Error: the database for modification date is broken");
    err = TRUE;
  }
  free(xml);
  return err ? FALSE : TRUE;
}


/* Delete an article existing in a database. */
int deletearticle(RBBS *rbbs, const char *id){
  ARTICLE *oart;
  CBLIST *olist, *nlist;
  const char *val;
  int i, err;
  assert(rbbs && id);
  if(!(oart = getarticle(rbbs, id))) return FALSE;
  err = FALSE;
  if(!dpout(rbbs->artdb, id, -1)){
    err = TRUE;
    if(!rbbs->errmsg)
      rbbs->errmsg = cbsprintf("DB-Error: the database for articles is broken or not writable");
  }
  if(!(olist = vlgetlist(rbbs->cdidx, oart->cdate, -1))) olist = cblistopen();
  nlist = cblistopen();
  for(i = 0; i < cblistnum(olist); i++){
    val = cblistval(olist, i, NULL);
    if(strcmp(val, id)) cblistpush(nlist, val, -1);
  }
  if(cblistnum(olist) == cblistnum(nlist)) err = TRUE;
  if(!vloutlist(rbbs->cdidx, oart->cdate, -1)) err = TRUE;
  if(cblistnum(nlist) > 0 && !vlputlist(rbbs->cdidx, oart->cdate, -1, nlist)) err = TRUE;
  cblistclose(nlist);
  cblistclose(olist);
  if(err && !rbbs->errmsg)
    rbbs->errmsg = cbsprintf("DB-Error: the database for creation date is broken");
  if(!(olist = vlgetlist(rbbs->mdidx, oart->mdate, -1))) olist = cblistopen();
  nlist = cblistopen();
  for(i = 0; i < cblistnum(olist); i++){
    val = cblistval(olist, i, NULL);
    if(strcmp(val, id)) cblistpush(nlist, val, -1);
  }
  if(cblistnum(olist) == cblistnum(nlist)) err = TRUE;
  if(!vloutlist(rbbs->mdidx, oart->mdate, -1)) err = TRUE;
  if(cblistnum(nlist) > 0 && !vlputlist(rbbs->mdidx, oart->mdate, -1, nlist)) err = TRUE;
  cblistclose(nlist);
  cblistclose(olist);
  if(err && !rbbs->errmsg)
    rbbs->errmsg = cbsprintf("DB-Error: the database for creation date is broken");
  freearticle(oart);
  return err ? FALSE : TRUE;
}


/* Make a object of article from an XML string. */
ARTICLE *makearticle(const char *xml){
  ARTICLE *art;
  CBLIST *elems, *stack;
  CBMAP *attrs;
  CBDATUM *cont;
  const char *elem, *name, *pname, *ntext, *celem, *val;
  char *rtext;
  int i, j, isart, ishead, isbody, istail, bstart, rstart;
  assert(xml);
  art = cbmalloc(sizeof(ARTICLE));
  art->id = NULL;
  art->password = NULL;
  art->flags = NULL;
  art->language = NULL;
  art->category = NULL;
  art->subject = NULL;
  art->author = NULL;
  art->cdate = NULL;
  art->mdate = NULL;
  art->body = cblistopen();
  art->tail = cblistopen();
  art->errmsg = NULL;
  art->hash = cbsprintf("%08X", dpouterhash(xml, -1));
  rtext = strnormalize(xml, FALSE, FALSE);
  elems = cbxmlbreak(rtext, TRUE);
  free(rtext);
  stack = cblistopen();
  isart = FALSE;
  ishead = FALSE;
  isbody = FALSE;
  istail = FALSE;
  bstart = -1;
  rstart = -1;
  cont = cbdatumopen("", 0);
  for(i = 0; i < cblistnum(elems); i++){
    elem = cblistval(elems, i, NULL);
    ntext = cblistval(elems, i + 1, NULL);
    pname = NULL;
    if(cblistnum(stack) > 0) pname = cblistval(stack, cblistnum(stack) - 1, NULL);
    if(!ntext || ntext[0] == '<') ntext = "";
    if(elem[0] == '<'){
      if(elem[1] == '?' || elem[1] == '!') continue;
      attrs = cbxmlattrs(elem);
      name = cbmapget(attrs, "", 0, NULL);
      if(elem[1] == '/'){
        if(!pname || strcmp(name, pname)){
          art->errmsg = cbsprintf("Invalid-XML: an end tag \"%s\" is mismatching", name);
        } else {
          if(bstart >= 0 && (!strcmp(pname, "topic") || !strcmp(pname, "para") ||
                             !strcmp(pname, "asis") || !strcmp(pname, "list"))){
            for(j = bstart; j <= i; j++){
              celem = cblistval(elems, j, NULL);
              cbdatumcat(cont, celem, -1);
            }
            cblistpush(art->body, cbdatumptr(cont), -1);
            bstart = -1;
            cbdatumsetsize(cont, 0);
          } else if(rstart >= 0 && !strcmp(pname, "response")){
            for(j = rstart; j <= i; j++){
              celem = cblistval(elems, j, NULL);
              cbdatumcat(cont, celem, -1);
            }
            rtext = strnormalize(cbdatumptr(cont), TRUE, TRUE);
            cblistpush(art->tail, rtext, -1);
            free(rtext);
            rstart = -1;
            cbdatumsetsize(cont, 0);
          }
          free(cblistpop(stack, NULL));
        }
      } else {
        if(!strcmp(name, "article")){
          if(pname){
            art->errmsg = cbsprintf("Invalid-XML: \"article\" must be root");
          } else if(isart){
            art->errmsg = cbsprintf("Invalid-XML: \"article\" must be root");
          } else if(cbstrbwmatch(elem, "/>")){
            art->errmsg = cbsprintf("Invalid-XML: \"article\" must not be empty");
          } else {
            isart = TRUE;
            if(!art->id && (val = cbmapget(attrs, "id", -1, NULL)) != NULL)
              art->id = cbmemdup(val, -1);
            if(!art->password && (val = cbmapget(attrs, "password", -1, NULL)) != NULL)
              art->password = cbmemdup(val, -1);
          }
        } else if(!strcmp(name, "head")){
          if(!pname || strcmp(pname, "article")){
            art->errmsg = cbsprintf("Invalid-XML: \"head\" must be a child of \"article\"");
          } else if(ishead){
            art->errmsg = cbsprintf("Invalid-XML: \"head\" must not be duplicated");
          } else if(cbstrbwmatch(elem, "/>")){
            art->errmsg = cbsprintf("Invalid-XML: \"head\" must not be empty");
          } else {
            ishead = TRUE;
          }
        } else if(!strcmp(name, "flags")){
          if(!pname || strcmp(pname, "head")){
            art->errmsg = cbsprintf("Invalid-XML: \"flags\" must be a child of \"head\"");
          } else if(art->flags){
            art->errmsg = cbsprintf("Invalid-XML: \"flags\" must not be duplicated");
          } else if(cbstrbwmatch(elem, "/>")){
            art->errmsg = cbsprintf("Invalid-XML: \"flags\" must not be empty");
          } else {
            rtext = cbxmlunescape(ntext);
            art->flags = strnormalize(rtext, TRUE, TRUE);
            free(rtext);
          }
        } else if(!strcmp(name, "language")){
          if(!pname || strcmp(pname, "head")){
            art->errmsg = cbsprintf("Invalid-XML: \"language\" must be a child of \"head\"");
          } else if(art->language){
            art->errmsg = cbsprintf("Invalid-XML: \"language\" must not be duplicated");
          } else if(cbstrbwmatch(elem, "/>")){
            art->errmsg = cbsprintf("Invalid-XML: \"language\" must not be empty");
          } else {
            rtext = cbxmlunescape(ntext);
            art->language = strnormalize(rtext, TRUE, TRUE);
            free(rtext);
          }
        } else if(!strcmp(name, "category")){
          if(!pname || strcmp(pname, "head")){
            art->errmsg = cbsprintf("Invalid-XML: \"category\" must be a child of \"head\"");
          } else if(art->category){
            art->errmsg = cbsprintf("Invalid-XML: \"category\" must not be duplicated");
          } else if(cbstrbwmatch(elem, "/>")){
            art->errmsg = cbsprintf("Invalid-XML: \"category\" must not be empty");
          } else {
            rtext = cbxmlunescape(ntext);
            art->category = strnormalize(rtext, TRUE, TRUE);
            free(rtext);
          }
        } else if(!strcmp(name, "subject")){
          if(!pname || strcmp(pname, "head")){
            art->errmsg = cbsprintf("Invalid-XML: \"subject\" must be a child of \"head\"");
          } else if(art->subject){
            art->errmsg = cbsprintf("Invalid-XML: \"subject\" must not be duplicated");
          } else if(cbstrbwmatch(elem, "/>")){
            art->errmsg = cbsprintf("Invalid-XML: \"subject\" must not be empty");
          } else {
            rtext = cbxmlunescape(ntext);
            art->subject = strnormalize(rtext, TRUE, TRUE);
            free(rtext);
          }
        } else if(!strcmp(name, "author")){
          if(!pname || strcmp(pname, "head")){
            art->errmsg = cbsprintf("Invalid-XML: \"author\" must be a child of \"head\"");
          } else if(art->author){
            art->errmsg = cbsprintf("Invalid-XML: \"author\" must not be duplicated");
          } else if(cbstrbwmatch(elem, "/>")){
            art->errmsg = cbsprintf("Invalid-XML: \"author\" must not be empty");
          } else {
            rtext = cbxmlunescape(ntext);
            art->author = strnormalize(rtext, TRUE, TRUE);
            free(rtext);
          }
        } else if(!strcmp(name, "cdate")){
          if(!pname || strcmp(pname, "head")){
            art->errmsg = cbsprintf("Invalid-XML: \"cdate\" must be a child of \"head\"");
          } else if(art->cdate){
            art->errmsg = cbsprintf("Invalid-XML: \"cdate\" must not be duplicated");
          } else if(cbstrbwmatch(elem, "/>")){
            art->errmsg = cbsprintf("Invalid-XML: \"cdate\" must not be empty");
          } else {
            rtext = cbxmlunescape(ntext);
            art->cdate = strnormalize(rtext, TRUE, TRUE);
            free(rtext);
          }
        } else if(!strcmp(name, "mdate")){
          if(!pname || strcmp(pname, "head")){
            art->errmsg = cbsprintf("Invalid-XML: \"mdate\" must be a child of \"head\"");
          } else if(art->mdate){
            art->errmsg = cbsprintf("Invalid-XML: \"mdate\" must not be duplicated");
          } else if(cbstrbwmatch(elem, "/>")){
            art->errmsg = cbsprintf("Invalid-XML: \"mdate\" must not be empty");
          } else {
            rtext = cbxmlunescape(ntext);
            art->mdate = strnormalize(rtext, TRUE, TRUE);
            free(rtext);
          }
        } else if(!strcmp(name, "body")){
          if(!pname || strcmp(pname, "article")){
            art->errmsg = cbsprintf("Invalid-XML: \"body\" must be a child of \"article\"");
          } else if(isbody){
            art->errmsg = cbsprintf("Invalid-XML: \"body\" must not be duplicated");
          } else if(cbstrbwmatch(elem, "/>")){
            art->errmsg = cbsprintf("Invalid-XML: \"body\" must not be empty");
          } else {
            isbody = TRUE;
          }
        } else if(!strcmp(name, "topic")){
          if(!pname || strcmp(pname, "body")){
            art->errmsg = cbsprintf("Invalid-XML: \"topic\" must be a child of \"body\"");
          } else if(cbstrbwmatch(elem, "/>")){
            art->errmsg = cbsprintf("Invalid-XML: \"topic\" must not be empty");
          } else {
            bstart = i;
          }
        } else if(!strcmp(name, "para")){
          if(!pname || strcmp(pname, "body")){
            art->errmsg = cbsprintf("Invalid-XML: \"para\" must be a child of \"body\"");
          } else if(cbstrbwmatch(elem, "/>")){
            art->errmsg = cbsprintf("Invalid-XML: \"para\" must not be empty");
          } else {
            bstart = i;
          }
        } else if(!strcmp(name, "asis")){
          if(!pname || strcmp(pname, "body")){
            art->errmsg = cbsprintf("Invalid-XML: \"asis\" must be a child of \"body\"");
          } else if(cbstrbwmatch(elem, "/>")){
            art->errmsg = cbsprintf("Invalid-XML: \"asis\" must not be empty");
          } else {
            bstart = i;
          }
        } else if(!strcmp(name, "list")){
          if(!pname || strcmp(pname, "body")){
            art->errmsg = cbsprintf("Invalid-XML: \"list\" must be a child of \"body\"");
          } else if(cbstrbwmatch(elem, "/>")){
            art->errmsg = cbsprintf("Invalid-XML: \"list\" must not be empty");
          } else {
            bstart = i;
          }
        } else if(!strcmp(name, "item")){
          if(!pname || strcmp(pname, "list")){
            art->errmsg = cbsprintf("Invalid-XML: \"item\" must be a child of \"list\"");
          } else if(cbstrbwmatch(elem, "/>")){
            art->errmsg = cbsprintf("Invalid-XML: \"item\" must not be empty");
          }
        } else if(!strcmp(name, "graph")){
          if(!pname || strcmp(pname, "body")){
            art->errmsg = cbsprintf("Invalid-XML: \"graph\" must be a child of \"body\"");
          } else if(!cbstrbwmatch(elem, "/>")){
            art->errmsg = cbsprintf("Invalid-XML: \"graph\" must be empty");
          } else if(!cbmapget(attrs, "data", -1, NULL)){
            art->errmsg = cbsprintf("Invalid-XML: \"graph\" must have the attribute \"data\"");
          } else {
            cblistpush(art->body, elem, -1);
          }
        } else if(!strcmp(name, "link")){
          if(!pname || bstart < 0){
            art->errmsg = cbsprintf("Invalid-XML: \"link\" must be in a block");
          } else if(cbstrbwmatch(elem, "/>")){
            art->errmsg = cbsprintf("Invalid-XML: \"link\" must not be empty");
          } else if(!cbmapget(attrs, "to", -1, NULL)){
            art->errmsg = cbsprintf("Invalid-XML: \"link\" must have the attribute \"to\"");
          }
        } else if(!strcmp(name, "emp")){
          if(!pname || bstart < 0){
            art->errmsg = cbsprintf("Invalid-XML: \"emp\" must be in a block");
          } else if(cbstrbwmatch(elem, "/>")){
            art->errmsg = cbsprintf("Invalid-XML: \"emp\" must not be empty");
          }
        } else if(!strcmp(name, "cite")){
          if(!pname || bstart < 0){
            art->errmsg = cbsprintf("Invalid-XML: \"cite\" must be in a block");
          } else if(cbstrbwmatch(elem, "/>")){
            art->errmsg = cbsprintf("Invalid-XML: \"cite\" must not be empty");
          }
        } else if(!strcmp(name, "ins")){
          if(!pname || bstart < 0){
            art->errmsg = cbsprintf("Invalid-XML: \"ins\" must be in a block");
          } else if(cbstrbwmatch(elem, "/>")){
            art->errmsg = cbsprintf("Invalid-XML: \"ins\" must not be empty");
          }
        } else if(!strcmp(name, "del")){
          if(!pname || bstart < 0){
            art->errmsg = cbsprintf("Invalid-XML: \"del\" must be in a block");
          } else if(cbstrbwmatch(elem, "/>")){
            art->errmsg = cbsprintf("Invalid-XML: \"del\" must not be empty");
          }
        } else if(!strcmp(name, "tail")){
          if(!pname || strcmp(pname, "article")){
            art->errmsg = cbsprintf("Invalid-XML: \"tail\" must be a child of \"article\"");
          } else if(istail){
            art->errmsg = cbsprintf("Invalid-XML: \"tail\" must not be duplicated");
          }
          istail = TRUE;
        } else if(!strcmp(name, "response")){
          if(!pname || strcmp(pname, "tail")){
            art->errmsg = cbsprintf("Invalid-XML: \"response\" must be in a block");
          } else if(cbstrbwmatch(elem, "/>")){
            art->errmsg = cbsprintf("Invalid-XML: \"response\" must not be empty");
          } else if(!cbmapget(attrs, "flags", -1, NULL)){
            art->errmsg =
              cbsprintf("Invalid-XML: \"response\" must have the attribute \"flags\"");
          } else if(!cbmapget(attrs, "author", -1, NULL)){
            art->errmsg =
              cbsprintf("Invalid-XML: \"response\" must have the attribute \"author\"");
          } else if(!cbmapget(attrs, "cdate", -1, NULL)){
            art->errmsg =
              cbsprintf("Invalid-XML: \"response\" must have the attribute \"cdate\"");
          } else {
            rstart = i;
          }
        } else {
          art->errmsg = cbsprintf("Invalid-XML: a tag \"%s\" is unknown", name);
        }
        if(!cbstrbwmatch(elem, "/>")) cblistpush(stack, name, -1);
      }
      cbmapclose(attrs);
    } else {
      if(strisprintable(elem)){
        if(!pname) pname = "";
        if(strcmp(pname, "flags") && strcmp(pname, "language") && strcmp(pname, "category") &&
           strcmp(pname, "subject") && strcmp(pname, "author") && strcmp(pname, "cdate") &&
           strcmp(pname, "mdate") && strcmp(pname, "topic") && strcmp(pname, "para") &&
           strcmp(pname, "asis") && strcmp(pname, "item") && strcmp(pname, "link") &&
           strcmp(pname, "emp") && strcmp(pname, "cite") && strcmp(pname, "ins") &&
           strcmp(pname, "del") && strcmp(pname, "response")){
          art->errmsg =
            cbsprintf("Invalid-XML: a printable character must not be included in a block");
        }
      }
    }
    if(art->errmsg) break;
  }
  if(!art->id){
    art->id = uniqueid();
  } else if(!validid(art->id)){
    if(!art->errmsg) art->errmsg = cbsprintf("Invalid-XML: the ID is invalid");
  }
  if(!art->password) art->password = cbmemdup("", 0);
  if(!art->flags) art->flags = cbmemdup("", 0);
  if(!art->language || art->language[0] == '\0'){
    if(!art->errmsg) art->errmsg = cbsprintf("Invalid-XML: the language is undefined");
    if(!art->language) art->language = cbmemdup("", 0);
  }
  if(!art->category || art->category[0] == '\0'){
    if(!art->errmsg) art->errmsg = cbsprintf("Invalid-XML: the category is undefined");
    if(!art->category) art->category = cbmemdup("", 0);
  }
  if(!art->subject || art->subject[0] == '\0'){
    if(!art->errmsg) art->errmsg = cbsprintf("Invalid-XML: the subject is undefined");
    if(!art->subject) art->subject = cbmemdup("", 0);
  }
  if(!art->author || art->author[0] == '\0'){
    if(!art->errmsg) art->errmsg = cbsprintf("Invalid-XML: the author is undefined");
    if(!art->author) art->author = cbmemdup("", 0);
  }
  if(!art->cdate || art->cdate[0] == '\0' || !validdate(art->cdate)){
    if(!art->errmsg)
      art->errmsg = cbsprintf("Invalid-XML: the creation date is undefined or malformed");
    if(!art->cdate) art->cdate = cbmemdup("1970-01-01 00:00:00", 0);
  }
  if(!art->mdate || art->mdate[0] == '\0' || !validdate(art->mdate)){
    if(!art->errmsg)
      art->errmsg = cbsprintf("Invalid-XML: the modification date is undefined or malformed");
    if(!art->mdate) art->mdate = cbmemdup("1970-01-01 00:00:00", 0);
  }
  cbdatumclose(cont);
  cblistclose(stack);
  cblistclose(elems);
  if(!art->errmsg && !validxmlent(xml))
    art->errmsg = cbsprintf("Invalid-XML: there is an unknown entity reference");
  return art;
}


/* Release resources of an article object. */
void freearticle(ARTICLE *art){
  assert(art);
  free(art->hash);
  free(art->errmsg);
  cblistclose(art->tail);
  cblistclose(art->body);
  free(art->mdate);
  free(art->cdate);
  free(art->author);
  free(art->subject);
  free(art->category);
  free(art->language);
  free(art->flags);
  free(art->password);
  free(art->id);
  free(art);
}


/* Check the article is OK. */
int articleisok(ARTICLE *art){
  assert(art);
  return art->errmsg ? FALSE : TRUE;
}


/* Get the error message of an article. */
const char *artemsg(ARTICLE *art){
  assert(art);
  if(!art->errmsg) return "unknown error";
  return art->errmsg;
}


/* Get the hash code of an article. */
const char *arthash(ARTICLE *art){
  assert(art);
  return art->hash;
}


/* Get the ID string of an article. */
const char *artid(ARTICLE *art){
  assert(art);
  return art->id;
}


/* Get the password of an article. */
const char *artpassword(ARTICLE *art){
  assert(art);
  return art->password;
}


/* Get the flags of an article. */
const char *artflags(ARTICLE *art){
  assert(art);
  return art->flags;
}


/* Get the language of an article. */
const char *artlanguage(ARTICLE *art){
  assert(art);
  return art->language;
}


/* Get the category of an article. */
const char *artcategory(ARTICLE *art){
  assert(art);
  return art->category;
}


/* Get the subject of an article. */
const char *artsubject(ARTICLE *art){
  assert(art);
  return art->subject;
}


/* Get the author of an article. */
const char *artauthor(ARTICLE *art){
  assert(art);
  return art->author;
}


/* Get the creation date of an article. */
const char *artcdate(ARTICLE *art){
  assert(art);
  return art->cdate;
}


/* Get the modification date of an article. */
const char *artmdate(ARTICLE *art){
  assert(art);
  return art->mdate;
}


/* Get the number of responses of an article. */
int artresnum(ARTICLE *art){
  assert(art);
  return cblistnum(art->tail);
}


/* Add a response to an article. */
void artaddres(ARTICLE *art, const char *flags, const char *author, const char *cdate,
               const char *text){
  CBDATUM *buf;
  assert(art && flags && author && cdate && text);
  if(strchr(art->flags, LONECHAR)) return;
  buf = cbdatumopen("", 0);
  xsprintf(buf, "<response flags=\"%@\" author=\"%@\" cdate=\"%@\">%@</response>",
           flags, author, cdate, text);
  cblistpush(art->tail, cbdatumptr(buf), -1);
  cbdatumclose(buf);
}


/* Set the creation date of an article. */
void artsetcdate(ARTICLE *art, const char *date){
  assert(art && date);
  free(art->cdate);
  art->cdate = cbmemdup(date, -1);
}


/* Set the modification date of an article. */
void artsetmdate(ARTICLE *art, const char *date){
  assert(art && date);
  free(art->mdate);
  art->mdate = cbmemdup(date, -1);
}


/* Serialize an article into a XML string. */
char *arttoxml(ARTICLE *art){
  CBDATUM *buf;
  CBLIST *elems;
  CBMAP *attrs;
  const char *blk, *elem, *name, *val, *flags, *author, *cdate;
  char *rtext;
  int i, j, inlist, initem;
  assert(art);
  buf = cbdatumopen("", 0);
  xsprintf(buf, "<article id=\"%@\" password=\"%@\">\n", art->id, art->password);
  xsprintf(buf, "<head>\n");
  xsprintf(buf, "<flags>%@</flags>\n", art->flags);
  xsprintf(buf, "<language>%@</language>\n", art->language);
  xsprintf(buf, "<category>%@</category>\n", art->category);
  xsprintf(buf, "<subject>%@</subject>\n", art->subject);
  xsprintf(buf, "<author>%@</author>\n", art->author);
  xsprintf(buf, "<cdate>%@</cdate>\n", art->cdate);
  xsprintf(buf, "<mdate>%@</mdate>\n", art->mdate);
  xsprintf(buf, "</head>\n");
  xsprintf(buf, "<body>\n");
  inlist = FALSE;
  initem = FALSE;
  for(i = 0; i < cblistnum(art->body); i++){
    blk = cblistval(art->body, i, NULL);
    elems = cbxmlbreak(blk, FALSE);
    for(j = 0; j < cblistnum(elems); j++){
      elem = cblistval(elems, j, NULL);
      if(elem[0] == '<'){
        attrs = cbxmlattrs(elem);
        name = cbmapget(attrs, "", -1, NULL);
        if(elem[1] == '/'){
          xsprintf(buf, "</%@>", name);
          if(!strcmp(name, "topic") || !strcmp(name, "para") || !strcmp(name, "asis") ||
             !strcmp(name, "list") || !strcmp(name, "item")) xsprintf(buf, "\n");
          if(!strcmp(name, "list")){
            inlist = FALSE;
          } else if(!strcmp(name, "item")){
            initem = FALSE;
          }
        } else {
          if(!strcmp(name, "graph")){
            if(!(val = cbmapget(attrs, "data", -1, NULL))) val = "";
            xsprintf(buf, "<graph data=\"%@\"/>\n", val);
          } else if(!strcmp(name, "link")){
            if(!(val = cbmapget(attrs, "to", -1, NULL))) val = "";
            xsprintf(buf, "<link to=\"%@\">", val);
          } else {
            xsprintf(buf, "<%@>", name);
            if(!strcmp(name, "list")){
              xsprintf(buf, "\n");
              inlist = TRUE;
            } else if(!strcmp(name, "item")){
              initem = TRUE;
            }
          }
        }
        cbmapclose(attrs);
      } else {
        if(!inlist || initem) xsprintf(buf, "%s", elem);
      }
    }
    cblistclose(elems);
  }
  xsprintf(buf, "</body>\n");
  xsprintf(buf, "<tail>\n");
  for(i = 0; i < cblistnum(art->tail); i++){
    blk = cblistval(art->tail, i, NULL);
    elems = cbxmlbreak(blk, FALSE);
    for(j = 0; j < cblistnum(elems); j++){
      elem = cblistval(elems, j, NULL);
      if(elem[0] == '<'){
        attrs = cbxmlattrs(elem);
        name = cbmapget(attrs, "", -1, NULL);
        if(!strcmp(name, "response")){
          if(elem[1] == '/'){
            xsprintf(buf, "</%@>\n", name);
          } else {
            if(!(flags = cbmapget(attrs, "flags", -1, NULL))) flags = "";
            if(!(author = cbmapget(attrs, "author", -1, NULL))) author = "";
            if(!(cdate = cbmapget(attrs, "cdate", -1, NULL))) cdate = "";
            xsprintf(buf, "<response flags=\"%@\" author=\"%@\" cdate=\"%@\">",
                     flags, author, cdate);
          }
        }
        cbmapclose(attrs);
      } else {
        if(strisprintable(elem)){
          rtext = strnormalize(elem, TRUE, TRUE);
          xsprintf(buf, "%s", rtext);
          free(rtext);
        }
      }
    }
    cblistclose(elems);
  }
  xsprintf(buf, "</tail>\n");
  xsprintf(buf, "</article>\n");
  return cbdatumtomalloc(buf, NULL);
}


/* Serialize an article into a string of plain text. */
char *arttotext(ARTICLE *art){
  CBDATUM *buf;
  CBLIST *elems;
  CBMAP *attrs;
  const char *blk, *elem, *name, *tmp;
  char *flags, *author, *cdate, *text, *rtext;
  int i, j;
  assert(art);
  buf = cbdatumopen("", 0);
  xsprintf(buf, "%s\n", art->flags);
  xsprintf(buf, "%s\n", art->language);
  xsprintf(buf, "%s\n", art->category);
  xsprintf(buf, "%s\n", art->subject);
  xsprintf(buf, "%s\n", art->author);
  xsprintf(buf, "%s\n", art->cdate);
  xsprintf(buf, "%s\n", art->mdate);
  xsprintf(buf, "\n");
  for(i = 0; i < cblistnum(art->body); i++){
    blk = cblistval(art->body, i, NULL);
    elems = cbxmlbreak(blk, FALSE);
    for(j = 0; j < cblistnum(elems); j++){
      elem = cblistval(elems, j, NULL);
      if(elem[0] == '<') continue;
      rtext = cbxmlunescape(elem);
      xsprintf(buf, "%s", rtext);
      free(rtext);
    }
    xsprintf(buf, "\n");
    cblistclose(elems);
  }
  xsprintf(buf, "\n");
  for(i = 0; i < cblistnum(art->tail); i++){
    flags = NULL;
    author = NULL;
    cdate = NULL;
    text = NULL;
    blk = cblistval(art->tail, i, NULL);
    elems = cbxmlbreak(blk, FALSE);
    for(j = 0; j < cblistnum(elems); j++){
      elem = cblistval(elems, j, NULL);
      if(elem[0] == '<'){
        attrs = cbxmlattrs(elem);
        name = cbmapget(attrs, "", -1, NULL);
        if(!strcmp(name, "response") && elem[1] != '/'){
          if(!flags && (tmp = cbmapget(attrs, "flags", -1, NULL)) != NULL)
            flags = cbmemdup(tmp, -1);
          if(!author && (tmp = cbmapget(attrs, "author", -1, NULL)) != NULL)
            author = cbmemdup(tmp, -1);
          if(!cdate && (tmp = cbmapget(attrs, "cdate", -1, NULL)) != NULL)
            cdate = dateformen(tmp);
        }
        cbmapclose(attrs);
      } else {
        if(!text && strisprintable(elem)){
          rtext = strnormalize(elem, TRUE, TRUE);
          text = cbxmlunescape(rtext);
          free(rtext);
        }
      }
    }
    cblistclose(elems);
    if(flags && author && cdate && text)
      xsprintf(buf, "%s\t%s\t%s\t%s\n", flags, author, cdate, text);
    free(text);
    free(cdate);
    free(author);
    free(flags);
  }
  return cbdatumtomalloc(buf, NULL);
}


/* Serialize an article into a string of HTML. */
char *arttohtml(ARTICLE *art, int bnum, int rnum){
  CBDATUM *buf;
  char *tmp;
  assert(art);
  buf = cbdatumopen("", 0);
  xsprintf(buf, "<div id=\"%@\" lang=\"%@\" xml:lang=\"%@\" class=\"article%s\">\n",
           art->id, art->language, art->language, artsubclass(art->flags));
  tmp = headtohtml(art);
  xsprintf(buf, "%s", tmp);
  free(tmp);
  tmp = bodytohtml(art, bnum);
  xsprintf(buf, "%s", tmp);
  free(tmp);
  tmp = tailtohtml(art, rnum);
  xsprintf(buf, "%s", tmp);
  free(tmp);
  xsprintf(buf, "</div>\n");
  return cbdatumtomalloc(buf, NULL);
}


/* Serialize an article into a string of Atom. */
char *arttoatom(ARTICLE *art, const char *prefix, int bnum, int rnum){
  CBDATUM *buf;
  char *tmp;
  assert(art && prefix);
  buf = cbdatumopen("", 0);
  xsprintf(buf, "<entry>\n");
  xsprintf(buf, "<id>%@%@</id>\n", prefix, art->id);
  xsprintf(buf, "<link rel=\"alternate\" type=\"text/html\" href=\"%@%@\"/>\n", prefix, art->id);
  xsprintf(buf, "<title>%@</title>\n", art->subject);
  xsprintf(buf, "<author>\n");
  xsprintf(buf, "<name>%@</name>\n", art->author);
  xsprintf(buf, "</author>\n");
  tmp = dateforwww(art->cdate, TRUE);
  xsprintf(buf, "<issued>%@</issued>\n", tmp);
  free(tmp);
  tmp = dateforwww(art->cdate, FALSE);
  xsprintf(buf, "<created>%@</created>\n", tmp);
  free(tmp);
  tmp = dateforwww(art->mdate, FALSE);
  xsprintf(buf, "<modified>%@</modified>\n", tmp);
  free(tmp);
  xsprintf(buf, "<content type=\"text/html\" mode=\"escaped\""
           " xml:base=\"%@%s\" xml:lang=\"%@\">\n", prefix, art->id, art->language);
  tmp = bodytohtml(art, bnum);
  xsprintf(buf, "%@", tmp);
  free(tmp);
  tmp = tailtohtml(art, rnum);
  xsprintf(buf, "%@", tmp);
  free(tmp);
  xsprintf(buf, "</content>\n");
  xsprintf(buf, "</entry>\n");
  return cbdatumtomalloc(buf, NULL);
}


/* Get the jet lag by minute. */
int jetlag(int min){
  static int lag = DEFJETLAG;
  struct tm *tp;
  time_t t, gt, lt;
  if(min < DEFJETLAG) lag = min;
  if(lag < DEFJETLAG) return lag;
  if((t = time(NULL)) < 0) return 0;
  if(!(tp = gmtime(&t))) return 0;
  gt = mktime(tp);
  if(!(tp = localtime(&t))) return 0;
  lt = mktime(tp);
  lag = (lt - gt) / 60;
  return lag;
}


/* Create the string for a current date in internal form. */
char *datecurrent(void){
  return datestr(time(NULL));
}


/* Create the string for a date in W3DTF. */
char *dateforwww(const char *date, int local){
  struct tm ts, *tp;
  time_t t;
  int jl;
  CBLIST *elems;
  char rdate[NUMBUFSIZ], tzone[NUMBUFSIZ];
  assert(date);
  if(!validdate(date)) return cbmemdup("1901-01-01T00:00:00Z", -1);
  elems = cbsplit(date, -1, "- :");
  memset(&ts, 0, sizeof(struct tm));
  ts.tm_year = atoi(cblistval(elems, 0, NULL)) - 1900;
  ts.tm_mon = atoi(cblistval(elems, 1, NULL)) - 1;
  ts.tm_mday = atoi(cblistval(elems, 2, NULL));
  ts.tm_hour = atoi(cblistval(elems, 3, NULL));
  ts.tm_min = atoi(cblistval(elems, 4, NULL));
  ts.tm_sec = atoi(cblistval(elems, 5, NULL));
  sprintf(tzone, "Z");
  if(local){
    jl = jetlag(DEFJETLAG);
    ts.tm_min += jl;
    t = mktime(&ts);
    if((tp = localtime(&t)) != NULL) ts = *tp;
    if(jl >= 0){
      tzone[0] = '+';
    } else {
      jl *= -1;
      tzone[0] = '-';
    }
    sprintf(tzone + 1, "%02d:%02d", jl / 60, jl % 60);
  }
  sprintf(rdate, "%04d-%02d-%02dT%02d:%02d:%02d%s",
          ts.tm_year + 1900, ts.tm_mon + 1, ts.tm_mday, ts.tm_hour, ts.tm_min, ts.tm_sec, tzone);
  cblistclose(elems);
  return cbmemdup(rdate, -1);
}


/* Create the string for a date in RFC1123. */
char *dateforhttp(const char *date, int local){
  struct tm ts, *tp;
  time_t t;
  int jl;
  CBLIST *elems;
  char rdate[NUMBUFSIZ], tzone[NUMBUFSIZ];
  assert(date);
  if(!validdate(date)) return cbmemdup("Thu, 01 Jan 1970 00:00:00 GMT", -1);
  elems = cbsplit(date, -1, "- :");
  memset(&ts, 0, sizeof(struct tm));
  ts.tm_year = atoi(cblistval(elems, 0, NULL)) - 1900;
  ts.tm_mon = atoi(cblistval(elems, 1, NULL)) - 1;
  ts.tm_mday = atoi(cblistval(elems, 2, NULL));
  ts.tm_hour = atoi(cblistval(elems, 3, NULL));
  ts.tm_min = atoi(cblistval(elems, 4, NULL));
  ts.tm_sec = atoi(cblistval(elems, 5, NULL));
  sprintf(tzone, "GMT");
  if(local){
    jl = jetlag(DEFJETLAG);
    ts.tm_min += jl;
    t = mktime(&ts);
    if((tp = localtime(&t)) != NULL) ts = *tp;
    if(jl >= 0){
      tzone[0] = '+';
    } else {
      jl *= -1;
      tzone[0] = '-';
    }
    sprintf(tzone + 1, "%02d%02d", jl / 60, jl % 60);
  }
  sprintf(rdate, "%s, %02d %s %04d %02d:%02d:%02d %s",
          wdayname(ts.tm_wday), ts.tm_mday, monname(ts.tm_mon), ts.tm_year + 1900,
          ts.tm_hour, ts.tm_min, ts.tm_sec, tzone);
  cblistclose(elems);
  return cbmemdup(rdate, -1);
}


/* Create the string for a date which is human readable. */
char *dateformen(const char *date){
  char *buf;
  buf = dateforwww(date, TRUE);
  buf[4] = '/';
  buf[7] = '/';
  buf[10] = ' ';
  buf[16] = '\0';
  return buf;
}


/* Normalize a string.
   `str' specifies a string.
   `nolf' specifies whether to convert a line feed to a space.
   `trim' specifies whether to cut the heading spaces and tailing spaces.
   The return value is a converted string. */
char *strnormalize(const char *str, int nolf, int trim){
  char *buf, *rp, *wp;
  int miss;
  assert(str);
  if((buf = cbiconv(str, -1, "UTF-8", "UTF-8", NULL, &miss)) != NULL && miss > 0){
    free(buf);
    buf = cbiconv(str, -1, "UTF-8", "US-ASCII", NULL, NULL);
  }
  if(!buf) buf = cbmemdup(str, -1);
  rp = buf;
  wp = buf;
  while(*rp != '\0'){
    if(*rp == '\n'){
      *(wp++) = nolf ? ' ' : '\n';
    } else if(*rp == '\t' || *rp == '\v' || *rp == '\f'){
      *(wp++) = ' ';
    } else if(*rp < 0 || (*rp >= ' ' && *rp <= '~')){
      *(wp++) = *rp;
    }
    rp++;
  }
  *wp = '\0';
  if(trim) cbstrtrim(buf);
  return buf;
}


/* Perform formatted conversion and write the result into a buffer. */
void xsprintf(CBDATUM *buf, const char *format, ...){
  va_list ap;
  char *tmp, numbuf[NUMBUFSIZ];
  unsigned char c;
  assert(buf && format);
  va_start(ap, format);
  while(*format != '\0'){
    if(*format == '%'){
      format++;
      switch(*format){
      case 'c':
        c = va_arg(ap, int);
        cbdatumcat(buf, (char *)&c, 1);
        break;
      case 's':
        tmp = va_arg(ap, char *);
        if(!tmp) tmp = "(null)";
        cbdatumcat(buf, tmp, -1);
        break;
      case 'd':
        sprintf(numbuf, "%d", va_arg(ap, int));
        cbdatumcat(buf, numbuf, -1);
        break;
      case '@':
        tmp = va_arg(ap, char *);
        if(!tmp) tmp = "(null)";
        while(*tmp){
          switch(*tmp){
          case '&': cbdatumcat(buf, "&amp;", -1); break;
          case '<': cbdatumcat(buf, "&lt;", -1); break;
          case '>': cbdatumcat(buf, "&gt;", -1); break;
          case '"': cbdatumcat(buf, "&quot;", -1); break;
          default:
            if(!((*tmp >= 0 && *tmp <= 0x8) || (*tmp >= 0x0e && *tmp <= 0x1f)))
              cbdatumcat(buf, tmp, 1);
            break;
          }
          tmp++;
        }
        break;
      case '?':
        tmp = va_arg(ap, char *);
        if(!tmp) tmp = "(null)";
        while(*tmp){
          c = *(unsigned char *)tmp;
          if((c >= 'A' && c <= 'Z') || (c >= 'a' && c <= 'z') ||
             (c >= '0' && c <= '9') || (c != '\0' && strchr("_-.", c))){
            cbdatumcat(buf, (char *)&c, 1);
          } else {
            sprintf(numbuf, "%%%02X", c);
            cbdatumcat(buf, numbuf, -1);
          }
          tmp++;
        }
        break;
      case '%':
        cbdatumcat(buf, "%", 1);
        break;
      }
    } else {
      cbdatumcat(buf, format, 1);
    }
    format++;
  }
  va_end(ap);
}


/* Get a DES hash string of a key string. */
const char *makecrypt(const char *key){
  const char tbl[] = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789./";
  const char *rp;
  char salt[3];
  int pid;
  assert(key);
  pid = getpid();
  salt[0] = tbl[((pid%65536)/256)%(sizeof(tbl)-1)];
  salt[1] = tbl[(pid%256)%(sizeof(tbl)-1)];
  salt[2] = '\0';
  if(!(rp = crypt(key, salt))) rp = "";
  return rp;
}


/* Check whether a key matches a DES hash string. */
int matchcrypt(const char *key, const char *code){
  const char *rp;
  char salt[3];
  assert(key && code);
  if(strlen(code) < 2) return FALSE;
  salt[0] = code[0];
  salt[1] = code[1];
  salt[2] = '\0';
  if(!(rp = crypt(key, salt))) return FALSE;
  return strcmp(rp, code) ? FALSE : TRUE;
}



/*************************************************************************************************
 * private objects
 *************************************************************************************************/


/* Check whether an XML text does include no invalid entity reference.
   `xml' specifies a string.
   The return value is true if no invalid entity reference, else it is false */
static int validxmlent(const char *xml){
  assert(xml);
  while(*xml != '\0'){
    if((*xml == '&') && !cbstrfwmatch(xml, "&amp;") &&
       !cbstrfwmatch(xml, "&lt;") && !cbstrfwmatch(xml, "&gt;") &&
       !cbstrfwmatch(xml, "&quot;") && !cbstrfwmatch(xml, "&apos;")) return FALSE;
    xml++;
  }
  return TRUE;
}


/* Create a unique ID.
   The return value is the string of a unique ID. */
static char *uniqueid(void){
  FILE *ifp;
  char str[PATHBUFSIZ];
  int i, rnd;
  rnd = 0;
  if((ifp = fopen("/dev/urandom", "r")) != NULL){
    for(i = 0; i < 4; i++){
      rnd = (rnd << 8) + fgetc(ifp);
    }
    fclose(ifp);
  }
  sprintf(str, "%s%010d%05d%05d",
          IDPREFIX, (int)time(NULL), (int)getpid() % 100000, (unsigned int)rnd % 100000);
  return cbmemdup(str, -1);
}


/* Check whether an ID string is valid.
   The return value is true if an ID string is valid, else, it is not. */
static int validid(const char *id){
  int i;
  assert(id);
  if(!cbstrfwmatch(id, IDPREFIX)) return FALSE;
  for(i = strlen(IDPREFIX); id[i] != '\0'; i++){
    if(id[i] < '0' || id[i] > '9') return FALSE;
  }
  return TRUE;
}


/* Create the string for a date.
   `t' specifies the UNIX time.
   The return value is the string of the date in the internal form (YYYY-MM-DD hh:mm:ss). */
static char *datestr(time_t t){
  struct tm *tp;
  char date[NUMBUFSIZ];
  if(t < 0 || !(tp = gmtime(&t))) return "1970-01-01 00:00:00";
  sprintf(date, "%04d-%02d-%02d %02d:%02d:%02d",
          tp->tm_year + 1900, tp->tm_mon + 1, tp->tm_mday, tp->tm_hour, tp->tm_min, tp->tm_sec);
  return cbmemdup(date, -1);
}


/* Check whether a date string is valid or not.
   `date' specifies a string for a date.
   The return value is true if it is valid, else it is false. */
static int validdate(const char *date){
  int i;
  assert(date);
  for(i = 0; i <= 3; i++){
    if(date[i] < '0' || date[i] > '9') return FALSE;
  }
  if(date[4] != '-') return FALSE;
  for(i = 5; i <= 6; i++){
    if(date[i] < '0' || date[i] > '9') return FALSE;
  }
  if(atoi(date + 5) > 12) return FALSE;
  if(date[7] != '-') return FALSE;
  for(i = 8; i <= 9; i++){
    if(date[i] < '0' || date[i] > '9') return FALSE;
  }
  if(atoi(date + 8) > 31) return FALSE;
  if(date[10] != ' ') return FALSE;
  for(i = 11; i <= 12; i++){
    if(date[i] < '0' || date[i] > '9') return FALSE;
  }
  if(atoi(date + 11) > 24) return FALSE;
  if(date[13] != ':') return FALSE;
  for(i = 14; i <= 15; i++){
    if(date[i] < '0' || date[i] > '9') return FALSE;
  }
  if(atoi(date + 14) > 60) return FALSE;
  if(date[16] != ':') return FALSE;
  for(i = 17; i <= 18; i++){
    if(date[i] < '0' || date[i] > '9') return FALSE;
  }
  if(atoi(date + 17) > 60) return FALSE;
  if(date[19] != '\0') return FALSE;
  return TRUE;
}


/* Get the name of a week day.
   `wday' specifies the number of a week day (0-6).
   The return value is the strint of the name of the week day. */
static const char *wdayname(int wday){
  switch(wday){
  case 1: return "Mon";
  case 2: return "Tue";
  case 3: return "Wed";
  case 4: return "Thu";
  case 5: return "Fri";
  case 6: return "Sat";
  }
  return "Sun";
}


/* Get the name of a month.
   `wday' specifies the number of a month (0-11).
   The return value is the strint of the name of the month. */
static const char *monname(int mon){
  switch(mon){
  case 1: return "Feb";
  case 2: return "Mar";
  case 3: return "Apr";
  case 4: return "May";
  case 5: return "Jun";
  case 6: return "Jul";
  case 7: return "Aug";
  case 8: return "Sep";
  case 9: return "Oct";
  case 10: return "Nov";
  case 11: return "Dec";
  }
  return "Jan";
}


/* Check whether a string has one or more printable characters.
   `str' specifies a string.
   The return value is true if a string has one or more printable characters, else it is false. */
static int strisprintable(const char *str){
  assert(str);
  while(*str != '\0'){
    if(!strchr(" \t\r\n\v\f\a\b", *str)) return TRUE;
    str++;
  }
  return FALSE;
}


/* Extract HTML from the head of an article.
   `art' specifies a handle of a article object.
   The return value is a string of HTML. */
static char *headtohtml(ARTICLE *art){
  CBDATUM *buf;
  char *tmp;
  assert(art);
  buf = cbdatumopen("", 0);
  xsprintf(buf, "<div id=\"%@H\" class=\"head\">\n", art->id);
  xsprintf(buf, "<div class=\"line\">\n");
  xsprintf(buf, "<h2 class=\"subject %@\">%@</h2>\n",
           hashclass("subject", art->subject), art->subject);
  xsprintf(buf, "<div class=\"category\">%@</div>\n", art->category);
  xsprintf(buf, "<div class=\"resnum\">%d</div>\n", cblistnum(art->tail));
  xsprintf(buf, "</div>\n");
  xsprintf(buf, "<div class=\"line\">\n");
  xsprintf(buf, "<div class=\"author %@\">%@</div>\n",
           hashclass("author", art->author), art->author);
  tmp = dateformen(art->cdate);
  xsprintf(buf, "<div class=\"cdate\">%@</div>\n", tmp);
  free(tmp);
  tmp = dateformen(art->mdate);
  xsprintf(buf, "<div class=\"mdate\">%@</div>\n", tmp);
  free(tmp);
  xsprintf(buf, "</div>\n");
  xsprintf(buf, "</div>\n");
  return cbdatumtomalloc(buf, NULL);
}


/* Extract HTML from the body of an article.
   `art' specifies a handle of a article object.
   `num' specifies the number of shown blocks in the body.  If it is zero, all responses are
   shown.  If it is negative, the absolute value is calculated and the lower blocks are shown.
   The return value is a string of HTML. */
static char *bodytohtml(ARTICLE *art, int num){
  CBDATUM *buf;
  CBLIST *elems;
  CBMAP *attrs;
  const char *blk, *elem, *name, *oname, *val;
  char *rtext;
  int i, j, start, end, inlist, initem;
  assert(art);
  buf = cbdatumopen("", 0);
  xsprintf(buf, "<div id=\"%@B\" class=\"body\">\n", art->id);
  start = 0;
  end = cblistnum(art->body);
  if(num > 0 && num < end){
    end = num;
  } else if(num < 0 && -num < end){
    start += end + num;
  }
  if(start > 0)
    xsprintf(buf, "<div class=\"abbrev\"><code>......</code></div>\n");
  inlist = FALSE;
  initem = FALSE;
  for(i = start; i < end; i++){
    blk = cblistval(art->body, i, NULL);
    elems = cbxmlbreak(blk, FALSE);
    for(j = 0; j < cblistnum(elems); j++){
      elem = cblistval(elems, j, NULL);
      if(elem[0] == '<'){
        attrs = cbxmlattrs(elem);
        name = cbmapget(attrs, "", -1, NULL);
        oname = name;
        if(!strcmp(name, "topic")){
          name = "h3";
        } else if(!strcmp(name, "para")){
          name = "p";
        } else if(!strcmp(name, "asis")){
          name = "pre";
        } else if(!strcmp(name, "list")){
          name = "ul";
        } else if(!strcmp(name, "item")){
          name = "li";
        } else if(!strcmp(name, "graph")){
          name = "img";
        } else if(!strcmp(name, "link")){
          name = "a";
        } else if(!strcmp(name, "emp")){
          name = "strong";
        } else if(!strcmp(name, "cite")){
          name = "cite";
        } else if(!strcmp(name, "ins")){
          name = "ins";
        } else if(!strcmp(name, "del")){
          name = "del";
        }
        if(elem[1] == '/'){
          xsprintf(buf, "</%@>", name);
          if(!strcmp(name, "h3") || !strcmp(name, "p") || !strcmp(name, "pre") ||
             !strcmp(name, "ul") || !strcmp(name, "li")) xsprintf(buf, "\n");
          if(!strcmp(name, "ul")){
            inlist = FALSE;
          } else if(!strcmp(name, "li")){
            initem = FALSE;
          }
        } else {
          if(!strcmp(name, "img")){
            if(!(val = cbmapget(attrs, "data", -1, NULL))) val = "";
            xsprintf(buf, "<div class=\"%@\"><img src=\"%@\" alt=\"[graph]\" /></div>\n",
                     oname, val);
          } else if(!strcmp(name, "a")){
            if(!(val = cbmapget(attrs, "to", -1, NULL))) val = "";
            xsprintf(buf, "<a href=\"%@\" class=\"%@\">", val, oname);
          } else {
            xsprintf(buf, "<%@ class=\"%@\">", name, oname);
            if(!strcmp(name, "ul")){
              xsprintf(buf, "\n");
              inlist = TRUE;
            } else if(!strcmp(name, "li")){
              initem = TRUE;
            }
          }
        }
        cbmapclose(attrs);
      } else {
        if(!inlist || initem){
          rtext = cbxmlunescape(elem);
          xsprintf(buf, "%@", rtext);
          free(rtext);
        }
      }
    }
    cblistclose(elems);
  }
  if(end < cblistnum(art->body))
    xsprintf(buf, "<div class=\"abbrev\"><code>......</code></div>\n");
  xsprintf(buf, "</div>\n");
  return cbdatumtomalloc(buf, NULL);
}


/* Extract HTML from the tail of an article.
   `art' specifies a handle of a article object.
   `num' specifies the number of shown responses.  If it is zero, all responses are shown.
   If it is negative, the absolute value is calculated and the lower responses are shown.
   The return value is a string of HTML. */
static char *tailtohtml(ARTICLE *art, int num){
  CBDATUM *buf;
  CBLIST *elems;
  CBMAP *attrs;
  const char *blk, *elem, *name, *tmp;
  char *flags, *author, *cdate, *text, *rtext;
  int i, j, start, end;
  assert(art);
  if(cblistnum(art->tail) < 1) return cbmemdup("", 0);
  buf = cbdatumopen("", 0);
  xsprintf(buf, "<div id=\"%@T\" class=\"tail\">\n", art->id);
  start = 0;
  end = cblistnum(art->tail);
  if(num > 0 && num < end){
    end = num;
  } else if(num < 0 && -num < end){
    start += end + num;
  }
  if(start > 0)
    xsprintf(buf, "<div class=\"abbrev\"><code>......</code></div>\n");
  for(i = start; i < end; i++){
    flags = NULL;
    author = NULL;
    cdate = NULL;
    text = NULL;
    blk = cblistval(art->tail, i, NULL);
    elems = cbxmlbreak(blk, FALSE);
    for(j = 0; j < cblistnum(elems); j++){
      elem = cblistval(elems, j, NULL);
      if(elem[0] == '<'){
        attrs = cbxmlattrs(elem);
        name = cbmapget(attrs, "", -1, NULL);
        if(!strcmp(name, "response") && elem[1] != '/'){
          if(!flags && (tmp = cbmapget(attrs, "flags", -1, NULL)) != NULL)
            flags = cbmemdup(tmp, -1);
          if(!author && (tmp = cbmapget(attrs, "author", -1, NULL)) != NULL)
            author = cbmemdup(tmp, -1);
          if(!cdate && (tmp = cbmapget(attrs, "cdate", -1, NULL)) != NULL)
            cdate = dateformen(tmp);
        }
        cbmapclose(attrs);
      } else {
        if(!text && strisprintable(elem)){
          rtext = strnormalize(elem, TRUE, TRUE);
          text = cbxmlunescape(rtext);
          free(rtext);
        }
      }
    }
    cblistclose(elems);
    if(flags && author && cdate && text){
      xsprintf(buf, "<div class=\"response%s\">\n", ressubclass(flags));
      xsprintf(buf, "<span class=\"author %@\">%@</span>\n", hashclass("author", author), author);
      xsprintf(buf, "<span class=\"text\">%@</span>\n", text);
      xsprintf(buf, "<span class=\"cdate\">%@</span>\n", cdate);
      xsprintf(buf, "</div>\n");
    }
    free(text);
    free(cdate);
    free(author);
    free(flags);
  }
  if(end < cblistnum(art->tail))
    xsprintf(buf, "<div class=\"abbrev\"><code>......</code></div>\n");
  xsprintf(buf, "</div>\n");
  return cbdatumtomalloc(buf, NULL);
}


/* Get the sub class string of an article.
   `flags' specifies the flags of an article.
   The return value is the sub class string of the article. */
const char *artsubclass(const char *flags){
  static char name[PATHBUFSIZ];
  char *wp;
  assert(flags);
  wp = name;
  if(strchr(flags, LONECHAR)) wp += sprintf(wp, " article-%s", LONESTR);
  *wp = '\0';
  return name;
}


/* Get the sub class string of a response.
   `flags' specifies the flags of a response.
   The return value is the sub class string of the response. */
const char *ressubclass(const char *flags){
  static char name[PATHBUFSIZ];
  char *wp;
  assert(flags);
  wp = name;
  if(strchr(flags, SINKCHAR)) wp += sprintf(wp, " response-%s", SINKSTR);
  *wp = '\0';
  return name;
}


/* Get the class name of a string.
   `str' specifies a string.
   The return value is the class name of the string. */
static const char *hashclass(const char *prefix, const char *str){
  static char name[PATHBUFSIZ];
  assert(str);
  sprintf(name, "%s-%d", prefix, ((unsigned int)(dpouterhash(str, -1) * str[0])) % HASHNUM + 1);
  return name;
}



/* END OF FILE */
