package jp.sourceforge.qrcode.reader;

import jp.sourceforge.qrcode.data.*;
import jp.sourceforge.qrcode.geom.*;
import jp.sourceforge.qrcode.exception.*;
import jp.sourceforge.qrcode.reader.pattern.*;
import jp.sourceforge.qrcode.util.*;

public class QRCodeImageReader {
	public static int DECIMAL_POINT = 23;
	public static final boolean POINT_DARK  = true;
	public static final boolean POINT_LIGHT = false;
	final boolean debug = true;

	public QRCodeImageReader() {}
	
	boolean[][] applyMedianFilter(boolean[][] image, int threshold) {
		boolean[][] filteredMatrix = new boolean[image.length][image[0].length];
		int numPointDark;
		for (int y = 1; y < image[0].length - 1; y++) {
		  for (int x = 1; x < image.length - 1; x++) {
			numPointDark = 0;
			for (int fy = -1; fy < 2; fy++) {
				for (int fx = -1; fx < 2; fx++) {
					if (image[x + fx][y + fy] == true) {
						numPointDark++;
					}
				}
			}
			if (numPointDark > threshold) 
				filteredMatrix[x][y] = POINT_DARK;
		  }
		}
		
		return filteredMatrix;
	}
	
	public QRCodeSymbol getQRCodeSymbol(boolean[][] image) throws FinderPatternNotFoundException, VersionInformationException, AlignmentPatternEdgeNotFoundException, UnsupportedVersionException {
		int longSide = (image.length < image[0].length) ? image[0].length : image.length;
		QRCodeImageReader.DECIMAL_POINT = 23 - QRCodeUtility.sqrt(longSide / 256);

		if (debug) System.out.println("Scanning Finder Pattern.");
		FinderPattern finderPattern = null;
		try {
			finderPattern = FinderPattern.findFinderPattern(image);
		} catch (FinderPatternNotFoundException e) {
			if (debug) System.out.println("Not found, retrying...");
			image = applyMedianFilter(image, 5);
		
			try {
				finderPattern = FinderPattern.findFinderPattern(image);
			} catch (FinderPatternNotFoundException e2) {
				if (debug) e2.printStackTrace();
				throw e2;
			} catch (VersionInformationException e3) {
				if (debug) e3.printStackTrace();
				throw e3;
			}
		} catch (VersionInformationException e4) {
			if (debug) e4.printStackTrace();
			throw e4;
		}

		if (debug) {	
			String finderPatternCoordinates = finderPattern.getCenter(FinderPattern.UL).toString() + finderPattern.getCenter(FinderPattern.UR).toString() + finderPattern.getCenter(FinderPattern.DL).toString();
			System.out.println("FinderPattern at: " + finderPatternCoordinates);
				
 			int[] sincos = finderPattern.getAngle();
			System.out.println("Angle: Sin = " + Integer.toString(sincos[0]) + "  " + "Cos = " + Integer.toString(sincos[1]));
		}
	
		Line[][][][] samplingGrid = new Line[1][1][1][1];
		int version = finderPattern.getVersion();

		if (debug) System.out.println("Version: " + Integer.toString(version));

		AlignmentPattern alignmentPattern = null;
		if (version > 1) {
			try {
				alignmentPattern = AlignmentPattern.findAlignmentPattern(image, finderPattern);
			} catch (AlignmentPatternEdgeNotFoundException e) {
				if (debug) e.printStackTrace();
				throw e;
			} catch (UnsupportedVersionException e) {
				if (debug) e.printStackTrace();
				throw e;
			}
			
			int matrixLength = alignmentPattern.getCenter().length;
			if (debug) System.out.println("AlignmentPatterns at: ");
			for (int y = 0; y < matrixLength; y++) {
				String alignmentPatternCoordinates = "";
				for (int x = 0; x < matrixLength; x++) {
					alignmentPatternCoordinates += alignmentPattern.getCenter()[x][y].toString();
				}
				if (debug) System.out.println(alignmentPatternCoordinates);
			}
		}

		if (debug) System.out.println("Creating sampling grid.");

		if (version == 1)
			samplingGrid = getSamplingGrid1(finderPattern);
		else if (version >= 2 && version <= 6)
			samplingGrid = getSamplingGrid2_6(finderPattern, alignmentPattern);
		else if (version >= 7 && version <= 13)
			samplingGrid = getSamplingGrid7_13(finderPattern, alignmentPattern);
		else {
			if (debug) System.out.println("Version > 13 not supported.");
			throw new UnsupportedVersionException();
		}
		
		if (debug) System.out.println("Reading grid.");
		boolean[][] qRCodeMatrix = getQRCodeMatrix(image, samplingGrid);
		
		return new QRCodeSymbol(qRCodeMatrix);
	}
	
	Line[][][][] getSamplingGrid1(FinderPattern finderPattern) {
		int sqrtNumArea = 1;
		int sqrtNumModules = finderPattern.getSqrtNumModules(); 
		int sqrtNumAreaModules = sqrtNumModules / sqrtNumArea;
		Point[] centers = finderPattern.getCenter();
		int logicalDistance = 14;
		Line[][][][] samplingGrid = new Line[sqrtNumArea][sqrtNumArea][2][sqrtNumAreaModules];
		Line baseLineX, baseLineY, gridLineX, gridLineY;

		int[] modulePitch = new int[2]; //up left 
		modulePitch[0] = getAreaModulePitch(centers[0], centers[1], logicalDistance);
		modulePitch[1] = getAreaModulePitch(centers[0], centers[2], logicalDistance);

		baseLineX = new Line(
					finderPattern.getCenter(FinderPattern.UL), 
					finderPattern.getCenter(FinderPattern.DL));

		int sin = finderPattern.getAngle()[0];
		int cos = finderPattern.getAngle()[1];

		Axis axis = new Axis(sin, cos, modulePitch[0]);
		axis.setOrigin(baseLineX.getP1());
		baseLineX.setP1(axis.translate(-3, -3));

		axis.setModulePitch(modulePitch[1]);
		axis.setOrigin(baseLineX.getP2());
		baseLineX.setP2(axis.translate(-3, 3));
						
	  	baseLineY = new Line(
	  				finderPattern.getCenter(FinderPattern.UL),
					finderPattern.getCenter(FinderPattern.UR));
	  
		axis.setModulePitch(modulePitch[1]);
		axis.setOrigin(baseLineY.getP1());
		baseLineY.setP1(axis.translate(-3, -3));

		axis.setModulePitch(modulePitch[1]);
		axis.setOrigin(baseLineY.getP2());
		baseLineY.setP2(axis.translate(3, -3));
		
		baseLineX.translate(1,1);
		baseLineY.translate(1,1);

		for (int i = 0; i < sqrtNumAreaModules; i++) {
			gridLineX = new Line(baseLineX.getP1(), baseLineX.getP2());
			
			axis.setOrigin(gridLineX.getP1());
			axis.setModulePitch(modulePitch[0]);
			gridLineX.setP1(axis.translate(i,0));

			axis.setOrigin(gridLineX.getP2());
			axis.setModulePitch(modulePitch[0]);
			gridLineX.setP2(axis.translate(i,0));
			
			gridLineY = new Line(baseLineY.getP1(), baseLineY.getP2());
			axis.setOrigin(gridLineY.getP1());
			axis.setModulePitch(modulePitch[1]);
			gridLineY.setP1(axis.translate(0,i));

			axis.setOrigin(gridLineY.getP2());
			axis.setModulePitch(modulePitch[1]);
			gridLineY.setP2(axis.translate(0,i));

			samplingGrid[0][0][0][i] = gridLineX;
			samplingGrid[0][0][1][i] = gridLineY;
		}
		
		return samplingGrid;
	}
	
	Line[][][][] getSamplingGrid2_6(FinderPattern finderPattern, AlignmentPattern alignmentPattern) {
		Point centers[][] = alignmentPattern.getCenter();
		centers[0][0] = finderPattern.getCenter(FinderPattern.UL);
		centers[1][0] = finderPattern.getCenter(FinderPattern.UR);
		centers[0][1] = finderPattern.getCenter(FinderPattern.DL);
		int sqrtNumModules = finderPattern.getSqrtNumModules(); 

		Line[][][][] samplingGrid = new Line[1][1][2][sqrtNumModules];
		Line baseLineX, baseLineY, gridLineX, gridLineY;
		
		int logicalDistance = alignmentPattern.getLogicalDistance();
		int sin = finderPattern.getAngle()[0];
		int cos = finderPattern.getAngle()[1];
		Axis axis = new Axis(sin, cos, finderPattern.getModuleSize());
	
		int[] modulePitch = new int[4]; //top left bottom right

		modulePitch[0] = getAreaModulePitch(centers[0][0], centers[1][0], logicalDistance + 6);
		modulePitch[1] = getAreaModulePitch(centers[0][0], centers[0][1], logicalDistance + 6);
		axis.setModulePitch(modulePitch[0]);
		axis.setOrigin(centers[0][1]);
		modulePitch[2] = getAreaModulePitch(axis.translate(0, -3), centers[1][1], logicalDistance + 3);
		axis.setModulePitch(modulePitch[1]);
		axis.setOrigin(centers[1][0]);
		modulePitch[3] = getAreaModulePitch(axis.translate(-3, 0), centers[1][1], logicalDistance + 3);

		baseLineX = new Line();
		baseLineY = new Line();
		
		axis.setOrigin(centers[0][0]);
		modulePitch[0] = getAreaModulePitch(centers[0][0], centers[1][0], logicalDistance + 6);
		modulePitch[1] = getAreaModulePitch(centers[0][0], centers[0][1], logicalDistance + 6);
		axis.setModulePitch(modulePitch[0]);
		axis.setOrigin(centers[0][1]);
		modulePitch[2] = getAreaModulePitch(axis.translate(0,-3), centers[1][1], logicalDistance + 3);
		axis.setModulePitch(modulePitch[1]);
		axis.setOrigin(centers[1][0]);
		modulePitch[3] = getAreaModulePitch(axis.translate(-3,0), centers[1][1], logicalDistance + 3);
		
		axis.setOrigin(centers[0][0]);
		axis.setModulePitch(modulePitch[0]);
		baseLineX.setP1(axis.translate(-3,-3));

		axis.setModulePitch(modulePitch[1]);
		baseLineY.setP1(axis.translate(-3,-3));
		
		axis.setOrigin(centers[0][1]);
		axis.setModulePitch(modulePitch[2]);
		baseLineX.setP2(axis.translate(-3,3));
		
		axis.setOrigin(centers[1][0]);
		axis.setModulePitch(modulePitch[3]);
		baseLineY.setP2(axis.translate(3,-3));
		
		baseLineX.translate(1,1);
		baseLineY.translate(1,1);
		
		for (int i = 0; i < sqrtNumModules; i++) {
			gridLineX = new Line(baseLineX.getP1(), baseLineX.getP2());

			axis.setOrigin(gridLineX.getP1());
			axis.setModulePitch(modulePitch[0]);
			gridLineX.setP1(axis.translate(i,0));

			axis.setOrigin(gridLineX.getP2());
			axis.setModulePitch(modulePitch[2]);
			gridLineX.setP2(axis.translate(i,0));
	
			gridLineY = new Line(baseLineY.getP1(), baseLineY.getP2());
			
			axis.setOrigin(gridLineY.getP1());
			axis.setModulePitch(modulePitch[1]);
			gridLineY.setP1(axis.translate(0,i));

			axis.setOrigin(gridLineY.getP2());
			axis.setModulePitch(modulePitch[3]);
			gridLineY.setP2(axis.translate(0,i));
			
			samplingGrid[0][0][0][i] = gridLineX;
			samplingGrid[0][0][1][i] = gridLineY;			
		}
		
		return samplingGrid;
	}
	
	Line[][][][] getSamplingGrid7_13(FinderPattern finderPattern, AlignmentPattern alignmentPattern) {
		Point centers[][] = alignmentPattern.getCenter();
		centers[0][0] = finderPattern.getCenter(FinderPattern.UL);
		centers[2][0] = finderPattern.getCenter(FinderPattern.UR);
		centers[0][2] = finderPattern.getCenter(FinderPattern.DL);
		int sqrtNumModules = finderPattern.getSqrtNumModules();
		int sqrtNumArea = 2;
		int sqrtNumAreaModules = sqrtNumModules / sqrtNumArea;
		sqrtNumAreaModules++;
		Line[][][][] samplingGrid = new Line[sqrtNumArea][sqrtNumArea][2][sqrtNumAreaModules];
		Line baseLineX, baseLineY, gridLineX, gridLineY;

		int logicalDistance = alignmentPattern.getLogicalDistance();
		int sin = finderPattern.getAngle()[0];
		int cos = finderPattern.getAngle()[1];
		Axis axis = new Axis(sin, cos, finderPattern.getModuleSize());
		int[] modulePitch;
		for (int ay = 0; ay < sqrtNumArea; ay++) {
			for (int ax = 0; ax < sqrtNumArea; ax++) {
				modulePitch = new int[4]; //top left bottom right
				baseLineX = new Line();
				baseLineY = new Line();
				axis.setModulePitch(finderPattern.getModuleSize());
				if (ax == 0 && ay == 0) {
					axis.setOrigin(centers[0][0]);
					modulePitch[0] = getAreaModulePitch(axis.translate(0,3), centers[1][0], logicalDistance + 3);
					modulePitch[1] = getAreaModulePitch(axis.translate(3,0), centers[0][1], logicalDistance + 3);
					axis.setModulePitch(modulePitch[0]);
					modulePitch[2] = getAreaModulePitch(centers[0][1], centers[1][1], logicalDistance);
					axis.setModulePitch(modulePitch[1]);
					modulePitch[3] = getAreaModulePitch(centers[1][0], centers[1][1], logicalDistance);
					
					axis.setModulePitch(modulePitch[0]);
					baseLineX.setP1(axis.translate(-3,-3));

					axis.setModulePitch(modulePitch[1]);
					baseLineY.setP1(axis.translate(-3,-3));
					
					axis.setOrigin(centers[0][1]);
					axis.setModulePitch(modulePitch[2]);
					baseLineX.setP2(axis.translate(-6,0));
					
					axis.setOrigin(centers[1][0]);
					axis.setModulePitch(modulePitch[3]);
					baseLineY.setP2(axis.translate(0,-6));
				}	else if (ax == 1 && ay == 0) {
					axis.setOrigin(centers[1][0]);
					modulePitch[0] = getAreaModulePitch(axis.translate(0,-3), centers[2][0], logicalDistance + 3);
					modulePitch[1] = getAreaModulePitch(centers[1][0], centers[1][1], logicalDistance);
					axis.setModulePitch(modulePitch[0]);
					modulePitch[2] = getAreaModulePitch(centers[1][1], centers[2][1], logicalDistance);
					axis.setModulePitch(modulePitch[1]);
					axis.setOrigin(centers[2][0]);
					modulePitch[3] = getAreaModulePitch(axis.translate(-3,0), centers[2][1], logicalDistance + 3);
					
					axis.setOrigin(centers[1][0]);
					axis.setModulePitch(modulePitch[1]);
					baseLineX.setP1(axis.translate(0,-6));

					baseLineY.setP1(axis.translate(0,-6));
					
					baseLineX.setP2(centers[1][1]);
					
					axis.setOrigin(centers[2][0]);
					axis.setModulePitch(modulePitch[3]);
					baseLineY.setP2(axis.translate(3,-3));
				} else if (ax == 0 && ay == 1) {
					modulePitch[0] = getAreaModulePitch(centers[0][1], centers[1][1], logicalDistance);
					axis.setOrigin(centers[0][2]);
					modulePitch[1] = getAreaModulePitch(centers[0][1], axis.translate(3,0), logicalDistance + 3);
					axis.setModulePitch(modulePitch[0]);
					modulePitch[2] = getAreaModulePitch(axis.translate(0,-3), centers[1][2], logicalDistance + 3);
					axis.setModulePitch(modulePitch[2]);
					modulePitch[3] = getAreaModulePitch(centers[1][1], centers[1][2], logicalDistance);
					
					axis.setOrigin(centers[0][1]);
					axis.setModulePitch(modulePitch[0]);
					baseLineX.setP1(axis.translate(-6,0));

					baseLineY.setP1(axis.translate(-6,0));
					
					axis.setOrigin(centers[0][2]);
					axis.setModulePitch(modulePitch[2]);
					baseLineX.setP2(axis.translate(-3, 3));
					
					baseLineY.setP2(centers[1][1]);					
				} else if (ax == 1 && ay == 1) {
					modulePitch[0] = getAreaModulePitch(centers[1][1], centers[2][1], logicalDistance);
					modulePitch[1] = getAreaModulePitch(centers[1][1], centers[1][2], logicalDistance);
					modulePitch[2] = getAreaModulePitch(centers[1][2], centers[2][2], logicalDistance);
					modulePitch[3] = getAreaModulePitch(centers[2][1], centers[2][2], logicalDistance);
					
					baseLineX.setP1(centers[1][1]);
					baseLineY.setP1(centers[1][1]);

					axis.setOrigin(centers[1][2]);
					axis.setModulePitch(modulePitch[1]);
					baseLineX.setP2(axis.translate(0,6));

					axis.setOrigin(centers[2][1]);
					axis.setModulePitch(modulePitch[0]);
					baseLineY.setP2(axis.translate(6,0));
				}

				baseLineX.translate(1,1);
				baseLineY.translate(1,1);
				
				for (int i = 0; i < sqrtNumAreaModules; i++) {
					gridLineX = new Line(baseLineX.getP1(), baseLineX.getP2());

					axis.setOrigin(gridLineX.getP1());
					axis.setModulePitch(modulePitch[0]);
					gridLineX.setP1(axis.translate(i,0));

					axis.setOrigin(gridLineX.getP2());
					axis.setModulePitch(modulePitch[2]);
					gridLineX.setP2(axis.translate(i,0));
						
					gridLineY = new Line(baseLineY.getP1(), baseLineY.getP2());
					
					axis.setOrigin(gridLineY.getP1());
					axis.setModulePitch(modulePitch[1]);
					gridLineY.setP1(axis.translate(0,i));

					axis.setOrigin(gridLineY.getP2());
					axis.setModulePitch(modulePitch[3]);
					gridLineY.setP2(axis.translate(0,i));
					
					samplingGrid[ax][ay][0][i] = gridLineX;
					samplingGrid[ax][ay][1][i] = gridLineY;				
				}
			}
		}
	
		return samplingGrid;
	}
	
	int getAreaModulePitch(Point start, Point end, int logicalDistance) {
		Line tempLine;
		tempLine = new Line(start, end);
		int realDistance = tempLine.getLength();
		int modulePitch = (realDistance << DECIMAL_POINT) / logicalDistance;
		return modulePitch;
	}
		
	boolean[][] getQRCodeMatrix(boolean[][] image, Line[][][][] gridLines) throws ArrayIndexOutOfBoundsException {
		int gridSize = gridLines.length * gridLines[0][0][0].length;
		if (gridLines.length >= 2) gridSize--;
		boolean[][] sampledMatrix = new boolean[gridSize][gridSize];
		
		for (int ay = 0; ay < gridLines[0].length; ay++) {
			for (int ax = 0; ax < gridLines.length; ax++) {
				for (int y = 0; y < gridLines[0][0][1].length; y++) {
					for (int x = 0; x < gridLines[0][0][0].length; x++) {
						int x1 = gridLines[ax][ay][0][x].x1;
						int y1 = gridLines[ax][ay][0][x].y1;
						int x2 = gridLines[ax][ay][0][x].x2;
						int y2 = gridLines[ax][ay][0][x].y2;
						int x3 = gridLines[ax][ay][1][y].x1;
						int y3 = gridLines[ax][ay][1][y].y1;
						int x4 = gridLines[ax][ay][1][y].x2;
						int y4 = gridLines[ax][ay][1][y].y2;
						
						int e = (y2 - y1) * (x3 - x4) - (y4 - y3) * (x1 - x2);
						int f = (x1 * y2 - x2 * y1) * (x3 - x4) - (x3 * y4 - x4 * y3) * (x1 - x2);
						int g = (x3 * y4 - x4 * y3) * (y2 - y1) - (x1 * y2 - x2 * y1) * (y4 - y3);
						
						try {
							sampledMatrix[ax * gridLines[0][0][0].length + x - ax][ay * gridLines[0][0][1].length + y - ay] = image[f / e][g / e];
						}
						catch (ArrayIndexOutOfBoundsException ex) {
							sampledMatrix[ax * gridLines[0][0][0].length + x - ax][ay * gridLines[0][0][1].length + y - ay] = POINT_LIGHT;					
						}
					}
				}
			}
		}

		return sampledMatrix;
	}
}
