package jp.sourceforge.qrcode.reader;

import java.io.ByteArrayOutputStream;

public class QRCodeDataBlockReader {
	int[] blocks;
	int version;
	int mode = -1;
	int blockPointer;
	int bitPointer;
	int dataLength;
	final boolean debug = true;
	final int MODE_NUMBER           = 1;
	final int MODE_ROMAN_AND_NUMBER = 2;
	final int MODE_8BIT_BYTE        = 4;
	final int MODE_KANJI            = 8;

	public QRCodeDataBlockReader(int[] blocks, int version) {
		blockPointer = 0;
		bitPointer = 7;
		dataLength = 0;
		this.blocks = blocks;
		this.version = version;
	}
	
	int getNextBits(int numBits) throws ArrayIndexOutOfBoundsException {
		if (numBits < bitPointer + 1) { 
			int mask = 0;
			for (int i = 0; i < numBits; i++) {
				mask += 1 << i;
			}
			mask <<= (bitPointer - numBits + 1);
			
			int bits = (blocks[blockPointer] & mask) >> (bitPointer - numBits + 1);
			bitPointer -= numBits;
			return bits;
		} else if (numBits < bitPointer + 1 + 8) { 
			int mask1 = 0;
			for (int i = 0; i < bitPointer + 1; i++) {
				mask1 += 1 << i;
			}
			int bits = (blocks[blockPointer] & mask1) << (numBits - (bitPointer + 1));
			blockPointer++;

			bits += (blocks[blockPointer]) >> (8 - (numBits - (bitPointer + 1)));
	
			bitPointer = bitPointer - numBits % 8;
			if (bitPointer < 0) {
				bitPointer = 8 + bitPointer;
			}
			return bits;	
		} else if (numBits < bitPointer + 1 + 16) { 
			int mask1 = 0; 
			int mask3 = 0; 
			for (int i = 0; i < bitPointer + 1; i++) {
				mask1 += 1 << i;
			}
			int bitsFirstBlock = (blocks[blockPointer] & mask1) << (numBits - (bitPointer + 1));
			blockPointer++;

			int bitsSecondBlock = blocks[blockPointer] << (numBits - (bitPointer + 1 + 8));
			blockPointer++;
			
			for (int i = 0; i < numBits - (bitPointer + 1 + 8); i++) {
				mask3 += 1 << i;
			}
			mask3 <<= 8 - (numBits - (bitPointer + 1 + 8));
			int bitsThirdBlock = (blocks[blockPointer] & mask3) >> (8 - (numBits - (bitPointer + 1 + 8)));
			
			int bits = bitsFirstBlock + bitsSecondBlock + bitsThirdBlock;
			bitPointer = bitPointer - (numBits - 8) % 8;
			if (bitPointer < 0) {
				bitPointer = 8 + bitPointer;
			}
			return bits;
		} else {
			if (debug) System.out.println("ERROR!");
			return 0;
		}
	}	
	
	int getNextMode() throws ArrayIndexOutOfBoundsException {
		return getNextBits(4);
	}

	int getDataLength(int mode) throws ArrayIndexOutOfBoundsException {
		switch (mode) {
		  case MODE_NUMBER:
			if (version <= 9)
				return getNextBits(10);
			else if (version >= 10 && version <= 13)
				return getNextBits(12);
		
		  case MODE_ROMAN_AND_NUMBER:
			if (version <= 9)
				return getNextBits(9);
			else if (version >= 10 && version <= 13)
				return getNextBits(11);
		
		  case MODE_8BIT_BYTE:
			if (version <= 9)
				return getNextBits(8);
			else if (version >= 10 && version <= 13)
				return getNextBits(16);
		
		  case MODE_KANJI:
			if (version <= 9)
				return getNextBits(8);
			else if (version >= 10 && version <= 13)
				return getNextBits(10);
		
		  default:
				return 0;
		}
	}	
	
	public String getDataString() throws ArrayIndexOutOfBoundsException {	
		String dataString = "";
		boolean firstMode = true;

		do {
			mode = getNextMode();
			if (debug) System.out.println("mode: " + mode);
			if (mode == 0 && !firstMode) break;
		
			if (mode != MODE_NUMBER && mode != MODE_ROMAN_AND_NUMBER && mode != MODE_8BIT_BYTE && mode != MODE_KANJI) {
				mode = MODE_8BIT_BYTE; //always assume it is MODE_8BIT_BYTE
				if (debug) System.out.println("reset mode to: " + mode);
			}	
			firstMode = false;
			dataLength = getDataLength(mode);
			if (debug) System.out.println("dataLength: " + dataLength);
					
			switch (mode) {
		  	  case MODE_NUMBER: 
				if (debug) System.out.println("Mode: Figure");
				dataString += getFigureString(dataLength);
				break;
			  
			  case MODE_ROMAN_AND_NUMBER: 
				if (debug) System.out.println("Mode: Roman&Figure");
				dataString += getRomanAndFigureString(dataLength);
				break;
			  
			  case MODE_8BIT_BYTE: 
				if (debug) System.out.println("Mode: 8bit Byte");
				dataString += get8bitByteString(dataLength);
				break;
			
			  case MODE_KANJI: 
				if (debug) System.out.println("Mode: Kanji");
				dataString += getKanjiString(dataLength);
				break;
			}
			
			if (debug) System.out.println("DataLength: " + dataLength);
			if (debug) System.out.println(dataString);
		} while (true);
	
		return dataString;
	}
		
	String getFigureString(int dataLength) throws ArrayIndexOutOfBoundsException {
		int length = dataLength;
		int intData = 0;
		String strData = "";
		do {
			if (length >= 3) {
				intData = getNextBits(10);
				if (intData < 100) strData += "0";
				if (intData < 10)  strData += "0";
				length -= 3;
			} else if (length == 2) {
				intData = getNextBits(7);
				if (intData < 10) strData += "0";
				length -= 2;
			} else if (length == 1) {
				intData = getNextBits(4);
				length -= 1;
			}
			if (intData == 0) break;
			strData += Integer.toString(intData);
		} while (length > 0);
		
		return strData;
	}
	
	String getRomanAndFigureString(int dataLength) throws ArrayIndexOutOfBoundsException  {
		int length = dataLength;
		int intData = 0;
		String strData = "";
		final char[] tableRomanAndFigure = {
			 '0', '1', '2', '3', '4', '5',
	 		 '6', '7', '8', '9', 'A', 'B',
			 'C', 'D', 'E', 'F', 'G', 'H',
			 'I', 'J', 'K', 'L', 'M', 'N',
			 'O', 'P', 'Q', 'R', 'S', 'T',
			 'U', 'V', 'W', 'X', 'Y', 'Z',
			 ' ', '$', '%', '*', '+', '-',
			 '.', '/', ':'
			 };
		do {
			if (length > 1) {
				intData = getNextBits(11);
				int firstLetter  = intData / 45;
				int secondLetter = intData % 45;
				strData += String.valueOf(tableRomanAndFigure[firstLetter]);
				strData += String.valueOf(tableRomanAndFigure[secondLetter]);
				length -= 2;
			} else if (length == 1) {
				intData = getNextBits(6);
				strData += String.valueOf(tableRomanAndFigure[intData]);
				length -= 1;
			}
		} while (length > 0);
		
		return strData;
	}

	String get8bitByteString(int dataLength) throws ArrayIndexOutOfBoundsException  {
		int length = dataLength;
		int intData = 0;
		ByteArrayOutputStream output=new ByteArrayOutputStream();

		boolean flag = true;
		do {
			intData = getNextBits(8);
			if (intData == 0) {
				flag = false;
			} else {
				if (debug) System.out.print(intData + " ");
				output.write((byte)intData);
			}
			length--;
			if (length == 0) flag = false;
		} while (flag);
		if (debug) System.out.println("");
		
		String strData = "";
		try {
			strData = new String(output.toByteArray(), "UTF-8");
		} catch (Exception e) {
			if (debug) e.printStackTrace();
			strData = new String(output.toByteArray());
		}
		
		return strData;
	}
	
	String getKanjiString(int dataLength) throws ArrayIndexOutOfBoundsException {
		int length = dataLength;
		int intData = 0;
		String unicodeString = "";
		
		do {
			intData = getNextBits(13);
			int lowerByte  = intData % 0xC0;
			int higherByte = intData / 0xC0;

			int tempWord = (higherByte << 8) + lowerByte;
			int shiftjisWord = 0;
			if (tempWord + 0x8140 <= 0x9FFC) { 
				shiftjisWord = tempWord + 0x8140;
			} else { 
				shiftjisWord = tempWord + 0xC140;
			}

			byte[] tempByte = new byte[2];
			tempByte[0] = (byte)(shiftjisWord >> 8);
			tempByte[1] = (byte)(shiftjisWord & 0xFF);
			unicodeString += new String(tempByte);
			length--;
		} while (length > 0);
			
		return unicodeString;
	}
}
