/* psync.c - Last modified: 14-Dec-2018 (kobayasy)
 *
 * Copyright (c) 2018 by Yuichi Kobayashi <kobayasy@kobayasy.com>
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation files
 * (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge,
 * publish, distribute, sublicense, and/or sell copies of the Software,
 * and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#include <limits.h>
#include <signal.h>
#include <stdbool.h>
#include <stddef.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <dirent.h>
#include <fcntl.h>
#include <libgen.h>
#include <poll.h>
#include <pthread.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/time.h>
#include "psync_utils.h"
#include "psync.h"

#define EXPIRE_DEFAULT (400*24*60*60)  /* [sec] */
#define BACKUP_DEFAULT   (3*24*60*60)  /* [sec] */
#define POLL_TIMEOUT 30000  /* [msec] */
#define LOADBUFFER_SIZE (16*1024)  /* [byte] */

#ifndef SYMLINK_MAX
#define SYMLINK_MAX PATH_MAX
#endif  /* #ifndef SYMLINK_MAX */
#if LOADBUFFER_SIZE < SYMLINK_MAX
#undef LOADBUFFER_SIZE
#define LOADBUFFER_SIZE SYMLINK_MAX
#endif  /* #if LOADBUFFER_SIZE < SYMLINK_MAX */

ssize_t write_sp(int fd, const void *buf, size_t count) {
    ssize_t status = -1;
    size_t size;
    struct pollfd fds;
    ssize_t n;

    size = count;
    fds.fd = fd, fds.events = POLLOUT;
    while (size > 0) {
        if (poll(&fds, 1, POLL_TIMEOUT) != 1)
            goto error;
        if (!(fds.revents & POLLOUT))
            goto error;
        n = write(fd, buf, size);
        switch (n) {
        case -1:
        case  0:  /* end of file */
            goto error;
        }
        size -= n;
        buf += n;
    }
    status = count;
error:
    return status;
}
ssize_t read_sp(int fd, void *buf, size_t count) {
    ssize_t status = -1;
    size_t size;
    struct pollfd fds;
    ssize_t n;

    size = count;
    fds.fd = fd, fds.events = POLLIN;
    while (size > 0) {
        if (poll(&fds, 1, POLL_TIMEOUT) != 1)
            goto error;
        if (!(fds.revents & POLLIN))
            goto error;
        n = read(fd, buf, size);
        switch (n) {
        case -1:
        case  0:  /* end of file */
            goto error;
        }
        size -= n;
        buf += n;
    }
    status = count;
error:
    return status;
}

int strcmp_sp(const char *s1, const char *s2) {
    int n = strcmp(s1, s2);

    if (n < 0) {
        if (!*s1)
            n = INT_MAX;
    }
    else if (n > 0) {
        if (!*s2)
            n = INT_MIN;
    }
    return n;
}

#define FST_UPLD  0x08
#define FST_DNLD  0x80
#define FST_LTYPE 0x07
#define  FST_LREG 0x01
#define  FST_LDIR 0x02
#define  FST_LLNK 0x03
#define FST_RTYPE 0x70
#define  FST_RREG 0x10
#define  FST_RDIR 0x20
#define  FST_RLNK 0x30
typedef struct {
    time_t revision;
    time_t mtime;
    mode_t mode;
    off_t size;
    uint8_t flags;
} FST;
typedef struct s_flist {
    struct s_flist *next, *prev;
    FST st;
    char name[1];
} FLIST;
typedef enum {
    FSETS_1AND2,
    FSETS_1NOT2,
    FSETS_2NOT1
} FSETS;

static FLIST *new_f(FLIST *flist) {
    LIST_NEW(flist);
    memset(&flist->st, 0, sizeof(flist->st));
    return flist;
}

static FLIST *add_f(FLIST *flist, const char *name) {
    FLIST *fnew = NULL;

    if (!*name)
        goto error;
    fnew = malloc(offsetof(FLIST, name) + strlen(name) + 1);
    if (fnew == NULL)
        goto error;
    strcpy(fnew->name, name);
    memset(&fnew->st, 0, sizeof(fnew->st));
    LIST_INSERT_NEXT(fnew, flist);
error:
    return fnew;
}

static int each_next_f(FLIST *flist,
                       int (*func)(FLIST *f, void *data),
                                             void *data,
                       volatile sig_atomic_t *stop ) {
    int status = INT_MIN;
    FLIST *f = flist;
    FLIST *fnext;

    if (*f->name)
        goto error;
    f = flist->next;
    while (*f->name) {
        ONSTOP(stop, -1);
        fnext = f->next;
        status = func(f, data);
        if (ISERR(status))
            goto error;
        f = fnext;
    }
    status = 0;
error:
    return status;
}

#if 0  // not used
static int each_prev_f(FLIST *flist,
                       int (*func)(FLIST *f, void *data),
                                             void *data,
                       volatile sig_atomic_t *stop ) {
    int status = INT_MIN;
    FLIST *f = flist;
    FLIST *fprev;

    if (*f->name)
        goto error;
    f = flist->prev;
    while (*f->name) {
        ONSTOP(stop, -1);
        fprev = f->prev;
        status = func(f, data);
        if (ISERR(status))
            goto error;
        f = fprev;
    }
    status = 0;
error:
    return status;
}
#endif  /* #if 0  // not used */

static int each_sets_f(FLIST *flist1, FLIST *flist2,
                       int (*func)(FSETS sets, FLIST *f1, FLIST *f2, void *data),
                                                                     void *data,
                       volatile sig_atomic_t *stop ) {
    int status = INT_MIN;
    FLIST *f1 = flist1;
    FLIST *f2 = flist2;
    FLIST *f1next, *f2next;
    int n;

    if (*f1->name || *f2->name)
        goto error;
    f1 = f1->next, f2 = f2->next;
    while (*f1->name || *f2->name) {
        ONSTOP(stop, -1);
        n = strcmp_sp(f1->name, f2->name);
        if (n < 0) {
            f1next = f1->next;
            status = func(FSETS_1NOT2, f1, f2, data);
            if (ISERR(status))
                goto error;
            f1 = f1next;
        }
        else if (n > 0) {
            f2next = f2->next;
            status = func(FSETS_2NOT1, f1, f2, data);
            if (ISERR(status))
                goto error;
            f2 = f2next;
        }
        else {
            f1next = f1->next, f2next = f2->next;
            status = func(FSETS_1AND2, f1, f2, data);
            if (ISERR(status))
                goto error;
            f1 = f1next, f2 = f2next;
        }
    }
    status = 0;
error:
    return status;
}

static int write_f(bool synced, FLIST *flist, int fd,
                   volatile sig_atomic_t *stop ) {
    int status = INT_MIN;
    size_t length, n;
    FST st;

    if (*flist->name) {
        status = -1;
        goto error;
    }
    for (flist = flist->next; *flist->name; flist = flist->next) {
        ONSTOP(stop, -1);
        n = length = strlen(flist->name);
        WRITE_ONERR(n, fd, -1);
        if (write_sp(fd, flist->name, length) != length) {
            status = -1;
            goto error;
        }
        st = flist->st;
        WRITE_ONERR(st.revision, fd, -1);
        WRITE_ONERR(st.mtime, fd, -1);
        WRITE_ONERR(st.mode, fd, -1);
        if (synced) {
            WRITE_ONERR(st.size, fd, -1);
            st.flags = st.flags >> 4 | st.flags << 4;
            WRITE_ONERR(st.flags, fd, -1);
        }
    }
    length = 0;
    WRITE_ONERR(length, fd, -1);
    status = 0;
error:
    return status;
}

static int read_f(bool synced, FLIST *flist, int fd,
                  volatile sig_atomic_t *stop ) {
    int status = INT_MIN;
    size_t length;
    char dirname[PATH_MAX];

    if (*flist->name) {
        status = -1;
        goto error;
    }
    READ_ONERR(length, fd, -1);
    while (length > 0) {
        ONSTOP(stop, -1);
        if (length > sizeof(dirname)-1) {
            status = -1;
            goto error;
        }
        if (read_sp(fd, dirname, length) != length) {
            status = -1;
            goto error;
        }
        dirname[length] = 0;
        flist = add_f(flist, dirname);
        if (flist == NULL) {
            status = -1;
            goto error;
        }
        READ_ONERR(flist->st.revision, fd, -1);
        READ_ONERR(flist->st.mtime, fd, -1);
        READ_ONERR(flist->st.mode, fd, -1);
        if (synced) {
            READ_ONERR(flist->st.size, fd, -1);
            READ_ONERR(flist->st.flags, fd, -1);
        }
        READ_ONERR(length, fd, -1);
    }
    status = 0;
error:
    return status;
}

static int delete_func(FLIST *f, void *data) {
    int status = INT_MIN;

    LIST_DELETE(f);
    free(f);
    status = 0;
    return status;
}

#define FILEID 0x01665370  /* 'p', 'S', 'f', 1 */
#define SYNCDIR  ".psync"
#define LASTFILE "last"
#define LOCKDIR  "lock"
#define BACKDIR  "%Y%m%d%H%M.%S%z"
#define UPFILE   "u%lu"
#define DOWNFILE "d%lu"
#define BACKFILE "%lu,%s"
typedef struct {
    time_t t;
    time_t expire;
    time_t backup;
    int fdin, fdout;
    int info;
    volatile sig_atomic_t *stop;
    time_t tlast;
    FLIST fsynced;
    FLIST flocal, fremote;
    char dirname[1];
} PRIV;

#define INFO(_priv, ...) \
    do { \
        if ((_priv)->info != -1) \
            dprintf((_priv)->info, __VA_ARGS__); \
    } while (0)

static int lock(PRIV *priv) {
    int status = INT_MIN;
    char loadname[PATH_MAX], *pload;

    pload = loadname, pload += sprintf(pload, "%s/"SYNCDIR, priv->dirname);
    mkdir(loadname, S_IRWXU);
    strcpy(pload, "/"LOCKDIR);
    status = mkdir(loadname, S_IRWXU) == -1 ? 1 : 0;
    return status;
}

static int unlock(PRIV *priv) {
    int status = INT_MIN;
    char loadname[PATH_MAX], *pload;
    char pathname[PATH_MAX], *p;
    struct timeval tv[2];

    pload = loadname, pload += sprintf(pload, "%s/"SYNCDIR"/"LOCKDIR, priv->dirname);
    tv[0].tv_sec = priv->t, tv[0].tv_usec = 0;
    tv[1].tv_sec = priv->t, tv[1].tv_usec = 0;
    utimes(loadname, tv);
    p = pathname, p += sprintf(p, "%s/"SYNCDIR"/", priv->dirname);
    strftime(p, sizeof(pathname) - (p - pathname), BACKDIR, localtime(&priv->t));
    status = rename(loadname, pathname) == -1 ? 1 : 0;
    return status;
}

static PRIV *new_priv(const char *dirname,
                      volatile sig_atomic_t *stop ) {
    PRIV *priv = NULL;
    time_t t;

    if (time(&t) == -1)
        goto error;
    priv = malloc(offsetof(PRIV, dirname) + strlen(dirname) + 1);
    if (priv == NULL)
        goto error;
    strcpy(priv->dirname, dirname);
    priv->t = t;
    priv->expire = t - EXPIRE_DEFAULT;
    priv->backup = t - BACKUP_DEFAULT;
    priv->fdin = -1, priv->fdout = -1;
    priv->info = -1;
    priv->stop = stop;
    priv->tlast = -1;
    new_f(&priv->fsynced);
    new_f(&priv->flocal);
    new_f(&priv->fremote);
    if (lock(priv)) {
        free(priv), priv = NULL;
        goto error;
    }
error:
    return priv;
}

static void free_priv(PRIV *priv) {
    unlock(priv);
    each_next_f(&priv->fsynced, delete_func, NULL, NULL);
    each_next_f(&priv->flocal, delete_func, NULL, NULL);
    each_next_f(&priv->fremote, delete_func, NULL, NULL);
    free(priv);
}

static int save_fsynced(PRIV *priv) {
    int status = INT_MIN;
    char pathname[PATH_MAX];
    int fd = -1;
    uint32_t id;
    struct timeval tv[2];

    sprintf(pathname, "%s/"SYNCDIR"/"LOCKDIR"/"LASTFILE, priv->dirname);
    fd = creat(pathname, S_IRUSR|S_IWUSR);
    if (fd == -1) {
        status = ERROR_DMAKE;
        goto error;
    }
    id = FILEID;
    WRITE_ONERR(id, fd, ERROR_DWRITE);
    status = write_f(false, &priv->fsynced, fd, priv->stop);
    ONSTOP(priv->stop, ERROR_STOP);
    if (ISERR(status)) {
        status = ERROR_DWRITE;
        goto error;
    }
    close(fd), fd = -1;
    tv[0].tv_sec = priv->t, tv[0].tv_usec = 0;
    tv[1].tv_sec = priv->t, tv[1].tv_usec = 0;
    if (utimes(pathname, tv) == -1) {
        status = ERROR_DWRITE;
        goto error;
    }
    status = 0;
error:
    if (fd != -1)
        close(fd);
    return status;
}

static int load_fsynced(PRIV *priv) {
    int status = INT_MIN;
    struct stat st;
    char pathname[PATH_MAX];
    int fd = -1;
    uint32_t id;
    int n;

    sprintf(pathname, "%s/"SYNCDIR"/"LASTFILE, priv->dirname);
    if (stat(pathname, &st) == -1) {
        status = ERROR_DREAD;
        goto error;
    }
    fd = open(pathname, O_RDONLY);
    if (fd == -1) {
        status = ERROR_DOPEN;
        goto error;
    }
    READ(id, fd, n);
    if (!ISERR(n) && id == FILEID) {
        status = read_f(false, &priv->fsynced, fd, priv->stop);
        ONSTOP(priv->stop, ERROR_STOP);
        if (ISERR(status)) {
            status = ERROR_DREAD;
            goto error;
        }
    }
    close(fd), fd = -1;
    priv->tlast = st.st_mtime;
    status = 0;
error:
    if (fd != -1)
        close(fd);
    return status;
}

static int get_flocal_r(FLIST **flocal, FLIST **flast, char *pathname, char *name,
                        volatile sig_atomic_t *stop ) {
    int status = INT_MIN;
    struct stat st;
    uint8_t ltype;
    int seek;
    FLIST *fprev;
    DIR *dir = NULL;
    char *p;
    FLIST *fdir;
    struct dirent *ent;

    if (lstat(pathname, &st) == -1) {
        status = ERROR_SREAD;
        goto error;
    }
    switch (st.st_mode & S_IFMT) {
    case S_IFREG:
        if (access(pathname, R_OK) != 0) {
            status = ERROR_FPERM;
            goto error;
        }
        ltype = FST_LREG;
        break;
    case S_IFDIR:
        if (access(pathname, R_OK|W_OK|X_OK) != 0) {
            status = ERROR_FPERM;
            goto error;
        }
        ltype = FST_LDIR;
        break;
    case S_IFLNK:
        ltype = FST_LLNK;
        break;
    default:
        status = ERROR_FTYPE;
        goto error;
    }
    LIST_SEEK_NEXT(*flocal, name, seek);
    LIST_SEEK_NEXT(*flast, name, seek);
    if (seek) {
        *flocal = add_f(*flocal, name);
        if (*flocal == NULL) {
            status = ERROR_MEMORY;
            goto error;
        }
        (*flocal)->st.revision = st.st_ctime > st.st_mtime ? st.st_ctime : st.st_mtime;
    }
    else {
        fprev = (*flast)->prev;
        LIST_DELETE(*flast);
        LIST_INSERT_NEXT(*flast, *flocal);
        *flocal = *flast, *flast = fprev;
        if ((*flocal)->st.mtime != st.st_mtime)
            (*flocal)->st.revision = st.st_ctime > st.st_mtime ? st.st_ctime : st.st_mtime;
    }
    (*flocal)->st.mtime = st.st_mtime;
    (*flocal)->st.mode = st.st_mode & (S_IFMT|S_IRWXU|S_IRWXG|S_IRWXO);
    (*flocal)->st.size = st.st_size;
    (*flocal)->st.flags |= ltype;
    switch (ltype & FST_LTYPE) {
    case FST_LDIR:
        dir = opendir(pathname);
        if (dir == NULL) {
            status = ERROR_FOPEN;
            goto error;
        }
        p = name, p += strlen(p), *p++ = '/';
        fdir = *flocal;
        while ((ent = readdir(dir)) != NULL) {
            ONSTOP(stop, ERROR_STOP);
            if (!strcmp(ent->d_name, ".") ||
                !strcmp(ent->d_name, "..") )
                continue;
            strcpy(p, ent->d_name);
            status = get_flocal_r(flocal, flast, pathname, name, stop);
            if (ISERR(status))
                goto error;
            if ((*flocal)->st.revision > fdir->st.revision)
                fdir->st.revision = (*flocal)->st.revision;
        }
        closedir(dir), dir = NULL;
        break;
    }
    status = 0;
error:
    if (dir != NULL)
        closedir(dir);
    return status;
}
static int get_flocal(PRIV *priv) {
    int status = INT_MIN;
    struct stat st;
    DIR *dir = NULL;
    char pathname[PATH_MAX], *p;
    FLIST *flocal, *flast;
    struct dirent *ent;

    if (stat(priv->dirname, &st) == -1) {
        status = ERROR_SREAD;
        goto error;
    }
    switch (st.st_mode & S_IFMT) {
    case S_IFDIR:
        if (access(priv->dirname, R_OK|W_OK|X_OK) != 0) {
            status = ERROR_FPERM;
            goto error;
        }
        break;
    default:
        status = ERROR_FTYPE;
        goto error;
    }
    dir = opendir(priv->dirname);
    if (dir == NULL) {
        status = ERROR_FOPEN;
        goto error;
    }
    p = pathname, p += sprintf(p, "%s/", priv->dirname);
    flocal = &priv->flocal, flast = &priv->fsynced;
    while ((ent = readdir(dir)) != NULL) {
        ONSTOP(priv->stop, ERROR_STOP);
        if (!strcmp(ent->d_name, ".") ||
            !strcmp(ent->d_name, "..") ||
            !strcmp(ent->d_name, SYNCDIR) )
            continue;
        strcpy(p, ent->d_name);
        status = get_flocal_r(&flocal, &flast, pathname, p, priv->stop);
        if (ISERR(status))
            goto error;
    }
    closedir(dir), dir = NULL;
    status = 0;
error:
    if (dir != NULL)
        closedir(dir);
    return status;
}

static int add_deleted_func(FSETS sets, FLIST *flocal, FLIST *flast, void *data) {
    int status = INT_MIN;
    PRIV *priv = data;

    switch (sets) {
    case FSETS_1AND2:
        LIST_DELETE(flast);
        free(flast);
        break;
    case FSETS_1NOT2:
        break;
    case FSETS_2NOT1:
        LIST_DELETE(flast);
        switch (flast->st.mode & S_IFMT) {
        case 0:  /* deleted */
            if (flast->st.revision > priv->expire)
                LIST_INSERT_PREV(flast, flocal);
            else
                free(flast);
            break;
        default:
            flast->st.revision = priv->tlast + 1;
            flast->st.mtime = 0;
            flast->st.mode = 0;
            flast->st.size = 0;
            LIST_INSERT_PREV(flast, flocal);
        }
        break;
    }
    status = 0;
    return status;
}

static int get_fsynced_func(FSETS sets, FLIST *flocal, FLIST *fremote, void *data) {
    int status = INT_MIN;
    PRIV *priv = data;

    switch (sets) {
    case FSETS_1AND2:
        if (flocal->st.revision < fremote->st.revision) {
            fremote->st.flags |= flocal->st.flags;
            if (flocal->st.mtime != fremote->st.mtime)
                fremote->st.flags |= FST_DNLD;
            LIST_DELETE(flocal);
            free(flocal);
            LIST_DELETE(fremote);
            LIST_INSERT_PREV(fremote, &priv->fsynced);
        }
        else {
            flocal->st.flags |= fremote->st.flags;
            if (flocal->st.revision > fremote->st.revision &&
                flocal->st.mtime != fremote->st.mtime )
                flocal->st.flags |= FST_UPLD;
            LIST_DELETE(fremote);
            free(fremote);
            LIST_DELETE(flocal);
            LIST_INSERT_PREV(flocal, &priv->fsynced);
        }
        break;
    case FSETS_1NOT2:
        flocal->st.flags |= FST_UPLD;
        LIST_DELETE(flocal);
        LIST_INSERT_PREV(flocal, &priv->fsynced);
        break;
    case FSETS_2NOT1:
        fremote->st.flags |= FST_DNLD;
        LIST_DELETE(fremote);
        LIST_INSERT_PREV(fremote, &priv->fsynced);
        break;
    }
    status = 0;
    return status;
}

static int preload(PRIV *priv) {
    int status = INT_MIN;
    intmax_t uploadsize, downloadsize;
    char pathname[PATH_MAX], *p;
    char loadname[PATH_MAX], *pload;
    unsigned long count;
    FLIST *fsynced;
    char buffer[SYMLINK_MAX];

    uploadsize = 0, downloadsize = 0;
    p = pathname, p += sprintf(p, "%s/", priv->dirname);
    pload = loadname, pload += sprintf(pload, "%s/"SYNCDIR"/"LOCKDIR"/", priv->dirname);
    count = 0;
    for (fsynced = priv->fsynced.next; *fsynced->name; fsynced = fsynced->next) {
        ONSTOP(priv->stop, ERROR_STOP);
        switch (fsynced->st.flags & (FST_UPLD|FST_LTYPE)) {
        case FST_UPLD|FST_LREG:
        case FST_UPLD|FST_LLNK:
            strcpy(p, fsynced->name);
            sprintf(pload, UPFILE, ++count);
            switch (fsynced->st.flags & FST_LTYPE) {
            case FST_LREG:
                if (link(pathname, loadname) == -1) {
                    status = ERROR_FLINK;
                    goto error;
                }
                break;
            case FST_LLNK:
                if (readlink(pathname, buffer, sizeof(buffer)) != fsynced->st.size) {
                    status = ERROR_FREAD;
                    goto error;
                }
                if (symlink(buffer, loadname) == -1) {
                    status = ERROR_FWRITE;
                    goto error;
                }
                break;
            }
            uploadsize += fsynced->st.size;
            break;
        }
        switch (fsynced->st.flags & (FST_DNLD|FST_RTYPE)) {
        case FST_DNLD|FST_RREG:
        case FST_DNLD|FST_RLNK:
            downloadsize += fsynced->st.size;
            break;
        }
    }
    INFO(priv, "P%+jd%+jd\n", uploadsize, downloadsize);
    status = 0;
error:
    return status;
}

static int upload(PRIV *priv) {
    int status = INT_MIN;
    intmax_t progress;
    char loadname[PATH_MAX], *pload;
    unsigned long count;
    FLIST *fsynced;
    off_t size;
    int fd = -1;
    ssize_t n;
    char buffer[LOADBUFFER_SIZE];

    progress = 0;
    pload = loadname, pload += sprintf(pload, "%s/"SYNCDIR"/"LOCKDIR"/", priv->dirname);
    count = 0;
    for (fsynced = priv->fsynced.next; *fsynced->name; fsynced = fsynced->next) {
        ONSTOP(priv->stop, ERROR_STOP);
        switch (fsynced->st.flags & (FST_UPLD|FST_LTYPE)) {
        case FST_UPLD|FST_LREG:
        case FST_UPLD|FST_LLNK:
            sprintf(pload, UPFILE, ++count);
            size = fsynced->st.size;
            switch (fsynced->st.flags & FST_LTYPE) {
            case FST_LREG:
                fd = open(loadname, O_RDONLY);
                if (fd == -1) {
                    status = ERROR_FOPEN;
                    goto error;
                }
                while (size > 0) {
                    ONSTOP(priv->stop, ERROR_STOP);
                    n = read_sp(fd, buffer, size > sizeof(buffer) ? sizeof(buffer) : size);
                    if (n == -1) {
                        status = ERROR_FREAD;
                        goto error;
                    }
                    if (write_sp(priv->fdout, buffer, n) != n) {
                        status = ERROR_FUPLD;
                        goto error;
                    }
                    size -= n;
                    INFO(priv, "U%+jd\n", progress += n);
                }
                close(fd), fd = -1;
                break;
            case FST_LLNK:
                if (readlink(loadname, buffer, size) != size) {
                    status = ERROR_FREAD;
                    goto error;
                }
                if (write_sp(priv->fdout, buffer, size) != size) {
                    status = ERROR_FUPLD;
                    goto error;
                }
                INFO(priv, "U%+jd\n", progress += size);
                break;
            }
            if (unlink(loadname) == -1) {
                status = ERROR_FREMOVE;
                goto error;
            }
            break;
        }
    }
    status = 0;
error:
    if (fd != -1)
        close(fd);
    return status;
}

static int download(PRIV *priv) {
    int status = INT_MIN;
    intmax_t progress;
    char loadname[PATH_MAX], *pload;
    unsigned long count;
    FLIST *fsynced;
    off_t size;
    int fd = -1;
    ssize_t n;
    char buffer[LOADBUFFER_SIZE];
    struct timeval tv[2];

    progress = 0;
    pload = loadname, pload += sprintf(pload, "%s/"SYNCDIR"/"LOCKDIR"/", priv->dirname);
    count = 0;
    for (fsynced = priv->fsynced.next; *fsynced->name; fsynced = fsynced->next) {
        ONSTOP(priv->stop, ERROR_STOP);
        switch (fsynced->st.flags & (FST_DNLD|FST_RTYPE)) {
        case FST_DNLD|FST_RREG:
        case FST_DNLD|FST_RLNK:
            sprintf(pload, DOWNFILE, ++count);
            size = fsynced->st.size;
            switch (fsynced->st.flags & FST_RTYPE) {
            case FST_RREG:
                fd = creat(loadname, S_IRUSR|S_IWUSR);
                if (fd == -1) {
                    status = ERROR_FMAKE;
                    goto error;
                }
                while (size > 0) {
                    ONSTOP(priv->stop, ERROR_STOP);
                    n = read_sp(priv->fdin, buffer, size > sizeof(buffer) ? sizeof(buffer) : size);
                    if (n == -1) {
                        status = ERROR_FDNLD;
                        goto error;
                    }
                    if (write_sp(fd, buffer, n) != n) {
                        status = ERROR_FWRITE;
                        goto error;
                    }
                    size -= n;
                    INFO(priv, "D%+jd\n", progress += n);
                }
                close(fd), fd = -1;
                if (chmod(loadname, fsynced->st.mode & (S_IRWXU|S_IRWXG|S_IRWXO)) == -1) {
                    status = ERROR_SWRITE;
                    goto error;
                }
                break;
            case FST_RLNK:
                if (size > sizeof(buffer)-1) {
                    status = ERROR_SYSTEM;
                    goto error;
                }
                if (read_sp(priv->fdin, buffer, size) != size) {
                    status = ERROR_FDNLD;
                    goto error;
                }
                buffer[size] = 0;
                if (symlink(buffer, loadname) == -1) {
                    status = ERROR_FWRITE;
                    goto error;
                }
                INFO(priv, "D%+jd\n", progress += size);
                break;
            }
            tv[0].tv_sec = fsynced->st.mtime, tv[0].tv_usec = 0;
            tv[1].tv_sec = fsynced->st.mtime, tv[1].tv_usec = 0;
            if (lutimes(loadname, tv) == -1) {
                status = ERROR_SWRITE;
                goto error;
            }
            break;
        }
    }
    status = 0;
error:
    if (fd != -1)
        close(fd);
    return status;
}

static int commit(PRIV *priv) {
    int status = INT_MIN;
    char pathname[PATH_MAX], *p;
    char loadname[PATH_MAX], *pload;
    unsigned long count;
    FLIST *fsynced;
    struct timeval tv[2];

    p = pathname, p += sprintf(p, "%s/", priv->dirname);
    pload = loadname, pload += sprintf(pload, "%s/"SYNCDIR"/"LOCKDIR"/", priv->dirname);
    count = 0;
    for (fsynced = priv->fsynced.prev; *fsynced->name; fsynced = fsynced->prev)
        switch (fsynced->st.flags & (FST_DNLD|FST_LTYPE)) {
        case FST_DNLD|FST_LREG:
            strcpy(p, fsynced->name);
            sprintf(pload, BACKFILE, ++count, basename(p));
            if (rename(pathname, loadname) == -1) {
                status = ERROR_FMOVE;
                goto error;
            }
            break;
        case FST_DNLD|FST_LLNK:
            strcpy(p, fsynced->name);
            if (unlink(pathname) == -1) {
                status = ERROR_FREMOVE;
                goto error;
            }
            break;
        case FST_DNLD|FST_LDIR:
            switch (fsynced->st.flags & FST_RTYPE) {
            case FST_RREG:
            case FST_RLNK:
            case 0:  /* deleted */
                strcpy(p, fsynced->name);
                if (rmdir(pathname) == -1) {
                    status = ERROR_FREMOVE;
                    goto error;
                }
                break;
            }
            break;
        }
    count = 0;
    for (fsynced = priv->fsynced.next; *fsynced->name; fsynced = fsynced->next)
        switch (fsynced->st.flags & (FST_DNLD|FST_RTYPE)) {
        case FST_DNLD|FST_RREG:
        case FST_DNLD|FST_RLNK:
            strcpy(p, fsynced->name);
            sprintf(pload, DOWNFILE, ++count);
            if (rename(loadname, pathname) == -1) {
                status = ERROR_FMOVE;
                goto error;
            }
            break;
        case FST_DNLD|FST_RDIR:
            switch (fsynced->st.flags & FST_LTYPE) {
            case FST_LREG:
            case FST_LLNK:
            case 0:  /* deleted */
                strcpy(p, fsynced->name);
                if (mkdir(pathname, fsynced->st.mode & (S_IRWXU|S_IRWXG|S_IRWXO)) == -1) {
                    status = ERROR_FMAKE;
                    goto error;
                }
                break;
            case FST_LDIR:
                strcpy(p, fsynced->name);
                if (chmod(pathname, fsynced->st.mode & (S_IRWXU|S_IRWXG|S_IRWXO)) == -1) {
                    status = ERROR_SWRITE;
                    goto error;
                }
                break;
            }
            break;
        }
    for (fsynced = priv->fsynced.prev; *fsynced->name; fsynced = fsynced->prev)
        switch (fsynced->st.flags & (FST_DNLD|FST_RTYPE)) {
        case FST_DNLD|FST_RDIR:
            strcpy(p, fsynced->name);
            tv[0].tv_sec = fsynced->st.mtime, tv[0].tv_usec = 0;
            tv[1].tv_sec = fsynced->st.mtime, tv[1].tv_usec = 0;
            if (lutimes(pathname, tv) == -1) {
                status = ERROR_SWRITE;
                goto error;
            }
            break;
        }
    strcpy(p, SYNCDIR"/"LASTFILE);
    strcpy(pload, LASTFILE);
    if (rename(loadname, pathname) == -1) {
        status = ERROR_DWRITE;
        goto error;
    }
    status = 0;
error:
    return status;
}

typedef struct {
    PRIV *priv;
    int status;
    pthread_t tid;
} LOAD_PARAM;
static void *upload_thread(void *data) {
    LOAD_PARAM *param = data;

    param->status = upload(param->priv);
    return NULL;
}
static int run(bool master, PRIV *priv) {
    int status = INT_MIN;
    LOAD_PARAM upload_param = {priv, INT_MIN};

    ONSTOP(priv->stop, ERROR_STOP);
    load_fsynced(priv);
    ONSTOP(priv->stop, ERROR_STOP);
    status = get_flocal(priv);
    if (ISERR(status))
        goto error;
    ONSTOP(priv->stop, ERROR_STOP);
    status = each_sets_f(&priv->flocal, &priv->fsynced, add_deleted_func, priv, priv->stop);
    ONSTOP(priv->stop, ERROR_STOP);
    if (ISERR(status))
        goto error;
    if (master) {
        status = read_f(true, &priv->fremote, priv->fdin, priv->stop);
        ONSTOP(priv->stop, ERROR_STOP);
        if (ISERR(status)) {
            status = ERROR_SDNLD;
            goto error;
        }
        status = each_sets_f(&priv->flocal, &priv->fremote, get_fsynced_func, priv, priv->stop);
        ONSTOP(priv->stop, ERROR_STOP);
        if (ISERR(status))
            goto error;
        status = write_f(true, &priv->fsynced, priv->fdout, priv->stop);
        ONSTOP(priv->stop, ERROR_STOP);
        if (ISERR(status)) {
            status = ERROR_SUPLD;
            goto error;
        }
    }
    else {
        status = write_f(true, &priv->flocal, priv->fdout, priv->stop);
        ONSTOP(priv->stop, ERROR_STOP);
        if (ISERR(status)) {
            status = ERROR_SUPLD;
            goto error;
        }
        status = each_next_f(&priv->flocal, delete_func, NULL, priv->stop);
        ONSTOP(priv->stop, ERROR_STOP);
        if (ISERR(status))
            goto error;
        status = read_f(true, &priv->fsynced, priv->fdin, priv->stop);
        ONSTOP(priv->stop, ERROR_STOP);
        if (ISERR(status)) {
            status = ERROR_SDNLD;
            goto error;
        }
    }
    status = save_fsynced(priv);
    if (ISERR(status))
        goto error;
    ONSTOP(priv->stop, ERROR_STOP);
    status = preload(priv);
    if (ISERR(status))
        goto error;
    ONSTOP(priv->stop, ERROR_STOP);
    if (pthread_create(&upload_param.tid, NULL, upload_thread, &upload_param) != 0) {
        status = ERROR_SYSTEM;
        goto error;
    }
    status = download(priv);
    if (ISERR(status))
        goto error;
    if (pthread_join(upload_param.tid, NULL) != 0) {
        status = ERROR_SYSTEM;
        goto error;
    }
    if (ISERR(upload_param.status)) {
        status = upload_param.status;
        goto error;
    }
    ONSTOP(priv->stop, ERROR_STOP);
    status = commit(priv);
    if (ISERR(status))
        goto error;
    ONSTOP(priv->stop, ERROR_STOP);
    status = 0;
error:
    return status;
}

static int clean_r(char *pathname, char *name,
                   volatile sig_atomic_t *stop ) {
    int status = INT_MIN;
    struct stat st;
    DIR *dir = NULL;
    struct dirent *ent;

    if (lstat(pathname, &st) == -1) {
        status = ERROR_DREAD;
        goto error;
    }
    switch (st.st_mode & S_IFMT) {
    case S_IFDIR:
        dir = opendir(pathname);
        if (dir == NULL) {
            status = ERROR_DOPEN;
            goto error;
        }
        *name++ = '/';
        while ((ent = readdir(dir)) != NULL) {
            ONSTOP(stop, ERROR_STOP);
            if (!strcmp(ent->d_name, ".") ||
                !strcmp(ent->d_name, "..") )
                continue;
            strcpy(name, ent->d_name);
            status = clean_r(pathname, name + strlen(name), stop);
            if (ISERR(status))
                goto error;
        }
        *--name = 0;
        closedir(dir), dir = NULL;
        if (rmdir(pathname) == -1) {
            status = ERROR_DREMOVE;
            goto error;
        }
        break;
    default:
        if (unlink(pathname) == -1) {
            status = ERROR_DREMOVE;
            goto error;
        }
    }
    status = 0;
error:
    if (dir != NULL)
        closedir(dir);
    return status;
}
static int clean(PRIV *priv) {
    int status = INT_MIN;
    char pathname[PATH_MAX], *p;
    DIR *dir = NULL;
    struct dirent *ent;
    struct stat st;

    p = pathname, p += sprintf(p, "%s/"SYNCDIR, priv->dirname);
    dir = opendir(pathname);
    if (dir == NULL) {
        status = ERROR_DOPEN;
        goto error;
    }
    *p++ = '/';
    while ((ent = readdir(dir)) != NULL) {
        ONSTOP(priv->stop, ERROR_STOP);
        if (!strcmp(ent->d_name, ".") ||
            !strcmp(ent->d_name, "..") ||
            !strcmp(ent->d_name, LASTFILE) ||
            !strcmp(ent->d_name, LOCKDIR) )
            continue;
        strcpy(p, ent->d_name);
        if (stat(pathname, &st) == -1) {
            status = ERROR_DREAD;
            goto error;
        }
        if (st.st_mtime > priv->backup)
            continue;
        if (ISERR(clean_r(pathname, p + strlen(p), priv->stop))) {
            status = ERROR_DREMOVE;
            goto error;
        }
    }
    *--p = 0;
    closedir(dir), dir = NULL;
    status = 0;
error:
    if (dir != NULL)
        closedir(dir);
    return status;
}

PSYNC *psync_new(const char *dirname,
                 volatile sig_atomic_t *stop ) {
    return (PSYNC *)new_priv(dirname, stop);
}

void psync_free(PSYNC *psync) {
    free_priv((PRIV *)psync);
}

int psync_run(bool master, PSYNC *psync) {
    return run(master, (PRIV *)psync);
}

int psync_clean(PSYNC *psync) {
    return clean((PRIV *)psync);
}
