# server.rb
# $Id: server.rb,v 1.1.2.1 2004/02/08 11:58:07 komatsu Exp $
#
# Copyright (C) 2001 Satoru Takabayashi <satoru@namazu.org>
# Copyright (C) 2004 Hiroyuki Komatsu <komatsu@taiyaki.org>
#     All rights reserved.
#     This is free software with ABSOLUTELY NO WARRANTY.
#
# You can redistribute it and/or modify it under the terms of 
# the GNU General Public License version 2.
#

require 'socket'
require 'thread'

class Server
  def initialize (session)
    @session = session
  end

  public
  def session_loop (io_in, io_out)
    loop {
      line = @session.get_line(io_in)
      logging(line)
      Mutex.new.synchronize {
        result = @session.execute(line)
        if result then
          io_out.print result
        else
          return
        end
      }
      return if io_in.closed? or io_out.closed?
    }
  end

  def set_debug ()
    @debug   = true
    Dir::ensure(PRIME_USER_DIR + "/logs")
    logfile = PRIME_USER_DIR + "/logs/debug_" +
              Time::new.strftime("%Y%m%d%H%M") + "_" + $$.to_s + ".log"
    @debug_io = File::open(logfile, "w")
    @debug_io.chmod(0600) if @debug_io::stat.owned?
  end

  def logging (line)
    if @debug then
      line.chomp()
      @debug_io.puts(line)
      @debug_io.flush()
    end
  end
end

class UnixSocketServer < Server
  def initialize (session, path)
    super(session)
    @server = UNIXServer.new(path)
  end

  def start
    loop {
      socket = @server.accept
      t = Thread.new {
        session_loop(socket, socket)
        socket.close()
      }
      t.abort_on_exception = true
    }
  end
end

class TaiyakiTCPServer < Server
  def initialize (session, port)
    super(session)
    @server = TCPServer.new(port)
  end

  def start
    loop {
      socket = @server.accept
      t = Thread.new {
        session_loop(socket, socket)
        socket.close()
      }
      t.abort_on_exception = true
    }
  end
end

class StdioServer < Server
  def initialize (session)
    super(session)
    $stdout.sync = true
  end

  def start
    session_loop($stdin, $stdout)
  end
end
