# prime/prime.rb
# $Id: prime.rb,v 1.2.2.26 2004/02/29 11:11:55 komatsu Exp $
#
# Copyright (C) 2002, 2003, 2004 Hiroyuki Komatsu <komatsu@taiyaki.org>
#     All rights reserved.
#     This is free software with ABSOLUTELY NO WARRANTY.
#
# You can redistribute it and/or modify it under the terms of 
# the GNU General Public License version 2.
#

require 'prime/prime-config'
require 'prime/taiyaki'
require 'prime/prime-japanese'
require 'prime/prime-mixed'

PRIME_CONFIGFILE_GLOBAL = File::join2(PRIME_CONFIG_DIR,  "Custom_prime.rb")
PRIME_CONFIGFILE_LOCAL  = (ENV['PRIME_CONFIG'] or 
                             File::join2(PRIME_USER_DIR, "Custom_prime.rb"))

class Prime
  include Debug
  include PrimeJapanese
  include PrimeMixed

  def initialize (engine_files = nil)
    @debug_mode = false
    Dir::ensure(PRIME_USER_DIR)
    initialize_rc()

    @engines = init_engines(engine_files)
    @context = nil

    initialize_prime_japanese()

    @flag_opened = true
  end

  def initialize_rc
    if File::exist?(PRIME_CONFIGFILE_GLOBAL) == false then
      $stderr.puts "PRIME-ERROR:"
      $stderr.puts "  The file Custom_prime.rb is not found."
      $stderr.puts "  You might need to install Custom_prime.rb by"
      $stderr.puts "  'make install-etc'.  Sorry for your inconvenience."
      Kernel::exit()
    end

    load(PRIME_CONFIGFILE_GLOBAL)
    if FileTest::exist?(PRIME_CONFIGFILE_LOCAL) then
      load(PRIME_CONFIGFILE_LOCAL)
    else
      Dir::ensure(File::dirname(PRIME_CONFIGFILE_LOCAL))
      `cp #{PRIME_CONFIGFILE_GLOBAL} #{PRIME_CONFIGFILE_LOCAL}`
    end

    case PRIME_ENV['typing_method']
    when 'tcode' then
      if PRIME_ENV.has_key?('style_mask_pending_chars') == false then
        PRIME_ENV['style_mask_pending_chars'] == true
      end
    end
  end

  def init_engines(engine_files = nil)
    engines = []
    def engines.command (method, *args)
      self.map {|engine|
	engine.send(method, *args)
      }
    end

    engine_files = (engine_files or PRIME_ENV['engines'])

    engine_files.each {|engine_file|
      require engine_file
      engines << eval($engine_class_name).new
    }
    debug_message("Init done.")
    return engines
  end

  def add_engine (engine)
    @engines << engine
  end

  def init
    @engines.command(:init)
  end

  def exit
    @engines.command(:exit)
  end

  def open
    @engines.command(:open)
  end

  def close
    if @flag_opened then
      @engines.command(:close)
      @flag_opened = false
    end
  end

  def update
    @engines.command(:update)
    refresh()
  end
  def refresh
    initialize_rc()
    PrimeTypeConv::refresh()
    @engines.command(:refresh)
  end

  def get_env (key)
    return PRIME_ENV[key]
  end

  def get_label (pattern)
    return PrimeTypeConv::convert(pattern).join()
  end

  def preedit_convert_input (pattern)
    return PrimeTypeConv::convert(pattern).join()
  end

  def set_context (context)
    @context = context
  end

  def learn_word (pron, literal, pos = "",
                  context = "", suffix = "", rest = "")
    pron    = (pron    or "").strip
    literal = (literal or "").strip
    pos     = (pos     or "").strip
    context = (context or "").strip
    suffix  = (suffix  or "").strip
    rest    = (rest    or "").strip

    @engines.command(:learn_word, pron, literal, pos, context, suffix, rest)
  end

  def lookup (string)
    if PRIME_ENV['typing_method'] == 'tcode' or 
        PRIME_ENV['typing_method'] == 'handwrite' then
      word_list = lookup_direct(string)
    else
#      word_list = lookup_compact(string)
      word_list = lookup_hybrid(string)
    end
    return word_list
  end
  def lookup_all (string)
    if PRIME_ENV['typing_method'] == 'tcode' or 
        PRIME_ENV['typing_method'] == 'handwrite' then
      word_list = lookup_direct_all(string)
    else
      word_list = lookup_hybrid_all(string)
    end
    return word_list
  end

  def lookup_hybrid (string)
    if @cache_hybrid_pattern == string then
      return @cache_hybrid
    end

    words  = search_hybrid(string, 10)
    result = PrimeWordList::merge_with_label(@context, words)

    @cache_hybrid_pattern = string
    @cache_hybrid = result
    return result
  end
  def lookup_hybrid_all (string)
    words  = search_hybrid(string)
    result = PrimeWordList::merge_with_label(@context, words)
    return result
  end

  def lookup_direct (string)
    results_expansion = lookup_expansion(string)
    results = PrimeWordList::merge_with_label(@context, results_expansion)
    return results[0,3]
  end

  def lookup_direct_all (string)
    #     results_prefix    = lookup_prefix(string)
    results_expansion = lookup_expansion(string)
    results_mixed     = lookup_mixed(string)
    words_japanese  = search_japanese(string)
    words_overall   = search_overall(string)

    results = PrimeWordList::merge_with_label(@context,
                                              # results_prefix[0, 3],
                                              results_expansion,
                                              results_mixed,
                                              words_japanese,
                                              words_overall)
    return PrimeWordList.new(results_expansion[0,3] | results)
  end

  def lookup_compact (string)
    words_compact = search_compact(string)
    words = PrimeWordList::merge_with_label(@context, words_compact)
    return words
  end
  def lookup_compact_all (string)
    words_compact  = search_compact(string)
    words_overall  = search_overall(string)
    words_japanese = search_japanese(string)
    
    results_conversion = PrimeWordList::merge_with_label(@context,
                                                         words_overall,
                                                         words_japanese)
    return PrimeWordList.new(words_compact | results_conversion)
  end

  def lookup_prefix (string)
    words_prefix = search_prefix(string)
    return PrimeWordList::merge_with_label(@context, words_prefix)
  end

  def lookup_exact (string)
    words_exact = search_exact(string)
    return PrimeWordList::merge_with_label(@context, words_exact)
  end

  def lookup_overall (string)
    words_overall = search_overall(string)
    return PrimeWordList::merge_with_label(@context, words_overall)
  end

  def lookup_expansion (string)
    # ͽͽ¬
    expands = PrimeTypeConv::expand(string)
    query = PrimeQuery.new(expands, nil, :literal_prefix, @context)
    results = lookup_internal(query)
    return results
  end

  def check_existence (pron, pos, literal)
    results = @engines.command(:check_existence, pron, literal, pos)
    results.each {|result|
      return true if result
    }
    return false
  end


  def search (query)
    query.input.uniq!()
    words_list = @engines.command(:search, query)
    return PrimeWordList::merge(words_list)
  end

  def search_prefix (string)
    if @cache_prefix_pattern == string then
      return @cache_prefix
    end

    # ֤袪ͽ¬
    expands = PrimeTypeConv::expand(string)
    query = PrimeQuery.new(expands, nil, :prefix, @context)
    words = search(query)

    @cache_prefix_pattern = string
    @cache_prefix = words

    return words
  end

  def search_exact (string)
    # ֤褽ͽ¬
    converted = PrimeTypeConv::convert(string).join()
    query = PrimeQuery.new([converted, string], nil, :exact, @context)
    words = search(query)
    return words
  end
  def search_raw (string)
    ## FIXME: This method is an ad-hoc routine for search_japanese.
    ## FIXME: <komatsu@taiyaki.org> (2004-02-28)
    query = PrimeQuery.new([string], nil, :exact, @context)
    words = search(query)
    return words
  end

  def search_overall (string)
    # 1+1=2, aiueo
    query = PrimeQuery.new([string], nil, :overall)
    words_overall = search(query)
    return words_overall
  end

  def search_compact (string)
    if @cache_compact_pattern == string then
      return @cache_compact
    end
    expands = PrimeTypeConv::expand(string)
    query   = PrimeQuery.new(expands, nil, :prefix, @context)
    words_compact = search(query)[0,1]

    if words_compact.length > 0 then
      predict_with_multi_clauses!(words_compact)

      words_compact[1..-1].each {|word|
        word.score = words_compact[0].score
      }
    end

    @cache_compact_pattern = string
    @cache_compact         = words_compact
    return words_compact
  end

  def search_hybrid (string, max = nil)
    words_prefix   = search_prefix(string)
    words_japanese = search_japanese(string)
    words_overall  = search_overall(string)

    num_prefix = 3
    num_max_prefix = 10
    if (num_max_prefix - words_japanese.length) > num_prefix then
      num_prefix = (num_max_prefix - words_japanese.length)
    end
    words = PrimeWordList::merge_with_label(@context, 
                                            words_prefix[0,num_prefix],
                                            words_overall,
                                            words_japanese)
    if max then
      words = words[0,max]
    end
    predict_with_multi_clauses!(words)

    return words
  end


  private
  def lookup_internal (query)
    query.input.uniq!()
    words_list = @engines.command(:search, query)
    return PrimeWordList::merge_with_label(@context, words_list)
  end

  def predict_with_multi_clauses!(results)
    if results.empty? then
      string.decrease {|string2|
        expands = PrimeTypeConv::expand(string2)
        if expands.length > 2 then
          next
        end
        query = PrimeQuery.new(expands, nil, :exact, @context)
        word1 = search(query).first
        
        if word1.non_nil? then
          expands = PrimeTypeConv::expand(string[(string2.length)..-1])
          query = PrimeQuery.new(expands, nil, :context, word1.literal)
          word2 = search(query).first
          if word2.non_nil? then
            results.push(_merge_words(word1, word2))
            break
          end
        end
      }
      word = PrimeWordList::merge_with_label(@context, results)[0,1]
    else
      word1 = results.first
      query = PrimeQuery.new([""], nil, :context, word1.literal)
      word2 = search(query).first
      if word2.non_nil? then
        results.push(_merge_words(word1, word2))
      end
    end
    return results
  end

  public
  def Prime::get_prefix (context, literal)
    if PRIME_ENV['style_auto_space'] and context and context != "" then
      if (context[-1] > 128 and literal[0] > 128) or
          (literal[0] < ?a and literal[0] > ?z and
             literal[0] < ?A and literal[0] > ?Z) or
          (context =~ /[(֡ءڡҡԡʡ̡ΡСȡ]$/) then
        return ""
      else
        return " "
      end
    else
      return ""
    end
  end
end


class PrimeQuery
  attr_accessor :input, :pos, :method, :context
  def initialize(input = [], pos = nil, method = :exact, context = nil)
    @input   = input
    @pos     = pos
    # method = {:prefix, :exact, :literal_prefix, :literal_exact,:overall, :context}
    @method  = method 
    @context = context
  end
end

class PrimeWord
  attr_reader   :pron, :literal, :pos, :index
  attr_accessor :category, :annotation, :data, :score, 
      :conjugation, :conjugation_pos, :rest, :prefix
  def initialize(pron, literal, pos, score,
                 category = nil, annotation = nil, *data)
    @pron        = (pron    or "")
    @literal     = (literal or "")
    @pos         = (pos     or "̤θ")
    @score       = score.to_i
    @category    = category
    @annotation  = annotation
    @data        = data
    @conjugation = ""
    @rest        = ""
    @prefix      = ""
#    @label       = [pron, literal].join("\t")
    @index       = [pron, literal, pos].join("\t")
  end

  def values
    return [@pron, @literal, @pos, @score, @category, @annotation, @data]
  end

  def label ()
    return (to_text_pron + "\t" + to_text_literal)
  end

  def to_text_literal
    return (@prefix + @literal + @conjugation + @rest)
  end
  def to_text_pron
    return (@pron + @conjugation + @rest)
  end

  def to_text
    return [to_text_pron(),
            to_text_literal(),
            "priority=#{@score}",
            "part=#{@pos}",
            "base=#{@literal}",
            "basekey=#{@pron}",
            ("conjugation=#{@conjugation}" unless @conjugation.empty?),
            ("suffix=#{@rest}"             unless @rest.empty?)
    ].compact.join("\t")
  end
end

class PrimeWordList < Array
  def PrimeWordList::merge (*words)
    ## merge should be sorted by inter-engine score.
    ## FIXME: Isn't it necessary to sort here?
    ## FIXME: <komatsu@taiyaki.org> (2004-01-24)
    result = words.flatten.compact.sort {|word1, word2| 
      (word2.score != word1.score) ? (word2.score <=> word1.score) :
             (word1.pron.length <=> word2.pron.length)
    }

    mark = {}
    merged = PrimeWordList.new
    result.each {|word|
      if mark[word.index] then
        word0 = mark[word.index]
        ## FIXME: Make this merging more intelligence.
        ## FIXME: <komatsu@taiyaki.org> (2004-01-24)
        word0.category   = (word0.category   or word.category)
        word0.annotation = (word0.annotation or word.annotation)
      else
	mark[word.index] = word
	merged.push(word)
      end
    }
    return merged
  end

  ## FIXME: Change the name of method.
  ## FIXME: <komatsu@taiyaki.org> (2004-01-26)
  def PrimeWordList::merge_with_label (context, *words)
    ## merge should be sorted by inter-engine score.
    ## FIXME: Isn't it necessary to sort here?
    ## FIXME: <komatsu@taiyaki.org> (2004-01-24)
    result = words.flatten.compact.sort {|word1, word2| 
      (word2.score != word1.score) ? (word2.score <=> word1.score) :
             (word1.pron.length <=> word2.pron.length)
    }

    mark = {}
    merged = PrimeWordList.new
    result.each {|word|
      if word.literal != "" then
        if mark[word.label()] then
          word0 = mark[word.label()]
          ## FIXME: Make this merging more intelligence.
          ## FIXME: <komatsu@taiyaki.org> (2004-01-24)
          word0.category   = (word0.category   or word.category)
          word0.annotation = (word0.annotation or word.annotation)
        else
          mark[word.label()] = word
          word.prefix = Prime::get_prefix(context, word.literal)
          merged.push(word)
        end
      end
    }
    return merged
  end

  def to_text
    texts = self.map {|word|
      word.to_text
    }
    return texts.join("\n")
  end
end
