# prime/prime.rb
# $Id: prime.rb,v 1.2.2.18 2004/01/29 04:20:31 komatsu Exp $
#
# Copyright (C) 2002, 2003, 2004 Hiroyuki Komatsu <komatsu@taiyaki.org>
#     All rights reserved.
#     This is free software with ABSOLUTELY NO WARRANTY.
#
# You can redistribute it and/or modify it under the terms of 
# the GNU General Public License version 2.
#

require 'prime/prime-config'
require 'prime/taiyaki'
require 'prime/prime-japanese'
require 'prime/prime-mixed'

PRIME_CONFIGFILE_GLOBAL = File::join2(PRIME_CONFIG_DIR,  "Custom_prime.rb")
PRIME_CONFIGFILE_LOCAL  = (ENV['PRIME_CONFIG'] or 
                             File::join2(PRIME_USER_DIR, "Custom_prime.rb"))

class Prime
  include Debug
  include PrimeJapanese
  include PrimeMixed

  def initialize (engine_files = nil)
    @debug_mode = false
    Dir::ensure(PRIME_USER_DIR)
    initialize_rc()

    @engines = init_engines(engine_files)
    @context = nil

    initialize_prime_japanese()
  end

  def initialize_rc
    if File::exist?(PRIME_CONFIGFILE_GLOBAL) == false then
      $stderr.puts "PRIME-ERROR:"
      $stderr.puts "  The file Custom_prime.rb is not found."
      $stderr.puts "  You might need to install Custom_prime.rb by"
      $stderr.puts "  'make install-etc'.  Sorry for your inconvenience."
      Kernel::exit()
    end

    load(PRIME_CONFIGFILE_GLOBAL)
    if FileTest::exist?(PRIME_CONFIGFILE_LOCAL) then
      load(PRIME_CONFIGFILE_LOCAL)
    else
      Dir::ensure(File::dirname(PRIME_CONFIGFILE_LOCAL))
      `cp #{PRIME_CONFIGFILE_GLOBAL} #{PRIME_CONFIGFILE_LOCAL}`
    end

    case PRIME_ENV['typing_method']
    when 'tcode' then
      if PRIME_ENV.has_key?('style_mask_pending_chars') == false then
        PRIME_ENV['style_mask_pending_chars'] == true
      end
    end
  end

  def init_engines(engine_files = nil)
    engines = []
    def engines.command (method, *args)
      self.map {|engine|
	engine.send(method, *args)
      }
    end

    engine_files = (engine_files or PRIME_ENV['engines'])

    engine_files.each {|engine_file|
      require engine_file
      engines << eval($engine_class_name).new
    }
    debug_message("Init done.")
    return engines
  end

  def add_engine (engine)
    @engines << engine
  end

  def init
    @engines.command(:init)
  end

  def exit
    @engines.command(:exit)
  end

  def open
    @engines.command(:open)
  end

  def close
    @engines.command(:close)
  end

  def refresh
    initialize_rc()
    PrimeTypeConv::refresh()
    @engines.command(:refresh)
  end

  def get_env (key)
    return PRIME_ENV[key]
  end

  def get_label (pattern)
    return PrimeTypeConv::convert(pattern).join()
  end

  def preedit_convert_input (pattern)
    return PrimeTypeConv::convert(pattern).join("\t")
  end

  def set_context (context)
    @context = context
  end

  def learn_word (pron, literal, pos = "",
                  context = "", suffix = "", rest = "")
    pron    = (pron    or "").strip
    literal = (literal or "").strip
    pos     = (pos     or "").strip
    context = (context or "").strip
    suffix  = (suffix  or "").strip
    rest    = (rest    or "").strip

    @engines.command(:learn_word, pron, literal, pos, context, suffix, rest)
  end

  def lookup (string)
    if PRIME_ENV['typing_method'] == 'tcode' or 
        PRIME_ENV['typing_method'] == 'handwrite' then
      word_list = lookup_direct(string)
    else
      word_list = lookup_hybrid(string)
    end
    return word_list
  end

  def lookup_hybrid (string)
    results_prefix   = lookup_prefix(string)
    results_japanese = lookup_japanese(string)
    results_overall  = lookup_overall(string)

    ## FIXME: Delete the Magic Numbers.
    prefix_length  = 10 - [results_japanese.length, 7].min
    results = PrimeWordList::merge_with_label(@context,
                                              results_prefix[0,prefix_length],
                                              results_overall,
                                              results_japanese)
    return results
  end

  def lookup_direct (string)
    #     results_prefix    = lookup_prefix(string)
    results_expansion = lookup_expansion(string)
    results_mixed     = lookup_mixed(string)
    results_japanese  = lookup_japanese(string)
    results_overall   = lookup_overall(string)

    results = PrimeWordList::merge_with_label(@context,
                                              # results_prefix[0, 3],
                                              results_expansion,
                                              results_mixed,
                                              results_japanese,
                                              results_overall)
    return results
  end

  def lookup_prefix (string)
    # ֤袪ͽ¬
    expands = PrimeTypeConv::expand(string)
    query = PrimeQuery.new(expands, nil, :prefix, @context)
    results = lookup_internal(query)
    return results
  end

  def lookup_exact (string)
    # ֤褽ͽ¬
    converted = PrimeTypeConv::convert(string).join()
    query = PrimeQuery.new([converted, string], nil, :exact, @context)
    results = lookup_internal(query)
    return results
  end

  def lookup_overall (string)
    # 1+1=2
    query = PrimeQuery.new([string], nil, :overall)
    results = lookup_internal(query)
    return results
  end

  def lookup_expansion (string)
    # ͽͽ¬
    expands = PrimeTypeConv::expand(string)
    query = PrimeQuery.new(expands, nil, :literal_prefix, @context)
    results = lookup_internal(query)
    return results
  end

  def check_existence (pron, pos, literal)
    results = @engines.command(:check_existence, pron, literal, pos)
    results.each {|result|
      return true if result
    }
    return false
  end

  def search (query)
    words_list = @engines.command(:search, query)
    return PrimeWordList::merge(words_list)
  end

  private
  def lookup_internal (query)
    words_list = @engines.command(:search, query)
    return PrimeWordList::merge_with_label(@context, words_list)
  end

  public
  def Prime::get_prefix (context, literal)
    if PRIME_ENV['style_auto_space'] and context then
      if (context[-1] > 128 and literal[0] > 128) or
          (literal[0] < ?a and literal[0] > ?z and
             literal[0] < ?A and literal[0] > ?Z) or
          (context =~ /[(֡ءڡҡԡʡ̡ΡСȡ]$/) then
        return ""
      else
        return " "
      end
    else
      return ""
    end
  end
end


class PrimeQuery
  attr_accessor :input, :pos, :method, :context
  def initialize(input = [], pos = nil, method = :exact, context = nil)
    @input   = input
    @pos     = pos
    # method = {:prefix, :exact, :literal_prefix, :literal_exact,:overall}
    @method  = method 
    @context = context
  end
end

class PrimeWord
  attr_reader   :pron, :literal, :pos, :index, :label
  attr_accessor :category, :annotation, :data, :score, 
      :conjugation, :conjugation_pos, :rest, :prefix
  def initialize(pron, literal, pos, score,
                 category = nil, annotation = nil, *data)
    @pron        = (pron    or "")
    @literal     = (literal or "")
    @pos         = (pos     or "̤θ")
    @score       = score.to_i
    @category    = category
    @annotation  = annotation
    @data        = data
    @label       = [pron, literal].join("\t")
    @index       = [pron, literal, pos].join("\t")
    @conjugation = ""
    @rest        = ""
    @prefix      = ""
  end

  def values
    return [@pron, @literal, @pos, @score, @category, @annotation, @data]
  end

  def to_text
    return [(@pron    + @conjugation + @rest),
            (@prefix + @literal + @conjugation + @rest),
            "priority=#{@score}",
            "part=#{@pos}",
            "base=#{@literal}",
            "basekey=#{@pron}",
            ("conjugation=#{@conjugation}" unless @conjugation.empty?),
            ("suffix=#{@rest}"             unless @rest.empty?)
    ].compact.join("\t")
  end
end

class PrimeWordList < Array
  def PrimeWordList::merge (*words)
    ## merge should be sorted by inter-engine score.
    ## FIXME: Isn't it necessary to sort here?
    ## FIXME: <komatsu@taiyaki.org> (2004-01-24)
    result = words.flatten.compact.sort {|word1, word2| 
      word2.score <=> word1.score
    }

    mark = {}
    merged = PrimeWordList.new
    result.each {|word|
      if mark[word.index] then
        word0 = mark[word.index]
        ## FIXME: Make this merging more intelligence.
        ## FIXME: <komatsu@taiyaki.org> (2004-01-24)
        word0.category   = (word0.category   or word.category)
        word0.annotation = (word0.annotation or word.annotation)
      else
	mark[word.index] = word
	merged.push(word)
      end
    }
    return merged
  end

  ## FIXME: Change the name of method.
  ## FIXME: <komatsu@taiyaki.org> (2004-01-26)
  def PrimeWordList::merge_with_label (context, *words)
    ## merge should be sorted by inter-engine score.
    ## FIXME: Isn't it necessary to sort here?
    ## FIXME: <komatsu@taiyaki.org> (2004-01-24)
    result = words.flatten.compact.sort {|word1, word2| 
      word2.score <=> word1.score
    }

    mark = {}
    merged = PrimeWordList.new
    result.each {|word|
      if mark[word.label] then
        word0 = mark[word.label]
        ## FIXME: Make this merging more intelligence.
        ## FIXME: <komatsu@taiyaki.org> (2004-01-24)
        word0.category   = (word0.category   or word.category)
        word0.annotation = (word0.annotation or word.annotation)
      else
	mark[word.label] = word
        word.prefix = Prime::get_prefix(context, word.literal)
	merged.push(word)
      end
    }
    return merged
  end

  def to_text
    texts = self.map {|word|
      word.to_text
    }
    return texts.join("\n")
  end
end
