;;;; prime.el: Emacs Lisp for PRIME
;;;; $Id: prime.el,v 1.4 2004/01/14 19:51:27 komatsu Exp $
;;;;
;;;; Copyright (C) 2002, 2003 Hiroyuki Komatsu <komatsu@taiyaki.org>
;;;;     All rights reserved.
;;;;     This is free software with ABSOLUTELY NO WARRANTY.
;;;;
;;;; You can redistribute it and/or modify it under the terms of 
;;;; the GNU General Public License version 2.

(require 'prime-init)

; - Ѵؿ -----------------------------------------------------------

(defun prime-set-minor-mode (name modeline &optional key-map)
  (make-variable-buffer-local name)
  (or (assq name minor-mode-alist)
      (setq minor-mode-alist (cons (list name modeline) minor-mode-alist)))
  (and key-map
       (or (assq name minor-mode-map-alist)
	   (setq minor-mode-map-alist  (cons (cons name key-map)
					     minor-mode-map-alist))))
  )

(defun prime-make-keymap (&optional default)
  (let ((map (make-sparse-keymap)))
    (and default
	 (if (functionp 'set-keymap-default-binding) ;; Emacs ˤϤʤ.
	     (set-keymap-default-binding map default)
	   (setq map (append map (list (cons t default))))))
    map
    ))

; ----------------------------------------------------------------------

(defun prime-mode (&optional arg)
  "Toggle PRIME mode.
PRIME is a Japanese Predictive Input Method Editor."
  (interactive "P")
  (if (consp arg) 
      (if (and (mell-transient-region-active-p)
	       (/= (region-beginning) (region-end)))

	 (call-interactively 'prime-fund-convert-region)
	(prime-convert-last-word))
    (setq prime-mode 
	  (if (null arg) (not prime-mode) (> (prefix-numeric-value arg) 0)))
    ;; ɤˡǤϤʤ줷ʤ.
    (add-hook 'post-command-hook 'prime-set-cursor-color)
    (add-hook 'minibuffer-setup-hook 'prime-set-cursor-color) ; For XEmacs
;  (prime-set-cursor-color)
    (if prime-mode
	(prime-mode-on t)
      (prime-mode-off t))
    (run-hooks 'prime-mode-hook)
    (and (mell-transient-region-active-p)
	 (/= (region-beginning) (region-end))
	 (call-interactively 'prime-fund-convert-region))
  ))

(defun prime-mode-on (&optional forcep)
  (if (or forcep (not prime-mode))
      (progn
	(setq inactivate-current-input-method-function 'prime-mode)
	(or (markerp prime-marker) (setq prime-marker (make-marker)))
	(prime-server-open)
	(setq prime-mode t)
	(prime-toggle-mode-line)
	(prime-reset-before-input)
	(prime-fund-mode-on)
	(and prime-style-kutouten-autochange-p
	     (prime-style-kutouten-set-automatically))
	)))

(defun prime-mode-off (&optional forcep)
  (if (or forcep prime-mode)
      (progn
	(prime-server-close)
	(setq prime-context nil)
	(set-marker prime-marker nil)
	(setq prime-mode nil)
	(prime-toggle-mode-line)
	(prime-fund-mode-off)
	(prime-input-mode-off)
;	(prime-server-dict-save)
	(prime-fix prime-curstr)
	)))

;;;; ------------------------------------------------------------
;;;; prime-cand-mode
;;;; ------------------------------------------------------------
(defun prime-cand-mode (&optional arg)
  (setq prime-cand-mode 
	(if (null arg) (not prime-cand-mode)
	  (> (prefix-numeric-value arg) 0)))
  (if prime-cand-mode
      (prime-cand-mode-on t)
    (prime-cand-mode-off t)
    ))

(defun prime-cand-mode-on (&optional forcep)
  (if (or forcep (not prime-cand-mode))
      (progn
	(prime-conv-mode-on)
	(setq prime-cand-mode t)
	(setq prime-cand-column 0)
	(prime-cand-set-column-list)
;	(prime-enum-mode-off)
	)))

(defun prime-cand-mode-off (&optional forcep)
  (if (or forcep prime-cand-mode)
      (progn
;	(prime-narrow-mode-off)
	(setq prime-cand-mode nil)
	)))

;;;; ------------------------------------------------------------
;;;; prime-narrow-mode
;;;; ------------------------------------------------------------
(defun prime-narrow-mode (&optional arg)
  (interactive)
  (setq prime-narrow-mode 
	(if (null arg) (not prime-narrow-mode)
	  (> (prefix-numeric-value arg) 0)))
  (if prime-narrow-mode
      (prime-narrow-mode-on t)
    (prime-narrow-mode-off t)
    )
  (prime-disp-narrow)
  )

(defun prime-narrow-mode-on (&optional forcep)
  (if (or forcep (not prime-narrow-mode))
      (progn
	(prime-conv-mode-off)
	(setq prime-narrow-mode t)
	(prime-narrow-mode-reset)
	(setq prime-narrow-orig-cands prime-cands)
	(setq prime-narrow-orig-nth-cand prime-nth-cand)
	(setq prime-nth-cand 0)
	)))

(defun prime-narrow-mode-off (&optional forcep)
  (if (or forcep prime-narrow-mode)
      (progn
	(setq prime-narrow-mode nil)
	(setq prime-cands prime-narrow-orig-cands)
	(setq prime-nth-cand (prime-get-nth prime-curstr prime-pat))

	;; prime-nth-cand  nil äν
	(or (and (integerp prime-nth-cand)
		 (>= prime-nth-cand 0)
		 (< prime-nth-cand (length prime-cands)))
	    (setq prime-nth-cand prime-narrow-orig-nth-cand))
	)))

(defun prime-narrow-mode-reset ()
  (setq prime-narrow-pat "")
  (setq prime-narrow-column 0)
  )

(defun prime-convert-region (beg end)
  (interactive "r")
  (let ((pattern (suikyo-convert-kana-romaji
		  (concat (japanese-hiragana (buffer-substring beg end))
			  " "))))
    (if (string-match "\\cj" pattern)
	(message (concat "ѴбƤʤʸ󤬴ޤޤƤޤ. "
			 "(" pattern ")"))
      (setq prime-start-pat (buffer-substring beg end))
      (setq prime-pat pattern)
      (prime-input-mode-on t)
      (delete-region beg end)
      (prime-disp-input))))

(defun prime-convert-last-word ()
  (interactive)
  (let* ((end (point))
	 (skip (min (prog1 (skip-chars-backward "a-zA-Z-") (goto-char end))
		    (prog1 (skip-chars-backward "-")  (goto-char end))
		    (skip-chars-backward "-"))))
    (and (< skip 0)
	 (prime-convert-region (+ end skip) end))
    ))


;;;; ------------------------------------------------------------
;;;; prime-cand-mode
;;;; ------------------------------------------------------------
;; prime-cand-column  prime-nth-cand å
;; ߤ column ֤.
(defun prime-cand-set-current-column ()
  (let ((list prime-cand-column-list))
    (while (not (eq prime-nth-cand (cdr (car list))))
      (setq list (cdr list)))
    (or (and (>= prime-cand-column (car (car list)))
	     (if (cdr (nth 1 list))
		 (< prime-cand-column (car (nth 1 list)))
	       (< prime-cand-column (car (nth 2 list)))))
	(setq prime-cand-column (/ (+ (car (car list)) (car (nth 1 list))) 2))
	)))

(defun prime-cand-get ()
  (let ((list (cdr prime-cand-column-list))
	(cur-data (car prime-cand-column-list))
	prev-data)
    (while (and cur-data (>= prime-cand-column (car cur-data)))
      (setq prev-data (if (cdr cur-data) cur-data prev-data))
      (setq cur-data (car list))
      (setq list (cdr list)))
    (cdr prev-data)))

(defun prime-cand-forward ()
  (interactive)
  (setq prime-nth-cand (% (1+ prime-nth-cand) (length prime-cands)))
  (prime-cand-set-current-column)
  (prime-disp-conv)
  )

(defun prime-cand-backward ()
  (interactive)
  (setq prime-nth-cand (% (+ (1- prime-nth-cand) (length prime-cands))
			  (length prime-cands)))
  (prime-cand-set-current-column)
  (prime-disp-conv)
  )

(defun prime-cand-next-line ()
  (interactive)
  (prime-cand-set-current-column)
  (setq prime-cand-column 
	(% (+ prime-cand-column (window-width)) prime-cand-column-length))
  (setq prime-nth-cand (prime-cand-get))
  (prime-disp-conv)
  )

(defun prime-cand-prev-line ()
  (interactive)
  (prime-cand-set-current-column)
  (setq prime-cand-column 
	(% (+ (- prime-cand-column (window-width)) prime-cand-column-length)
	   prime-cand-column-length))
  (setq prime-nth-cand (prime-cand-get))
  (prime-disp-conv)
  )

(defun prime-cand-beginning-of-line ()
  (interactive)
  (prime-cand-set-current-column)
  (setq prime-cand-column
	(- prime-cand-column (% prime-cand-column (window-width))))
  (setq prime-nth-cand (prime-cand-get))
  (prime-disp-conv)
  )

(defun prime-cand-end-of-line ()
  (interactive)
  (prime-cand-set-current-column)
  (setq prime-cand-column
	(+ (- prime-cand-column (% prime-cand-column (window-width)))
	   (1- (window-width))))
  (setq prime-nth-cand (prime-cand-get))
  (prime-disp-conv)
  )

;;;; ------------------------------------------------------------
;;;; prime-narrow-mode
;;;; ------------------------------------------------------------
(defun prime-narrow-keyin (&optional char)
  (interactive)
  (setq char (or char last-input-char))

  (cond
   ((prime-enum-keyin char))

   ((prime-keyin-direct char))

   (prime-conv-mode
    (prime-fix prime-curstr)
    (prime-after-fix char)
    )

   ((prime-keyin-capital char))

   (t
    (prime-narrow-insert-char char)
    (prime-disp-narrow))
   ))

(defun prime-narrow-compare-list (list-target list-pattern)
  (mapcar 
   '(lambda (target)
      (catch 'match
	(let ((list list-pattern)
	      (target-regexp 
	       ;; Ҥ餬, , ѿ '[', ']' ϽƤ
	       (concat "["
		       (mell-string-delete-regexp
			target "[a-z0-9]\\|\\cH\\|\\cK\\|\\[\\|\\]")
		       "]")))
	  (and (> (length target-regexp) 2)
	       (while (car list)
		 (and (string-match target-regexp (car list))
		      (throw 'match target))
		 (setq list (cdr list))))
	  nil
	  )))
   list-target
   ))

(defun prime-narrow-insert-char (char &optional column)
  (or column (setq column prime-narrow-column))
  (setq prime-narrow-pat (concat (substring prime-narrow-pat 0 column)
			  (char-to-string char)  (substring prime-narrow-pat column)))
  (setq prime-narrow-column 
	(+ prime-narrow-column (length (char-to-string char))))
  )

;;;; prime-input-* ȤۤȤɤäʤΤ礷.
(defun prime-narrow-backward-char ()
  (interactive)
  (setq prime-narrow-column (max 0 (1- prime-narrow-column)))
  (prime-disp-narrow)
  )
	
(defun prime-narrow-forward-char ()
  (interactive)
  (setq prime-narrow-column (min (length prime-narrow-pat) (1+ prime-narrow-column)))
  (prime-disp-narrow)
  )

(defun prime-narrow-end-of-pattern ()
  (interactive)
  (setq prime-narrow-column (length prime-narrow-pat))
  (prime-disp-narrow)
  )

(defun prime-narrow-beginning-of-pattern ()
  (interactive)
  (setq prime-narrow-column 0)
  (prime-disp-narrow)
  )

(defun prime-narrow-transpose-chars ()
  (interactive)
  (setq prime-narrow-pat
	(mell-string-transpose-chars prime-narrow-pat prime-narrow-column))
  (and (> prime-narrow-column 0)
       (< prime-narrow-column (length prime-narrow-pat))
       (setq prime-narrow-column (1+ prime-narrow-column)))
  (prime-disp-narrow)
  )

;; Think "undo" !
(defun prime-narrow-delete-char ()
  (interactive)
  (or (= prime-narrow-column (length prime-narrow-pat))
      (setq prime-narrow-pat (concat (substring prime-narrow-pat 0 prime-narrow-column)
			      (substring prime-narrow-pat (1+ prime-narrow-column)))))
  (prime-disp-narrow)
  )

(defun prime-narrow-delete-backward-char ()
  (interactive)
  (if prime-conv-mode
      (prime-conv-prev)
    (or (= prime-narrow-column 0)
	(progn
	  (setq prime-narrow-pat (concat (substring prime-narrow-pat
						    0 (1- prime-narrow-column))
					 (substring prime-narrow-pat 
						    prime-narrow-column)))
	  (setq prime-narrow-column (1- prime-narrow-column))))
    (prime-disp-narrow))
  )

;;;; ----------------------------------------------------------------

(defun prime-set-cands (pattern &optional forcep)
  (if (and (string= pattern prime-prev-pattern) (not forcep))
      prime-cands
    (setq prime-prev-pattern pattern)
    (setq prime-cands (prime-search pattern))
    ))


(defun prime-search (pattern)
  (prime-server-search pattern prime-context)
  )

(provide 'prime)
(require 'prime-engine)

(prime-init)
