# prime/prime.rb
# $Id: prime.rb,v 1.6 2004/01/08 06:54:01 komatsu Exp $
#
# Copyright (C) 2002 Hiroyuki Komatsu <komatsu@taiyaki.org>
#     All rights reserved.
#     This is free software with ABSOLUTELY NO WARRANTY.
#
# You can redistribute it and/or modify it under the terms of 
# the GNU General Public License version 2.
#

require 'prime/prime-config'
require 'prime/taiyaki'
require 'suikyo/suikyo'

PRIME_CONFIGFILE_GLOBAL = File::join2(PRIME_CONFIG_DIR,  "Custom_prime.rb")
PRIME_CONFIGFILE_LOCAL  = (ENV['PRIME_CONFIG'] or 
                             File::join2(PRIME_USER_DIR, "Custom_prime.rb"))

class Prime
  include Debug

  def initialize (engine_files = nil)
    @debug_mode = false
    Dir::ensure(PRIME_USER_DIR)
    initialize_rc()

    @input = PrimeInput.new()
    @engines = init_engines(engine_files)
  end

  def initialize_rc
    if File::exist?(PRIME_CONFIGFILE_GLOBAL) == false then
      $stderr.puts "PRIME-ERROR:"
      $stderr.puts "  The file Custom_prime.rb is not found."
      $stderr.puts "  You might need to install Custom_prime.rb by"
      $stderr.puts "  'make install-etc'.  Sorry for your inconvenience."
      Kernel::exit()
    end

    load(PRIME_CONFIGFILE_GLOBAL)
    if FileTest::exist?(PRIME_CONFIGFILE_LOCAL) then
      load(PRIME_CONFIGFILE_LOCAL)
    else
      Dir::ensure(File::dirname(PRIME_CONFIGFILE_LOCAL))
      `cp #{PRIME_CONFIGFILE_GLOBAL} #{PRIME_CONFIGFILE_LOCAL}`
    end
  end

  def init_engines(engine_files = nil)
    engines = []
    def engines.command (method, *args)
      self.map {|engine|
	engine.send(method, *args)
      }
    end

    engine_files = (engine_files or PRIME_ENV['engines'])

    engine_files.each {|engine_file|
      require engine_file
      engines << eval($engine_class_name).new
    }
    debug_message("Init done.")
    return engines
  end

  def add_engine (engine)
    @engines << engine
  end

  def init
    @engines.command(:init)
  end

  def exit
    @engines.command(:exit)
  end

  def open
    @engines.command(:open)
  end

  def close
    @engines.command(:close)
  end

  def refresh
    initialize_rc()
    @input = PrimeInput.new()
#    @engines.command(:refresh)
  end

  def get_env (key)
    return PRIME_ENV[key]
  end

  def get_label (pattern)
    return @input.label(pattern)
  end

  def preedit_convert_input (pattern)
    return @input.convert_input(pattern).join("\t")
  end

  def set_context (context)
    @input.set_context(context)
  end

  def learn_word (key, value, part = nil,
		  context = nil, suffix = nil, rest = nil)
    part    = (part    or "")
    context = (context or "")
    suffix  = (suffix  or "")
    rest    = (rest    or "")
    @engines.command(:learn_word, key, value, part, context, suffix, rest)
  end

  def lookup_prefix (pattern)
    lookup_internal(pattern, :lookup_prefix)
  end

  def lookup_exact (pattern)
    lookup_internal(pattern, :lookup_exact)
  end

  def lookup_hybrid (pattern)
    results_prefix = lookup_internal(pattern, :lookup_prefix)
    results_exact  = lookup_internal(pattern, :lookup_exact)
    ## FIXME: Delete the Magic Numbers.
    prefix_length  = 10 - [results_exact.length, 7].min
    result = PrimeResult::merge(results_prefix[0,prefix_length], results_exact)
    return result
#    return result.to_text
  end

  def check_existence (pron, pos, literal)
    pattern = [pron, pos, literal].join("\t")
    result = lookup_internal(pattern, :lookup_exact)
    return (result.length > 0)
  end

  private
#   def lookup (pattern, method = :lookup_prefix)
#     result = lookup_internal(pattern, method)
#     return result.to_text
#   end

  def lookup_internal (pattern, method = :lookup_prefix)
    @input.set(pattern)
    results = @engines.command(method, @input)
    return PrimeResult::merge(results)
  end
end

# PrimeInput is a class for input data by a user.
class PrimeInput
  attr_reader :original, :base, :expands, :context

  def initialize
    @suikyo = initialize_suikyo
    @context = nil
  end

  def initialize_suikyo
    if PRIME_ENV['suikyo_use_cache'] then
      threshold_time = [File::mtime(File::expand_path(__FILE__)),
                        File::mtime(PRIME_CONFIGFILE_GLOBAL),
                        File::mtime(PRIME_CONFIGFILE_LOCAL)].max
    else
      threshold_time = Time.new()
    end
    suikyo = Marshal::init_file(File::join2(PRIME_USER_DIR, "suikyo.rbo"),
				threshold_time) {
      suikyo = Suikyo.new
      PRIME_ENV['suikyo_tables'].each {|table|
	suikyo.table.loadfile(table)
      }
      suikyo
    }
    PRIME_ENV['suikyo_tables'] = suikyo.table.table_files
    return suikyo
  end

  def label (input = nil)
    string = input.nil? ? @original : input
    if @suikyo.valid?(string) then
      return @suikyo.convert(string)
    else
      return string
    end    
  end

  def convert_input (string)
    (conversion, pending, node) = @suikyo.convert_internal(string)
    return [conversion, pending]
  end

  def set (input, context = nil)
    @original = input
    (@base, @expands) = @suikyo.expand(input)
    @expands << input

    set_context(context) if context
    return true
  end

  def set_context (context)
    @context = context
    return true
  end
end

