# session.rb
# $Id: session.rb,v 1.7 2004/12/11 07:53:01 komatsu Exp $
#
# Copyright (C) 2001 Satoru Takabayashi <satoru@namazu.org>
# Copyright (C) 2002, 2003, 2004 Hiroyuki Komatsu <komatsu@taiyaki.org>
#     All rights reserved.
#     This is free software with ABSOLUTELY NO WARRANTY.
#
# You can redistribute it and/or modify it under the terms of 
# the GNU General Public License version 2.
#

require 'suikyo/suikyo-composer'

class Command
  attr_reader :name, :args, :nargs, :min_nargs, :description

  def initialize (name, args, description, min_nargs = nil)
    @name = name
    @args = args
    @nargs = args.length
    @min_nargs = (min_nargs or @nargs)
    @description = description
  end
end

class SessionCore
  def initialize (prime)
    @prime = prime
  end

  def get_line (io_in)
    return io_in.gets()
  end

  def execute (line)
  end
end

class Session < SessionCore
  def initialize (prime, version)
    super(prime)
    @version = version
    @debug   = false
    @command_table = Hash.new
    init_command_table()
  end

  def init_command_table
    add_command(:close,   [], "close the connection")
    add_command(:help,    [], "print the help message")
    add_command(:version, [], "show the version number")
  end

  def add_command (name, args, description, min_nargs = nil)
    command = Command.new(name, args, description, min_nargs)
    @command_table[name] = command
  end

  def reply_successful (result = nil)
    if result.nil? or result.empty? then
      result = ""
    else
      result += "\n"
    end
    return ("ok\n" + result + "\n")
  end

  def reply_unsuccessful (result = nil)
    if result.nil? or result.empty? then
      result = ""
    else
      result += "\n"
    end
    return ("error\n" + result + "Try `help' for protocol information.\n\n")
  end

  def help
#     commands = @command_table.values.sort {|a, b| 
#       (a.nargs <=> b.nargs).nonzero? || a.name.to_s <=> b.name.to_s
#     }
    commands = @command_table.values.sort {|a, b| 
      (a.name.to_s <=> b.name.to_s).nonzero? || a.nargs <=> b.nargs
    }
    help = ""
    commands.each {|c|
      help += format("%-16s %s - %s\n",
                     c.name, c.args.join(" "), c.description)
    }
    help += "Note: Use TAB for delimiters."
    return reply_successful(help)
  end

  def execute (line)
    if line.nil? then
      return false
    end
    chunks = line.gsub(/[\r\n]/, "").split("\t")
    if chunks.length > 0 then
      name = chunks.shift
      args = chunks
      return send_command(name, args)
    end
  end

  def send_command (name, args)
    if name.empty? then
      return reply_unsuccessful("Empty command:")
    end

    command = @command_table[name.intern]
    if command.nil? then
      return reply_unsuccessful("Unknown command: #{name}")
    elsif command.min_nargs <= args.length and args.length <= command.nargs
      return send(command.name, *args)
    else
      error_msg = "Wrong number of arguments (expecting #{command.nargs})"
      return reply_unsuccessful(error_msg)
    end
  end

  def close
    return false
  end

  def version
    return reply_successful(@version)
  end
end

class SessionPrime < Session
  def initialize (prime, version)
    super

    add_command(:l, [:PATTERN],
		"look up PATTERN", 0)
    add_command(:lookup, [:PATTERN],
		"look up PATTERN", 0)
    add_command(:lookup_all, [:PATTERN],
		"look up PATTERN", 0)
    add_command(:lookup_compact, [:PATTERN],
		"look up PATTERN and return one or two candidate", 0)
    add_command(:lookup_compact_all, [:PATTERN],
		"look up PATTERN and return one or two candidate", 0)
    add_command(:lookup_direct, [:PATTERN],
		"look up PATTERN for direct typing method", 0)
    add_command(:lookup_direct_all, [:PATTERN],
		"look up PATTERN for direct typing method", 0)
    add_command(:lookup_hybrid, [:PATTERN],
		"look up PATTERN with hybrid matching", 0)
    add_command(:lookup_hybrid_all, [:PATTERN],
		"look up PATTERN with hybrid matching", 0)
    add_command(:lookup_prefix, [:PATTERN],
		"look up PATTERN with prefix matching", 0)
    add_command(:lookup_prefix_ex, [:PATTERN],
		"look up PATTERN with prefix matching", 0)
    add_command(:lookup_exact, [:PATTERN],
		"look up PATTERN with exact matching", 0)
    add_command(:lookup_expansion, [:PATTERN],
                "look up PATTERN from literal dictionaries", 0)
    add_command(:lookup_mixed, [:PATTERN],
                "look up PATTERN.  PATTERN can be mixed with pron and literal",
                0)
    add_command(:learn_word, [:KEY, :VALUE, :PART, :CONTEXT, :SUFFIX, :REST],
		"learn and record a word to the user dictionary", 2)
    add_command(:reset_context, [], "reset context")
    add_command(:set_context, [:CONTEXT], "set context to CONTEXT")
    add_command(:get_env, [:KEY], "get a variable associated with KEY")
    add_command(:get_label, [:PATTERN],
		"get a label string (e.g. hiragana) from a user input.")
    add_command(:preedit_convert_input, [:PATTERN],
                "convert a PATTERN and return a pair of a converted and a pending string.")
    add_command(:refresh, [],
                "refresh the statuses of the conversion engines.")
  end

  def learn_word (key, value, part = nil,
                  context = nil, suffix = nil, rest = nil)
    @prime.learn_word(key, value, part, context, suffix, rest)
    return reply_successful()
  end

  def get_env (key)
    env = @prime.get_env(key)
    if env.kind_of?(String) then
      output = format("string\t%s", env)
    elsif env.kind_of?(Array) then
      output = format("array\t%s", env.join("\t"))
    elsif env.kind_of?(TrueClass) or env.kind_of?(FalseClass) then
      output = format("boolean\t%s", env ? "true" : "false")
    elsif env == nil then
      output = 'nil'
    else
      output = 'unknown'
    end

    return reply_successful(output)
  end

  def get_label (pattern)
    return reply_successful(@prime.get_label(pattern))
  end

  def preedit_convert_input (pattern)
    return reply_successful(@prime.preedit_convert_input(pattern))
  end

  def set_context (context)
    @prime.set_context(context)
    return reply_successful()
  end
  def reset_context ()
    @prime.set_context(nil)
    return reply_successful()
  end
  
  def l (pattern = "")
    return reply_successful(@prime.lookup(pattern).to_text)
  end
  def lookup (pattern = "")
    return reply_successful(@prime.lookup(pattern).to_text)
  end
  def lookup_all (pattern = "")
    return reply_successful(@prime.lookup_all(pattern).to_text)
  end

  def lookup_compact (pattern = "")
    return reply_successful(@prime.lookup_compact(pattern).to_text)
  end
  def lookup_compact_all (pattern = "")
    return reply_successful(@prime.lookup_compact_all(pattern).to_text)
  end
  def lookup_direct (pattern = "")
    return reply_successful(@prime.lookup_direct(pattern).to_text)
  end
  def lookup_direct_all (pattern = "")
    return reply_successful(@prime.lookup_direct_all(pattern).to_text)
  end
  def lookup_hybrid (pattern = "")
    return reply_successful(@prime.lookup_hybrid(pattern).to_text)
  end
  def lookup_hybrid_all (pattern = "")
    return reply_successful(@prime.lookup_hybrid_all(pattern).to_text)
  end
  def lookup_prefix (pattern = "")
    return reply_successful(@prime.lookup_prefix(pattern).to_text)
  end
  def lookup_prefix_ex (pattern = "")
    return reply_successful(@prime.lookup_prefix_ex(pattern).to_text)
  end
  def lookup_exact (pattern = "")
    return reply_successful(@prime.lookup_exact(pattern).to_text)
  end
  def lookup_expansion (pattern = "")
    return reply_successful(@prime.lookup_expansion(pattern).to_text)
  end
  def lookup_mixed (pattern = "")
    return reply_successful(@prime.lookup_mixed(pattern).to_text)
  end

  def refresh ()
    @prime.refresh()
    return reply_successful()
  end
end

class SessionPrime2 < SessionPrime
  def initialize (prime, version)
    super(prime, version)
    @sessions = {}
    @session_no = 0

    add_command(:session_start, [],
                "start a session and return the session id.")
    add_command(:session_end, [:SESSION],
                "close the session specified with the session id.")
    add_command(:edit_insert, [:SESSION, :STRING],
                "insert this string into the preediting string.")
    add_command(:edit_delete, [:SESSION],
                "delete a character from the preediting string.")
    add_command(:edit_backspace, [:SESSION],
                "delete a character backward from the preediting string.")
    add_command(:edit_erase, [:SESSION],
                "erase the preediting string.")
    add_command(:edit_undo, [:SESSION],
                "undo the preediting string.")
    add_command(:edit_cursor_right, [:SESSION],
                "move the cursor right")
    add_command(:edit_cursor_left, [:SESSION],
                "move the cursor left")
    add_command(:edit_cursor_right_edge, [:SESSION],
                "move the cursor the end of the preediting string.")
    add_command(:edit_cursor_left_edge, [:SESSION],
                "move the cursor the beginning of the preediting string.")
    add_command(:edit_get_preedition, [:SESSION],
                "return a list fo the preediting string [left, cursor, right]")
    add_command(:edit_get_query_string, [:SESSION],
                "return a query string for lookup functions. (temporal)")

    add_command(:edit_set_mode, [:SESSION, :MODE],
                "set display mode of the preedition.\n" +
                "        MODE = [default, katakana, half_katakana, \n" + 
                "                wide_ascii, raw]")

#     add_command(:convert_start, [:SESSION],
#                 "convert the preedition string.")
#     add_command(:predict, [:METHOD],
#                 "predict candidate words with the method", 0)
  end

#   def predict (method = nil)
#   end

  ##
  ## Conversion methods
#   def convert_start (session)
#     composer = session_get_composer(session)
#     composer.


  ## 
  ## Session methods
  ##
  def session_start ()
    ## FIXME: Revise the following code.
    ## FIXME: (2004-12-06) <Hiro>
    suikyo = PrimeTypeConv::initialize_suikyo()
    composer = SuikyoComposer.new(suikyo.table)
    composer.set_reverse_table(PRIME_ENV['suikyo_reverse_tables'])

    ## Setting hybrid_typing.
    if PRIME_ENV['hybrid_typing'].nil? then
      if PRIME_ENV['typing_method'] == 'romaji' then
        PRIME_ENV['hybrid_typing'] = true
        composer.hybrid_typing     = true
      end
    else
      composer.hybrid_typing = PRIME_ENV['hybrid_typing']
    end

    @session_no += 1
    @sessions[@session_no.to_s] = composer

    return reply_successful( @session_no.to_s )
  end

  def session_end (session)
    @sessions[session] = nil
    return reply_successful()
  end

  ## 
  ## Composition methods
  ##
  def edit_insert (session, string)
    composer = session_get_composer(session)
    composer.edit_insert(string)
    return reply_with_preediting_string(composer)
  end
  def edit_delete (session)
    composer = session_get_composer(session)
    composer.edit_delete()
    return reply_with_preediting_string(composer)
  end
  def edit_backspace (session)
    composer = session_get_composer(session)
    composer.edit_backspace()
    return reply_with_preediting_string(composer)
  end
  def edit_erase (session)
    composer = session_get_composer(session)
    composer.edit_erase()
    if PRIME_ENV['hybrid_typing'] then
      composer.set_mode_hybrid()
    end
    return reply_with_preediting_string(composer)
  end
  def edit_undo (session)
    composer = session_get_composer(session)
    composer.undo()
    return reply_with_preediting_string(composer)
  end
  def edit_cursor_right (session)
    composer = session_get_composer(session)
    composer.cursor_right()
    return reply_with_preediting_string(composer)
  end
  def edit_cursor_left (session)
    composer = session_get_composer(session)
    composer.cursor_left()
    return reply_with_preediting_string(composer)
  end
  def edit_cursor_right_edge (session)
    composer = session_get_composer(session)
    composer.cursor_right_edge()
    return reply_with_preediting_string(composer)
  end
  def edit_cursor_left_edge (session)
    composer = session_get_composer(session)
    composer.cursor_left_edge()
    return reply_with_preediting_string(composer)
  end
  def edit_get_preedition (session)
    composer = session_get_composer(session)
    return reply_with_preediting_string(composer)
  end
  def edit_get_query_string (session)
    composer = session_get_composer(session)
    return reply_successful( composer.edit_get_query_string() )
  end

  def edit_set_mode (session, mode)
    composer = session_get_composer(session)
    case mode
    when "hybrid" then         # Original
      composer.set_mode_hybrid()
    when "default" then        # F6
      composer.set_mode_default()
    when "katakana" then       # F7
      composer.set_mode_katakana()
    when "half_katakana" then  # F8
      composer.set_mode_half_katakana()
    when "wide_ascii" then     # F9
      composer.set_mode_wide_ascii()
    when "raw" then            # F10
      composer.set_mode_raw()
    else
      error_message = "Unknown mode.  Valid modes are: \n" +
        "[default, katakana, half_katakana, wide_ascii, raw]"
      return reply_unsuccessful(error_message)
    end
    return reply_with_preediting_string(composer)
  end

  private
  def reply_with_preediting_string (composer)
    result = composer.edit_get_preediting_string().join("\t")
    return reply_successful(result)
  end

  def session_get_composer (session)
    composer = @sessions[session]
    return composer
  end
end

# ----

class SessionSKK < SessionCore
  def initialize (prime, portnum)
    super(prime)
    @portnum = portnum
  end

  def get_line (io_in)
    line = ""
    loop {
      char = io_in.getc()
      if char.nil? then
        return
      end
      if char.chr == " " or char.chr == "\n" then
        return line
      else
        line += char.chr
      end
    }
  end

  def lookup (pattern)
    return @prime.lookup_japanese(pattern)
  end

  def execute (line)
    line.chomp!()

    case line[0,1]
    when "0" then
      return false
    when "1" then
      pattern = line.chomp[1..-1]
      results = lookup(pattern)
      if results.empty? then
        return "4" + pattern + "\n"
      else
        result_line = results.map {|result|
          result.to_text_literal()
        }.join('/')
        return "1/" + result_line + "/\n"
      end
    when "2" then
      # FIXME
      return "prime-#{PRIME_VERSION} "
    when "3" then
      # FIXME
      return "localhost:#{@protnum} "
    else
      return ""
    end
  end
end


# POBox Protocol
# ----
# Close          "0"                     "<none>"
# GetWords       "1<query>"              "1/<word1>/<word2>/.../\n" or
#                                        "0\n" (error) or
#                                        "4\n" (no word)
# GetVersion     "2"                     "<major>.<minor> "
# GetHostName    "3"                     "<hostname>:<port> "
# SetContext     "4<context>"            "1"
# RegisterWord   "5<word>\t<pattern>"    "1"
# DeleteWord     "6<word>"               "1"
# SaveDict       "7"                     "1"
# SelectWord     "8<number>"             "1"

class SessionPOBox < SessionSKK
  def lookup (pattern)
    return @prime.lookup(pattern)
  end

  def execute (line)
    ## The detail of POBox protocol is here; 
    ## <http://pitecan.com/OpenPOBox/server/protocol.html>

    line.chomp!()

    case line[0,1]
    when "0" then
      return false
    when "1" then
      pattern = line.chomp[1..-1]
      results = lookup(pattern)
      if results.empty? then
        return "4" + pattern + "\n"
      else
        result_line = results.map {|result|
          result.to_text_literal()
        }.join("\t")
        return "1\t" + result_line + "\n"
      end
    when "2" then
      # FIXME
      return "prime-#{PRIME_VERSION} "
    when "3" then
      # FIXME
      return "localhost:#{@protnum} "
    when "4" then
      context = line.chomp[1..-1]
      if context == "" then
        @prime.set_context(nil)
      else
        @prime.set_context(context)
      end
      return "1 "
    when "5" then
      # Not implemented yet.
      return "1 "
    when "6" then
      # Not implemented yet.
      return "1 "
    when "7" then
      # Not implemented yet.
      return "1 "
    when "8" then
      # Not implemented yet.
      return "1 "
    else
      super
    end
  end
end
