#include <stdlib.h>
#include <stdio.h>
#include "esrh.h"

extern TCpuState    gCpuState;
extern TCpuFunction gCpuFunction;

// PC $B$+$i0z$-?t$G;XDj$5$l$k%P%$%H@h$N%G!<%?$rFI$`(B
unsigned char FetchNextByte( int aDist ){
  return gCpuState.iExtRam[gCpuState.iPc + aDist];
}
// $B0z$-?t$G;XDj$5$l$?FbIt(B RAM $B$+$i(B 1byte $BFI$`(B
unsigned char LoadInnerRamByte( unsigned char aAddr ){
  return gCpuState.iInnerRam[ aAddr ];
}
// $B0z$-?t$G;XDj$5$l$?FbIt(B RAM $B$+$i(B 1word $BFI$`(B
unsigned short LoadInnerRamWord( unsigned char aAddr ){
  return gCpuState.iInnerRam[aAddr] + (gCpuState.iInnerRam[aAddr+1]<<8);
}
// $BFbIt(B RAM $B$K(B 1byte $B=q$-9~$`(B
void StoreInnerRamByte( unsigned char aAddr, unsigned char aData ){
  gCpuState.iInnerRam[aAddr] = aData;
}
// $BFbIt(B RAM $B$K(B 1word $B=q$-9~$`(B
void StoreInnerRamWord( unsigned char aAddr, unsigned short aData ){
  gCpuState.iInnerRam[aAddr]   = aData&0xff;
  gCpuState.iInnerRam[aAddr+1] = aData>>8;
}
// $B30It(B RAM $B$+$i(B 1byte $BFI$`(B
#define LoadExtRamByte( aAddr ) (*gCpuFunction.iFuncReadExternalRam)( aAddr )
// $B30It(B RAM $B$+$i(B 1byte $BFI$`(B (PC)
#define LoadExtRamBytePc( aAddr ) (*gCpuFunction.iFuncReadExternalRamPc)( aAddr )

// $B30It(B RAM $B$K(B 1byte $B=q$-9~$`(B
#define StoreExtRamByte(a,d) (*gCpuFunction.iFuncWriteExternalRam)(a,d)

// $B30It(B RAM $B$+$i(B 1word $BFI$_9~$`(B(big-endian)
unsigned short LoadExtRamWord( unsigned short aAddr ){
  return ((*gCpuFunction.iFuncReadExternalRam)(aAddr)<<8)|
      (*gCpuFunction.iFuncReadExternalRam)(aAddr+1);
}
// $B30It(B RAM $B$+$i(B 1word $BFI$_9~$`(B(PC) (big-endian)
unsigned short LoadExtRamWordPc( unsigned short aAddr ){
  return ((*gCpuFunction.iFuncReadExternalRamPc)(aAddr)<<8)|
      (*gCpuFunction.iFuncReadExternalRamPc)(aAddr+1);
}
// $B30It(B RAM $B$K(B 1word $B=q$-9~$`(B(big-endian)
void StoreExtRamWord( unsigned short aAddr, unsigned short aData ){
  (*gCpuFunction.iFuncWriteExternalRam)(aAddr,aData>>8);
  (*gCpuFunction.iFuncWriteExternalRam)(aAddr+1,aData&0xff);
}

// 2$B?t$H%*!<%P!<%U%m!<$r(B BCD $B2C;;$9$k!#(B
int AddBcd( unsigned char a1, unsigned char a2, unsigned char aOv ){
  int retVal;
  retVal = (a1&0x0f)+(a2&0x0f)+aOv;
  if( retVal >= 0x0a ){
    retVal -= 0x0a;
    retVal += (a1&0xf0)+(a2&0xf0)+0x10;
  }else{
    retVal += (a1&0xf0)+(a2&0xf0);
  }
  if( retVal >= 0xa0 ){
    retVal = retVal-0xa0+0x100;
  }
  return retVal;
}
// 2$B?t$H%"%s%@!<%U%m!<$r(B BCD $B8:;;$9$k!#(B(a1-a2-aOv)
// $B$?$@$7Ez$,Ii$N>l9g$O(B +0x100 $B$5$l$F$$$k!#(B(16bit$BL\$,Id9f(B)
int SubBcd( unsigned char a1, unsigned char a2, unsigned char aOv ){
  int retVal;
  retVal = (a1&0x0f)-(a2&0x0f)-aOv;
  if( retVal < 0 ){
    retVal += 0x0a;
    retVal += (a1&0xf0)-(a2&0xf0)-0x10;
  }else{
    retVal += (a1&0xf0)-(a2&0xf0);
  }
  if( retVal < 0 ){
    retVal = retVal+0xa0+0x100;
  }
  return retVal;
}


void ExecLII( void ){
  gCpuState.iInnerRam[kRamI] = FetchNextByte(1);
  gCpuState.iPc += 2;
}

void ExecLIJ( void ){
  gCpuState.iInnerRam[kRamJ] = FetchNextByte(1);
  gCpuState.iPc += 2;
}

void ExecLIA( void ){
  gCpuState.iInnerRam[kRamA] = FetchNextByte(1);
  gCpuState.iPc += 2;
}

void ExecLIB( void ){
  gCpuState.iInnerRam[kRamB] = FetchNextByte(1);
  gCpuState.iPc += 2;
}

void ExecIX( void ){
  gCpuState.iDp = LoadInnerRamWord( kRamXl );
  gCpuState.iDp++;
  StoreInnerRamWord( kRamXl, gCpuState.iDp );
  gCpuState.iQ   = kRamXh;
  gCpuState.iPc += 1;
}
void ExecDX( void ){
  gCpuState.iDp = LoadInnerRamWord( kRamXl );
  gCpuState.iDp--;
  StoreInnerRamWord( kRamXl, gCpuState.iDp );
  gCpuState.iQ   = kRamXh;
  gCpuState.iPc += 1;
}
void ExecIY( void ){
  gCpuState.iDp = LoadInnerRamWord( kRamYl );
  gCpuState.iDp++;
  StoreInnerRamWord( kRamYl, gCpuState.iDp );
  gCpuState.iQ   = kRamYh;
  gCpuState.iPc += 1;
}
void ExecDY( void ){
  gCpuState.iDp = LoadInnerRamWord( kRamYl );
  gCpuState.iDp--;
  StoreInnerRamWord( kRamYl, gCpuState.iDp );
  gCpuState.iQ   = kRamYh;
  gCpuState.iPc += 1;
}

void ExecMVW(void){
  int len = gCpuState.iInnerRam[kRamI];
  do{
    StoreInnerRamByte( gCpuState.iP, LoadInnerRamByte( gCpuState.iQ ) );
    gCpuState.iP = (gCpuState.iP+1)&0x7f;
    gCpuState.iQ = (gCpuState.iQ+1)&0x7f;
  }while( len-- );
  gCpuState.iPc += 1;
}

void ExecEXW(void){
  int len = gCpuState.iInnerRam[kRamI];
  /*fprintf(stderr,"EXW I:%02x ->",gCpuState.iInnerRam[0] );
   */
  unsigned short tmp;
  do{
    /*fprintf(stderr,"P:%02x Q:%02x len:%d\n",gCpuState.iP,gCpuState.iQ,len );
     */
    tmp = LoadInnerRamByte( gCpuState.iP );
    StoreInnerRamByte( gCpuState.iP, LoadInnerRamByte( gCpuState.iQ ) );
    StoreInnerRamByte( gCpuState.iQ, tmp );
    gCpuState.iP = (gCpuState.iP+1)&0x7f;
    gCpuState.iQ = (gCpuState.iQ+1)&0x7f;
  }while( len-- );
  gCpuState.iPc += 1;
  /*fprintf(stderr,"%02x\n",gCpuState.iInnerRam[2] );
   */
  
}

void ExecMVB(void){
  int len = gCpuState.iInnerRam[kRamJ];
  do{
    StoreInnerRamByte( gCpuState.iP, LoadInnerRamByte( gCpuState.iQ ) );
    gCpuState.iP = (gCpuState.iP+1)&0x7f;
    gCpuState.iQ = (gCpuState.iQ+1)&0x7f;
  }while( len-- );
  gCpuState.iPc += 1;
}

void ExecEXB(void){
  int len = gCpuState.iInnerRam[kRamJ];
  unsigned short tmp;
  do{
    tmp = LoadInnerRamByte( gCpuState.iP );
    StoreInnerRamByte( gCpuState.iP, LoadInnerRamByte( gCpuState.iQ ) );
    StoreInnerRamByte( gCpuState.iQ, tmp );
    gCpuState.iP = (gCpuState.iP+1)&0x7f;
    gCpuState.iQ = (gCpuState.iQ+1)&0x7f;
  }while( len-- );
  gCpuState.iPc += 1;
}

void ExecADN(void){
  int len      = gCpuState.iInnerRam[kRamI];
  int overflow = 0;
  int isZero   = 1;
  int ans;
  gCpuState.iCFlag = 0;

  ans = AddBcd( gCpuState.iInnerRam[gCpuState.iP],
		gCpuState.iInnerRam[kRamA],
		0 );
  if( ans&0xff ) isZero = 0;
  overflow = ans>>8;
  gCpuState.iInnerRam[ gCpuState.iP ] = ans&0xff;
  gCpuState.iP = (gCpuState.iP-1)&0x7f;
  while( len-- ){
    ans = AddBcd( gCpuState.iInnerRam[gCpuState.iP],0,overflow );
    if( ans&0xff ) isZero = 0;
    overflow = ans>>8;
    gCpuState.iInnerRam[ gCpuState.iP ] = ans&0xff;
    gCpuState.iP = (gCpuState.iP-1)&0x7f;
  }
  if( overflow ) gCpuState.iCFlag = 1;
  gCpuState.iZFlag = isZero;
  gCpuState.iPc += 1;
}

void ExecSBN(void){
  int len      = gCpuState.iInnerRam[kRamI];
  int overflow = 0;
  int isZero   = 1;
  int ans;
  gCpuState.iCFlag = 0;

  ans = SubBcd( gCpuState.iInnerRam[gCpuState.iP],
	        gCpuState.iInnerRam[kRamA],
	        0 );
  if( ans&0xff ) isZero = 0;
  overflow = ans>>8;
  gCpuState.iInnerRam[ gCpuState.iP ] = ans&0xff;
  gCpuState.iP = (gCpuState.iP-1)&0x7f;
  while( len-- ){
    ans = SubBcd( gCpuState.iInnerRam[gCpuState.iP],0,overflow );
    if( ans&0xff ) isZero = 0;
    overflow = ans>>8;
    gCpuState.iInnerRam[ gCpuState.iP ] = ans&0xff;
    gCpuState.iP = (gCpuState.iP-1)&0x7f;
  }
  if( overflow ) gCpuState.iCFlag = 1;
  gCpuState.iZFlag = isZero;
  gCpuState.iPc += 1;
}

void ExecADW(void){
  int len      = gCpuState.iInnerRam[kRamI];
  int overflow = 0;
  int isZero   = 1;
  int ans;
  gCpuState.iCFlag = 0;
  do{
    ans = AddBcd( gCpuState.iInnerRam[gCpuState.iP],
		  gCpuState.iInnerRam[gCpuState.iQ],
		  overflow );
    gCpuState.iInnerRam[gCpuState.iP] = ans&0xff;
    if( ans&0xff ) isZero = 0;
    overflow = ans>>8;
    gCpuState.iP = (gCpuState.iP-1)&0x7f;
    gCpuState.iQ = (gCpuState.iQ-1)&0x7f;
  }while( len-- );
  if( overflow ) gCpuState.iCFlag = 1;
  gCpuState.iZFlag = isZero;
  gCpuState.iPc += 1;
}

void ExecSBW(void){
  int len      = gCpuState.iInnerRam[kRamI];
  int overflow = 0;
  int isZero   = 1;
  int ans;
  gCpuState.iCFlag = 0;
  do{
    ans = SubBcd( gCpuState.iInnerRam[gCpuState.iP],
		  gCpuState.iInnerRam[gCpuState.iQ],
		  overflow );
    gCpuState.iInnerRam[gCpuState.iP] = ans&0xff;
    if( ans&0xff ) isZero = 0;
    overflow = ans>>8;
    gCpuState.iP = (gCpuState.iP-1)&0x7f;
    gCpuState.iQ = (gCpuState.iQ-1)&0x7f;
  }while( len-- );
  if( overflow ) gCpuState.iCFlag = 1;
  gCpuState.iZFlag = isZero;
  gCpuState.iPc += 1;
}

void ExecLIDP(void){
  gCpuState.iDp = (FetchNextByte(1)<<8) + FetchNextByte(2);
  gCpuState.iPc += 3;
}

void ExecLIDL(void){
  gCpuState.iDp &= 0xff00;
  gCpuState.iDp |= FetchNextByte(1);
  gCpuState.iPc += 2;
}
  
void ExecLIP(void){
  gCpuState.iP = FetchNextByte(1) & 0x7f;
  gCpuState.iPc += 2;
}

void ExecLIQ(void){
  gCpuState.iQ = FetchNextByte(1) & 0x7f;
  gCpuState.iPc += 2;
}

void ExecADB(void){
  int ab,p;
  gCpuState.iCFlag = 0;
  gCpuState.iZFlag = 0;
  ab = LoadInnerRamWord(kRamA);
  p  = LoadInnerRamWord( gCpuState.iP );
  p += ab;
  StoreInnerRamWord( gCpuState.iP, p&0xffff );
  if( p>0xffff )  gCpuState.iCFlag = 1;
  if(!(p&0xffff)) gCpuState.iZFlag = 1;
  gCpuState.iP = (gCpuState.iP+1)&0x7f;
  gCpuState.iPc += 1;
}

void ExecSBB(void){
  int ab,p;
  gCpuState.iCFlag = 0;
  gCpuState.iZFlag = 0;
  ab = LoadInnerRamWord(kRamA);
  p  = LoadInnerRamWord( gCpuState.iP );
  p -= ab;
  StoreInnerRamWord( gCpuState.iP, p&0xffff );
  if( p<0 )  gCpuState.iCFlag = 1;
  if( p==0 ) gCpuState.iZFlag = 1;
  gCpuState.iP = (gCpuState.iP+1)&0x7f;
  gCpuState.iPc += 1;
}

void ExecMVWD(void){
  int len = gCpuState.iInnerRam[kRamI];
  do{
    StoreInnerRamByte( gCpuState.iP, LoadExtRamByte(gCpuState.iDp) );
    gCpuState.iP = (gCpuState.iP+1)&0x7f;
    gCpuState.iDp= (gCpuState.iDp+1)&0xffff;
  }while( len-- );
  gCpuState.iPc += 1;
}

void ExecEXWD(void){
  int len      = gCpuState.iInnerRam[kRamI];
  unsigned char tmp;
  do{
    tmp = LoadExtRamByte(gCpuState.iDp);
    StoreExtRamByte( gCpuState.iDp, LoadInnerRamByte( gCpuState.iP ) );
    StoreInnerRamByte( gCpuState.iP, tmp );
    gCpuState.iP = (gCpuState.iP+1)&0x7f;
    gCpuState.iDp= (gCpuState.iDp+1)&0xffff;
  }while( len-- );
  gCpuState.iPc += 1;
}

void ExecMVBD(void){
  int len      = gCpuState.iInnerRam[kRamJ];
  do{
    StoreInnerRamByte( gCpuState.iP, LoadExtRamByte(gCpuState.iDp) );
    gCpuState.iP = (gCpuState.iP+1)&0x7f;
    gCpuState.iDp= (gCpuState.iDp+1)&0xffff;
  }while( len-- );
  gCpuState.iPc += 1;
}

void ExecEXBD(void){
  int len      = gCpuState.iInnerRam[kRamJ];
  unsigned char tmp;
  do{
    tmp = LoadExtRamByte(gCpuState.iDp);
    StoreExtRamByte( gCpuState.iDp, LoadInnerRamByte( gCpuState.iP ) );
    StoreInnerRamByte( gCpuState.iP, tmp );
    gCpuState.iP = (gCpuState.iP+1)&0x7f;
    gCpuState.iDp= (gCpuState.iDp+1)&0xffff;
  }while( len-- );
  gCpuState.iPc += 1;
}

void ExecSRW(void){
  int len      = gCpuState.iInnerRam[kRamI];
  unsigned char tmp,tmp2;
  tmp2 = 0;
  do{
    tmp = LoadInnerRamByte( gCpuState.iP );
    StoreInnerRamByte( gCpuState.iP, (tmp>>4)|tmp2 );
    tmp2 = tmp<<4;
    gCpuState.iP = (gCpuState.iP+1)&0x7f;
  }while( len-- );
  gCpuState.iPc += 1;
}

void ExecSLW(void){
  int len      = gCpuState.iInnerRam[kRamI];
  unsigned char tmp,tmp2;
  tmp2 = 0;
  do{
    tmp = LoadInnerRamByte( gCpuState.iP );
    StoreInnerRamByte( gCpuState.iP, (tmp<<4)|tmp2 );
    tmp2 = tmp>>4;
    gCpuState.iP = (gCpuState.iP-1)&0x7f;
  }while( len-- );
  gCpuState.iPc += 1;
}

void ExecFILM(void){
  int len      = gCpuState.iInnerRam[kRamI];
  do{
    StoreInnerRamByte( gCpuState.iP, gCpuState.iInnerRam[kRamA] );
    gCpuState.iP = (gCpuState.iP+1)&0x7f;
  }while( len-- );
  gCpuState.iPc += 1;
}

void ExecFILD(void){
  int len      = gCpuState.iInnerRam[kRamI];
  do{
    StoreExtRamByte( gCpuState.iDp, gCpuState.iInnerRam[kRamA] );
    gCpuState.iDp = (gCpuState.iDp+1)&0xffff;
  }while( len-- );
  gCpuState.iPc += 1;
}

void ExecLDP(void){
  gCpuState.iInnerRam[kRamA] = gCpuState.iP;
  gCpuState.iPc += 1;
}
void ExecLDQ(void){
  gCpuState.iInnerRam[kRamA] = gCpuState.iQ;
  gCpuState.iPc += 1;
}
void ExecLDR(void){
  gCpuState.iInnerRam[kRamA] = gCpuState.iR;
  gCpuState.iPc += 1;
}
void ExecCLRA(void){
  gCpuState.iInnerRam[kRamA] = 0;
  gCpuState.iPc += 1;
}

void ExecIXL(void){
  gCpuState.iDp = LoadInnerRamWord(kRamXl);
  gCpuState.iDp++;
  StoreInnerRamWord( kRamXl, gCpuState.iDp );
  StoreInnerRamByte( kRamA,  LoadExtRamByte( gCpuState.iDp ) );
  gCpuState.iQ = kRamXh;
  gCpuState.iPc += 1;
}

void ExecDXL(void){
  gCpuState.iDp = LoadInnerRamWord(kRamXl);
  gCpuState.iDp--;
  StoreInnerRamWord( kRamXl, gCpuState.iDp );
  StoreInnerRamByte( kRamA,  LoadExtRamByte( gCpuState.iDp ) );
  gCpuState.iQ = 5;
  gCpuState.iPc += 1;
}

void ExecIYS(void){
  gCpuState.iDp = LoadInnerRamWord(kRamYl);
  gCpuState.iDp++;
  StoreInnerRamWord( kRamYl, gCpuState.iDp );
  StoreExtRamByte( gCpuState.iDp,  gCpuState.iInnerRam[kRamA] );
  gCpuState.iQ = 7;
  gCpuState.iPc += 1;
}

void ExecDYS(void){
  gCpuState.iDp = LoadInnerRamWord(kRamYl);
  gCpuState.iDp--;
  StoreInnerRamWord( kRamYl, gCpuState.iDp );
  StoreExtRamByte( gCpuState.iDp,  gCpuState.iInnerRam[kRamA] );
  gCpuState.iQ = 7;
  gCpuState.iPc += 1;
}

void ExecJRNZP(void){
  gCpuState.iPc += gCpuState.iZFlag ? 2 : FetchNextByte(1)+1;
}
void ExecJRNZM(void){
  gCpuState.iPc += gCpuState.iZFlag ? 2 : -FetchNextByte(1)+1;
}
void ExecJRNCP(void){
  gCpuState.iPc += gCpuState.iCFlag ? 2 : FetchNextByte(1)+1;
}
void ExecJRNCM(void){
  gCpuState.iPc += gCpuState.iCFlag ? 2 : -FetchNextByte(1)+1;
}
void ExecJRP(void){
  gCpuState.iPc += FetchNextByte(1)+1;
}
void ExecJRM(void){
  gCpuState.iPc += -FetchNextByte(1)+1;
}
void ExecLOOP(void){
  gCpuState.iPc += gCpuState.iInnerRam[gCpuState.iR]?-FetchNextByte(1)+1:2;
  gCpuState.iInnerRam[ gCpuState.iR ]--;
  if( gCpuState.iInnerRam[gCpuState.iR]==0xff ){
    gCpuState.iR++;
  }
}
void ExecSTP(void){
  gCpuState.iP = gCpuState.iInnerRam[ kRamA ]&0x7f;
  gCpuState.iPc += 1;
}
void ExecSTQ(void){
  gCpuState.iQ = gCpuState.iInnerRam[ kRamA ]&0x7f;
  gCpuState.iPc += 1;
}
void ExecSTR(void){
  gCpuState.iR = gCpuState.iInnerRam[ kRamA ]&0x7f;
  gCpuState.iPc += 1;
}

void ExecNOPT(void){
  gCpuState.iPc += 1;
}

void ExecPUSH(void){
  gCpuState.iR = (gCpuState.iR-1)&0x7f;
  StoreInnerRamByte( gCpuState.iR, gCpuState.iInnerRam[ kRamA ] );
  gCpuState.iPc += 1;
}

void ExecMVWP(void){ // also known as DATA or RST
  int len = gCpuState.iInnerRam[kRamI];
  unsigned short ba;
  ba = LoadInnerRamWord( kRamA );
  do{
    StoreInnerRamByte( gCpuState.iP, LoadExtRamBytePc( ba++ ) );
//    StoreInnerRamWord( kRamA, ba );
    gCpuState.iP = ( gCpuState.iP+1 )&0x7f;
  }while( len-- );
  gCpuState.iPc += 1;
}

void ExecRTN( void ){
  gCpuState.iPc = LoadInnerRamWord( gCpuState.iR );
  gCpuState.iR = (gCpuState.iR+2)&0x7f;
}

void ExecJRZP(void){
  gCpuState.iPc += gCpuState.iZFlag ?  FetchNextByte(1)+1 : 2;
}
void ExecJRZM(void){
  gCpuState.iPc += gCpuState.iZFlag ?  -FetchNextByte(1)+1 : 2;
}
void ExecJRCP(void){
  gCpuState.iPc += gCpuState.iCFlag ?  FetchNextByte(1)+1 : 2;
}
void ExecJRCM(void){
  gCpuState.iPc += gCpuState.iCFlag ?  -FetchNextByte(1)+1 : 2;
}

void ExecINCI(void){
  gCpuState.iInnerRam[kRamI]++;
  gCpuState.iCFlag = gCpuState.iZFlag =
      gCpuState.iInnerRam[kRamI]?0:1;
  gCpuState.iQ = kRamI;
  gCpuState.iPc += 1;
}

void ExecDECI(void){
  gCpuState.iCFlag = gCpuState.iInnerRam[kRamI]?0:1;
  gCpuState.iInnerRam[kRamI]--;
  gCpuState.iZFlag = gCpuState.iInnerRam[kRamI]?0:1;
  gCpuState.iQ = kRamI;
  gCpuState.iPc += 1;
}

void ExecINCA(void){
  gCpuState.iInnerRam[kRamA]++;
  gCpuState.iCFlag = gCpuState.iZFlag =
      gCpuState.iInnerRam[kRamA]?0:1;
  gCpuState.iQ = kRamA;
  gCpuState.iPc += 1;
}

void ExecDECA(void){
  gCpuState.iCFlag = gCpuState.iInnerRam[kRamA]?0:1;
  gCpuState.iInnerRam[kRamA]--;
  gCpuState.iZFlag = gCpuState.iInnerRam[kRamA]?0:1;
  gCpuState.iQ = kRamA;
  gCpuState.iPc += 1;
}

void ExecADM(void){
  int ans = LoadInnerRamByte( gCpuState.iP );
  ans += gCpuState.iInnerRam[kRamA];
  StoreInnerRamByte( gCpuState.iP,(unsigned char)ans&0xff );
  gCpuState.iZFlag = (ans&0xff)?0:1;
  gCpuState.iCFlag = (ans>0xff)?1:0;
  gCpuState.iPc += 1;
}

void ExecSBM(void){
  unsigned short ans = LoadInnerRamByte( gCpuState.iP );
  ans -= gCpuState.iInnerRam[kRamA];
  StoreInnerRamByte( gCpuState.iP,(unsigned char)(ans&0xff) );
  gCpuState.iZFlag = (ans&0xff)?0:1;
  gCpuState.iCFlag = (ans&0xff00)?1:0;
  gCpuState.iPc += 1;
}

void ExecANMA(void){
  unsigned char ans = LoadInnerRamByte( gCpuState.iP );
  ans &= gCpuState.iInnerRam[kRamA];
  StoreInnerRamByte( gCpuState.iP, ans );
  gCpuState.iZFlag = ans?0:1;
  gCpuState.iPc += 1;
}

void ExecORMA(void){
  unsigned char ans = LoadInnerRamByte( gCpuState.iP );
  ans |= gCpuState.iInnerRam[kRamA];
  StoreInnerRamByte( gCpuState.iP, ans );
  gCpuState.iZFlag = ans?0:1;
  gCpuState.iPc += 1;
}

void ExecINCK(void){
  gCpuState.iInnerRam[kRamK]++;
  gCpuState.iCFlag = gCpuState.iZFlag =
    gCpuState.iInnerRam[kRamK]?0:1;
  gCpuState.iQ = kRamK;
  gCpuState.iPc += 1;
}

void ExecDECK(void){
  gCpuState.iCFlag = gCpuState.iInnerRam[kRamK]?0:1;
  gCpuState.iInnerRam[kRamK]--;
  gCpuState.iZFlag = gCpuState.iInnerRam[kRamK]?0:1;
  gCpuState.iQ = kRamK;
  gCpuState.iPc += 1;
}

void ExecINCM(void){
  gCpuState.iInnerRam[kRamM]++;
  gCpuState.iCFlag = gCpuState.iZFlag =
    gCpuState.iInnerRam[kRamM]?0:1;
  gCpuState.iQ = kRamM;
  gCpuState.iPc += 1;
}

void ExecDECM(void){
  gCpuState.iCFlag = gCpuState.iInnerRam[kRamM]?0:1;
  gCpuState.iInnerRam[kRamM]--;
  gCpuState.iZFlag = gCpuState.iInnerRam[kRamM]?0:1;
  gCpuState.iQ = kRamM;
  gCpuState.iPc += 1;
}

void ExecINA( void ){
  gCpuState.iInnerRam[kRamA] = (*gCpuFunction.iFuncReadPortA)();
  gCpuState.iZFlag = (gCpuState.iInnerRam[kRamA])?0:1;
  gCpuState.iPc += 1;
}

void ExecNOPW(void){
  gCpuState.iPc += 1;
}

void ExecWAIT(void){
  gCpuState.iPc += 2;
}

void ExecCUP(void){
  int len = gCpuState.iInnerRam[kRamI];
  gCpuState.iZFlag = 1;
  do{
    gCpuState.iP = (gCpuState.iP+1)&0x7f;
    if( (*gCpuFunction.iFuncReadXin)() ){
      gCpuState.iZFlag = 0;
      break;
    }
  }while(len--);
  gCpuState.iPc += 1;
}

void ExecINCP(void){
  gCpuState.iP = (gCpuState.iP+1)&0x7f;
  gCpuState.iPc += 1;
}

void ExecDECP(void){
  gCpuState.iP = (gCpuState.iP-1)&0x7f;
  gCpuState.iPc += 1;
}

void ExecSTD(void){
  StoreExtRamByte( gCpuState.iDp, gCpuState.iInnerRam[kRamA] );
  gCpuState.iPc += 1;
}

void ExecMVDM(void){
  StoreExtRamByte( gCpuState.iDp, LoadInnerRamByte( gCpuState.iP ) );
  gCpuState.iPc += 1;
}

void ExecMVMP(void){  // (P):=(PC+1)
  StoreInnerRamByte( gCpuState.iP, FetchNextByte(1) );
  gCpuState.iPc += 1;
}

void ExecMVMD(void){
  StoreInnerRamByte( gCpuState.iP, LoadExtRamByte(gCpuState.iDp) );
  gCpuState.iPc += 1;
}

void ExecLDPC(void){  // A:=(PC+1)
  gCpuState.iInnerRam[kRamA]=FetchNextByte(1);
  gCpuState.iPc += 1;
}

void ExecLDD(void){
  gCpuState.iInnerRam[kRamA]=LoadExtRamByte(gCpuState.iDp);
  gCpuState.iPc += 1;
}

void ExecSWP(void){
  gCpuState.iInnerRam[kRamA]=
      (gCpuState.iInnerRam[kRamA]<<4)|(gCpuState.iInnerRam[kRamA]>>4);
  gCpuState.iPc += 1;
}

void ExecLDM(void){
  gCpuState.iInnerRam[kRamA]=LoadInnerRamByte(gCpuState.iP);
  gCpuState.iPc += 1;
}

void ExecSL(void){
  int tmpC = gCpuState.iCFlag;
  gCpuState.iCFlag = gCpuState.iInnerRam[kRamA]>>7;
  gCpuState.iInnerRam[kRamA]=(gCpuState.iInnerRam[kRamA]<<1)+tmpC;
  gCpuState.iPc += 1;
}

void ExecPOP(void){
  gCpuState.iInnerRam[kRamA]=LoadInnerRamByte( gCpuState.iR );
  gCpuState.iR = (gCpuState.iR+1)&0x7f;
  gCpuState.iPc += 1;
}

void ExecOUTA(void){
  (*(gCpuFunction.iFuncWritePortA))(LoadInnerRamByte(kRamPortA));
  gCpuState.iPc += 1;
}

void ExecOUTF(void){
  (*(gCpuFunction.iFuncWritePortF))(LoadInnerRamByte(kRamPortF));
  gCpuState.iPc += 1;
}

void ExecANIM(void){
  unsigned char tmp;
  tmp = LoadInnerRamByte( gCpuState.iP ) & FetchNextByte(1);
  StoreInnerRamByte( gCpuState.iP, tmp );
  gCpuState.iZFlag = tmp?0:1;
  gCpuState.iPc += 2;
}
  
void ExecORIM(void){
  unsigned char tmp;
  tmp = LoadInnerRamByte( gCpuState.iP );
  tmp |= FetchNextByte(1);
  StoreInnerRamByte( gCpuState.iP, tmp );
  gCpuState.iZFlag = tmp?0:1;
  gCpuState.iPc += 2;
}
  
void ExecTSIM(void){
  gCpuState.iZFlag = (LoadInnerRamByte(gCpuState.iP)&FetchNextByte(1))?0:1;
  gCpuState.iPc += 2;
}

void ExecCPIM(void){
  unsigned char p = LoadInnerRamByte(gCpuState.iP);
  unsigned char n = FetchNextByte(1);
  gCpuState.iZFlag = (p==n)?1:0;
  gCpuState.iCFlag = (p<n)?1:0;
  gCpuState.iPc += 2;
}

void ExecANIA(void){
  gCpuState.iZFlag=(gCpuState.iInnerRam[kRamA]&=FetchNextByte(1))?0:1;
  gCpuState.iPc += 2;
}

void ExecORIA(void){
  gCpuState.iZFlag=(gCpuState.iInnerRam[kRamA]|=FetchNextByte(1))?0:1;
  gCpuState.iPc += 2;
}
  
void ExecTSIA(void){
  gCpuState.iZFlag=(gCpuState.iInnerRam[kRamA]&FetchNextByte(1))?0:1;
  gCpuState.iPc += 2;
}

void ExecCPIA(void){
  unsigned char a = gCpuState.iInnerRam[kRamA];
  unsigned char n = FetchNextByte(1);
  gCpuState.iZFlag = (a==n)?1:0;
  gCpuState.iCFlag = (a<n)?1:0;
  gCpuState.iPc += 2;
}

void ExecCASE1(void){
  gCpuState.iCaseNum = FetchNextByte(1);
  gCpuState.iR = (gCpuState.iR-2)&0x7f;
  StoreInnerRamWord( gCpuState.iR, LoadExtRamWordPc(gCpuState.iPc+2) );
  gCpuState.iPc += 4;
}

void ExecCASE2(void){
  int i;
  unsigned short dest;
  dest=LoadExtRamWordPc(gCpuState.iPc+3*gCpuState.iCaseNum+1);
  for(i=0; i<gCpuState.iCaseNum; i++){
    if(gCpuState.iInnerRam[kRamA]==LoadExtRamBytePc(gCpuState.iPc+3*i+1)){
      dest=LoadExtRamWordPc(gCpuState.iPc+3*i+2);
      break;
    }
  }
  gCpuState.iPc = dest;
}

void ExecTEST(void){
  gCpuState.iZFlag=( (*(gCpuFunction.iFuncReadTest))()&FetchNextByte(1) )?0:1;
  gCpuState.iPc += 2;
}

void ExecCDN(void){
  int len = gCpuState.iInnerRam[kRamI];
  gCpuState.iZFlag = 1;
  do{
    gCpuState.iP = (gCpuState.iP+1)&0x7f;
    if( !(*gCpuFunction.iFuncReadXin)() ){
      gCpuState.iZFlag = 0;
      break;
    }
  }while(len--);
  gCpuState.iPc += 1;
}

void ExecADIM(void){
  int ans;
  ans = LoadInnerRamByte(gCpuState.iP)+FetchNextByte(1);
  gCpuState.iZFlag = (ans&0xff)?0:1;
  gCpuState.iCFlag = (ans>0xff)?1:0;
  StoreInnerRamByte(gCpuState.iP,ans&0xff);
  gCpuState.iPc += 2;
}

void ExecSBIM(void){
  int ans;
  ans = LoadInnerRamByte(gCpuState.iP)-FetchNextByte(1);
  gCpuState.iZFlag = (ans&0xff)?0:1;
  gCpuState.iCFlag = (ans<0)?1:0;
  StoreInnerRamByte(gCpuState.iP,ans&0xff);
  gCpuState.iPc += 2;
}

void ExecADIA(void){
  int ans;
  ans = gCpuState.iInnerRam[kRamA]+FetchNextByte(1);
  gCpuState.iZFlag = (ans&0xff)?0:1;
  gCpuState.iCFlag = (ans>0xff)?1:0;
  gCpuState.iInnerRam[kRamA]=ans;
  gCpuState.iPc += 2;
}

void ExecSBIA(void){
  int ans;
  ans = gCpuState.iInnerRam[kRamA]-FetchNextByte(1);
  gCpuState.iZFlag = (ans&0xff)?0:1;
  gCpuState.iCFlag = (ans<0)?1:0;
  gCpuState.iInnerRam[kRamA]=ans;
  gCpuState.iPc += 2;
}

void ExecCALL(void){
  gCpuState.iR=(gCpuState.iR-2)&0x7f;
  StoreInnerRamWord(gCpuState.iR,gCpuState.iPc+3);
  gCpuState.iPc=LoadExtRamWordPc(gCpuState.iPc+1);
}

void ExecCAL(void){
  gCpuState.iR=(gCpuState.iR-2)&0x7f;
  StoreInnerRamWord(gCpuState.iR,gCpuState.iPc+2);
  gCpuState.iPc=LoadExtRamWordPc(gCpuState.iPc)-0xe000;
}

void ExecJP(void){
  gCpuState.iPc=LoadExtRamWordPc(gCpuState.iPc+1);
}

void ExecJPNZ(void){
  if(!gCpuState.iZFlag){
    gCpuState.iPc=LoadExtRamWordPc(gCpuState.iPc+1);
  }else{
    gCpuState.iPc+=3;
  }
}

void ExecJPNC(void){
  if(!gCpuState.iCFlag){
    gCpuState.iPc=LoadExtRamWordPc(gCpuState.iPc+1);
  }else{
    gCpuState.iPc+=3;
  }
}

void ExecJPZ(void){
  if(gCpuState.iZFlag){
    gCpuState.iPc=LoadExtRamWordPc(gCpuState.iPc+1);
  }else{
    gCpuState.iPc+=3;
  }
}

void ExecJPC(void){
  if(gCpuState.iCFlag){
    gCpuState.iPc=LoadExtRamWordPc(gCpuState.iPc+1);
  }else{
    gCpuState.iPc+=3;
  }
}

void ExecLP(void){
  gCpuState.iP=gCpuState.iCurrentInstruction-0x80;
  gCpuState.iPc += 1;
}

void ExecINCJ(void){
  gCpuState.iInnerRam[kRamJ]++;
  gCpuState.iCFlag = gCpuState.iZFlag =
      gCpuState.iInnerRam[kRamJ]?0:1;
  gCpuState.iQ = kRamJ;
  gCpuState.iPc += 1;
}

void ExecDECJ(void){
  gCpuState.iCFlag = gCpuState.iInnerRam[kRamJ]?0:1;
  gCpuState.iInnerRam[kRamJ]--;
  gCpuState.iZFlag = gCpuState.iInnerRam[kRamJ]?0:1;
  gCpuState.iQ = kRamJ;
  gCpuState.iPc += 1;
}

void ExecINCB(void){
  gCpuState.iInnerRam[kRamB]++;
  gCpuState.iCFlag = gCpuState.iZFlag =
      gCpuState.iInnerRam[kRamB]?0:1;
  gCpuState.iQ = kRamB;
  gCpuState.iPc += 1;
}

void ExecDECB(void){
  gCpuState.iCFlag = gCpuState.iInnerRam[kRamB]?0:1;
  gCpuState.iInnerRam[kRamB]--;
  gCpuState.iZFlag = gCpuState.iInnerRam[kRamB]?0:1;
  gCpuState.iQ = kRamB;
  gCpuState.iPc += 1;
}

void ExecADCM(void){
  int ans;
  ans = gCpuState.iInnerRam[kRamA]
        +LoadInnerRamByte(gCpuState.iP)
	+gCpuState.iCFlag;
  gCpuState.iCFlag=(ans>0xff)?1:0;
  gCpuState.iZFlag=(ans&0xff)?0:1;
  StoreInnerRamByte(gCpuState.iP,ans&0xff);
  gCpuState.iPc += 1;
}

void ExecSBCM(void){
  int ans;
  ans = LoadInnerRamByte(gCpuState.iP)
        -gCpuState.iInnerRam[kRamA]
	-gCpuState.iCFlag;
  gCpuState.iCFlag=(ans<0   )?1:0;
  gCpuState.iZFlag=(ans&0xff)?0:1;
  StoreInnerRamByte(gCpuState.iP,ans&0xff);
  gCpuState.iPc += 1;
}

void ExecTSMA(void){
  gCpuState.iZFlag = (LoadInnerRamByte(gCpuState.iP)&gCpuState.iInnerRam[kRamA])?0:1;
  gCpuState.iPc += 1;
}

void ExecCPMA(void){
  int ans;
  ans = LoadInnerRamByte(gCpuState.iP)-gCpuState.iInnerRam[kRamA];
  gCpuState.iCFlag = (ans<0   )?1:0;
  gCpuState.iZFlag = (ans&0xff)?0:1;
  gCpuState.iPc += 1;
}
  
void ExecINCL(void){
  gCpuState.iInnerRam[kRamL]++;
  gCpuState.iCFlag = gCpuState.iZFlag =
      gCpuState.iInnerRam[kRamL]?0:1;
  gCpuState.iQ = kRamL;
  gCpuState.iPc += 1;
}

void ExecDECL(void){
  gCpuState.iCFlag = gCpuState.iInnerRam[kRamL]?0:1;
  gCpuState.iInnerRam[kRamL]--;
  gCpuState.iZFlag = gCpuState.iInnerRam[kRamL]?0:1;
  gCpuState.iQ = kRamL;
  gCpuState.iPc += 1;
}

void ExecINCN(void){
  gCpuState.iInnerRam[kRamN]++;
  gCpuState.iCFlag = gCpuState.iZFlag =
      gCpuState.iInnerRam[kRamN]?0:1;
  gCpuState.iQ = kRamN;
  gCpuState.iPc += 1;
}

void ExecDECN(void){
  gCpuState.iCFlag = gCpuState.iInnerRam[kRamN]?0:1;
  gCpuState.iInnerRam[kRamN]--;
  gCpuState.iZFlag = gCpuState.iInnerRam[kRamN]?0:1;
  gCpuState.iQ = kRamN;
  gCpuState.iPc += 1;
}

void ExecINB(void){
  gCpuState.iInnerRam[kRamA] = (*gCpuFunction.iFuncReadPortB)();
  gCpuState.iZFlag = (gCpuState.iInnerRam[kRamA])?0:1;
  gCpuState.iPc += 1;
}

void ExecSC(void){
  gCpuState.iCFlag = 1;
  gCpuState.iZFlag = 1;
  gCpuState.iPc += 1;
}

void ExecRC(void){
  gCpuState.iCFlag = 0;
  gCpuState.iZFlag = 1;
  gCpuState.iPc += 1;
}

void ExecSR(void){
  int tmpC = gCpuState.iCFlag<<7;
  gCpuState.iCFlag = gCpuState.iInnerRam[kRamA]&1;
  gCpuState.iInnerRam[kRamA]=(gCpuState.iInnerRam[kRamA]>>1)|tmpC;
  gCpuState.iPc += 1;
}

void ExecANID(void){
  // $BK\Ev$O%9%?%C%/$r(B1$BCJ>CHq$9$k$i$7$$(B
  unsigned char ans;
  ans = LoadExtRamByte( gCpuState.iDp ) & FetchNextByte(1);
  StoreExtRamByte( gCpuState.iDp,ans );
  gCpuState.iZFlag = ans?0:1;
  gCpuState.iPc += 2;
}

void ExecORID(void){
  // $BK\Ev$O%9%?%C%/$r(B1$BCJ>CHq$9$k$i$7$$(B
  unsigned char ans;
  ans = LoadExtRamByte( gCpuState.iDp ) | FetchNextByte(1);
  StoreExtRamByte( gCpuState.iDp,ans );
  gCpuState.iZFlag = ans?0:1;
  gCpuState.iPc += 2;
}

void ExecTSID(void){
  // $BK\Ev$O%9%?%C%/$r(B1$BCJ>CHq$9$k$i$7$$(B
  gCpuState.iZFlag =(LoadExtRamByte( gCpuState.iDp )&FetchNextByte(1))?0:1;
  gCpuState.iPc += 2;
}

void ExecLEAVE(void){
  StoreInnerRamByte( gCpuState.iR, 0 );
  gCpuState.iPc += 1;
}

void ExecEXAB(void){
  unsigned char tmp;
  tmp = gCpuState.iInnerRam[kRamA];
  gCpuState.iInnerRam[kRamA] = gCpuState.iInnerRam[kRamB];
  gCpuState.iInnerRam[kRamB] = tmp;
  gCpuState.iPc += 1;
}

void ExecEXAM(void){
  unsigned char tmp;
  tmp = gCpuState.iInnerRam[kRamA];
  gCpuState.iInnerRam[kRamA] = LoadInnerRamByte( gCpuState.iP );
  StoreInnerRamByte( gCpuState.iP, tmp );
  gCpuState.iPc += 1;
}

void ExecOUTB(void){
  (*(gCpuFunction.iFuncWritePortB))(LoadInnerRamByte(kRamPortB));
  gCpuState.iPc += 1;
}

void ExecOUTC(void){
  (*(gCpuFunction.iFuncWritePortC))(LoadInnerRamByte(kRamPortC));
  gCpuState.iPc += 1;
}

void Panic( void ){
  printf( "\nESR-H Panic!\n"
	  " PC:%04x  DP:%04x   Instructuion:%02x\n"
	  " P:%02x Q:%02x R:%02x\n"
	  ,gCpuState.iPc, gCpuState.iDp,
	  gCpuState.iCurrentInstruction,
	  gCpuState.iP,gCpuState.iQ,gCpuState.iR );
  exit(0);
}


void (*gExecuteInst[])(void)={
  // 0x00,
  ExecLII, ExecLIJ, ExecLIA, ExecLIB, ExecIX,  ExecDX,  ExecIY,  ExecDY,
  ExecMVW, ExecEXW, ExecMVB, ExecEXB, ExecADN, ExecSBN, ExecADW, ExecSBW,
  // 0x10,
  ExecLIDP,ExecLIDL,ExecLIP, ExecLIQ, ExecADB, ExecSBB, Panic,   Panic,
  ExecMVWD,ExecEXWD,ExecMVBD,ExecEXBD,ExecSRW, ExecSLW, ExecFILM,ExecFILD,
  // 0x20,
  ExecLDP, ExecLDQ, ExecLDR, ExecCLRA,ExecIXL, ExecDXL, ExecIYS, ExecDYS,
  ExecJRNZP,ExecJRNZM,ExecJRNCP,ExecJRNCM,ExecJRP,ExecJRM,Panic, ExecLOOP,
  // 0x30,
  ExecSTP, ExecSTQ, ExecSTR, ExecNOPT,ExecPUSH,ExecMVWP,Panic,   ExecRTN,
  ExecJRZP,ExecJRZM,ExecJRCP,ExecJRCM,Panic,   Panic,   Panic,   Panic,
  // 0x40,
  ExecINCI,ExecDECI,ExecINCA,ExecDECA,ExecADM, ExecSBM, ExecANMA,ExecORMA,
  ExecINCK,ExecDECK,ExecINCM,ExecDECM,ExecINA, ExecNOPW,ExecWAIT,ExecCUP,
  // 0x50,
  ExecINCP,ExecDECP,ExecSTD, ExecMVDM,ExecMVMP,ExecMVMD,ExecLDPC,ExecLDD,
  ExecSWP, ExecLDM, ExecSL,  ExecPOP, Panic,   ExecOUTA,Panic,   ExecOUTF,
  // 0x60
  ExecANIM,ExecORIM,ExecTSIM,ExecCPIM,ExecANIA,ExecORIA,ExecTSIA,ExecCPIA,
  ExecNOPT,ExecCASE2,ExecNOPT,ExecTEST,Panic,  Panic,   Panic,   ExecCDN,
  // 0x70
  ExecADIM,ExecSBIM,Panic,   Panic,   ExecADIA,ExecSBIA,Panic,   Panic,
  ExecCALL,ExecJP,  ExecCASE1,Panic,  ExecJPNZ,ExecJPNC,ExecJPZ, ExecJPC,
  // 0x80 - 0xaf
  ExecLP,  ExecLP,  ExecLP,  ExecLP,  ExecLP,  ExecLP,  ExecLP,  ExecLP,  
  ExecLP,  ExecLP,  ExecLP,  ExecLP,  ExecLP,  ExecLP,  ExecLP,  ExecLP,  
  ExecLP,  ExecLP,  ExecLP,  ExecLP,  ExecLP,  ExecLP,  ExecLP,  ExecLP,  
  ExecLP,  ExecLP,  ExecLP,  ExecLP,  ExecLP,  ExecLP,  ExecLP,  ExecLP,  
  ExecLP,  ExecLP,  ExecLP,  ExecLP,  ExecLP,  ExecLP,  ExecLP,  ExecLP,  
  ExecLP,  ExecLP,  ExecLP,  ExecLP,  ExecLP,  ExecLP,  ExecLP,  ExecLP,  
  // 0xb0
  ExecLP,  ExecLP,  ExecLP,  ExecLP,  ExecLP,  ExecLP,  ExecLP,  ExecLP,  
  ExecLP,  ExecLP,  ExecLP,  ExecLP,  ExecLP,  ExecLP,  ExecLP,  ExecLP,  
  // 0xc0
  ExecINCJ,ExecDECJ,ExecINCB,ExecDECB,ExecADCM,ExecSBCM,ExecTSMA,ExecCPMA,
  ExecINCL,ExecDECL,ExecINCN,ExecDECN,ExecINB, ExecNOPW,ExecNOPT,Panic,
  // 0xd0
  ExecSC,  ExecRC,  ExecSR,  ExecNOPW,ExecANID,ExecORID,ExecTSID,Panic,
  ExecLEAVE,ExecNOPW,ExecEXAB,ExecEXAM,Panic,  ExecOUTB,Panic,   ExecOUTC,
  // 0xe0 - 0xff
  ExecCAL, ExecCAL, ExecCAL, ExecCAL, ExecCAL, ExecCAL, ExecCAL, ExecCAL, 
  ExecCAL, ExecCAL, ExecCAL, ExecCAL, ExecCAL, ExecCAL, ExecCAL, ExecCAL, 
  ExecCAL, ExecCAL, ExecCAL, ExecCAL, ExecCAL, ExecCAL, ExecCAL, ExecCAL, 
  ExecCAL, ExecCAL, ExecCAL, ExecCAL, ExecCAL, ExecCAL, ExecCAL, ExecCAL
};

unsigned char ReadPortDummy(void){
  return 0;
}
#ifdef __cplusplus
void WritePortDummy(unsigned char /*aData*/){
#else
void WritePortDummy(unsigned char aData){
#endif
}

void esrhInitialize(void){
  gCpuFunction.iFuncReadPortA  = ReadPortDummy;
  gCpuFunction.iFuncReadPortB  = ReadPortDummy;
  gCpuFunction.iFuncReadXin    = ReadPortDummy;
  gCpuFunction.iFuncReadTest   = ReadPortDummy;
  gCpuFunction.iFuncWritePortA = WritePortDummy;
  gCpuFunction.iFuncWritePortB = WritePortDummy;
  gCpuFunction.iFuncWritePortC = WritePortDummy;
  gCpuFunction.iFuncWritePortF = WritePortDummy;
  esrhReset();
}

void esrhReset( void ){
  gCpuState.iPc = 0x0000;
  gCpuState.iZFlag = 0;
  gCpuState.iCFlag = 0;
}

void esrhAssignExtRam( unsigned char *aRamTopAddress ){
  gCpuState.iExtRam = aRamTopAddress;
}

void FetchAndExecute( void ){
  gCpuState.iCurrentInstruction = gCpuState.iExtRam[ gCpuState.iPc ];
  (*(gExecuteInst[gCpuState.iCurrentInstruction]))();
}

void esrhAssignFuncReadPortA( unsigned char (*aFunc)(void) ){
  gCpuFunction.iFuncReadPortA = aFunc;
}
void esrhAssignFuncReadPortB( unsigned char (*aFunc)(void) ){
  gCpuFunction.iFuncReadPortB = aFunc;
}
void esrhAssignFuncReadPortXin( unsigned char (*aFunc)(void) ){
  gCpuFunction.iFuncReadXin = aFunc;
}
void esrhAssignFuncReadPortTest( unsigned char (*aFunc)(void) ){
  gCpuFunction.iFuncReadTest = aFunc;
}
void esrhAssignFuncWritePortA( void (*aFunc)(unsigned char aData) ){
  gCpuFunction.iFuncWritePortA = aFunc;
}
void esrhAssignFuncWritePortB( void (*aFunc)(unsigned char aData) ){
  gCpuFunction.iFuncWritePortB = aFunc;
}
void esrhAssignFuncWritePortC( void (*aFunc)(unsigned char aData) ){
  gCpuFunction.iFuncWritePortC = aFunc;
}
void esrhAssignFuncWritePortF( void (*aFunc)(unsigned char aData) ){
  gCpuFunction.iFuncWritePortF = aFunc;
}

void esrhAssignFuncWriteExternalRam( void (*aFunc)(unsigned short aAddress,
						   unsigned char aData) ){
  gCpuFunction.iFuncWriteExternalRam = aFunc;
}

void esrhAssignFuncReadExternalRam( unsigned char (*aFunc)(unsigned short aAddress) ){
  gCpuFunction.iFuncReadExternalRam = aFunc;
}
void esrhAssignFuncReadExternalRamPc( unsigned char (*aFunc)(unsigned short aAddress) ){
  gCpuFunction.iFuncReadExternalRamPc = aFunc;
}

void esrhRun( void ){
  FetchAndExecute();
}


