/*
 * Copyright (c) 2009 OrangeSignal.com All rights reserved.
 * 
 * これは Apache ライセンス Version 2.0 (以下、このライセンスと記述) に
 * 従っています。このライセンスに準拠する場合以外、このファイルを使用
 * してはなりません。このライセンスのコピーは以下から入手できます。
 * 
 * http://www.apache.org/licenses/LICENSE-2.0.txt
 * 
 * 適用可能な法律がある、あるいは文書によって明記されている場合を除き、
 * このライセンスの下で配布されているソフトウェアは、明示的であるか暗黙の
 * うちであるかを問わず、「保証やあらゆる種類の条件を含んでおらず」、
 * 「あるがまま」の状態で提供されるものとします。
 * このライセンスが適用される特定の許諾と制限については、このライセンス
 * を参照してください。
 */

package jp.sf.orangesignal.csv.handlers;

import java.io.IOException;
import java.lang.reflect.Field;

import jp.sf.orangesignal.csv.CsvListHandler;

/**
 * Java プログラム要素のリストと区切り文字形式データアクセスを行うハンドラの基底クラスを提供します。
 * 
 * @author 杉澤 浩二
 */
public abstract class BeanListHandlerSupport<T> implements CsvListHandler<T> {

	/**
	 * Java プログラム要素の型を保持します。
	 */
	private Class<T> type;

	/**
	 * 区切り文字形式データの項目値コンバータを保持します。
	 */
	private CsvValueConverter converter = new SimpleCsvValueConverter();

	/**
	 * コンストラクタです。
	 * 
	 * @param type Java プログラム要素の型
	 * @throws IllegalArgumentException <code>type</code> が <code>null</code> の場合
	 */
	protected BeanListHandlerSupport(final Class<T> type) {
		if (type == null) {
			throw new IllegalArgumentException("Class must not be null");
		}
		this.type = type;
	}

	// ------------------------------------------------------------------------

	/**
	 * 区切り文字形式データの項目値コンバータを設定します。
	 * 
	 * @param converter 区切り文字形式データの項目値コンバータ
	 * @return このオブジェクトへの参照
	 * @throws IllegalArgumentException <code>converter</code> が <code>null</code> の場合
	 */
	public BeanListHandlerSupport<T> converter(final CsvValueConverter converter) {
		if (converter == null) {
			throw new IllegalArgumentException("CsvValueConverter must not be null");
		}
		this.converter = converter;
		return this;
	}

	/**
	 * 区切り文字形式データの項目値コンバータを返します。
	 * 
	 * @return 区切り文字形式データの項目値コンバータ
	 */
	public CsvValueConverter getConverter() { return converter; }

	/**
	 * Java プログラム要素の型を返します。
	 * 
	 * @return Java プログラム要素の型
	 */
	public Class<T> getType() { return type; }

	// ------------------------------------------------------------------------

	protected T createBean() throws IOException {
		try {
			return type.newInstance();
		} catch (InstantiationException e) {
			throw new IOException(e.getMessage(), e);
		} catch (IllegalAccessException e) {
			throw new IOException(e.getMessage(), e);
		}
	}

	// ------------------------------------------------------------------------

	protected static Field getField(final Class<?> type, final String name) throws IOException {
		try {
			return type.getDeclaredField(name);
		} catch (SecurityException e) {
			throw new IOException(e.getMessage(), e);
		} catch (NoSuchFieldException e) {
			throw new IOException(e.getMessage(), e);
		}
	}

	protected static void setFieldValue(final Object obj, final Field f, final Object value) throws IOException {
		if (!f.isAccessible()) {
			f.setAccessible(true);
		}
		try {
			f.set(obj, value);
		} catch (IllegalArgumentException e) {
			throw new IOException(e.getMessage(), e);
		} catch (IllegalAccessException e) {
			throw new IOException(e.getMessage(), e);
		}
	}

	protected static Object getFieldValue(final Object obj, final Field f) throws IOException {
		if (!f.isAccessible()) {
			f.setAccessible(true);
		}
		try {
			return f.get(obj);
		} catch (IllegalArgumentException e) {
			throw new IOException(e.getMessage(), e);
		} catch (IllegalAccessException e) {
			throw new IOException(e.getMessage(), e);
		}
	}

}
