/*
 * Copyright (c) 2006-2009 OrangeSignal.com All rights reserved.
 */

package jp.sourceforge.orangesignal.trading.order;

import java.io.Serializable;
import java.util.Date;

/**
 * 注文情報の基底クラスを提供します。
 * 
 * @author 杉澤 浩二
 */
abstract class AbstractOrder implements Order, Serializable {

	private static final long serialVersionUID = -833003275626738986L;

	/**
	 * デフォルトコンストラクタです。
	 */
	protected AbstractOrder() {}

	/**
	 * コンストラクタです。
	 * 
	 * @param symbol シンボル
	 * @param label ラベル
	 * @param date 注文日時
	 * @param quantity 数量
	 * @param findId 決済注文ID
	 * @param findLabel 決済注文ラベル
	 * @throws NullPointerException パラメーターに <code>null</code> を指定した場合
	 */
	protected AbstractOrder(
			final String symbol,
			final String label,
			final Date date,
			final int quantity,
			final Integer findId,
			final String findLabel)
	{
		if (date == null)
			throw new NullPointerException();
		this.symbol = symbol;
		this.label = label;
		this.date = date;
		this.quantity = quantity;
		this.findId = findId;
		this.findLabel = findLabel;
	}

	/**
	 * シンボルを保持します。
	 */
	private String symbol;

	@Override public String getSymbol() { return symbol == null ? "" : symbol; }

	/**
	 * ラベルを保持します。
	 */
	private String label;

	@Override public final String getLabel() { return label; }

	/**
	 * 注文日時を保持します。
	 */
	private Date date;

	@Override public final Date getDate() { return date; }

	/**
	 * 注文数量を保持します。
	 */
	private int quantity;

	@Override public final int getQuantity() { return quantity; }

	/**
	 * 決済注文IDを保持します。
	 */
	private Integer findId;

	@Override public Integer getFindId() { return findId; }

	/**
	 * 決済注文IDを設定します。
	 * 
	 * @param findId 決済注文ID
	 */
	public void setFindId(final Integer findId) { this.findId = findId; }

	/**
	 * 決済注文ラベルを保持します。
	 */
	private String findLabel;

	@Override public String getFindLabel() { return findLabel; }

	/**
	 * 決済注文ラベルを設定します。
	 * 
	 * @param findLabel 決済注文ラベル
	 */
	public void setFindLabel(final String findLabel) { this.findLabel = findLabel; }

	@Override public final int compareTo(final Order o) { return date.compareTo(o.getDate()); }

}
