/*-
 * Copyright (C) 2008 Speecys Corporation
 *		Toshiaki Nozawa <nozawa@speecys.com>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *        This product includes software developed by the Speecys Corporation.
 * 4. Neither the name of The Speecys Corporation nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE SPEECYS CORPORATION AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE CORPORATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <assert.h>
#include <string.h>

#include "libutil.h"
#include "libservo.h"
#include "rs485devs.h"

size_t
editSrvRst(
    u_char id,		/** dev ID	*/
    size_t bufLen,	/** length of buf	*/
    u_char buf[bufLen]	/** edit buf	*/
)
{
    size_t len;

    assert(id <= (u_char)MaxRs485Servo);
    len = rs485_make_packet(buf, id, (u_char)0x20, (u_char)0xff, (u_char)0
	, (u_char)0, NULL);
    assert(len <= bufLen);
    return len;
}

size_t
editSrvAng(
    bool isLong,	/**
    			 * true: edit long-packet data,
			 *
			 * false: edit short-packet data with rs485_make_packet
			 */
    u_char id,		/** dev ID.	*/
    uint16_t angle,	/** angle(deci-deg)	*/
    uint16_t time,	/** time(centi-sec)	*/
    size_t bufLen,	/** length of buf	*/
    u_char buf[bufLen]	/** edit buf	*/
)
{
    u_char	    data[4];
    size_t len;

    assert(id <= (u_char)MaxRs485Servo);

    /* make packet */
    (void)set485devShort(&data[0], angle);
    (void)set485devShort(&data[2], time);

    if (isLong) {
	len = sizeof id + sizeof data;
	buf[0] = id;
	memcpy(&buf[1], data, sizeof data);
    } else {
	len = rs485_make_packet(buf, id, (u_char)0, (u_char)SMM_GPos_L
	    , (u_char)sizeof data, (u_char)1, data);
    }
    assert(len <= bufLen);
    return len;
}

size_t
editSrvPwr(
    bool isLong,	/**
    			 * true: edit long-packet data,
			 *
			 * false: edit short-packet data with rs485_make_packet
			 */
    u_char id,		/** dev ID	*/
    u_char sw,		/** power on/off switch	*/
    size_t bufLen,	/** length of buf	*/
    u_char buf[bufLen]	/** edit buf	*/
)
{
    size_t len;

    assert(id <= (u_char)MaxRs485Servo);
    if (isLong) {
	len = sizeof id + sizeof sw;
	buf[0] = id;
	buf[1] = sw;
    } else {
	len = rs485_make_packet(buf, id, (u_char)0, (u_char)SMM_EnblTorque
	    , (u_char)sizeof sw, (u_char)1, &sw);
    }
    assert(len <= bufLen);
    return len;
}

size_t
editSrvCmgn(
    bool isLong,	/**
    			 * true: edit long-packet data,
			 *
			 * false: edit short-packet data with rs485_make_packet
			 */
    u_char id,		/** dev ID	*/
    u_char mgn,		/** compliance mergin	*/
    size_t bufLen,	/** length of buf	*/
    u_char buf[bufLen]	/** edit buf	*/
)
{
    u_char data[2] = {mgn, mgn};
    size_t len;

    assert(id <= (u_char)MaxRs485Servo);
    if (isLong) {
	len = sizeof id + sizeof data;
	buf[0] = id;
	memcpy(&buf[1], data, len);
    } else {
	len = rs485_make_packet(buf, id, (u_char)0, (u_char)SMM_MarginCW
	    , (u_char)sizeof data, (u_char)1, data);
    }
    assert(len <= bufLen);
    return len;
}


size_t
editSrvCslp(
    bool isLong,	/**
    			 * true: edit long-packet data,
			 *
			 * false: edit short-packet data with rs485_make_packet
			 */
    u_char id,		/** dev ID	*/
    u_char slp,		/** compliance slope	*/
    size_t bufLen,	/** length of buf	*/
    u_char buf[bufLen]	/** edit buf	*/
)
{
    u_char	    data[2] = {slp, slp};
    size_t len;

    assert(id <= (u_char)MaxRs485Servo);
    if (isLong) {
	len = sizeof id + sizeof data;
	buf[0] = id;
	memcpy(&buf[1], data, len);
    } else {
	len = rs485_make_packet(buf, id, (u_char)0, (u_char)SMM_SlopeCW
	    , (u_char)sizeof data, (u_char)1, data);
    }
    assert(len <= bufLen);
    return len;
}

size_t
editSrvPunch(
    bool isLong,	/**
    			 * true: edit long-packet data,
			 *
			 * false: edit short-packet data with rs485_make_packet
			 */
    u_char id,		/** dev ID	*/
    uint16_t pnc,	/** punch	*/
    size_t bufLen,	/** length of buf	*/
    u_char buf[bufLen]	/** edit buf	*/
)
{
    u_char	    data[sizeof pnc];
    size_t len;

    assert(id <= (u_char)MaxRs485Servo);
    (void)set485devShort(data, pnc);
    if (isLong) {
	len = sizeof id + sizeof data;
	buf[0] = id;
	memcpy(&buf[1], data, len);
    } else {
	len = rs485_make_packet(buf, id, (u_char)0, (u_char)SMM_Punch_L
	    , (u_char)sizeof data, (u_char)1, data);
    }
    assert(len <= bufLen);
    return len;
}

size_t
editSrvParams(
    bool isLong,	/**
    			 * true: edit long-packet data,
			 *
			 * false: edit short-packet data with rs485_make_packet
			 */
    u_char id,		/** dev ID	*/
    u_char mgn,		/** compliance mergin	*/
    u_char slp,		/** compliance slope	*/
    u_short pnc,	/** punch	*/
    size_t bufLen,	/** length of buf	*/
    u_char buf[bufLen]	/** edit buf	*/
)
{
    const size_t b = SMM_Resv23;
    u_char data[SMM_Punch_H - b];
    size_t len;

    assert(id <= (u_char)MaxRs485Servo);

    data[SMM_MarginCW - b] = mgn;
    data[SMM_MarginCCW - b] = mgn;
    data[SMM_SlopeCW - b] = slp;
    data[SMM_SlopeCCW - b] = slp;
    (void)set485devShort(&data[SMM_Punch_L - b], pnc);

    if (isLong) {
	len = sizeof id + sizeof data;
	buf[0] = id;
	memcpy(&buf[1], data, len);
    } else {
	len = rs485_make_packet(buf, id, (u_char)0, (u_char)SMM_MarginCW
	    , (u_char)sizeof data, (u_char)1, data);
    }
    assert(len <= bufLen);
    return len;
}

size_t
editSrvTrq(
    bool isLong,	/**
    			 * true: edit long-packet data,
			 *
			 * false: edit short-packet data with rs485_make_packet
			 */
    u_char id,		/** dev ID	*/
    u_char trq,		/** torque	*/
    size_t bufLen,	/** length of buf	*/
    u_char buf[bufLen]	/** edit buf	*/
)
{
    size_t len;

    assert(id <= (u_char)MaxRs485Servo);
    if (isLong) {
	len = sizeof id + sizeof trq;
	buf[0] = id;
	buf[1] = trq;
    } else {
	len = rs485_make_packet(buf, id, (u_char)0, (u_char)SMM_MaxTorque
	    , (u_char)sizeof trq, (u_char)1, &trq);
    }
    assert(len <= bufLen);
    return len;
}

size_t
editPwmSrvAng(
    u_char id,		/** dev ID	*/
    short angle,	/** angle(deci-deg)	*/
    u_short time,	/** time(centi-sec)	*/
    size_t bufLen,	/** length of buf	*/
    u_char buf[bufLen]	/** edit buf	*/
)
{
    u_char srvMv[2];
    static const u_char bias = (u_char)0x40;
    static const u_char maxDeg = bias * (u_char)2;
    static const u_char maxTim = (u_char)0x64;
    size_t len;

    assert(id == (u_char)rs485ID_LED_LEFT || id == (u_char)rs485ID_LED_RIGHT);

    if ((srvMv[0] = (u_char)(angle / 10 + bias)) > maxDeg) {
	//clipping wrist servo angle
	srvMv[0] = (angle > 0) ? maxDeg : (u_char)0;
    }
    srvMv[1] = (u_char)(time / 10);

    //clipping wrist servo move time
    if (srvMv[1] == (u_char)0)		srvMv[1] = (u_char)1;
    if (srvMv[1] > maxTim)	srvMv[1] = maxTim;

    //move
    len = rs485_make_packet(buf, id, (u_char)0, (u_char)LMM_Position, (u_char)2
	, (u_char)1, srvMv);
    assert(len <= bufLen);

    return len;
}
