/*
 * OpenI2CRADIO
 * Power switch.
 * Copyright (C) 2013-06-10 K.Ohta <whatisthis.sowhat ai gmail.com>
 * License: GPL2+LE
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2,
 *  or (at your option) any later version.
 *  This library / program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *  See the GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this library; see the file COPYING. If not, write to the
 *  Free Software Foundation, 51 Franklin Street, Fifth Floor, Boston,
 *  MA 02110-1301, USA.
 *
 *  As a special exception, if you link this(includeed from sdcc) library
 *  with other files, some of which are compiled with SDCC,
 *  to produce an executable, this library does not by itself cause
 *  the resulting executable to be covered by the GNU General Public License.
 *  This exception does not however invalidate any other reasons why
 *  the executable file might be covered by the GNU General Public License.
 */

#include "commondef.h"
#include "lcd_acm1602.h"
#include "power.h"

/*
 * Detect reset condition.
 */
unsigned char chk_reset(void)
{

#ifdef __SDCC
    union {
        __RCONbits_t b;
        unsigned char byte;
    } rcon;
    union {
        __STKPTRbits_t b;
        unsigned char byte;
    } stkptr;
#else
    union {
        RCONbits_t b;
        unsigned char byte;
    } rcon;
    union {
        STKPTRbits_t b;
        unsigned char byte;
    } stkptr;
#endif
    rcon.byte = RCON;
    stkptr.byte = STKPTR;
    RCON |= 0b01010011; // SBOREN,*RI,*POR,*BOR = '1'.
    if((rcon.b.POR ==0) && (rcon.b.BOR == 0)) {
        return RESET_POR;
    }

    if(rcon.b.BOR == 0) return RESET_BOR;
    if(rcon.b.RI == 0) return RESET_SOFTWARE;
    if(rcon.b.TO == 0) return RESET_WDT;
    if(rcon.b.PD == 0) return RESET_POWERDOWN;
    if(stkptr.b.STKOVF == 1) return RESET_STACK_FULL;
    if(stkptr.b.STKUNF == 1) return RESET_STACK_UNDER;
    return RESET_MCLR;
}

void power_on_inthook(void)
{

    IOCB = 0x00;
    IOCB |= 0x10; // IOCB4 ONLY.
#ifdef __SDCC
//    INTCON &= ~(_RBIF | _RBIE);
//    INTCON |= _GIE;
//    INTCON |= (_GIE | _PEIE);
    INTCONbits.RBIE = 0;
    INTCONbits.RBIF = 0;
    INTCONbits.GIE = 1;
    INTCONbits.PEIE = 1;

#else
    INTCON &= ~(_INTCON_RBIF_MASK | _INTCON_RBIE_MASK);
//    INTCON |= _GIE;
    INTCON |= (_INTCON_GIE_MASK | _INTCON_PEIE_MASK);
#endif
}

void power_on(unsigned char f)
{
    if(f == 0x00){
        set_examp_mute(1);
        idle_time_ms(200);
        set_radiopower(0);
    } else {
        set_examp_mute(0);
        idle_time_ms(100);
        set_radiopower(1);
    }
}

void rfamp_power(unsigned char b)
{
    __bitops_t bx;
    
    bx.byte = b;
    if(bx.b0) {
        _RFAMP_SEL_B0 = 1;
    } else {
        _RFAMP_SEL_B0 = 0;
    }
    if(bx.b1) {
        _RFAMP_SEL_B1 = 1;
    } else {
        _RFAMP_SEL_B1 = 0;
    }

}

unsigned char chk_powerbutton(void)
{
    unsigned char count = 0;
    char i;
    for(i = 0; i < 24; i++) {  // 10*32=320ms.
        if(PORTBbits.RB4 == 0) count++;
        _POWER_DELAY();
        ClrWdt();
    }
    if(count > 15) {
        return 0xff; // Pressed
    }
    return 0; // Not pressed
}

void shutdown(unsigned char save)
{
    _CLS();
    _LOCATE(0,0);
//    printstr("Bye...");
    idle_time_ms(1500); // 1.5Sec
    lcd_setbacklight(0, 0);
    set_powerlamp(0);
    _AM_STATLAMP = 0;
    _FM_STATLAMP = 0;
//    _CLS();
    acm1602_suspend(LCD_I2CADDR);
    power_off(save);
}

void power_off(unsigned char save)
{
    unsigned char sts;
    if(save != 0) save_eeprom();
    WDTCONbits.SWDTEN = 0; // Lame WDT OFF.

    do {
        IOCB = 0x00;
        IOCB |= 0x10; // IOCB4 ONLY.
        OSCCONbits.IDLEN = 0; // Not Idle.
        INTCON2bits.RBIP = 1; // Priority = HIGH;
        INTCONbits.RBIF = 0;
        INTCONbits.GIE = 0;
        INTCONbits.PEIE = 0;
        INTCONbits.RBIE = 1;
        power_on(0);
        Sleep();
        // Wake up
        if(INTCONbits.RBIF == 1) {
            sts = chk_powerbutton();
        } else {
            sts = 0;
        }
    } while(sts == 0);
    INTCONbits.RBIF = 0;
    INTCONbits.RBIE = 0;

    INTCONbits.GIE = 1;
    INTCONbits.PEIE = 1;

    // Button pressed, software reset.
    Reset();
}