/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.model;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.nio.charset.CharacterCodingException;					// 6.3.1.0 (2015/06/28)

import org.opengion.fukurou.system.OgRuntimeException ;				// 6.4.2.0 (2016/01/29)
import org.opengion.fukurou.system.OgCharacterException ;			// 6.5.0.1 (2016/10/21)
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.FileUtil;
import org.opengion.fukurou.system.Closer;							// 6.2.0.0 (2015/02/27)
import static org.opengion.fukurou.system.HybsConst.CR;
import static org.opengion.fukurou.system.HybsConst.BUFFER_MIDDLE;

/**
 * 指定の区切り記号(初期値:タブ区切り)ファイルの読み取りクラスです。
 *
 * txt形式のテキストファイルを、イベント方式でテキストデータを読み取ります。
 * タブ区切りテキストファイルで、セパレータと文字コードを外部から指定できます。
 *
 * @og.rev 6.2.0.0 (2015/02/27) 新規追加
 * @og.group ファイル入力
 *
 * @version  6.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK6.0,
 */
public class EventReader_TEXT implements EventReader {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.5.0.1 (2016/10/21)" ;

	private String	separator	= "\t";			// 項目区切り文字
	private String	encode		= "UTF-8";		// Shift_JIS,MS932,Windows-31J,UTF-8,ISO-8859-1,UnicodeLittle･･･

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public EventReader_TEXT() { super(); }		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。

	/**
	 * 引数ファイル(Text)を、イベントモデルを使用してテキスト化します。
	 *
	 * TableModelHelperは、EXCEL/TEXT読み取り処理用の統一されたイベント処理クラスです。
	 * openGion特有のEXCEL/TEXT処理方法(#NAME , 先頭行#コメントなど)を実装しています。
	 * これは、EXCELやTEXTの処理を、統一的なイベントモデルで扱うためです。
	 * あくまで、読み取り限定であれば、こちらのイベントモデルで十分です。
	 *
	 * @og.rev 6.2.0.0 (2015/02/27) 新規作成
	 * @og.rev 6.3.1.0 (2015/06/28) nioを使用すると UTF-8とShuft-JISで、エラーになる。
	 * @og.rev 6.5.0.1 (2016/10/21) CharacterCodingException は、OgCharacterException に変換する。
	 *
	 * @param	file 入力ファイル
	 * @param	helper イベント処理するオブジェクト
	 */
	@Override	// EventReader
	public void eventReader( final File file , final TableModelHelper helper ) {
		BufferedReader	reader	= null ;

		try {
			helper.startFile( file );

			// 6.2.0.0 (2015/02/27) TableModelHelper 変更に伴う修正
			reader = FileUtil.getBufferedReader( file,encode );

			String line;
			int rowNo = 0;
			final char sepa = separator.charAt(0);

			final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );
			while((line = reader.readLine()) != null) {
				if( helper.isSkip( rowNo ) ) { break; }

				// 5.2.0.0 (2010/09/01) ""で囲われているデータに改行が入っていた場合の対応
				int quotCount = StringUtil.countChar( line, '"' );
				if( quotCount % 2 != 0 ) {
					String addLine = null;
					buf.setLength(0);
					buf.append( line );
					while(quotCount % 2 != 0 && (addLine = reader.readLine()) != null) {
						buf.append( CR ).append( addLine );
						quotCount += StringUtil.countChar( addLine, '"' );
					}
					line = buf.toString();
				}
				helper.value( line, rowNo++, sepa );
	//			// オーバーフロー時は、HybsOverflowException が throw される。
	//			if( helper.isOverflow() ) { break; }		// オーバーフローで打ち切り
			}
		}
		// 6.3.1.0 (2015/06/28) nioを使用すると UTF-8とShuft-JISで、エラーになる。
		catch( final CharacterCodingException ex ) {
			final String errMsg = "文字のエンコード・エラーが発生しました。" + CR
								+	"  ファイルのエンコードが指定のエンコードと異なります。" + CR
								+	" [" + file.getPath() + "] , Encode=[" + encode + "]" ;
			throw new OgCharacterException( errMsg,ex );	// 6.5.0.1 (2016/10/21)
		}
		catch( final IOException ex ) {
			final String errMsg = "ファイル読込みエラーが発生しました。" + CR
								+	" [" + file.getPath() + "] , Encode=[" + encode + "]" ;
			throw new OgRuntimeException( errMsg,ex );
		}
		finally {
			Closer.ioClose( reader );
			helper.endFile( file );					// 6.2.0.0 (2015/02/27)
		}
	}

	/**
	 * データを読み込む場合の,区切り文字をセットします。
	 *
	 * なお，このメソッドは,サブクラスによっては,使用しない場合があります。
	 * もし，使用しないサブクラスを作成する場合は, UnsupportedOperationException
	 * を throw するように,サブクラスで実装して下さい。
	 *
	 * @param   sep 区切り文字
	 */
	public void setSeparator( final String sep ) {
		if( sep != null ) { separator = sep; }
	}

	/**
	 * 読み取り元ファイルのエンコード文字列を指定します。
	 * ファイルは、BufferedReader で受け取る為、本来は、エンコードは不要ですが、
	 * 固定長ファイルの読み取り時のバイトコード分割時に、指定のエンコードで
	 * 分割する必要があります。(例えば、半角文字は、Shift_JIS では、１バイト)
	 *
	 * @param   enc ファイルのエンコード文字列
	 */
	public void setEncode( final String enc ) {
		encode = enc;
	}
}
