/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.fukurou.util.XHTMLTag;
import org.opengion.fukurou.util.ToString;						// 6.1.1.0 (2015/01/17)
import org.opengion.fukurou.util.ArraySet;						// 6.4.3.4 (2016/03/11)

import java.util.Set;											// 6.4.3.4 (2016/03/11)

/**
 * ボタンを作成するHTML拡張タグです。
 *
 * ボタンごとに必要な画面を選び出して制御を転送することができます。
 * BODY 部分にラベルや強調文字などのタグを記述すれば、そのままボタンとして表示されます。
 * [ヒント]
 * buttonタグとinputタグが生成するボタンは、機能的には同等ですが、buttonタグの方が
 * レタリング能力があり、内容を持つことができます。
 * 例えば、画像を内容に持つbutton機能は、type属性がimageのinputタグと
 * そっくり同じですが、buttonタグは内容を持つことができます。
 *
 * @og.formSample
 * ●形式：&lt;og:button type="…" name="…" value="…" lbl="…" /&gt;
 * ●body：あり(EVAL_BODY_BUFFERED:BODYを評価し、{&#064;XXXX} を解析します)
 *
 * ●Tag定義：
 *   &lt;og:button
 *       type             ○【HTML】ボタンのタイプ[submit/reset/button]を指定します(必須)。
 *       name               【HTML】ボタンの名称を指定します
 *       value              【HTML】ボタンの値を指定します
 *       caseKey            【TAG】このタグ自体を利用するかどうかの条件キーを指定します(初期値:null)
 *       caseVal            【TAG】このタグ自体を利用するかどうかの条件値を指定します(初期値:null)
 *       caseNN             【TAG】指定の値が、null/ゼロ文字列 でない場合(Not Null=NN)は、このタグは使用されます(初期値:判定しない)
 *       caseNull           【TAG】指定の値が、null/ゼロ文字列 の場合は、このタグは使用されます(初期値:判定しない)
 *       caseIf             【TAG】指定の値が、true/TRUE文字列の場合は、このタグは使用されます(初期値:判定しない)
 *       lbl                【TAG】ラベルリソースのラベルIDを指定します
 *       id                 【HTML】要素に対して固有の名前(id)をつける場合に設定します
 *       lang               【HTML】要素の内容と他の属性値の言語(lang,xml:lang)を指定します
 *       dir                【HTML】文字表記の方向(dir)を指定します
 *       title              【HTML】要素に対する補足的情報(title)を設定します
 *       style              【HTML】この要素に対して適用させるスタイルシート(style)を設定します
 *       disabled           【TAG】その部品に対して、選択や変更が出来ないように(disabled)指定します(サーバーに送信されない)
 *       tabindex           【HTML】タブの移動順(tabindex)を指定します(0 ～ 32767)
 *       accesskey          【HTML】アクセスキー(alt+キーで直接指定)を割り当てます
 *       clazz              【HTML】要素に対して class 属性を設定します
 *       language           【TAG】タグ内部で使用する言語コード[ja/en/zh/…]を指定します
 *       onClick            【HTML】JavaScriptのイベント onClick を設定します(例:onClick="renew('query.jsp','QUERY');")
 *       onBlur             【HTML】JavaScriptのイベント onBlur を設定します(例:onBlur="this.value=value.toUpperCase();")
 *       onFocus            【HTML】JavaScriptのイベント onFocus を設定します
 *       ondblClick         【HTML】JavaScriptのイベント ondblClick を設定します
 *       onMouseDown        【HTML】JavaScriptのイベント onMouseDown を設定します
 *       onMouseUp          【HTML】JavaScriptのイベント onMouseUp を設定します
 *       onMouseMove        【HTML】JavaScriptのイベント onMouseMove を設定します
 *       onMouseOut         【HTML】JavaScriptのイベント onMouseOut を設定します
 *       onMouseOver        【HTML】JavaScriptのイベント onMouseOver を設定します
 *       autofocus          【HTML5】指定した入力欄にカーソルが当たって自動的にフォーカスされます。
 *       roles              【TAG】ロールをセットします
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   &gt;   ... Body ...
 *   &lt;/og:button&gt;
 *
 * ●使用例
 * &lt;og:button type="submit" name="test" value="1" lbl="test" /&gt;     LabelResource プロパティのラベル使用したいとききはlbl属性に、LabelResourceを使います。
 * &lt;og:button type="submit" name="test" value="1" msg="MSG0103" /&gt;  MessageResource プロパティのラベル使用したいときはmsg属性を使います。
 *
 * @og.group 画面部品
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class ButtonTag extends HTMLTagSupport {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.4.3.4 (2016/03/11)" ;
	private static final long serialVersionUID = 643420160311L ;

	// 6.3.4.0 (2015/08/01) String から String配列 に置き換え
	// 6.4.3.4 (2016/03/11) String配列 から、Setに置き換えます。
	private static final Set<String> TYPE_SET = new ArraySet<>( "submit" , "reset" , "button" );

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public ButtonTag() { super(); }		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。ボディが存在する場合のみボディが呼ばれる対応。
	 * @og.rev 5.2.2.0 (2010/11/01) caseKey 、caseVal 属性対応
	 *
	 * @return	後続処理の指示( EVAL_BODY_BUFFERED )
	 */
	@Override
	public int doStartTag() {
		// 5.2.2.0 (2010/11/01) caseKey 、caseVal 属性対応
		if( useTag() ) {
			set( "body",getLongLabel() );
			return EVAL_BODY_BUFFERED ;	// Body を評価する。( extends BodyTagSupport 時)
		}
		return SKIP_BODY ;				// Body を評価しない
	}

	/**
	 * Taglibのタグ本体を処理する doAfterBody() を オーバーライドします。
	 *
	 * @og.rev 3.1.1.0 (2003/03/28) ボディの内容を取得する処理を、CommonTagSupport で行う。
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。ボディが存在する場合のみボディが呼ばれる対応。
	 *
	 * @return	後続処理の指示(SKIP_BODY)
	 */
	@Override
	public int doAfterBody() {
		final String label = getBodyString();

		if( label != null && label.length() > 0 ) {
			set( "body",label );
		}

		return SKIP_BODY ;
	}

	/**
	 * ボタンを作成します。
	 *
	 * &lt;button type="submit" name="名前" value="値"&gt;
	 *
	 * @return  ボタンタグ文字列
	 * @og.rtnNotNull
	 */
	@Override
	protected String makeTag() {
		return XHTMLTag.button( getAttributes() );
	}

	/**
	 * 【HTML】ボタンのタイプ[submit/reset/button]を指定します。
	 *
	 * @og.tag
	 *
	 * submit:提出ボタンを作成します。
	 * reset:リセットボタンを作成します。
	 * button:押しボタンを作成します。
	 *
	 * @og.rev 6.3.4.0 (2015/08/01) Arrays.toString から String.join に置き換え。
	 * @og.rev 6.4.3.4 (2016/03/11) String配列 から、Setに置き換えます。
	 *
	 * @param   type ボタンのタイプ [submit/reset/button]
	 */
	public void setType( final String type ) {
		final String tp = getRequestParameter( type );
		if( check( tp , TYPE_SET ) ) {
			set( "type",tp );
		}
		else {
			final String errMsg = "type は、下記の範囲より指定してください。type=["
						 + tp + "]  in [" + String.join( ", " , TYPE_SET ) + "]";
			throw new HybsSystemException( errMsg );
		}
	}

	/**
	 * 【HTML】ボタンの名称を指定します。
	 *
	 * @og.tag
	 * @param	name	ボタンの名称
	 */
	public void setName( final String name ) {
		set( "name",getRequestParameter( name ) );
	}

	/**
	 * 【HTML】ボタンの値を指定します。
	 *
	 * @og.tag
	 * @param   value ボタンの値
	 */
	public void setValue( final String value ) {
		set( "value",getRequestParameter( value ) );
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION	)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
