/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */

package org.opengion.fukurou.queue;

import java.util.Locale;

import javax.jms.QueueSession;

import org.opengion.fukurou.util.StringUtil;

/**
 * キュータイプ別のクラス生成 
 * キュータイプ別のクラス生成用ファクトリークラスです。
 * 
 * @og.rev 5.10.14.0 (2019/08/01) 新規作成
 * 
 */
public final class QueueSendFactory {
	private static final int BUFFER_MIDDLE = 200;

	/**
	 * デフォルトコンストラクターをprivateにして、
	 * オブジェクトの生成をさせないようにします。
	 */
	private QueueSendFactory() {
	}
	
	/**
	 * キュー送信クラス生成
	 * 
	 * 引数のキュータイプのキュー送信クラスを生成します。
	 * MQ：Apache ActiveMq、amazonMQの場合に設定します。
	 * SQS：Amazon SQSの場合に設定します。
	 * 
	 * @param queueType キュータイプ
	 * @return キュータイプのキュー送信クラス
	 */
	public static QueueSend newQueueSend(final String queueType) {
//		QueueSend queueSend = null;
//		String setQueueType = null;

		// 2. 生成クラスの文字列生成
		final StringBuilder path = new StringBuilder(BUFFER_MIDDLE)
			.append("org.opengion.fukurou.queue.")
			.append("QueueSend_");

		// 1. 前処理
		// 大文字変換
		if (!StringUtil.isNull(queueType)) {
//			setQueueType = queueType.toUpperCase(Locale.JAPAN);
			path.append(queueType.toUpperCase(Locale.JAPAN));
		}

//		// 2. 生成クラスの文字列生成
//		path.append("org.opengion.fukurou.queue.");
//		path.append("QueueSend_");
//		path.append(setQueueType);

		final QueueSend queueSend;
		try {
			// 3. 対象クラスの生成
//			queueSend = (QueueSend) Class.forName(path.toString()).newInstance();							// 警告:[deprecation] ClassのnewInstance()は推奨されません
			queueSend = (QueueSend) Class.forName(path.toString()).getDeclaredConstructor().newInstance();
		} catch (final Throwable th) {
			// キャッチしたエラー情報をスロー
			throw new RuntimeException(th);
		}

		return queueSend;
	}

	/**
	 * バッチ処理用(mqのサンプル)
	 * バッチ処理用として、mainメソッドを実装します。
	 * 
	 * @param args メイン引数配列
	 */
	public static void main(final String[] args) {
		final boolean mqflug=true; // サンプルメソッドはMQ利用が標準としておきます
		System.out.println("main start");
		// 送信情報の設定
		final String url;
//				"tcp://localhost:61616";
//				"ssl://b-92d59238-9eb5-41a9-9bc3-bb6e08a49c4a-1.mq.ap-northeast-1.amazonaws.com:61617";
		final QueueInfo queueInfo = new QueueInfo();
//		QueueSend queueSend = null;
		final QueueSend queueSend;
		if(mqflug) {
			// MQサーバへの接続例
			// -DmqUserId -DmqPasswordに認証情報の設定が必要です。
			url = "tcp://localhost:61616";
			queueInfo.setMqQueueName("test01");
			queueInfo.setMqTransacted(false);
			queueInfo.setMqAcknowledgeMode(QueueSession.AUTO_ACKNOWLEDGE);
			queueSend = QueueSendFactory.newQueueSend("mq");
		}else {
			// SQSサーバへの接続例
			// EC2ロール認証か、-DaccessKey -DsecretKeyに認証情報の設定が必要です。
			url = "https://sqs.ap-northeast-1.amazonaws.com/716527209952/MyFifo.fifo";
			queueInfo.setSqsFifoGroupId("grp01");
			queueSend = QueueSendFactory.newQueueSend("sqs");
		}
		
		queueInfo.setMessage("送信メッセージ");
		// setBatchにtrueを設定することで、バッチ実行できるように実装しています。
		// (falseの場合はtomcatのjndi接続)
		queueSend.setBatchFlg(true);
		
		// ジョブ実行の場合、HybsSystemException内でnullエラー?
		try {
//			queueSend.connect(url);
			queueSend.connect(url,null,null);
			queueSend.sendMessage(queueInfo);
		}catch(final Exception e) {
			System.out.println(e.getMessage());
		} finally {
			queueSend.close();
		}
		System.out.println("main end");
	}
}
