/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.view;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.html.AbstractViewForm;
import org.opengion.fukurou.model.Formatter;
import org.opengion.fukurou.util.StringUtil;

/**
 * 検索結果を自動的に表形式に変換する、テーブル作成クラスです。
 *
 * AbstractViewForm により、setter/getterメソッドのデフォルト実装を提供しています。
 * 各HTMLのタグに必要な setter/getterメソッドのみ，追加定義しています。
 *
 * AbstractViewForm を継承している為,ロケールに応じたラベルを出力させる事が出来ます。
 *
 * @og.group 画面表示
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class ViewForm_HTMLTable extends AbstractViewForm {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "5.3.4.0 (2011/04/01)" ;

//	private String	caption 	 = null;		// 表のタイトル		5.1.6.0 (2010/05/01) 削除
	private String	cacheTag 	 = null;		// 3.5.5.5 (2004/04/23)
	private boolean	isCache  	 = false;		// 3.5.5.5 (2004/04/23)
	private int[]	popupClmNo	 = null;		// 3.8.6.1 (2006/10/20)

//	protected int[]		rowNo		 = null;	// 5.1.6.0 (2010/05/01) 削除
	/** ヘッダー部分のキャッシュ変数 */
	protected String	headerLine	 = null;	// 3.5.2.0 (2003/10/20)

	private final boolean	useRowId	 = HybsSystem.sysBool( "USE_CHECKBOX_ROW_ID" ) ;	// 3.6.0.0 (2004/09/17)

	// 5.2.3.0 (2010/12/01) テーブル罫線対応
	private final String	tableParam	 = HybsSystem.sys( "HTML_TABLE_PARAM" ) ;
	
	// 5.9.3.3 (2015/12/24) HTML5 で colgroup が効かない対応。
	protected final boolean useIE7Header	= HybsSystem.sysBool( "USE_IE7_HEADER" );	


	// 3.6.0.0 (2004/09/17) チェック済みの行の先頭に、フォーカスを当てる処理
	private static final String LAYER_FOCUS = "<a href=\"#top\" name=\"h_fcs\" id=\"h_fcs\" ></a>" ;

	// 4.3.4.4 (2009/01/01)
//	/**
//	 * デフォルトコンストラクター
//	 *
//	 */
//	public ViewForm_HTMLTable() {
//		super();
//		super.clear();
//	}

	/**
	 * 内容をクリア(初期化)します。
	 *
	 * @og.rev 3.1.1.0 (2003/03/28) 同期メソッド(synchronized付き)を非同期に変更する。
	 * @og.rev 3.5.2.0 (2003/10/20) ヘッダー繰り返し属性( headerSkipCount )を採用
	 * @og.rev 3.5.5.5 (2004/04/23) cacheTag,isCache 属性追加(チェックボックス作成用)
	 * @og.rev 3.5.6.2 (2004/07/05) bgColorCycle 変数の削除。
	 * @og.rev 3.8.6.1 (2006/10/20) popup の値を返す為のカラム番号配列(popupClmNo)追加
	 * @og.rev 5.1.6.0 (2010/05/01) caption 属性が使われていないため、削除する。
	 */
	@Override
	public void clear() {
		super.clear();
//		caption 	 = null;								// 表のタイトル
		cacheTag 	 = null;		// 3.5.5.5 (2004/04/23)
		isCache  	 = false;		// 3.5.5.5 (2004/04/23)
		popupClmNo	 = null;		// 3.8.6.1 (2006/10/20)
	}

	/**
	 * DBTableModel から HTML文字列を作成して返します。
	 * startNo(表示開始位置)から、pageSize(表示件数)までのView文字列を作成します。
	 * 表示残りデータが pageSize 以下の場合は,残りのデータをすべて出力します。
	 *
	 * @og.rev 3.5.2.0 (2003/10/20) ヘッダー繰り返し属性( headerSkipCount )を採用
	 * @og.rev 3.5.3.1 (2003/10/31) skip属性を採用。headerLine のキャッシュクリア
	 * @og.rev 3.5.5.0 (2004/03/12) No 欄そのものの作成判断ロジックを追加
	 * @og.rev 3.5.6.4 (2004/07/16) ヘッダーとボディー部をJavaScriptで分離
	 * @og.rev 3.7.0.3 (2005/03/01) getBgColorCycleClass に、選択行マーカーを採用
	 * @og.rev 4.3.1.0 (2008/09/08) 編集行のみを表示する属性(isSkipNoEdit)追加
	 * @og.rev 4.3.3.0 (2008/10/01) noTransition属性対応
	 *
	 * @param  startNo	  表示開始位置
	 * @param  pageSize   表示件数
	 *
	 * @return	DBTableModelから作成された HTML文字列
	 */
	public String create( final int startNo, final int pageSize )  {
		if( getRowCount() == 0 ) { return ""; }	// 暫定処置

		headerLine	 = null;
		int lastNo = getLastNo( startNo, pageSize );
		int blc = getBackLinkCount();
		int hsc = getHeaderSkipCount();		// 3.5.2.0 (2003/10/20)
		int hscCnt = 1;						// 3.5.2.0 (2003/10/20)

		StringBuilder out = new StringBuilder( HybsSystem.BUFFER_LARGE );

		out.append( getCountForm( startNo,pageSize ) );
		out.append( getHeader() );

		String ckboxTD = "  <td>";

		out.append("<tbody>").append( HybsSystem.CR );
		int bgClrCnt = 0;
		int clmCnt = getColumnCount();	// 3.5.5.7 (2004/05/10)
		for( int row=startNo; row<lastNo; row++ ) {
//			if( isSkip( row ) ) { continue; }		// 3.5.3.1 (2003/10/31)
			if( isSkip( row ) || isSkipNoEdit( row ) ) { continue; } // 4.3.1.0 (2008/09/08)
			out.append("<tr").append( getBgColorCycleClass( bgClrCnt++,row ) );
			if( isNoTransition() ) { // 4.3.3.0 (2008/10/01)
				out.append( getHiddenRowValue( row ) );
			}
			out.append(">");	// 3.7.0.3 (2005/03/01)
			out.append( HybsSystem.CR );
			// 3.5.5.0 (2004/03/12) No 欄そのものの作成判断追加
			if( isNumberDisplay() ) {
				out.append( makeCheckbox( ckboxTD, row, blc ) ).append( HybsSystem.CR );
			}
			for(int column = 0; column < clmCnt; column++) {
				if( isColumnDisplay( column ) ) {
					out.append("  <td>");
					out.append( getValueLabel(row,column) );
					out.append("</td>").append( HybsSystem.CR );
				}
			}
			out.append("</tr>").append( HybsSystem.CR );

		// 3.5.2.0 (2003/10/20) ヘッダー繰り返し属性( headerSkipCount )を採用
			if( hsc > 0 && hscCnt % hsc == 0 ) {
				out.append( getHeadLine() );
				hscCnt = 1;
			}
			else {
				hscCnt ++ ;
			}
		}
		out.append("</tbody>").append( HybsSystem.CR );
		out.append("</table>").append( HybsSystem.CR );

		out.append( getScrollBarEndDiv() );	// 3.8.0.3 (2005/07/15)

		return out.toString();
	}

	/**
	 * 選択用のチェックボックスと行番号と変更タイプ(A,C,D)を表示します。
	 *
	 * @og.rev 3.5.1.0 (2003/10/03) Noカラムに、numberType 属性を追加
	 * @og.rev 3.5.4.0 (2003/11/25) Formatter をローカル変数に変更
	 * @og.rev 3.5.4.1 (2003/12/01) table オブジェクトのセット廃止
	 * @og.rev 3.5.5.5 (2004/04/23) Attributes オブジェクトのセット廃止
	 * @og.rev 3.6.0.0 (2004/09/17) ガントチャートの移動時にチェックするためのIDを追加
	 * @og.rev 3.6.0.0 (2004/09/17) チェック済みの行の先頭に、フォーカスを当てる処理を追加
	 * @og.rev 3.8.6.1 (2006/10/20) popup 用の値を設定する機能を追加
	 *
	 * @param	ckboxTD チェックボックスのタグ(マルチカラム時のrowspan対応)
	 * @param	row	 行番号
	 * @param	blc	 バックラインカウント(先頭へ戻るリンク間隔)
	 *
	 * @return	tdタグで囲まれたチェックボックスのHTML文字列
	 */
	protected String makeCheckbox( final String ckboxTD,final int row,final int blc ) {
		StringBuilder out = new StringBuilder( HybsSystem.BUFFER_MIDDLE );

		DBTableModel table = getDBTableModel() ;

		String opAtt = null;
		String fmt = getOptionTypeAttributes();
		if( fmt != null ) {
			Formatter format = new Formatter( table );
			format.setFormat( fmt );
			opAtt = format.getFormatString( row );
		}

		out.append( ckboxTD );
		out.append( table.getModifyType( row ) );
		out.append("</td>");
		out.append( ckboxTD );

		// 3.5.5.5 (2004/04/23) Attributes オブジェクトのセット廃止
		String inputTag = makeChboxTag();
		if( isWritable( row ) && inputTag != null ) {
			out.append( inputTag );
			if( opAtt != null ) { out.append( opAtt ); }
			if( isChecked( row ) ) {
				out.append( " checked=\"checked\"" );
			}
			if( useRowId ) {	// 3.6.0.0 (2004/09/17)
				out.append( " id=\"" );
				out.append( HybsSystem.ROW_ID_KEY );
				out.append( row ).append( "\"" );
			}
			if( popupClmNo != null ) {	// 3.8.6.1 (2006/10/20)
				out = makePopupReturn( out,row );	// 内部で StringBuilder に append しています。
			}
			out.append( " value=\"" ).append( row ).append( "\" />" );
		}
		out.append("</td>");
		out.append( ckboxTD );

		// 3.5.1.0 (2003/10/03) Noカラムに、numberType 属性を追加
		if( blc != 0 && (row+1) % blc == 0 ) {
			out.append( "<a href=\"#top\">" ).append( getNumberData( row ) ).append( "</a>");
		} else {
			out.append( getNumberData( row ) );
		}

		// 4.3.5.3 (2008/02/22) Focus2のアンカー位置をtdタグ内に変更
		if( isFirstChecked( row ) ) {
			out.append( LAYER_FOCUS );
		}
		out.append("</td>");
		// if( isFirstChecked( row ) ) {
		// 	out.insert( 0,LAYER_FOCUS );
		// }

		return out.toString();
	}

	/**
	 * DBTableModel から テーブルのヘッダータグ文字列を作成して返します。
	 *
	 * @og.rev 3.5.2.0 (2003/10/20) ヘッダーそのもののキャッシュはしない。
	 *
	 * @return	テーブルのヘッダータグ文字列
	 */
	protected String getHeader() {
		return getTableTag() + getTableHead() ;
	}

	/**
	 * DBTableModel から テーブルのタグ文字列を作成して返します。
	 *
	 * @og.rev 3.5.6.4 (2004/07/16) ヘッダーとボディー部をJavaScriptで分離
	 * @og.rev 3.6.0.0 (2004/09/17) ヘッダー固定スクロールの簡素化(スクロールバーを右に出す)
	 * @og.rev 3.6.0.5 (2004/10/18) 印刷時の罫線出力関連機能の追加。id 属性を出力します。
	 * @og.rev 4.0.0.0 (2005/08/31) テーブル表示の CSSファイル利用の有無
	 * @og.rev 5.1.6.0 (2010/05/01) caption 属性が使われていないため、削除する。
	 * @og.rev 5.2.3.0 (2010/12/01) テーブル罫線対応
	 * @og.rev 5.3.4.0 (2011/04/01) テーブル罫線の初期値チェック変更
	 *
	 * @return	テーブルのタグ文字列
	 */
	protected String getTableTag() {
		StringBuilder buf = new StringBuilder( HybsSystem.BUFFER_MIDDLE );

		buf.append( getScrollBarStartDiv() );	// 3.8.0.3 (2005/07/15)

		buf.append("<table ");
		// 5.2.3.0 (2010/12/01) テーブル罫線対応
//		if( tableParam != null ) {		// 5.3.4.0 (2011/04/01)
		if( tableParam != null && !tableParam.isEmpty() ) {
			buf.append( tableParam );
			buf.append( " " );
		}
		buf.append("id=\"viewTable\" ");
		buf.append("class=\"").append( getTableClass() ).append( "\"" );

		buf.append(">").append( HybsSystem.CR );

//		if( caption != null ) {
//			buf.append("<caption>");
//			buf.append( caption );
//			buf.append("</caption>").append( HybsSystem.CR );
//		}

		return buf.toString();
	}

	
	/**
	 * HTML5 で colgroup が効かない暫定対応の、処理をまとめるメソッド
	 * 
	 * #viewTable td:nth-child(n) { text-align:right; } CSSを出力するメソッドをまとめました。
	 * 呼び出し元の、getTableHead() のコードをすっきりさせるためのメソッドです。
	 *
	 * @og.rev 5.9.3.3 (2015/12/26) 6.1.2.0での対応を移植
	 * @og.rev 5.9.5.3 (2016/02/26) viewTableにtableIdのクラスを追加する対応
	 * @og.rev 5.9.9.0 (2016/06/03) getViewClass
	 *
	 * @param buf		タグ文字列をappendしていくStringBuilder
	 * @param ad		TDタグの番号 nth-child(n)設置する場合の n の値
	 * @param dbType	text-alignを決めるためのDBTYPE
	 *                  (R,S9,X9→text-align:right; , BIT→text-align:center; , 他素通り)
	 *
	 * @return	テーブルのタグ文字列(入力と同じStringBuilder)
	 * @og.rtnNotNull
	 * @see		#getTableHead()
	 */
	protected StringBuilder makeNthChild( final StringBuilder buf , final int ad , final String dbType ) {
		if( "R".equalsIgnoreCase( dbType ) || "S9".equalsIgnoreCase( dbType ) || "X9".equalsIgnoreCase( dbType ) ) {
//			buf.append( "  #viewTable td:nth-child(" ).append( ad ).append( ") { text-align:right; }" );
//			buf.append( "  #viewTable."+ getTableClazz() +" td:nth-child(" ).append( ad ).append( ") { text-align:right; }" ); // 5.9.5.3 (2016/02/26)
			buf.append( "  #viewTable."+ getViewClass() +" td:nth-child(" ).append( ad ).append( ") { text-align:right; }" ); // 5.9.9.0
			buf.append( HybsSystem.CR );
		}
		else if( "BIT".equalsIgnoreCase( dbType ) ) {	// BIT は 真ん中
//			buf.append( "  #viewTable td:nth-child(" ).append( ad ).append( ") { text-align:center; }" );
//			buf.append( "  #viewTable."+ getTableClazz() + " td:nth-child(" ).append( ad ).append( ") { text-align:center; }" );// 5.9.5.3 (2016/02/26)
			buf.append( "  #viewTable."+ getViewClass() + " td:nth-child(" ).append( ad ).append( ") { text-align:center; }" );// 5.9.9.0
			buf.append( HybsSystem.CR );
		}
		// 上記(R,S9,X9,BIT)以外のdbTypeの場合は、素通りします。(入力のStringBuilderをそのまま返すだけ)

		return buf ;
	}
	
	/**
	 * DBTableModel から テーブルのタグ文字列を作成して返します。
	 *
	 * @og.rev 3.5.1.0 (2003/10/03) Noカラムに、numberType 属性を追加
	 * @og.rev 3.5.2.0 (2003/10/20) ヘッダー繰り返し部をgetHeadLine()へ移動
	 * @og.rev 3.5.3.1 (2003/10/31) VERCHAR2 を VARCHAR2 に修正。
	 * @og.rev 3.5.5.0 (2004/03/12) No 欄そのものの作成判断ロジックを追加
	 * @og.rev 3.5.6.5 (2004/08/09) thead に、id="header" を追加
	 * @og.rev 4.0.0.0 (2005/01/31) DBColumn の 属性(CLS_NM)から、DBTYPEに変更
	 * @og.rev 4.0.0.0 (2005/01/31) 新規作成(getColumnClassName ⇒ getColumnDbType)
	 * @og.rev 5.9.1.2 (2015/10/23) 自己終了警告対応
	 * @og.rev 5.9.3.3 (2015/12/26) colgroup対応
	 *
	 * @return	テーブルのタグ文字列
	 */
	protected String getTableHead() {
		StringBuilder buf = new StringBuilder( HybsSystem.BUFFER_MIDDLE );
		
		// 5.9.3.3 (2015/12/26) HTML5 で colgroup が効かない対応
		// 互換モードでない場合専用。処理速度を気にするより、処理を一か所にまとめておきます。
		if( !useIE7Header ) {
			buf.append( "<style type=\"text/css\">" )
				.append( HybsSystem.CR );
			int ad ;
			if( isNumberDisplay() ) {
				makeNthChild( buf,2,"BIT" );
				makeNthChild( buf,3,"S9"  );
				ad = 4;
			}
			else {
				ad = 1;		
			}

			final int clmCnt = getColumnCount();
			for( int column=0; column<clmCnt; column++ ) {
				if( isColumnDisplay( column ) ) {
					makeNthChild( buf,ad,getColumnDbType(column) );
					ad++ ;			// tdタグの順番なので、表示する場合のみカウントする。
				}
			}
			buf.append( "</style>" )
				.append( HybsSystem.CR );
		}

		// 3.5.5.0 (2004/03/12) No 欄そのものの作成判断追加
		if( isNumberDisplay() ) {
//			buf.append("<colgroup class=\"X\" />");		// 4.0.0 (2005/01/31)
//			buf.append("<colgroup class=\"BIT\" />");
//			buf.append("<colgroup class=\"S9\" />");		// 4.0.0 (2005/01/31)
			buf.append("<colgroup class=\"X\" ><!-- --></colgroup>");		// 5.9.1.2 (2015/10/23)
			buf.append("<colgroup class=\"BIT\" ><!-- --></colgroup>");
			buf.append("<colgroup class=\"S9\" ><!-- --></colgroup>");	
			buf.append( HybsSystem.CR );
		}

		int clmCnt = getColumnCount();	// 3.5.5.7 (2004/05/10)
		for(int column = 0; column < clmCnt; column++) {
			if( isColumnDisplay( column ) ) {
				buf.append("<colgroup class=\"" );
				buf.append( getColumnDbType(column) );		// 4.0.0 (2005/01/31)
//				buf.append("\"/>");
				buf.append("\"><!-- --></colgroup>");		// 5.9.1.2 (2015/10/23)
				buf.append( HybsSystem.CR );
			}
		}

	// 3.5.2.0 (2003/10/20) ヘッダー繰り返し部をgetHeadLine()へ移動
		buf.append("<thead id=\"header\">").append( HybsSystem.CR );	// 3.5.6.5 (2004/08/09)
		buf.append( getHeadLine() );
		buf.append("</thead>").append( HybsSystem.CR );

		return buf.toString();
	}

	/**
	 * ヘッダー繰り返し部を、getTableHead()メソッドから分離。
	 *
	 * @og.rev 3.5.4.5 (2004/01/23) 実装をgetHeadLine( String thTag )に移動
	 *
	 * @return	テーブルのタグ文字列
	 */
	protected String getHeadLine() {
		return getHeadLine( "<th" ) ;
	}

	/**
	 * ヘッダー繰り返し部を、getTableHead()メソッドから分離。
	 *
	 * @og.rev 3.5.2.0 (2003/10/20) 新規作成
	 * @og.rev 3.5.4.3 (2004/01/05) useCheckControl 属性の機能を追加
	 * @og.rev 3.5.4.5 (2004/01/23) thタグの属性設定出来る様に新規追加。
	 * @og.rev 3.5.4.6 (2004/01/30) numberType="none" 時の処理を追加(Noラベルを出さない)
	 * @og.rev 3.5.4.7 (2004/02/06) ヘッダーにソート機能用のリンクを追加します。
	 * @og.rev 3.7.0.1 (2005/01/31) 全件チェックコントロール処理変更
	 *
	 * @param	thTag タグの文字列
	 *
	 * @return	テーブルのタグ文字列
	 */
	protected String getHeadLine( final String thTag ) {
		if( headerLine != null ) { return headerLine; }		// キャッシュを返す。

		StringBuilder buf = new StringBuilder( HybsSystem.BUFFER_MIDDLE );

		buf.append("<tr class=\"row_h\"").append(" >").append( HybsSystem.CR );

		// 3.5.5.0 (2004/03/12) No 欄そのものの作成判断追加
		if( isNumberDisplay() ) {
			// 3.5.4.3 (2004/01/05) 追加分
			if( isUseCheckControl() && "checkbox".equals( getSelectedType() ) ) {
			// 3.5.4.5 (2004/01/23) thタグの属性設定出来る様に変更。
				buf.append( thTag ).append("></th>");
				buf.append( thTag ).append(">").append( getAllCheckControl() ).append("</th>");
				buf.append( thTag ).append(">").append( getNumberHeader() ).append("</th>");	// 3.5.4.6 (2004/01/30)
			}
			else {
			// 3.5.4.5 (2004/01/23) thタグの属性設定出来る様に変更。
				buf.append( thTag ).append(" colspan='3'>").append( getNumberHeader() ).append("</th>");	// 3.5.4.6 (2004/01/30)
			}
		}

		buf.append( HybsSystem.CR );
		int clmCnt = getColumnCount();	// 3.5.5.7 (2004/05/10)
		for(int column = 0; column < clmCnt; column++) {
			if( isColumnDisplay( column ) ) {
			// 3.5.4.5 (2004/01/23) thタグの属性設定出来る様に変更。
				buf.append( thTag ).append(">");
				buf.append( getSortedColumnLabel(column) );
				buf.append("</th>").append( HybsSystem.CR );
			}
		}
		buf.append("</tr>").append( HybsSystem.CR );

		headerLine = buf.toString();
		return headerLine;
	}

	/**
	 * テーブルのタイトルをセットします。
	 *
	 * @og.rev 3.1.1.0 (2003/03/28) 同期メソッド(synchronized付き)を非同期に変更する。
	 * @og.rev 5.1.6.0 (2010/05/01) caption 属性が使われていないため、削除する。
	 *
	 * @param  caption 属性
	 */
//	public void setCaption( final String caption ) {
//		this.caption = caption;
//	}

	/**
	 * フォーマットメソッドを使用できるかどうかを問い合わせます。
	 *
	 * @return  使用可能(true)/ 使用不可能(false)
	 */
	public boolean canUseFormat() {
		return false;
	}

	/**
	 * ogPopup で検索結果の値を返すキーを、CSV形式で指定します。
	 *
	 * popup の検索結果を返す画面で、結果のラジオボタンにイベントセットします。
	 * この場合、オープンもとのwindow に値を返しますが、そのキーをCSV形式で
	 * 指定します。なお、設定は、init 以降(つまり、DBTableModelは設定済み)の
	 * 状態で呼び出してください。(エラーにしません)
	 * なお、このメソッドは、一覧表示(HTMLTable)関係のビューのみでサポートして
	 * いますが、チェックメソッドの関係で、それ以外のビューに適用しても素通り
	 * するようにします。(エラーにしません)
	 *
	 * @og.rev 3.8.6.1 (2006/10/20) 新規追加
	 *
	 * @param  rtnKeys ogPopupで値を返すカラム文字列(CSV形式)
	 */
	@Override
	public void setPopupReturnKeys( final String rtnKeys ) {
		DBTableModel table = getDBTableModel() ;
		if( table != null && table.getRowCount() > 0 && rtnKeys != null ) {
			String[] clmNames = StringUtil.csv2Array( rtnKeys );
			popupClmNo = new int[clmNames.length];
			for( int i=0; i<clmNames.length; i++ ) {
				int no = table.getColumnNo( clmNames[i] );
				if( no >= 0 ) { popupClmNo[ i ] = no; }
			}
		}
	}

	/**
	 * ogPopup で検索結果の値を返すキーを、CSV形式で指定します。
	 *
	 * popup の検索結果を返す画面で、結果のラジオボタンにイベントセットします。
	 * この場合、オープンもとのwindow に値を返しますが、そのキーをCSV形式で
	 * 指定します。なお、設定は、init 以降(つまり、DBTableModelは設定済み)の
	 * 状態で呼び出してください。(エラーにしません)
	 * なお、このメソッドは、一覧表示(HTMLTable)関係のビューのみでサポートして
	 * いますが、チェックメソッドの関係で、それ以外のビューに適用しても素通り
	 * するようにします。(エラーにしません)
	 * rtnPopup に値を渡す場合に、ダブルクオート(")、シングルクオート(')は、
	 * それぞれ、ASCII コード(￥ｘ２２、￥ｘ２７)に置き換えます。
	 *
	 * @og.rev 3.8.6.1 (2006/10/20) 新規追加
	 *
	 * @param  buf StringBuilder 追加するStringBuilderオブジェクト
	 * @param	rowNo	列番号
	 *
	 * @return	引数にデータを追加した後の同一オブジェクト
	 */
	private StringBuilder makePopupReturn( final StringBuilder buf,final int rowNo ) {
		buf.append( " onClick=\"rtnPopup(new Array(" );
		int clmNo = popupClmNo[ 0 ];
		String val = StringUtil.quoteFilter( getValue( rowNo,clmNo ) );
		buf.append( "'" ).append( val ).append( "'" );
		for( int i=1; i<popupClmNo.length; i++ ) {
			clmNo = popupClmNo[ i ];
			val = StringUtil.quoteFilter( getValue( rowNo,clmNo ) );
			buf.append( ",'" ).append( val ).append( "'" );
		}
		buf.append( "));\"" );

		return buf;
	}

	/**
	 * 選択用のチェックボックスの input タグを作成します。
	 *
	 * @og.rev 3.5.5.5 (2004/04/23) 新規作成
	 *
	 * @return	チェックボックスのinputタグ
	 */
	private String makeChboxTag() {
		if( isCache ) { return cacheTag; }
		isCache = true;

		String type = getSelectedType();		// "checkbox"/"radio"/"hidden"

		if( type != null ) {
			StringBuilder buf = new StringBuilder();

			buf.append( "<input type=\"" ).append( type ).append( "\" " );
			buf.append( "name=\"" ).append( HybsSystem.ROW_SEL_KEY ).append( "\"" );

			cacheTag = buf.toString();
		}
		return cacheTag ;
	}

	/**
	 * カラムのラベル名(短)を返します。
	 * カラムの項目名に対して,見える形の文字列を返します。
	 * 一般には,リソースバンドルと組合せて,各国ロケール毎にラベルを
	 * 切替えます。
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) 新規追加( shortLabel を返します。)
	 *
	 * @param	column カラム番号
	 *
	 * @return	カラムのラベル名(短)
	 */
	@Override
	protected String getColumnLabel( final int column ) {
		return getDBColumn( column ).getShortLabel();
	}
}
