/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.TagBuffer;

/**
 * ガントチャートを表示するに当たり、ガント全体（ページ全体一つ設定）の指定を行います。
 *
 * iGantt タグは、ガントチャート全体に一つ指定する事で、最大行数、固定カラム数、
 * 番号列の出力有無を指定するのに使用されます。
 * 通常は、単独で使用するのではなく、iGanttBar タグと組み合わせて使用します。
 * 通常の view タグの後に記述します。
 *
 * ガントには、通常のガントと積上ガント、内部積上ガントがあります。
 * 通常ガントは、通常の view をガント化します。
 * 積上ガントは、ViewにHTMLstackedGanttTableを指定する必要があります。
 * この方式の利点は、積上と通常のガントバーを混在できることです。例えば、物件予定をガントバーで、表示し、
 * 個人の予定工数を積上るなどの表示が可能です。ただし、件数が多いと、処理に時間がかかります。
 * 内部積上ガントは、エンジン内部で積上計算しますので、処理は早いのですが、ガントバーとの
 * 混在ができません。これらは、うまく使い分けを行う必要があります。
 *
 * @og.formSample
 * ●形式：&lt;og:iGantt  ... /&gt;
 * ●body：なし
 * ●前提：headタグで、adjustEvent="Gantt" を指定してください。
 *
 * ●Tag定義：
 *   &lt;og:iGantt
 *       margeRows          【TAG】前後の行データが一致している場合、マージするかどうか[true/false]指定します(初期値:false)
 *       fixedCols          【TAG】左の固定列の列数（テーブル２分割機能）を指定します
 *       viewNumberType     【TAG】viewタグの出力に番号列が出力されているかかどうか（出力されていない場合:deleteを指定）
 *       verticalShift      【TAG】ガントの上下ずらし表示を行うかどうか[true/false]指定します(初期値=true:行う)
 *       paddingLeft        【TAG】ガントバーの間の左区切りスペースを指定します(初期値=null)
 *       paddingRigth       【TAG】ガントバーの間の右区切りスペースを指定します(初期値=null)
 *       useBgColor         【TAG】一覧の背景色の縞々模様を再作成するか[true/false]指定します（margeRows='true'の場合は使用する）
 *       viewGantt          【TAG】積上ガント:ガント部分の表示を行うかどうか[true/false]指定します(初期値=true:表示する)
 *       stackHoliday       【TAG】積上ガント:休日に積上げるかどうか[true/false]指定します（zoom=DAYの場合のみ有効。初期値=true:積上げる)
 *       viewMode           【TAG】積上ガント:1:行の最大値を基準に積上げ高さの計算を行う/0:能力設定値を基準に積上げ高さの計算を行う。
 *       stdUpper           【TAG】積上ガント:正常範囲の上限となる工数です。これを超えると積上げの色が変化します(初期値:1)
 *       stdCost            【TAG】積上ガント:この工数が行の2/3の高さとなります(初期値:1)
 *       stdLower           【TAG】積上ガント:正常範囲の下限となる工数です。これを下回ると積上げの色が変化します(初期値:0)
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   /&gt;
 *
 * ●使用例
 *   (通常ガント)
 *  &lt;og:view
 *      viewFormType = "HTMLCustomTable"
 *      command      = "{&#064;command}"
 *      writable     = "false"
 *      useScrollBar = "false"
 *  &gt;
 *    &lt;og:thead rowspan="2"&gt;
 *      &lt;tr&gt;
 *          &lt;td&gt;[NOORDER]&lt;/td&gt;
 *          &lt;td rowspan="2" class="gantt zoom{&#064;VZOOM}" style="line-height:normal;text-align:left;background-image: url('../image/ganttBG{&#064;VZOOM}.gif');" &gt;
 *              &lt;div style="position:relative;left:0px;z-index:101; " height="100%" width="100%" &gt;
 *                  &lt;og:ganttHeader
 *                      startDate       = "{&#064;startDate}"
 *                      endDate         = "{&#064;endDate}"
 *                      zoom            = "{&#064;VZOOM}"
 *                      daySpan         = "{&#064;daySpan}"
 *                  /&gt;
 *              &lt;/div&gt;
 *          &lt;/td&gt;
 *      &lt;/tr&gt;
 *    &lt;/og:thead&gt;
 *    &lt;og:tbody rowspan="2"&gt;
 *      &lt;tr&gt;
 *          &lt;td&gt;[NOORDER]&lt;/td&gt;
 *          &lt;td rowspan="2" class="gantt zoom{&#064;VZOOM}" style="text-align:left;background-image: url('../image/ganttBG{&#064;VZOOM}.gif');" &gt;
 *              &lt;og:iGanttBar type="1" colorNo="[COLOR]" start="[DYORDER]" end="[DYNOKI]" text="[PN]" /&gt;
 *              &lt;og:iGanttBar type="0" src="../image/dia_blue.gif" start="[DYORDER]"      text="開始" /&gt;
 *              &lt;og:iGanttBar type="0" src="../image/dia_red.gif"  end="[DYNOKI]"         text="終了" /&gt;
 *          &lt;/td&gt;
 *      &lt;/tr&gt;
 *    &lt;/og:tbody&gt;
 *  &lt;/og:view&gt;
 *    
 *   &lt;og:iGantt
 *       margeRows      = "true"
 *       fixedCols      = "1"
 *   /&gt;
 *
 *   (積上ガント)
 *  積上ガントを利用する場合は、ViewにHTMLstackedGanttTableを利用する必要があります。
 *  &lt;og:view
 *      viewFormType = "HTMLStackedGanttTable"
 *      command      = "{&#064;command}"
 *      writable     = "false"
 *      useScrollBar = "false"
 *      useParam     = "true"
 *      numberType   = "none"
 *  &gt;
 *  &lt;og:stackParam
 *      stackColumns = "NOORDER"
 *  /&gt;
 * 
 *    &lt;og:thead rowspan="2"&gt;
 *      &lt;tr&gt;
 *          &lt;td&gt;[NOORDER]&lt;/td&gt;
 *          &lt;td&gt;[PN]&lt;/td&gt;
 *          &lt;td rowspan="2" class="gantt zoom{&#064;SZOOM}" style="text-align:left;background-image: url('../image/ganttBG{&#064;SZOOM}.gif');" &gt;
 *              &lt;div style="position:relative;left:0px;z-index:101; " height="100%" width="100%" &gt;
 *                  &lt;og:ganttHeader
 *                      startDate   = "{&#064;startDate}"
 *                      endDate     = "{&#064;endDate}"
 *                      zoom        = "{&#064;SZOOM}"
 *                      calDB       = "GE13"
 *                      arg1        = "A"
 *                  /&gt;
 *              &lt;/div&gt;
 *          &lt;/td&gt;
 *      &lt;/tr&gt;
 *    &lt;/og:thead&gt;
 *    &lt;og:tbody rowspan="2"&gt;
 *      &lt;tr&gt;
 *          &lt;td&gt;[NOORDER]&lt;/td&gt;
 *          &lt;td&gt;[PN]&lt;/td&gt;
 *          &lt;td rowspan="2" class="gantt zoom{&#064;VZOOM}" style="text-align:left;background-image: url('../image/ganttBG{&#064;VZOOM}.gif');" &gt;
 *              &lt;og:iGanttBar type="1" colorNo="[COLOR]" start="[DYORDER]" end="[DYNOKI]"  text="[COSTCLM]"
 *                    cost="[COSTCLM]" capacity="[CAPACITY]" /&gt;
 *              &lt;og:iGanttBar type="0" src="../image/dia_blue.gif" start="[DYORDER]"      text="開始" /&gt;
 *          &lt;/td&gt;
 *      &lt;/tr&gt;
 *    &lt;/og:tbody&gt;
 *  &lt;/og:view&gt;
 *    
 *   &lt;og:iGantt
 *       margeRows      = "true"
 *       fixedCols      = "1"
 *       verticalShift  = "false"
 *       viewGantt      = "{&#064;viewGantt}"
 *       stackHoliday   = "{&#064;stackHoliday}"
 *       useBgColor     = "true"
 *       viewMode       = "2"
 *   /&gt;
 *
 * @og.rev 5.6.3.2 (2013/04/12) 新規作成
 * @og.group 画面部品
 *
 * @version  5.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK6.0,
 */
public class ViewIGanttTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "5.6.4.2 (2013/05/17)" ;

	private static final long serialVersionUID = 564220130517L ;

	private TagBuffer tag = new TagBuffer( "iGantt" ) ;

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 5.8.1.0 (2014/11/07) HTML5対応。javaScriptで、BODYがないと入れ子になってしまう。
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)

		tag.setBody( "<!-- -->" );		// 5.8.1.0 (2014/11/07) HTML5対応。
		jspPrint( tag.makeTag() );

		return(EVAL_PAGE);		// ページの残りを評価する。
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
		tag = new TagBuffer( "iGantt" );
	}

	/**
	 * 【TAG】前後の行データが一致している場合、マージするかどうか[true/false]指定します(初期値:false)
	 *
	 * @og.tag
	 * これは、ガントデータが同一グループの場合に、ガントとして一連に表示するようにします。
	 * この段階では、階段並べのままです。
	 * 前後に付ける場合（一列表示）は、verticalShift = "true" を指定してください。
	 * (adjustGanttTable.jsの)初期値は、false:行わない です。
	 *
	 * @param   margeRows 複数行のマージを行うかどうか[true/false]
	 * @see		#setVerticalShift( String )
	 */
	public void setMargeRows( final String margeRows ) {
		tag.add( "margeRows",StringUtil.nval( getRequestParameter( margeRows ),null ) );
	}

	/**
	 * 【TAG】左の固定列の列数（テーブル２分割機能）を指定します。
	 *
	 * @og.tag
	 * １段組でも２段組でも、固定したいカラム数を指定します。
	 *
	 * @param   fixedCols 固定したいカラム数
	 */
	public void setFixedCols( final String fixedCols ) {
		tag.add( "fixedCols",StringUtil.nval( getRequestParameter( fixedCols ),null ) );
	}

	/**
	 * 【TAG】viewタグの出力に番号列が出力されているかどうか（出力されていない場合:delete）を指定します。
	 *
	 * @og.tag
	 * viewタグで、numberType属性で、行番号を表示しない設定を行った場合、ここでも、viewNumberType="delete" を
	 * 指定する必要があります。
	 *
	 * @param   viewNumberType viewタグの番号列が出力されているかかどうか
	 */
	public void setViewNumberType( final String viewNumberType ) {
		tag.add( "viewNumberType",StringUtil.nval( getRequestParameter( viewNumberType ),null ) );
	}

	/**
	 * 【TAG】ガントの上下ずらし表示を行うかどうか[true/false]指定します(初期値=true:行う)
	 *
	 * @og.tag
	 * 通常のガント表示では、データは階段並べで表示されます。
	 * 同一属性のガントを横一列に表示したい場合は、この属性に、"true" を指定します。
	 * (adjustGanttTable.jsの)初期値は、true:行う です。
	 *
	 * @param   verticalShift 上下ずらし表示を行うかどうか[true/false]
	 */
	public void setVerticalShift( final String verticalShift ) {
		tag.add( "verticalShift",StringUtil.nval( getRequestParameter( verticalShift ), null ) );
	}

	/**
	 * 【TAG】ガントバーの間の左区切りスペースをピクセルで指定します(初期値:null)
	 *
	 * @og.tag
	 * ガント表示で、margeRows="true" (複数行のマージを行う)場合、前後のガントが同一色の
	 * 場合、くっついて表示されます。これを、verticalShift="true" (ガントの上下ずらし表示を行う)
	 * 場合は、個々のバーが判別可能ですが、そうしたくないケースでは、個々の判別ができません。
	 * そこで、特殊なケースとして、個々の判別が付く様に、ガントバーの長さを調整したいケースが
	 * あります。
	 *
	 * この属性は、バーの左に指定の空欄を用意します。
	 * 初期値は、null(属性を出力しない) です。
	 *
	 * @og.rev 5.6.4.2 (2013/05/17) 新規追加
	 *
	 * @param   paddingLeft 左区切りスペース
	 * @see		#setPaddingRigth( String )
	 */
	public void setPaddingLeft( final String paddingLeft ) {
		tag.add( "paddingLeft",StringUtil.nval( getRequestParameter( paddingLeft ),null ) );
	}

	/**
	 * 【TAG】ガントバーの間の右区切りスペースをピクセルで指定します(初期値:null)
	 *
	 * @og.tag
	 * ガント表示で、margeRows="true" (複数行のマージを行う)場合、前後のガントが同一色の
	 * 場合、くっついて表示されます。これを、verticalShift="true" (ガントの上下ずらし表示を行う)
	 * 場合は、個々のバーが判別可能ですが、そうしたくないケースでは、個々の判別ができません。
	 * そこで、特殊なケースとして、個々の判別が付く様に、ガントバーの長さを調整したいケースが
	 * あります。
	 *
	 * この属性は、バーの右に指定の空欄を用意します。
	 * 初期値は、null(属性を出力しない) です。
	 *
	 * @og.rev 5.6.4.2 (2013/05/17) 新規追加
	 *
	 * @param   paddingRigth 左区切りスペース
	 * @see		#setPaddingLeft( String )
	 */
	public void setPaddingRigth( final String paddingRigth ) {
		tag.add( "paddingRigth",StringUtil.nval( getRequestParameter( paddingRigth ),null ) );
	}

	/**
	 * 【TAG】積上ガント:ガント部分の表示を行うかどうか[true/false]指定します(初期値=true:表示する)
	 *
	 * @og.tag
	 * falseとするとガント部分を表示せず、積上げのみ表示します。
	 * (adjustGanttTable.jsの)初期値は、true:表示する。
	 *
	 * @param   viewGantt ガント部分の表示を行うかどうか[true/false]
	 */
	public void setViewGantt( final String viewGantt ) {
		tag.add( "viewGantt",StringUtil.nval( getRequestParameter( viewGantt ), null ) );
	}

	/**
	 * 【TAG】積上ガント:休日に積上げるかどうか[true/false]指定します（初期値=true:積上げる)
	 *
	 * @og.tag
	 * 休日に積上る場合、平日、休日を合わせた日数で、工数の平準化が行われます。
	 * false:積上ない を指定した場合、平日のみで工数が加算されます。
	 * 積上は、日付関係の場合(zoom=DAY)のみ有効で、時間単位の積上機能はありません。
	 * (adjustGanttTable.jsの)初期値は、true:積上げる。
	 *
	 * @param   stackHoliday ガントの表示を行うかどうか[true/false]
	 */
	public void setStackHoliday( final String stackHoliday ) {
		tag.add( "stackHoliday",StringUtil.nval( getRequestParameter( stackHoliday ), null ) );
	}

	/**
	 * 【TAG】一覧の背景色の縞々模様を再作成するか[true/false]指定します(初期値:true)
	 *
	 * @og.tag
	 * 背景色の縞々模様(ゼブラ模様)を作成する場合は、"true" にセットします。
	 * margeRows='true'の場合は使用します。
	 * (adjustGanttTable.jsの)初期値は、true:再作成する。
	 *
	 * @param   useBgColor 背景色ゼブラを行うかどうか[true/false]
	 * @see		#setMargeRows( String )
	 */
	public void setUseBgColor( final String useBgColor ) {
		tag.add( "useBgColor",StringUtil.nval( getRequestParameter( useBgColor ), null ) );
	}

	/**
	 * 【TAG】積上ガント:積上げ高さの計算方法[0:設定値基準/1:最大値基準]指定します（初期値:1:最大値基準）
	 *
	 * @og.tag
	 * 積上ガントの大きさを、設定値を基準にするか、最大値を基準にするか指定します。
	 * 1:最大値基準は、高さ固定と考えられます。つまり、積上ガントの最大が1.0の場合、
	 * 設定値が、0.1 なら、0.1 分の値としてつみあがります。１００分率での表示に適しています。
	 * 0:設定値基準は、高さ可変です。つまり、積上ガントの設定値の最大が 0.2 の場合、
	 * 0.1 なら、半分の所まで積みあがります。値に最大値がなく、各積上結果の相対レベルが
	 * 見たい場合に、適しています。
	 * 
	 * 0:能力設定値を基準に積上げ高さの計算を行う。
	 * 1:行の最大値を基準に積上げ高さの計算を行う。
	 * (adjustGanttTable.jsの)初期値は、1:最大値基準 です。
	 *
	 * @param   viewMode 積上げ高さの計算方法[0:設定値基準/1:最大値基準]
	 */
	public void setViewMode( final String viewMode ) {
		tag.add( "viewMode",StringUtil.nval( getRequestParameter( viewMode ),null ) );
	}

	/**
	 * 【TAG】積上ガント:正常範囲の上限となる工数を指定します(初期値:1)。
	 *
	 * @og.tag
	 * 正常範囲の上限となる工数を超えると積上げの色が変化します。
	 * (adjustGanttTable.jsの)初期値は、1 です。
	 *
	 * @param   stdUpper 正常範囲の上限となる工数
	 * @see		#setStdCost( String )
	 */
	public void setStdUpper( final String stdUpper ) {
		tag.add( "stdUpper",StringUtil.nval( getRequestParameter( stdUpper ),null ) );
	}

	/**
	 * 【TAG】積上ガント:行の2/3の高さとなる工数を指定します(初期値:1)。
	 *
	 * @og.tag
	 * 正常範囲の上限を初期値の "1" に設定し、この値を初期値の "1" を使うと、
	 * "1" の高さは、行の2/3の高さになるように計算されます。つまり、オーバー分は、
	 * 全体の 1/3 以下の場合に、ちょうど良い感じになります。
	 * オーバーする量との関係で指定します。
	 * (adjustGanttTable.jsの)初期値は、1 です。
	 *
	 * @param   stdCost 行の2/3の高さとなる工数
	 * @see		#setStdUpper( String )
	 */
	public void setStdCost( final String stdCost ) {
		tag.add( "stdCost",StringUtil.nval( getRequestParameter( stdCost ),null ) );
	}

	/**
	 * 【TAG】積上ガント:正常範囲の下限となる工数を指定します(初期値:0)。
	 *
	 * @og.tag
	 * 正常範囲の下限となる工数を下回ると積上げの色が変化します。
	 * (adjustGanttTable.jsの)初期値は、0 です。
	 *
	 * @param   stdLower 背景色ゼブラを行うかどうか[true/false]
	 */
	public void setStdLower( final String stdLower ) {
		tag.add( "stdLower",StringUtil.nval( getRequestParameter( stdLower ),null ) );
	}

	/**
	 * タグの名称を、返します。
	 * 自分自身のクラス名より、自動的に取り出せないため、このメソッドをオーバーライドします。
	 *
	 * @return  タグの名称
	 */
	@Override
	protected String getTagName() {
		return "iGantt" ;
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	@Override
	public String toString() {
		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION	)
				.println( "tag"			,tag.makeTag()	)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
