/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.hayabusa.html.JsonReader;
import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.util.StringUtil;
import static org.opengion.fukurou.util.StringUtil.nval;

import java.util.Locale;

/**
 * JSONを DBTableModelオブジェクトに読み取るタグです。
 *
 * JSONの読み取りにはpluginのJsonReaderを利用します。
 * 標準はエンジンの出力形式の読み取りを行うJsonReader_Defaultです。
 *
 * 入力件数を"DB.COUNT" キーでリクエストにセットしています。
 *
 * @og.formSample
 * ●形式：
 *     &lt;og:readJSON
 *         command      = "NEW"
 *         maxRowCount  = "10000"               読取最大件数(0:[無制限])
 *     &gt;
 *      ...
 *     &lt;og:readJSON /&gt;
 * ●body：なし
 *
 * ●Tag定義：
 *   &lt;og:readTable
 *       readerClass        【TAG】実際に読み出すクラス名の略称(TableReader_**** の ****)をセットします(初期値:Default)
 *       JSONData           【TAG】読み取るJSONデータをセットします。タグのBODYでもセット可能です。
 *       maxRowCount        【TAG】読取時の最大取り込み件数をセットします (初期値:DB_MAX_ROW_COUNT[=1000])(0:[無制限])
 *       tableId            【TAG】(通常使いません)sessionから所得する DBTableModelオブジェクトの ID
 *       command            【TAG】コマンド(NEW,RENEW)をセットします(初期値:NEW)
 *       modifyType         【TAG】ファイル取り込み時の モディファイタイプ(A(追加),C(更新),D(削除))を指定します
 *       displayMsg         【TAG】query の結果を画面上に表示するメッセージIDを指定します(初期値:MSG0033[　件検索しました])
 *       notfoundMsg        【TAG】検索結果がゼロ件の場合に表示するメッセージリソースIDを指定します(初期値:MSG0077[対象データはありませんでした])
 *       checkColumns       【TAG】読み取り元ファイルの整合性チェックを行うカラム列をカンマ指定します
 *       nullCheck          【TAG】NULL チェックすべきカラム列をカンマ区切り(CVS形式)で指定します
 *       language           【TAG】タグ内部で使用する言語コード[ja/en/zh/…]を指定します
 *       stopZero           【TAG】読込件数が０件のとき処理を続行するかどうか[true/false]を指定します(初期値:false[続行する])
 *       scope              【TAG】キャッシュする場合のスコープ[request/page/session/applicaton]を指定します(初期値:session)
 *       mainTrans          【TAG】(通常使いません)タグで処理される処理がメインとなるトランザクション処理かどうかを指定します(初期値:false)
 *       skipRowCount       【TAG】(通常は使いません)データの読み飛ばし件数を設定します
 *       useRenderer        【TAG】読取処理でラベルをコードリソースに逆変換を行うかどうかを指定します (初期値:false)
 *       caseKey            【TAG】このタグ自体を利用するかどうかの条件キーを指定します(初期値:null) 
 *       caseVal            【TAG】このタグ自体を利用するかどうかの条件値を指定します(初期値:null) 
 *       caseNN             【TAG】指定の値が、null/ゼロ文字列 でない場合(Not Null=NN)は、このタグは使用されます(初期値:true) 
 *       caseNull           【TAG】指定の値が、null/ゼロ文字列 の場合は、このタグは使用されます(初期値:true) 
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   /&gt;
 *
 * ●使用例
 *
 *     &lt;og:readJSON
 *         command        = "NEW"
 *         readerClass    = "Default"             
 *         maxRowCount    = "10000"               読取最大件数(0:[無制限])
 *         checkColumns   = "OYA,KO,HJO,SU"       整合性チェックするカラム列("*" で全カラム)
 *         nullCheck      = "OYA,KO,SU"           NULLチェックを実行します("*" で全カラム)
 *         stopZero       = "true"                取得0件の場合に以降の処理を停止します
 *     &gt;
 *       [JSON Data]
 *     &lt;og:readJSON /&gt;
 *
 * @og.group ファイル入力
 *
 * @version  4.0
 * @author   Takahashi Masakazu
 * @since    JDK5.0,
 */
public class ReadJSONTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String		VERSION				= "5.7.7.2 (2014/06/20)";

	private static final long		serialVersionUID	= 577220140620L;

	private static final int		ERROR_ROW_COUNT		= 200;													// 4.0.0 (2007/05/25)

	/** command 引数に渡す事の出来る コマンド  新規作成 {@value} */
	public static final String		CMD_NEW				= "NEW";
	/** command 引数に渡す事の出来る コマンド  再検索 {@value} */
	public static final String		CMD_RENEW			= "RENEW";
	private static final String[]	COMMAND_LIST		= new String[] { CMD_NEW, CMD_RENEW };

	private String					jsonData			= null;												//変換JSONデータ
	private String					readerClass			= "Default";
	private int						maxRowCount			= -1;
	private String					displayMsg			= HybsSystem.sys( "VIEW_DISPLAY_MSG" );
	private String					notfoundMsg			= "MSG0077";											// 対象データはありませんでした。
	private int						executeCount		= -1;													// 検索/実行件数
	private String					modifyType			= null;
	private String					checkColumns		= null;												//  取り込み時チェック
	private String					nullCheck			= null;												//  nullチェック確認
	private transient DBTableModel	table				= null;
	private String					command				= CMD_NEW;
	private String					tableId				= HybsSystem.TBL_MDL_KEY;
	private boolean					stopZero			= false;												//  stopZero属性追加
	private boolean					isMainTrans			= true;												//  DBLastSqlの処理の見直し
	private int						skipRowCount		= 0;													//  データの読み飛ばし設定

	//  読取処理でコードリソースのラベル変換を行うかどうか。
	private boolean					useRenderer			= false;

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 *
	 * @return	後続処理の指示( EVAL_BODY_BUFFERED )
	 */
	@Override
	public int doStartTag() {
		if( useTag() ) {

			return EVAL_BODY_BUFFERED ;	// Body を評価する。( extends BodyTagSupport 時)
		}
		return  SKIP_BODY ;				// Body を評価しない
	}
	
	/**
	 * Taglibのタグ本体を処理する doAfterBody() を オーバーライドします。
	 *
	 * @return	後続処理の指示(SKIP_BODY)
	 */
	@Override
	public int doAfterBody() {
		if( jsonData == null || jsonData.length() <= 0 ) {
			jsonData = getBodyString();
		}
		return SKIP_BODY;
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();
		if( !useTag() ) { return EVAL_PAGE; }

		if( jsonData != null && jsonData.length() > 0 && check( command, COMMAND_LIST ) ) {
			useMainTrans( isMainTrans );
			startQueryTransaction( tableId );

			if( "session".equals( getScope() ) ) {
				removeSessionAttribute( tableId );
				removeSessionAttribute( HybsSystem.VIEWFORM_KEY );
			}

			if( maxRowCount < 0 ) {
				maxRowCount = sysInt( "DB_MAX_ROW_COUNT" );
			}

			create( jsonData );

			if( table != null ) {
				//  checkTableColumn 前に、modifyType 設定を行います。
				executeCount = table.getRowCount();
				if( modifyType != null ) {
					for( int row = 0; row < executeCount; row++ ) {
						table.setModifyType( row, modifyType );
					}
				}

				ErrorMessage errMsg = checkTableColumn( table );
				if( errMsg != null && !errMsg.isOK() ) {
					jspPrint( TaglibUtil.makeHTMLErrorTable( errMsg, getResource() ) );
					return SKIP_PAGE;
				}

			}
			// トランザクションチェックを行います。
			if( !commitTableObject( tableId, table ) ) {
				jspPrint( "ReadTableTag Query処理が割り込まれました。DBTableModel は登録しません。" );
				return SKIP_PAGE;
			}

			StringBuilder buf = new StringBuilder( HybsSystem.BUFFER_SMALL );

			// 実行件数の表示 command="NEW" のときのみ、displayMsg を表示させます。
			if( CMD_NEW.equals( command ) ) {
				if( executeCount > 0 && displayMsg != null && displayMsg.length() > 0 ) {
					buf.append( executeCount );
					buf.append( getResource().getLabel( displayMsg ) );
					buf.append( HybsSystem.BR );
				}
				else if( executeCount == 0 && notfoundMsg != null && notfoundMsg.length() > 0 ) {
					buf.append( getResource().getLabel( notfoundMsg ) );
					buf.append( HybsSystem.BR );
				}
			}

			// 読込件数を、"DB.COUNT" キーでリクエストにセットする。
			setRequestAttribute( "DB.COUNT", String.valueOf( executeCount ) );

			jspPrint( buf.toString() );
		}

		// stopZero機能を追加
		final int rtnCode;
		if( executeCount < 0 && stopZero ) {
			rtnCode = SKIP_PAGE;
		}
		else {
			rtnCode = EVAL_PAGE;
		}

		return rtnCode;
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
		readerClass = "Default";
		maxRowCount = -1;
		displayMsg = HybsSystem.sys( "VIEW_DISPLAY_MSG" );
		notfoundMsg = "MSG0077"; // 対象データはありませんでした。
		executeCount = -1; // 検索/実行件数
		modifyType = null;
		command = CMD_NEW;
		table = null;
		tableId = HybsSystem.TBL_MDL_KEY;
		checkColumns = null; // 取り込み時チェック
		nullCheck = null;
		stopZero = false;
		isMainTrans = true;
		skipRowCount = 0;
		useRenderer = false;
		jsonData = null;
	}

	/**
	 * TableReader の実オブジェクトを生成して，BufferedReader に書き込みます。
	 *
	 *
	 * @param	out	出力するBufferedReaderオブジェクト
	 */
	protected void create( final String out ) {
		String className = HybsSystem.sys( "JsonReader_" + readerClass );
		JsonReader reader = (JsonReader) HybsSystem.newInstance( className );
		reader.setResourceManager( getResource() );
		reader.setMaxRowCount( maxRowCount );
		reader.setSkipRowCount( skipRowCount );
		reader.setUseRenderer( useRenderer );
		reader.setDebug( isDebug() );
		reader.readDBTable( out );
		table = reader.getDBTableModel();
	}

	/**
	 * カラム文字列(CSV形式)から、カラム番号配列を作成します。
	 * 簡易メソッドです。
	 * 引数が、"*" の場合は、全カラムを指定したことになります。
	 * null の場合は、サイズが ０ の配列を返します。
	 *
	 *
	 * @param	clms 	カラム文字列(CSV形式)
	 * @param	table	DBTableModelオブジェクト
	 *
	 * @return	カラム番号配列(無い場合は、長さ０の配列)
	 */
	private int[] makeClmNos( final String clms, final DBTableModel table ) {
		final int[] clmNo;

		if( clms == null ) {
			clmNo = new int[0];
		}
		else if( "*".equals( clms ) ) {
			int size = table.getColumnCount();
			clmNo = new int[size];
			for( int i = 0; i < size; i++ ) {
				clmNo[i] = i;
			}
		}
		else {
			String[] clmStr = StringUtil.csv2Array( clms );
			int size = clmStr.length;
			clmNo = new int[size];
			for( int i = 0; i < size; i++ ) {
				clmNo[i] = table.getColumnNo( clmStr[i] );
			}
		}

		return clmNo;
	}

	/**
	 * checkColumns に指定されたカラムをチェックします。
	 * カラムオブジェクトのDBType属性に対応したチェックを行います。
	 * チェック結果で、エラーが発生した場合は、ErrorMessage オブジェクトを
	 * 返します。
	 * DBColumn#valueCheck( String ) の結果のErrorMessageをすべて append
	 * していきます。
	 * debug=true で、エラー時の詳細なデータを出力します。
	 *
	 * @param	table	DBTableModelオブジェクト
	 *
	 * @return	カラムキー ＋ 値 のエラーメッセージオブジェクト
	 */
	private ErrorMessage checkTableColumn( final DBTableModel table ) {
		ErrorMessage errMsg = new ErrorMessage( "Check Columns Error!" );

		int rowCnt = table.getRowCount();
		int[] chkClmNo = makeClmNos( checkColumns, table );
		int[] nllclmNo = makeClmNos( nullCheck, table );

		for( int row = 0; row < rowCnt; row++ ) {
			String[] vals = table.getValues( row );
			DBColumn[] dbClms = table.getDBColumns();
			boolean isError = false;

			// checkColumns 処理
			for( int i = 0; i < chkClmNo.length; i++ ) {
				int no = chkClmNo[i];
				ErrorMessage msg = dbClms[no].valueCheck( vals[no] );
				if( msg.getKekka() > ErrorMessage.OK ) {
					isError = true;
					errMsg.append( row + 1, dbClms[no].valueCheck( vals[no] ) );
				}
			}

			// nullCheck 処理
			for( int i = 0; i < nllclmNo.length; i++ ) {
				int no = nllclmNo[i];
				if( vals[no] == null || vals[no].length() == 0 ) {
					isError = true;
					String label = dbClms[no].getLabel();
					// ERR0012 : 指定のデータがセットされていません。(NULLエラー)。key={0}
					errMsg.addMessage( row + 1, ErrorMessage.NG, "ERR0012", label );
				}
			}

			// エラー時のデバッグ出力
			if( isDebug() && isError ) {
				errMsg.addMessage( row + 1, ErrorMessage.OK, "Debug Info", java.util.Arrays.toString( table.getValues( row ) ) );
			}

			if( errMsg.size() > ERROR_ROW_COUNT ) {
				break;
			}
		}

		return errMsg;
	}

	/**
	 * 【TAG】(通常は使いません)結果のDBTableModelを、sessionに登録するときのキーを指定します
	 *		(初期値:HybsSystem#TBL_MDL_KEY[={@og.value org.opengion.hayabusa.common.HybsSystem#TBL_MDL_KEY}])。
	 *
	 * @og.tag
	 * 検索結果より、DBTableModelオブジェクトを作成します。これを、下流のviewタグ等に
	 * 渡す場合に、通常は、session を利用します。その場合の登録キーです。
	 * query タグを同時に実行して、結果を求める場合、同一メモリに配置される為、
	 * この tableId 属性を利用して、メモリ空間を分けます。
	 *		(初期値:HybsSystem#TBL_MDL_KEY[={@og.value org.opengion.hayabusa.common.HybsSystem#TBL_MDL_KEY}])。
	 *
	 * @param	id sessionに登録する時の ID
	 */
	public void setTableId( final String id ) {
		tableId = nval( getRequestParameter( id ), tableId );
	}

	/**
	 * 【TAG】実際に読み出すクラス名の略称(JsonReader_**** の ****)をセットします({@og.doc03Link readerClass 初期値:Default})。
	 *
	 * @og.tag
	 * 実際に読み出すクラス名(の略称)をセットします。
	 * これは、org.opengion.plugin.json 以下の JsonReader_**** クラスの **** を
	 * 与えます。これらは、JsonReader インターフェースを継承したサブクラスです。
	 * 
	 * @param   readerClass クラス名(の略称)
	 */
	public void setReaderClass( final String readerClass ) {
		this.readerClass = nval( getRequestParameter( readerClass ), this.readerClass );
	}

	/**
	 * 【TAG】読取時の最大取り込み件数をセットします
	 *		(初期値:DB_MAX_ROW_COUNT[={@og.value org.opengion.hayabusa.common.SystemData#DB_MAX_ROW_COUNT}])。
	 *
	 * @og.tag
	 * DBTableModelのデータとして登録する最大件数をこの値に設定します。
	 * サーバーのメモリ資源と応答時間の確保の為です。
	 * 0 をセットすると、無制限(Integer.MAX_VALUE)になります。
	 * (初期値:ユーザー定数のDB_MAX_ROW_COUNT[={@og.value org.opengion.hayabusa.common.SystemData#DB_MAX_ROW_COUNT}])。
	 *
	 *
	 * @param   count 読取時の最大取り込み件数
	 * @see		org.opengion.hayabusa.common.SystemData#DB_MAX_ROW_COUNT
	 */
	public void setMaxRowCount( final String count ) {
		maxRowCount = nval( getRequestParameter( count ), maxRowCount );
		if( maxRowCount == 0 ) {
			maxRowCount = Integer.MAX_VALUE;
		}
	}

	/**
	 * 【TAG】コマンド(NEW,RENEW)をセットします(初期値:NEW)。
	 *
	 * @og.tag
	 * コマンドは,HTMLから(get/post)指定されますので,CMD_xxx で設定される
	 * フィールド定数値のいづれかを、指定できます。
	 * 何も設定されない、または、null の場合は、"NEW" が初期値にセットされます。
	 *
	 * @param	cmd コマンド(public static final 宣言されている文字列)
	 * @see		<a href="../../../../constant-values.html#org.opengion.hayabusa.taglib.ReadTableTag.CMD_NEW">コマンド定数</a>
	 */
	public void setCommand( final String cmd ) {
		String cmd2 = getRequestParameter( cmd );
		if( cmd2 != null && cmd2.length() > 0 ) {
			command = cmd2.toUpperCase( Locale.JAPAN );
		}
	}

	/**
	 * 【TAG】query の結果を画面上に表示するメッセージIDを指定します(初期値:MSG0033[　件検索しました])。
	 *
	 * @og.tag
	 * ここでは、検索結果の件数や登録された件数をまず出力し、
	 * その次に、ここで指定したメッセージをリソースから取得して
	 * 表示します。
	 * 表示させたくない場合は, displayMsg = "" をセットしてください。
	 * 初期値は、検索件数を表示します。
	 *
	 * @param   id ディスプレイに表示させるメッセージ ID
	 */
	public void setDisplayMsg( final String id ) {
		if( id != null ) {
			displayMsg = id;
		}
	}

	/**
	 * 【TAG】検索結果がゼロ件の場合に表示するメッセージリソースIDを指定します(初期値:MSG0077[対象データはありませんでした])。
	 *
	 * @og.tag
	 * ここでは、検索結果がゼロ件の場合のみ、特別なメッセージを表示させます。
	 * 従来は、displayMsg と兼用で、『0　件検索しました』という表示でしたが、
	 * displayMsg の初期表示は、OFF になりましたので、ゼロ件の場合のみ別に表示させます。
	 * 表示させたくない場合は, notfoundMsg = "" をセットしてください。
	 * 初期値は、MSG0077[対象データはありませんでした]です。
	 *
	 * @param	id ディスプレイに表示させるメッセージ ID
	 */
	public void setNotfoundMsg( final String id ) {
		String ids = getRequestParameter( id );
		if( ids != null ) {
			notfoundMsg = ids;
		}
	}

	/**
	 * 【TAG】取り込み時の モディファイタイプ(A(追加),C(更新),D(削除))を指定します。
	 *
	 * @og.tag
	 * JSON読み込み時に、そのデータをA(追加)、C(更新)、D(削除)の
	 * モディファイタイプをつけた状態にします。
	 * その状態で、そのまま、update する事が可能になります。
	 *
	 * @param   type ファイル取り込み時の モディファイタイプ(A,C,D属性)
	 */
	public void setModifyType( final String type ) {
		modifyType = getRequestParameter( type );
	}

	/**
	 * 【TAG】整合性チェックを行うカラム列をカンマ指定します。
	 *
	 * @og.tag
	 * カラムオブジェクトのDBType属性に対応したチェックを行います。
	 * 指定のカラム名をカンマ区切り(CSV)で複数指定できます。
	 * 全てのカラムのチェックを行う場合は、allColumnCheck = "true" を
	 * 指定して下さい。
	 * 分解方法は、通常のパラメータ取得後に、CSV分解します。
	 *
	 *
	 * @param   clms 整合性チェックを行うカラム列(カンマ区切り文字)
	 */
	public void setCheckColumns( final String clms ) {
		checkColumns = nval( getRequestParameter( clms ), checkColumns );
	}

	/**
	 * 【TAG】NULL チェックすべきカラム列をカンマ区切り(CVS形式)で指定します。
	 *
	 * @og.tag nullCheck="AAA,BBB,CCC,DDD"
	 * 分解方法は、通常のパラメータ取得後に、CSV分解します。
	 *
	 * @param   clms カラム列(CVS形式)
	 */
	public void setNullCheck( final String clms ) {
		nullCheck = nval( getRequestParameter( clms ), nullCheck );
	}

	/**
	 * 【TAG】読込件数が０件のとき処理を続行するかどうか[true/false]を指定します(初期値:false[続行する])。
	 *
	 * @og.tag
	 * 初期値は、false(続行する)です。
	 *
	 *
	 * @param  cmd 読込件数が０件のとき、処理を [true:中止する/false:続行する]
	 */
	public void setStopZero( final String cmd ) {
		stopZero = nval( getRequestParameter( cmd ), stopZero );
	}

	/**
	 * 【TAG】(通常使いません)タグで処理される処理がメインとなるトランザクション処理かどうかを指定します(初期値:false)。
	 *
	 * @og.tag
	 * この値は、ファイルダウンロード処理に影響します。この値がtrueに指定された時にcommitされたDBTableModelが
	 * ファイルダウンロードの対象の表になります。
	 *
	 * このパラメーターは、通常、各タグにより実装され、ユーザーが指定する必要はありません。
	 * 但し、1つのJSP内でDBTableModelが複数生成される場合に、前に処理したDBTableModelについてファイルダウンロードをさせたい
	 * 場合は、後ろでDBTableModelを生成するタグで、明示的にこの値をfalseに指定することで、ファイルダウンロード処理の対象から
	 * 除外することができます。
	 *
	 *
	 * @param  flag メイントランザクションかどうか
	 */
	public void setMainTrans( final String flag ) {
		isMainTrans = nval( getRequestParameter( flag ), isMainTrans );
	}

	/**
	 * 【TAG】(通常は使いません)データの読み飛ばし件数を設定します。
	 *
	 * @og.tag
	 * データの読み始めの初期値を指定します。
	 * 先頭行が、０行としてカウントしますので、設定値は、読み飛ばす
	 * 件数になります。(１と指定すると、１件読み飛ばし、２行目から読み込みます。)
	 * 読み飛ばしは、コメント行などは、無視しますので、実際の行数分読み飛ばします。
	 * ＃NAME属性や、columns 属性は、有効です。
	 *
	 *
	 * @param	count 読み始めの初期値
	 */
	public void setSkipRowCount( final String count ) {
		skipRowCount = nval( getRequestParameter( count ), skipRowCount );
	}

	/**
	 * 【TAG】読取処理でラベルをコードリソースに逆変換を行うかどうかを指定します
	 *
	 * @og.tag
	 * コードリソースのカラムに対して、ラベルからコードを求める逆変換を行うことで、
	 * Renderer 系で出力したデータを取り込むことができるようにします。
	 *
	 * JSONでの出力ではRenderer系で出力する事は少ないと思われるため、初期値はfalseとしておきます。
	 *
	 *
	 * @param  flag コードリソースのラベル逆変換を行うかどうか
	 */
	public void setUseRenderer( final String flag ) {
		useRenderer = nval( getRequestParameter( flag ), useRenderer );
	}

	/**
	 * 【TAG】テーブルモデルに取り込むJSONデータ
	 *
	 * @og.tag
	 * テーブルモデルに取り込むためのJSONデータを指定します。
	 * タグのBODYでも指定可能です。
	 *
	 * @param json 取り込むJSON形式データ
	 */
	public void setJSONData( final String json ) {
		jsonData = nval( getRequestParameter( json ), jsonData );
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	@Override
	public String toString() {
		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() ).println( "VERSION", VERSION ).println( "readerClass", readerClass ).println( "maxRowCount", maxRowCount ).println( "displayMsg", displayMsg ).println( "executeCount", executeCount ).println( "modifyType", modifyType ).println( "checkColumns", checkColumns ).println( "nullCheck", nullCheck ).println( "command", command ).println( "tableId", tableId ).println( "Other...", getAttributes().getAttribute() ).fixForm().toString();
	}
}
