/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import static org.opengion.fukurou.util.StringUtil.*;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;

/**
 * Where句を作成するための条件を指定します。
 *
 * このタグのvalue 値に、{&#064;XXXX} 変数が含まれている場合、そのリクエスト値が
 * ない場合は、このタグそのものがなにも出力しません。(つまり条件から消えます。)
 * startKeyは、value を連結する場合の頭に置かれる文字列で、where句の最初には表示されず、
 * それ以降について、表示されます。(つまり、where VALUE1 and VALUE2 and VALUE3 … です。)
 * startKey の初期値は、"and" です。
 * multi は、{&#064;XXXX} 変数に、値が複数含まれている場合の処理を規定します。
 * 複数の値とは、同一nameでチェックボックス指定や、メニューでの複数指定した場合、
 * リクエストが配列で送られます。multi="true" とすると、'xx1','xx2','xx3', ･･･ という
 * 形式に変換されます。
 * 具体的には、"where PN in ( {&#064;PN} )" という文字列に対して、
 * "where PN in ( 'xx1','xx2','xx3' )" を作成することができます。
 * multi の初期値は、"false" です。
 * SystemData の USE_SQL_INJECTION_CHECK が true か、quotCheck 属性が true の場合は、
 * ＳＱＬインジェクション対策用のクォーティションチェックを行います。リクエスト引数に
 * クォーティション(')が含まれると、エラーになります。
 * 同様にUSE_XSS_CHECKがtrueか、xssCheck属性がtrueの場合は、
 * クロスサイトススクリプティング(XSS)対策のためless/greater than signのチェックを行います。
 *
 * 各属性は、{&#064;XXXX} 変数が使用できます。
 * これは、ServletRequest から、XXXX をキーに値を取り出し,この変数に割り当てます。
 * つまり、このXXXXをキーにリクエストすれば、この変数に値をセットすることができます。
 *
 * @og.formSample
 * ●形式：&lt;og:and startKey="[and|or|…]" value="…" multi="[false|true]" /&gt;
 * ●body：あり(EVAL_BODY_BUFFERED:BODYを評価し、{&#064;XXXX} を解析します)
 *
 * ●Tag定義：
 *   &lt;og:and
 *       startKey           【TAG】SQL条件句の最初の演算子を指定します(初期値:and)
 *       value              【TAG】条件の値を セットします
 *       multi              【TAG】複数の引数に対して処理するかどうか[true/false]を設定します(初期値:false)
 *       separator          【TAG】multi アクション時の文字列を分割する項目区切り文字をセットします
 *       quotCheck          【TAG】リクエスト情報の クォーティション(') 存在チェックを実施するかどうか[true/false]を設定します (初期値:USE_SQL_INJECTION_CHECK[=true])
 *       instrVals          【TAG】スペースで区切られた複数の値すべてを含む条件を作成します
 *       instrType          【TAG】instrValsで複数の値を条件にする際の方法を指定します(初期値:and)
 *       xssCheck           【TAG】リクエスト情報の HTMLTag開始/終了文字(&gt;&lt;) 存在チェックを実施するかどうか[true/false]を設定します (初期値:USE_XSS_CHECK[=true])
 *       caseKey            【TAG】このタグ自体を利用するかどうかの条件キーを指定します(初期値:null)
 *       caseVal            【TAG】このタグ自体を利用するかどうかの条件値を指定します(初期値:null)
 *       caseNN             【TAG】指定の値が、null/ゼロ文字列 でない場合(Not Null=NN)は、このタグは使用されます(初期値:true)
 *       caseNull           【TAG】指定の値が、null/ゼロ文字列 の場合は、このタグは使用されます(初期値:true)
 *       placeHolder        【TAG】value の?に設定する値を指定します。(queryType="JDBCPrepared"専用)
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   &gt;   ... Body ...
 *   &lt;/og:and&gt;
 *
 * ●使用例
 *     &lt;og:query command="NEW"&gt;
 *             select PN,YOBI,NMEN,HINM from XX01
 *         &lt;og:where&gt;
 *             &lt;og:and value="PN   =    '{&#064;PN}'"    /&gt;
 *             &lt;og:and value="YOBI like '{&#064;YOBI}%'" /&gt;
 *         &lt;/og:where&gt;
 *             order by PN
 *     &lt;/og:query&gt;
 *
 *          ・検索条件が入力された時(PN=AAA , YOBI=BBB)
 *            作成されるSQL文⇒select PN,YOBI,NMEN,HINM from XX01 where PN = 'AAA' and YOBI like 'BBB%' order by PN
 *
 *          ・検索条件が片方入力されなかった時(PNがNULLのとき, YOBI=BBB)
 *            作成されるSQL文⇒select PN,YOBI,NMEN,HINM from XX01 where YOBI like 'BBB%' order by PN
 *
 *          ・検索条件が入力されなかった時(PNがNULL, YOBIがNULL) WHERE句がなくなる。
 *            作成されるSQL文⇒select PN,YOBI,NMEN,HINM from XX01 order by PN
 *
 *        注意:WhereTagを使わない場合に、検索条件が入力されなかった場合は、下記のようになります。
 *            select PN,YOBI,NMEN,HINM from XX01 where PN = '' and YOBI like '%' order by PN
 *
 *    --------------------------------------------------------------------------------------------------------------
 *
 *     &lt;og:query command="NEW"&gt;
 *             select PN,YOBI,NMEN,HINM from XX01 where PN="11111"
 *         &lt;og:where startKey="and"&gt;
 *             &lt;og:and value="YOBI in   ({&#064;YOBI})" multi="true" /&gt;
 *             &lt;og:and value="HINM like '{&#064;HINM}%'"             /&gt;
 *         &lt;/og:where&gt;
 *             order by PN
 *     &lt;/og:query&gt;
 *
 *          ・YOBI を複数選択し、in で検索する時(YOBI=AA,BB,CC を選択)
 *            作成されるSQL文⇒select PN,YOBI,NMEN,HINM from XX01 where PN = '11111'
 *                             and YOBI in ( 'AA','BB','CC' ) and HINM like 'BBB%' order by PN
 *
 *    --------------------------------------------------------------------------------------------------------------
 *    placeHolderを利用する場合の利用例。
 *    queryタグのsqlTypeはJDBCPrepared専用です。
 *    
 *		&lt;og:query command="NEW" sqlType="JDBCPrepared"&gt;
 * 			SELECT * FROM XX01
 *			&lt;og:where&gt;
 *				&lt;og:and value="K01 = ?" placeHolder="{@VAL1}" /&gt;
 *				&lt;og:and value="K02 LIKE ?" placeHolder="{@VAL2}%" /&gt;
 *				&lt;og:and value="K03 IN (?)" multi="true" placeHolder="{@VAL3}" /&gt;
 *				&lt;og:and value="K04 = ? || ?" placeHolder="{@VAL4},{@VAL5}" /&gt;
 *				&lt;og:and value="K05 IN (?,?)" multi="true" placeHolder="{@VAL6},{@VAL7}" /&gt;
 *			&lt;/og:where&gt;
 *		&lt;/og:query&gt;
 *
 *		※SELECT句に?を利用する場合は、下記のように、queryタグのnames属性を利用してください。
 *
 *		&lt;og:query command="NEW" sqlType="JDBCPrepared" names="VAL1"&gt;
 *			SELECT ?, A.*  FROM XX01 A
 *			&lt;og:and value="A.K01 = ?" placeHolder="{@VAL2}" /&gt;
 *		&lt;/og:query&gt;
 *
 * @og.group 画面部品
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class SqlAndTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "5.5.1.1 (2012/04/06)" ;

	private static final long serialVersionUID = 551120120406L ;

	private String	startKey	= "and";
	private String	value		= "";
	private String	instrVals	= null;		// 3.8.8.1 (2007/01/06)
	private String	instrType	= "and";	// 5.4.1.0 (2011/11/01)
	private boolean	multi		= false;
	private boolean	quotCheck	= HybsSystem.sysBool( "USE_SQL_INJECTION_CHECK" );	// 4.0.0 (2005/08/31)
	private boolean	xssCheck	= HybsSystem.sysBool( "USE_XSS_CHECK" );	// 5.0.0.2 (2009/09/15)

	private boolean	allNull	 	= false;	// 5.0.0.2 (2009/09/15)

//	private String	matchKey	= null;		// 5.1.9.0 (2010/08/01) ⇒ 5.2.2.0 (2010/11/01) 廃止(caseKey,caseVal属性を使用してください。)

//	private String	matchVal	= null;		// 5.1.9.0 (2010/08/01) ⇒ 5.2.2.0 (2010/11/01) 廃止(caseKey,caseVal属性を使用してください。)

	private String	separator	= null;		// 5.2.2.0 (2010/11/01) 項目区切り文字
	
	// 5.10.2.1 (2018/08/18)
	private String placeHolder 		= null;
	private final String regex2		= "\\?";					// multipleの?追加
	private final String afRegex2		= "?,?";
	private Pattern pattern_multi		= Pattern.compile(regex2);
	private final String regex3		= "\\{@.+?\\}";				// 文字列から、{@XXX}を抽出(multipleの複数指定用)
	private Pattern pattern_trg		= Pattern.compile(regex3);
	
	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @og.rev 4.0.0.0 (2006/12/05) BODY 部の値を value に使用する機能追加
	 * @og.rev 4.0.0.0 (2005/08/31) useQuotCheck() によるＳＱＬインジェクション対策
	 * @og.rev 5.0.0.2 (2009/09/15) XSS対策
	 * @og.rev 5.2.2.0 (2010/11/01) caseKey 、caseVal 属性対応
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doStartTag() {
		// 5.2.2.0 (2010/11/01) caseKey 、caseVal 属性対応
		if( useTag() ) {
			useQuotCheck( quotCheck );
			// 5.0.0.2 (2009/09/15) XSS対策
			useXssCheck( xssCheck );

			value = getRequestParameter( value );
			
			if( value == null || value.isEmpty() ) {
				return( EVAL_BODY_BUFFERED );	// Body を評価する。( extends BodyTagSupport 時)
			}

	//		if( value != null && value.length() > 0 ) {
	//			return( SKIP_BODY );			// Body を評価しない
	//		}
	//		else {
	//			return( EVAL_BODY_BUFFERED );	// Body を評価する。( extends BodyTagSupport 時)
	//		}
		}
		return( SKIP_BODY );			// Body を評価しない
	}

	/**
	 * Taglibのタグ本体を処理する doAfterBody() を オーバーライドします。
	 *
	 * @og.rev 4.0.0.0 (2006/12/05) BODY 部の値を value に使用する機能追加
	 *
	 * @return	後続処理の指示(SKIP_BODY)
	 */
	@Override
	public int doAfterBody() {
		value = getBodyString();
		return(SKIP_BODY);
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.8.8.1 (2007/01/06) makeInstrVals を加味する。
	 * @og.rev 5.0.0.2 (2009/09/15) multi時のallNull対応
	 * @og.rev 5.1.9.0 (2010/08/01) matchKey 、matchVal 対応 ⇒ 5.2.2.0 (2010/11/01) 廃止
	 * @og.rev 5.2.2.0 (2010/11/01) caseKey 、caseVal 属性対応
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)
		// 5.2.2.0 (2010/11/01) caseKey 、caseVal 属性対応
		if( useTag() ) {
			SqlWhereTag where = (SqlWhereTag)findAncestorWithClass( this,SqlWhereTag.class );
			if( where == null ) {
//				String errMsg = "<b>このタグは、where タグの内部におく必要があります。</b>";
				String errMsg = "<b>" + getTagName() + "タグは、where タグの内部におく必要があります。</b>";
				throw new HybsSystemException( errMsg );
			}

			// 5.1.9.0 (2010/08/01) matchKey 、matchVal 対応 ⇒ 5.2.2.0 (2010/11/01) 廃止
	//		boolean flag = (matchKey == null) || (matchVal == null) || matchKey.matches( matchVal ) ;

			// 5.10.2.1 (2018/08/18) 
			// プレースホルダーが設定されている場合は、プレースホルーダの括弧整合性チェック
			if( placeHolder!=null && placeHolder.length() > 0 ) {
				getRequestParameter( placeHolder );
			}
			
			// if( ! isNull() ) {
	//		if( ! isNull() && ! allNull ) {			// 5.0.0.2 (2009/09/15)
	//		if( ! isNull() && ! allNull && flag ) { // 5.1.9.0 (2010/08/01)
			if( ! isNull() && ! allNull ) {			// 5.2.2.0 (2010/11/01)
//				value = makeInstrVals( instrVals,value );	// 3.8.8.1 (2007/01/06)
				value = makeInstrVals( instrVals,instrType,value );	// 5.4.1.0 (2011/11/01)
				
				if( value != null ) {
					// 5.10.2.1 (2018/08/18)
					// プレースホルダー指定の場合の処理
					if( placeHolder!=null && placeHolder.length() > 0 ) {
						QueryTag query = null;
						query = (QueryTag)findAncestorWithClass( this, QueryTag.class);
						if(multi) {
							// multiの場合は、?の数を増やして、設定する値も増やす
							// placeHolder属性から、対象文字列を取得。
							// 複数指定の場合を、考慮して、繰り返し処理を行う。
							Matcher matcher_trg = pattern_trg.matcher(placeHolder);
							while(matcher_trg.find()) {
								String trgPlaceHolder = matcher_trg.group();
								String[] list = getRequestParameterValues(trgPlaceHolder);
								// ２つ以上の値につき、,?を追加
								for(int i=1; i<list.length; i++) {
									Matcher matcher = pattern_multi.matcher(value);
									value = matcher.replaceFirst(afRegex2);
								}
								
								for(int i=0; i<list.length; i++) {
									query.addListPlaceValue(list[i]);
								}
							}
						}else {
							query.addListPlaceValue(placeHolder);
						}
					}
					
					// whereタグへの通常設定処理
					set( "keyWord", startKey );
					set( "value"  , value );
					where.setAttributes( getAttributes() ); 
				}
			}
		}
		
		return(EVAL_PAGE);
	}

	
	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 2.0.0.4 (2002/09/27) カスタムタグの release() メソッドを、追加
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.8.8.1 (2007/01/06) instrVals 属性追加
	 * @og.rev 4.0.0.0 (2005/08/31) quotCheck 属性の追加
	 * @og.rev 5.0.0.2 (2009/09/15) XSS対応
	 * @og.rev 5.0.0.2 (2009/09/15) multi時のallNull対応
	 * @og.rev 5.1.9.0 (2010/08/01) matchKey、matchVal 属性の追加
	 * @og.rev 5.2.2.0 (2010/11/01) separator , isMatch 属性の追加
	 * @og.rev 5.2.2.0 (2010/11/01) matchKey、matchVal 属性廃止(caseKey,caseVal属性を使用してください。)
	 * @og.rev 5.4.1.0 (2011/11/01) instrType属性追加
	 * @og.rev 5.10.2.1 (2018/08/18) placeHolder属性追加
	 */
	@Override
	protected void release2() {
		super.release2();
		startKey	= "and";
		value		= "";
		instrVals	= null;		// 3.8.8.1 (2007/01/06)
		instrType	= "and";	// 5.4.1.0 (2011/11/01)
		multi		= false;
		quotCheck	= HybsSystem.sysBool( "USE_SQL_INJECTION_CHECK" );	// 4.0.0 (2005/08/31)
		xssCheck	= HybsSystem.sysBool( "USE_XSS_CHECK" );	// 5.0.0.2 (2009/09/15)
		allNull	 	= false;	// 5.0.0.2 (2009/09/15)
//		matchKey	= null;		// 5.1.9.0 (2010/08/01) 新規追加
//		matchVal	= null;		// 5.1.9.0 (2010/08/01) 新規追加
		separator	= null;		// 5.2.2.0 (2010/11/01) 項目区切り文字
		placeHolder	= null;		// 5.10.2.1 (2018/08/18) プレースホルダー判定用
	}

	/**
	 * リクエスト情報の文字列を取得します。
	 *
	 * これは、通常のgetRequestParameter 処理の中で呼ばれる getRequestValue を
	 * オーバーライトしています。
	 *
	 * @og.rev 5.0.0.2 (2009/09/15) valuesの全NULL/空文字をisNull扱いにする
	 * @og.rev 5.3.8.0 (2011/08/01) Attribute等からも値が取得できるようにする。の対応時の特殊処理
	 *
	 * @param    key キー
	 *
	 * @return   リクエスト情報の文字列
	 */
	@Override
	protected String getRequestValue( final String key ) {
		String rtn = "";

		if( multi ) {
			// 5.3.8.0 (2011/08/01) getRequestValues の中で、getRequestValue を呼び出すためこのままでは
			// 再帰呼び出しが永遠に続くので、２回目以降は、再帰しないように、強制的に multi の値を書き換えます。
			multi = false;	// 5.3.8.0 (2011/08/01) 再帰しないように、強制的に値を書き換え
			String[] array = getRequestValues( key );
			allNull = true; // 5.0.0.2 (2009/09/15) arrayの内容が全てnull/空文字か
			if( ! isNull() ) {
				// 5.0.0.2 (2009/09/15) 全てnull/空文字の場合はnullと扱い
				for( int i = 0; i < array.length; i++ ) {
					if( array[i] != null && array[i].length() > 0 ) {
						allNull = false;
						break;
					}
				}
				if( ! allNull ){
					rtn = makeCSVvalue( array );
				}
			}
			multi = true;	// 5.3.8.0 (2011/08/01) 強制的に書き換えた値を元に戻す。
		}
		else {
			rtn = super.getRequestValue( key );
		}
		return rtn ;
	}

	/**
	 * 複数の値を 'xx1','xx2','xx3', ･･･ という形式に変換します。
	 *
	 * この処理は、in などで使用するためのリクエストを配列で受け取って処理
	 * する場合の文字列を加工します。
	 *
	 * @og.rev 5.2.2.0 (2010/11/01) separator 対応
	 *
	 * @param	array	元の配列文字列
	 *
	 * @return  連結後の文字列
	 */
	private String makeCSVvalue( final String[] array ) {
		if( array == null || array.length == 0 ) {
			String errMsg = "array 引数に、null や、サイズゼロの配列は使用できません。";
			throw new HybsSystemException( errMsg );
		}

		StringBuilder buf = new StringBuilder( HybsSystem.BUFFER_MIDDLE );

		if( separator != null ) {
			for(int i=0; i < array.length; i++) {
				String[] ary = array[i].split( separator );
				for( int j=0; j<ary.length; j++ ) {
					buf.append( "'" );
					buf.append( ary[j] );
					buf.append( "'," );
				}
			}
			buf.deleteCharAt( buf.length()-1 );		// 最後の ピリオドを削除する。
		}
		else {
			for(int i=0; i < array.length; i++) {
				buf.append( "'" );
				buf.append( array[i] );
				buf.append( "'," );
			}
			buf.deleteCharAt( buf.length()-1 );		// 最後の ピリオドを削除する。
		}
//		buf.append( "'" );
//		buf.append( array[0] );
//		buf.append( "'" );
//		for(int i=1; i < array.length; i++) {
//			buf.append( ",'" );
//			buf.append( array[i] );
//			buf.append( "'" );
//		}
		return buf.toString();
	}

	/**
	 * スペースで区切られた複数の値を and 接続で連結します。
	 *
	 * value="CLM" instrVals="ABC DEF GHI" と指定すると、
	 * value="CLM LIKE '%ABC%' AND CLM LIKE '%DEF%'  AND CLM LIKE '%GHI%' "
	 * という文字列を作成します。
	 * 個別にLIKE検索項目を AND 連結する為、現れる場所に依存しません。
	 * 逆に、現れる順序を指定する場合は、ABC%DEF の様に指定可能です。
	 * ただし、columnMarker の instrVals で、複数文字のマーカーを行う場合、
	 * ABC%DEF という文字列は、オリジナルでないので、マークアップされません。
	 *
	 * @og.rev 5.4.1.0 (2011/11/01) instrType属性対応
	 * @og.rev 5.5.1.1 (2012/04/06) notin対応
	 *
	 * @param	instrVals 	繰返し処理を行う 値
	 * @param	instrType 	連結方法
	 * @param	value   	繰返し処理を行う value
	 *
	 * @return  連結後の文字列
	 * @see		#setInstrVals( String )
	 * @see		ColumnMarkerTag#setInstrVals( String )
	 */
	private String makeInstrVals( final String instrVals, final String instrType , final String value ) {
		if( instrVals == null || instrVals.length() == 0 ) { return value; }

		String reqVals = nval( getRequestParameter( instrVals ),null );
		if( reqVals == null || reqVals.length() == 0 ) { return null; }

		final String[] vals ;
		if( multi ) {
			// multi のときは、makeCSVvalue で加工された値になっている。
			vals = StringUtil.csv2Array( reqVals,',' );
			// 前後の ' はずし
			for( int i=0; i<vals.length; i++ ) {
				vals[i] = vals[i].substring( 1,vals[i].length()-1 );
			}
		}
		else {
			vals = StringUtil.csv2Array( reqVals,' ' );
		}

		if( vals == null || vals.length == 0 ) { return null; }

		StringBuilder buf = new StringBuilder( HybsSystem.BUFFER_SMALL );

//		buf.append( value );
//		buf.append( " LIKE '%" );
//		buf.append( vals[0] );
//		buf.append( "%'" );

		buf.append( " (" );
		// 条件:and ⇒ 各値をandのlike条件で結合(%あり)
		if( instrType.startsWith( "a" ) ) {
			for(int i=0; i < vals.length; i++) {
				if( i > 0 ) { buf.append( " and " ); }
				buf.append( value );
				buf.append( " LIKE '%" ).append( vals[i] ).append( "%'" );
			}
		}
		// 条件:or ⇒ 各値をorのlike条件で結合(%あり)
		else if( instrType.startsWith( "o" ) ) {
			for(int i=0; i < vals.length; i++) {
				if( i > 0 ) { buf.append( " or " ); }
				buf.append( value );
				buf.append( " LIKE '%" ).append( vals[i] ).append( "%'" );
			}
		}
		// 条件:in ⇒ 各値をorのlike条件で結合(%なし)
		else if( instrType.startsWith( "i" ) ) {
			for(int i=0; i < vals.length; i++) {
				if( i > 0 ) { buf.append( " or " ); }
				buf.append( value );
				buf.append( " LIKE '" ).append( vals[i] ).append( "'" );
			}
		}
		// 条件:notin ⇒ 各値をandのnot like条件で結合(%なし) 5.5.1.1(2012/04/05)
		else if( instrType.startsWith( "n" ) ) {
			for(int i=0; i < vals.length; i++) {
				if( i > 0 ) { buf.append( " and " ); }
				buf.append( value );
				buf.append( " NOT LIKE '" ).append( vals[i] ).append( "'" );
			}
		}
		else {
			String errMsg = "instrTypeには、'and','or','in','notin'のいずれかを指定して下さい。instrType=[" + instrType + "]";
			throw new HybsSystemException( errMsg );
		}
		buf.append( ") " );

		return buf.toString();
	}

	/**
	 * 【TAG】SQL条件句の最初の演算子を指定します(初期値:and)。
	 *
	 * @og.tag
	 * value を連結する場合の頭に置かれる文字列で、where句の最初には表示されず、
	 * それ以降について、表示されます。
	 * (つまり、where VALUE1 and VALUE2 and VALUE3 … です。)
	 * startKey の初期値は、"and" です。
	 *
	 * @param	skey 条件句の最初の演算子
	 */
	public void setStartKey( final String skey ) {
		if( skey != null && skey.length() > 0 ) { startKey = skey; }
	}

	/**
	 * 【TAG】条件の値を セットします。
	 *
	 * @og.tag
	 * 条件値に、{&#064;XXXX} 変数が含まれている場合、そのリクエスト値がない場合は、
	 * このタグそのものがなにも出力しません。(つまり条件から消えます。)
	 * BODY 部に記述することが可能です。その場合は、条件属性になにも設定できません。
	 *
	 * @param	val 条件値
	 */
	public void setValue( final String val ) {
		value = val;
	}

	/**
	 * 【TAG】スペースで区切られた複数の値すべてを含む条件を作成します。
	 *
	 * @og.tag
	 * 通常、value="CLM LIKE 'ABC%'" という文字列を指定しますが、
	 * value="CLM" instrVals="ABC DEF GHI" と指定すると、
	 * value="CLM LIKE '%ABC%' AND CLM LIKE '%DEF%'  AND CLM LIKE '%GHI%' "
	 * という文字列を作成します。
	 * これは、instrVals に指定した引数に対して、スペース区切りで分割し、
	 * 前方の value に複数のAND検索を同時に実現できるように指定します
	 * 個別にLIKE検索項目を AND 連結する為、現れる場所に依存しません。
	 * 逆に、現れる順序を指定する場合は、ABC%DEF の様に指定可能です。
	 * ただし、columnMarker の instrVals で、複数文字のマーカーを行う場合、
	 * ABC%DEF という文字列は、オリジナルでないので、マークアップされません。
	 * ※instrType属性の指定により条件の生成方法を変更することができます。
	 *   詳細については、instrType属性のドキュメントを参照下さい。
	 *
	 * @param	val 複合条件作成のための設定値
	 * @see		#setInstrType
	 * @see		ColumnMarkerTag#setInstrVals( String )
	 */
	public void setInstrVals( final String val ) {
		instrVals = val;
	}

	/**
	 *  【TAG】valueの条件作成を判定します。(JDBCParepared用)
	 *  
	 *  @og.tag
	 *  value="CLM=?" placeHolder="{@CLM}"と指定されていた場合、
	 *  {@CLM}に値が存在する場合のみ、CLM=?が指定されます。
	 *  ※?に設定されえる値はplaceHolderを指定する値には関係なく、
	 *  親のqueryタグに指定した、names属性の値になります。
	 *  
	 *  {@xxx}形式での指定のみ可能です。
	 *  
	 * @og.rev 5.10.2.1 (2018/08/18) 新規追加
	 *
	 * @param val 値
	 */
	public void setPlaceHolder( final String val) {
		// リクエスト変数対応はここでは行わない
		placeHolder = val;
	}
	
	/**
	 * 【TAG】instrValsで複数の値を条件にする際の方法を指定します(初期値:and)。
	 *
	 * @og.tag
	 * 通常、instrValsに指定された値は、スペース区切りで分割した各値をLIKE条件としてand結合します。
	 * しかし、instrType属性を変更することで、この条件式の生成方法を変更
	 * することができます。
	 * 具体的には、以下の通りです。
	 * ①instrTypeに"and"が指定されている場合(初期値)
	 *   タグの記述 : value="CLM" instrVals="ABC DEF GHI"
	 *   生成文字列 :       "( CLM LIKE '%ABC%' AND CLM LIKE '%DEF%' AND CLM LIKE '%GHI%' )"
	 * ②instrTypeに"or"が指定されている場合
	 *   タグの記述 : value="CLM" instrVals="ABC DEF GHI"
	 *   生成文字列 :       "( CLM LIKE '%ABC%' OR CLM LIKE '%DEF%' OR CLM LIKE '%GHI%' )"
	 * ③instrTypeに"in"が指定されている場合
	 *   タグの記述 : value="CLM" instrVals="ABC DEF GHI"
	 *   生成文字列 :       "( CLM LIKE 'ABC' OR CLM LIKE 'DEF5' OR CLM LIKE 'GHI' )"
	 * ④instrTypeに"notin"が指定されている場合
	 * 	 タグの記述 : value="CLM" instrVals="ABC DEF GHI"
	 *   生成文字列 :       "( CLM NOT LIKE 'ABC' AND CLM NOT LIKE 'DEF5' AND CLM NOT LIKE 'GHI' )"
	 * ※この属性を指定しない場合は、①のLIKE条件でのand結合となります。
	 * ※③④について、LIKE条件で%を自動付加しないことにより、画面からの入力値に応じて、
	 *   前方一致、後方一致、前後方一致の制御を行うことができます。
	 *
	 * @og.rev 5.5.1.1 (2012/04/06) notin対応(コメント修正)
	 *
	 * @param	tp 条件方法[and/or/in/notin]
	 * @see		#setInstrVals( String )
	 */
	public void setInstrType( final String tp ) {
		instrType = nval( getRequestParameter( tp ),tp );
	}

	/**
	 * 【TAG】複数の引数に対して処理するかどうか[true/false]を設定します(初期値:false)。
	 *
	 * @og.tag
	 * {&#064;XXXX} 変数に、値が複数含まれている場合の処理を規定します。
	 * multi="true" に設定すると、複数の引数は、'xx1','xx2','xx3', ･･･ という
	 * 形式に変換します。
	 * where 条件で言うと、 "where PN in ( {&#064;PN} )" という文字列に対して、
	 * "where PN in ( 'xx1','xx2','xx3' )" を作成することになります。
	 * 初期値は、 false (マルチ変換しない) です。
	 *
	 * @param   flag マルチ変換 [true:する/それ以外:しない
	 * @see		#setSeparator( String )
	 */
	public void setMulti( final String flag ) {
		multi = nval( flag,multi );
	}

	/**
	 * 【TAG】multi アクション時の文字列を分割する項目区切り文字をセットします。
	 *
	 * @og.tag
	 * multi="true" の場合、複数のリクエストを連結して、 in 句で問合せを行う文字列を
	 * 作成しますが、separator を指定すると、さらに、separator で文字列を分割して、
	 * in 句の引数を構築します。
	 * 具体的には、分割後の文字列が、複数の個々のリクエスト変数と同じ形式に加工されます。
	 * この機能は、multi="true" を指定した場合のみ有効になります。
	 * 初期値は、null です。つまり、分割処理は行いません。
	 *
	 * @og.rev 5.2.2.0 (2010/11/01) 新規追加
	 *
	 * @param   sepa 項目区切り文字
	 * @see		#setMulti( String )
	 */
	public void setSeparator( final String sepa ) {
		separator = nval( getRequestParameter( sepa ),separator );
	}

	/**
	 * 【TAG】リクエスト情報の クォーティション(') 存在チェックを実施するかどうか[true/false]を設定します
	 *		(初期値:USE_SQL_INJECTION_CHECK[={@og.value org.opengion.hayabusa.common.SystemData#USE_SQL_INJECTION_CHECK}])。
	 *
	 * @og.tag
	 * ＳＱＬインジェクション対策の一つとして、暫定的ではありますが、SQLのパラメータに
	 * 渡す文字列にクォーティション(') を許さない設定にすれば、ある程度は防止できます。
	 * 数字タイプの引数には、 or 5=5 などのクォーティションを使用しないコードを埋めても、
	 * 数字チェックで検出可能です。文字タイプの場合は、必ず (')をはずして、
	 * ' or 'A' like 'A のような形式になる為、(')チェックだけでも有効です。
	 * (') が含まれていたエラーにする(true)／かノーチェックか(false)を指定します。
	 * (初期値:システム定数のUSE_SQL_INJECTION_CHECK[={@og.value org.opengion.hayabusa.common.SystemData#USE_SQL_INJECTION_CHECK}])。
	 *
	 * @og.rev 4.0.0.0 (2005/08/31) 新規追加
	 *
	 * @param   flag クォーティションチェック [true:する/それ以外:しない]
	 * @see		org.opengion.hayabusa.common.SystemData#USE_SQL_INJECTION_CHECK
	 */
	public void setQuotCheck( final String flag ) {
		quotCheck = nval( getRequestParameter( flag ),quotCheck );
	}

	/**
	 * 【TAG】リクエスト情報の HTMLTag開始/終了文字(&gt;&lt;) 存在チェックを実施するかどうか[true/false]を設定します
	 *		(初期値:USE_XSS_CHECK[={@og.value org.opengion.hayabusa.common.SystemData#USE_XSS_CHECK}])。
	 *
	 * @og.tag
	 * クロスサイトスクリプティング(XSS)対策の一環としてless/greater than signについてのチェックを行います。
	 * (&gt;&lt;) が含まれていたエラーにする(true)／かノーチェックか(false)を指定します。
	 * (初期値:システム定数のUSE_XSS_CHECK[={@og.value org.opengion.hayabusa.common.SystemData#USE_XSS_CHECK}])。
	 *
	 * @og.rev 5.0.0.2 (2009/09/15) 新規追加
	 *
	 * @param	flag	XSSチェック [true:する/false:しない]
	 * @see		org.opengion.hayabusa.common.SystemData#USE_XSS_CHECK
	 */
	public void setXssCheck( final String flag ) {
		xssCheck = nval( getRequestParameter( flag ),xssCheck );
	}

	/**
	 * 【TAG】このタグ自体を利用するかどうかの条件キーを指定します。
	 *
	 * @og.tag
	 * matchKey.matches( matchVal ) の値が、true の場合は、このタグは使用されます。
	 * false の場合は、このタグは使用されません。
	 * value 値に、{&#064;XXXX} 変数が含まれている場合と同じ効果を得られます。
	 * 何も指定しない場合、または、引数が null の場合は、true と同じで使用されます。
	 *
	 * @og.rev 5.1.9.0 (2010/08/01) 新規追加
	 * @og.rev 5.2.2.0 (2010/11/01) 廃止(caseKey,caseVal属性を使用してください。)
	 *
	 * @param	mkey String
	 */
//	public void setMatchKey( final String mkey ) {
//		matchKey = nval( getRequestParameter( mkey ),matchKey );
//	}

	/**
	 * 【TAG】このタグ自体を利用するかどうかの条件値を指定します。
	 *
	 * @og.tag
	 * matchKey.matches( matchVal ) の値が、true の場合は、このタグは使用されます。
	 * false の場合は、このタグは使用されません。
	 * value 値に、{&#064;XXXX} 変数が含まれている場合と同じ効果を得られます。
	 * 何も指定しない場合、または、引数が null の場合は、true と同じで使用されます。
	 *
	 * @og.rev 5.1.9.0 (2010/08/01) 新規追加
	 * @og.rev 5.2.2.0 (2010/11/01) 廃止(caseKey,caseVal属性を使用してください。)
	 *
	 * @param	mval String
	 */
//	public void setMatchVal( final String mval ) {
//		matchVal = nval( getRequestParameter( mval ),matchVal );
//	}

	/**
	 * タグの名称を、返します。
	 * 自分自身のクラス名より、自動的に取り出せないため、このメソッドをオーバーライドします。
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) 新規追加
	 *
	 * @return  タグの名称
	 */
	@Override
	protected String getTagName() {
		return "and" ;
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	@Override
	public String toString() {
		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() )
				.println( "VERSION"			,VERSION	)
				.println( "startKey"		,startKey	)
				.println( "value"			,value		)
				.println( "instrVals"		,instrVals	)
				.println( "multi"			,multi		)
				.println( "quotCheck"		,quotCheck	)
				.println( "placeHolder"		,placeHolder)	// 5.10.2.0 (2018/08/03) ADD
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
