/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.AbstractEditor;
import org.opengion.hayabusa.db.CellEditor;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.hayabusa.db.SelectionCellEditor;					// 6.2.2.0 (2015/03/27)
import org.opengion.hayabusa.db.Selection;
import org.opengion.hayabusa.db.SelectionFactory;
import org.opengion.fukurou.util.StringFormat;
import org.opengion.fukurou.util.XHTMLTag;
import org.opengion.fukurou.util.Attributes;
import org.opengion.fukurou.util.TagBuffer;

import static org.opengion.fukurou.util.StringUtil.isNull;				// 6.1.1.0 (2015/01/17)

/**
 * INDBMENU エディターは、カラムの表示パラメーターのＳＱＬ文を実行結果より、
 * 作成したプルダウンメニューと、テキストフィールドによる入力の両方をサポートする、
 * 編集に使用するクラスです。
 *
 * JavaScript によりテキストフィールドとメニュー(コンボボックス)を重ね合わせて
 * 表示しておき、メニューで選択した値を、テキストフィールドに設定しています。
 * このエディタを使用するには、jsp/common/inputMenu.js を予め使用できるように
 * 設定しておく必要があります。
 *
 * ここでは、AAA:BBB:CCC:DDD という値を、$1,$2,$3,$4 に割り当てなおして、
 * QUERYを実行します。また、$1 は、本来の値として、メニューの初期値設定等に
 * 使用します。上記の例では、AAA が値で、それ以降は、引数になります。
 * さらに、元の文字列"AAA:BBB:CCC:DDD"は、$0 に割り当てられます。割り当てがない
 * 変数は、""(ゼロ文字列)として、扱われます。
 *
 * このエディタはeventColumnに対応していません。
 *
 *  カラムの表示に必要な属性は, DBColumn オブジェクト より取り出します。
 * このクラスは、DBColumn オブジェクト毎に１つ作成されます。
 *
 * @og.rev 3.5.6.2 (2004/07/05) 新規作成
 * @og.rev 6.2.2.0 (2015/03/27) SelectionCellEditor I/Fを追加
 * @og.group データ編集
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Editor_INDBMENU extends AbstractEditor implements SelectionCellEditor {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "7.0.5.1 (2019/09/27)" ;

	private static final String SEL1 = "<script type=\"text/javascript\">makeInputMenu('" ;
	private static final String SEL2 = "');</script>" ;

	private final String query ;
	private final String dbid ;
	private final String lang ;				// 4.0.0 (2006/11/15)
	private final boolean addNoValue ;		// 3.5.5.7 (2004/05/10)
	private final boolean seqFlag ;			// 3.6.0.6 (2004/10/22)
	private final String useSLabel ;		// 6.2.0.0 (2015/02/27) SLABEL 対応
	private final String addKeyLabel ;		// 6.2.0.0 (2015/02/27) キー:ラベル形式

	private final TagBuffer selTagBuffer = new TagBuffer() ;

	/**
	 * デフォルトコンストラクター。
	 * このコンストラクターで、基本オブジェクトを作成します。
	 *
	 * @og.rev 3.6.0.6 (2004/10/22) シーケンスアクセス機能(seqFlag)を追加します
	 * @og.rev 6.2.0.0 (2015/02/27) SLABEL 対応
	 * @og.rev 6.2.0.0 (2015/02/27) キー:ラベル形式で表示するかどうかを、指定できるようにします。
	 * @og.rev 6.4.4.2 (2016/04/01) nameのfinal化
	 */
	public Editor_INDBMENU() {
		super();		// 6.4.1.1 (2016/01/16) PMD refactoring. It is a good practice to call super() in a constructor
		// 4.3.4.4 (2009/01/01)
		query		= null;
		dbid		= null;
		lang		= null;			// 4.0.0 (2006/11/15)
		addNoValue	= false;		// 3.5.5.7 (2004/05/10)
		seqFlag		= false;		// 3.6.0.6 (2004/10/22)
	//	name		= null;			// 4.3.4.0 (2008/12/01)
		useSLabel	= "auto";		// 6.2.0.0 (2015/02/27) SLABEL 対応
		addKeyLabel = null;			// 6.2.0.0 (2015/02/27) キー:ラベル形式
	}

	/**
	 * DBColumnオブジェクトを指定したprivateコンストラクター。
	 *
	 * @og.rev 3.6.0.6 (2004/10/22) シーケンスアクセス機能(seqFlag)を追加します
	 * @og.rev 4.0.0.0 (2006/11/24) TextField分の属性設定
	 * @og.rev 5.6.3.0 (2013/04/01) プルダウンのonChangeの設定場所を変更
	 * @og.rev 6.2.0.0 (2015/02/27) SLABEL 対応
	 * @og.rev 6.2.0.0 (2015/02/27) キー:ラベル形式で表示するかどうかを、指定できるようにします。
	 * @og.rev 6.4.4.2 (2016/04/01) nameのfinal化
	 * @og.rev 7.0.5.1 (2019/09/27) optionAttributes が二重に設定されていたため、削除
	 *
	 * @param	clm	DBColumnオブジェクト
	 */
	private Editor_INDBMENU( final DBColumn clm ) {
		super( clm );
		tagBuffer.add( XHTMLTag.inputAttri( attributes ) );

		addNoValue	= clm.isAddNoValue() ;		// 3.5.5.7 (2004/05/10)
		query		= clm.getEditorParam();
		dbid		= clm.getDbid();
		lang		= clm.getLang();			// 4.0.0.0 (2006/11/15)
		seqFlag		= false;					// 3.6.0.6 (2004/10/22)
	//	name		= clm.getName(); 			// 4.3.4.0 (2008/12/01) , 6.4.4.2 (2016/04/01) nameのfinal化
		useSLabel	= clm.getUseSLabel() ;		// 6.2.0.0 (2015/02/27) SLABEL 対応
		addKeyLabel = clm.getAddKeyLabel();		// 6.2.0.0 (2015/02/27) キー:ラベル形式

		if( query == null || query.isEmpty() ) {
			final String errMsg = "INDBMENU Editor では、編集パラメータは必須です。"
					+ " name=[" + name + "]" + CR ;
			throw new HybsSystemException( errMsg );
		}

		final String disabled = clm.isWritable() ? null : "disabled" ;

		// 6.1.1.0 (2015/01/17) Attributesの連結記述
		final Attributes selAttri = new Attributes()
			.set( "disabled"	,disabled )
			.set( clm.getEditorAttributes() )				// #addAttributes( Attributes ) の代替え
			.set( "onChange" ,"selChanged(this);" );		// INDBMENU 特有のJavaScript 5.6.3.0 (2013/03/01) 場所変更

		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
		// 7.0.5.1 (2019/09/27) optionAttributes が二重に設定されていたため、削除
		selTagBuffer.add( XHTMLTag.selectAttri( selAttri ) );
//					.add( selAttri.get( "optionAttributes" ) );		// 6.0.4.0 (2014/11/28)
	}

	/**
	 * 各オブジェクトから自分のインスタンスを返します。
	 * 自分自身をキャッシュするのか、新たに作成するのかは、各サブクラスの実装に
	 * まかされます。
	 *
	 * @param	clm	DBColumnオブジェクト
	 *
	 * @return	CellEditorオブジェクト
	 * @og.rtnNotNull
	 */
	public CellEditor newInstance( final DBColumn clm ) {
		return new Editor_INDBMENU( clm );
	}

	/**
	 * データの編集用文字列を返します。
	 *
	 * ここでは、AAA:BBB:CCC:DDD という値を、$1,$2,$3,$4 に割り当てなおして、
	 * QUERYを実行します。また、$1 は、本来の値として、メニューの初期値設定等に
	 * 使用します。上記の例では、AAA が値で、それ以降は、引数になります。
	 * さらに、元の文字列"AAA:BBB:CCC:DDD"は、$0 に割り当てられます。割り当てがない
	 * 変数は、""(ゼロ文字列)として、扱われます。
	 *
	 * @og.rev 3.8.5.3 (2006/06/30) 位置を絶対位置指定(position:absolute;)
	 * @og.rev 4.3.7.2 (2009/06/15) 属性でidが出力される場合は、idを出力しない
	 * @og.rev 6.2.0.0 (2015/02/27) SLABEL 対応
	 * @og.rev 6.4.5.3 (2016/05/13) value は、コロン区切りの先頭だけ分離する。
	 *
	 * @param	value 入力値
	 *
	 * @return	データの編集用文字列
	 * @og.rtnNotNull
	 */
	@Override
	public String getValue( final String value ) {
		final String newValue = StringFormat.getValue( value );				// 6.4.5.3 (2016/05/13) コロン区切りの先頭だけ

		// input タグの作成
		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
		final String intag = new TagBuffer( "input" )
						.add( "name"	, name )
						.add( "id"		, name , isNull( attributes.get( "id" ) ) )	// 4.3.7.2 (2009/06/15)
						.add( "value"	, newValue )								// 6.4.5.3 (2016/05/13)
						.add( "size"	, size1 )
						.add( tagBuffer.makeTag() )
						.makeTag();

		final boolean useSlbl = "true".equalsIgnoreCase( useSLabel );		// 6.2.0.0 (2015/02/27)

		// select タグの作成
		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
		final String seltag = getOption(
					new TagBuffer( "select" )
						.add( "id"		, name + ".sel" )			// INDBMENU 特有のJavaScript用のキー
						.add( "style"	, "position:absolute;" )	// 3.8.5.3 (2006/06/30) 位置を絶対位置指定
						.add( selTagBuffer.makeTag() )
					, value
					, useSlbl							// 6.2.0.0 (2015/02/27) SLABEL 対応
			).makeTag() ;

		return intag + CR + seltag + CR + SEL1 + name + SEL2;
	}

	/**
	 * name属性を変えた、データ表示/編集用のHTML文字列を作成します。
	 * テーブル上の name に 行番号を付加して、名前_行番号 で登録するキーを作成し,
	 * リクエスト情報を１つ毎のフィールドで処理できます。
	 *
	 * ここでは、AAA:BBB:CCC:DDD という値を、$1,$2,$3,$4 に割り当てなおして、
	 * QUERYを実行します。また、$1 は、本来の値として、メニューの初期値設定等に
	 * 使用します。上記の例では、AAA が値で、それ以降は、引数になります。
	 * さらに、元の文字列"AAA:BBB:CCC:DDD"は、$0 に割り当てられます。割り当てがない
	 * 変数は、""(ゼロ文字列)として、扱われます。
	 *
	 * @og.rev 3.8.5.1 (2006/04/28) makeInputMenu 呼び出し時の引数記述ミスを修正
	 * @og.rev 3.8.5.3 (2006/06/30) 位置を絶対位置指定(position:absolute;)
	 * @og.rev 4.3.7.2 (2009/06/15) 属性でidが出力される場合は、idを出力しない
	 * @og.rev 6.2.0.0 (2015/02/27) SLABEL 対応
	 * @og.rev 6.4.5.3 (2016/05/13) value は、コロン区切りの先頭だけ分離する。
	 *
	 * @param	row   行番号
	 * @param	value 入力値
	 *
	 * @return	データ表示/編集用の文字列
	 * @og.rtnNotNull
	 */
	@Override
	public String getValue( final int row,final String value ) {
		final String name2 = name + HybsSystem.JOINT_STRING + row ;
		final String newValue = StringFormat.getValue( value );				// 6.4.5.3 (2016/05/13) コロン区切りの先頭だけ

		// input タグの作成
		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
		final String intag = new TagBuffer( "input" )
						.add( "name"	, name2 )
						.add( "id"		, name2 , isNull( attributes.get( "id" ) ) )	// 4.3.7.2 (2009/06/15)
						.add( "value"	, newValue )			// 6.4.5.3 (2016/05/13)
						.add( "size"	, size2 )
						.add( tagBuffer.makeTag() )
						.makeTag( row,newValue );				// 6.4.5.3 (2016/05/13) [V] に割り当てる値。

		final boolean useSlbl = "auto".equalsIgnoreCase( useSLabel ) || "true".equalsIgnoreCase( useSLabel );		// 6.2.0.0 (2015/02/27)

		// select タグの作成
		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
		final String seltag = getOption(
					new TagBuffer( "select" )
						.add( "id"		, name2 + ".sel" )			// INDBMENU 特有のJavaScript用のキー
						.add( "style"	, "position:absolute;" )	// 3.8.5.3 (2006/06/30) 位置を絶対位置指定
						.add( selTagBuffer.makeTag() )
					, value
					, useSlbl							// 6.2.0.0 (2015/02/27) SLABEL 対応
			).makeTag( row,newValue ) ;					// 6.4.5.3 (2016/05/13) [V] に割り当てる値。

		return intag + CR + seltag + CR + SEL1 + name2 + SEL2;
	}

	/**
	 * 初期値が選択済みの 選択肢(オプション)をTagBuffer に反映します。
	 * このオプションは、引数の値を初期値とするオプションタグ作成し、TagBuffer
	 * に値を設定して返します。
	 *
	 * ここでは、AAA:BBB:CCC:DDD という値を、$1,$2,$3,$4 に割り当てなおして、
	 * QUERYを実行します。また、$1 は、本来の値として、メニューの初期値設定等に
	 * 使用します。上記の例では、AAA が値で、それ以降は、引数になります。
	 * さらに、元の文字列"AAA:BBB:CCC:DDD"は、$0 に割り当てられます。割り当てがない
	 * 変数は、""(ゼロ文字列)として、扱われます。
	 * 又、$Cには自分自身のカラム名を割り当てます。
	 *
	 * @og.rev 3.5.5.7 (2004/05/10) getOption( String value )の廃止を受けて、新規作成
	 * @og.rev 3.6.0.6 (2004/10/22) シーケンスアクセス機能(seqFlag)を追加します
	 * @og.rev 4.0.0.0 (2006/11/15) SelectionFactory に lang 属性を追加します。
	 * @og.rev 4.3.4.0 (2008/12/01) $C対応
	 * @og.rev 6.2.0.0 (2015/02/27) SLABEL 対応
	 * @og.rev 6.2.0.0 (2015/02/27) キー:ラベル形式で表示するかどうかを、指定できるようにします。
	 *
	 * @param	buf  反映させるTagBufferオブジェクト
	 * @param	value 選択されている値
	 * @param   useSlbl ラベル(短)をベースとしたオプション表示を行うかどうか。
	 *
	 * @return	オプションタグ
	 * @og.rtnNotNull
	 */
	private TagBuffer getOption( final TagBuffer buf,final String value,final boolean useSlbl ) {

		// StringFormat format = new StringFormat( query,value );
		final StringFormat format = new StringFormat( query, value, name );		// 4.3.4.0 (2008/12/01)
		final String newQuery = format.format();
		final String newValue = format.getValue();

		// 6.2.0.0 (2015/02/27) キー:ラベル形式
		final Selection selection = SelectionFactory.newDBSelection( newQuery, dbid, lang, addKeyLabel );

		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
		return buf.addBody( Selection.NO_VALUE_OPTION , addNoValue )				// 5.5.1.0 (2012/04/03)
				  .addBody( selection.getOption( newValue, seqFlag, useSlbl ) );	// 6.2.0.0 (2015/02/27) SLABEL 対応
	}
}
