/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import static org.opengion.fukurou.util.StringUtil.nval;

import java.io.File;
import java.io.PrintWriter;
import java.sql.Connection;

import org.opengion.fukurou.db.ConnectionFactory;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.ToString;						// 6.1.1.0 (2015/01/17)
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.common.SystemInstaller;

/**
 * システムのインストールを行うためのタグです。
 *
 * システムインストール用のアーカイブ(ZIP)形式を指定することで、以下の設定を自動で行います。
 * ①webapps以下へのコンテキストファイルの展開T
 * ②Tomcatのコンテキスト設定
 * ③各種DB環境のインストール
 *
 * 5.6.7.0 (2013/07/27) 
 * ※ インストールするデータベースは、dbid パラメータで指定できますが、リソースを登録するデータベースは
 * 実行している コンテキストの RESOURCE_DBID で、外部から指定できません。ご注意ください。
 *
 * このアーカイブは、ルートディレクトリにコンテキストパスのドキュメントベースが配置されている必要があります。
 *
 * [アーカイブの構成]
 * xx.zip - gf - db - ...               (DB環境インストール用のスクリプト)
 *             - filetemp - ...         (一時ファイルの保存場所 ※中身は空)
 *             - jsp - ...              (画面JSP)
 *             - log - ...              (ログファイルの保存場所 ※中身は空)
 *             - WEB-INF -  ...         (接続先情報、openGionのjarファイル)
 *                       - [CONTEXT].xml(コンテキスト設定ファイル)
 *　
 * ※WEB-INF直下の[CONTEXT].xmlは、Tomcatのconfディレクトリ以下に、コンテキスト設定ファイルとして、コピーされます。
 *   なお、この[CONTEXT].xmlがアーカイブに含まれていない場合、インストールは行われません。
 *
 * @og.formSample
 * ●形式：&lt;og:sysInstall fileURL="…" insFile="…" /&gt;
 * ●body：なし
 *
 * ●Tag定義：
 *   &lt;og:sysInstall
 *       insFile          ○【TAG】インストールファイルを指定します(必須)。
 *       fileURL            【TAG】操作するファイルのディレクトリを指定します (初期値:FILE_URL[=filetemp/])
 *       dbid               【TAG】(通常は使いません)インストールを行うDB接続IDを指定します。
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   /&gt;
 *
 * ●使用例
 *    ・Calc(ods)ファイルをPDFに変換
 *        &lt;og:sysInstall insFile="context.zip" /&gt;
 *
 * @og.group その他部品
 *
 * @version  4.0
 * @author	 Hiroki Nakamura
 * @since    JDK5.0,
 */
public class SysInstallTag extends CommonTagSupport {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.4.2.1 (2016/02/05)" ;
	private static final long serialVersionUID = 642120160205L ;

	private String	insFile		;
	private String	fileURL 	= HybsSystem.sys( "FILE_URL" );
	private String	dbid		;				// 5.5.4.5 (2012/07/27) 新規追加

	private final String RESOURCE_DBID = HybsSystem.sys( "RESOURCE_DBID" );		// 5.6.7.0 (2013/07/27) リソース系DBID

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public SysInstallTag() { super(); }		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @return	後続処理の指示( SKIP_BODY )
	 */
	@Override
	public int doStartTag() {
		return SKIP_BODY ;
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 5.5.4.5 (2012/07/27) dbid 新規追加
	 * @og.rev 5.6.7.0 (2013/07/27) リソース登録用とアプリケーション登録用のコネクションを分ける
	 * @og.rev 6.4.2.1 (2016/02/05) HybsSystem.url2dir に引数追加。
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();

		final File buildFile = new File( HybsSystem.url2dir( fileURL , insFile ) );					// 6.4.2.1 (2016/02/05)

		// 5.6.7.0 (2013/07/27) リソース登録用とアプリケーション登録用のコネクションを分ける
		// エラー処理も合わせて、修正しておきます。
		Connection defConn = null;
		Connection rscConn = null;
		boolean errFlag = true;
		try {
			defConn = ConnectionFactory.connection( dbid, getApplicationInfo() );
			rscConn = ConnectionFactory.connection( RESOURCE_DBID, getApplicationInfo() );
			final PrintWriter out = new PrintWriter( pageContext.getOut() );
			final SystemInstaller installer = new SystemInstaller( defConn , rscConn , out );		// 5.6.7.0 (2013/07/27)
			installer.install( buildFile );
			errFlag = false;		// エラーではない
		}
		catch( final Throwable ex ) {
			final String errMsg = "インストール中に、DB処理でエラーが発生しました。"
						+ ex.getMessage()  + CR ;
			throw new HybsSystemException( errMsg, ex );
		}
		finally {
			// 以下のコーディングの見直しを考える必要有り。
			// Connection はプールしている為、close() しても再利用されるだけ。
			if( errFlag ) {
				ConnectionFactory.remove( defConn,dbid );			// 削除
				ConnectionFactory.remove( rscConn,RESOURCE_DBID );
			}
			else {
				ConnectionFactory.close( defConn,dbid );			// 返却
				ConnectionFactory.close( rscConn,RESOURCE_DBID );
			}
		}

		return EVAL_PAGE ;
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 5.5.4.5 (2012/07/27) dbid 新規追加
	 */
	@Override
	protected void release2() {
		super.release2();
		insFile		= null;
		fileURL 	= HybsSystem.sys( "FILE_URL" );
		dbid		= null;				// 5.5.4.5 (2012/07/27) 新規追加
	}

	/**
	 * 【TAG】操作するファイルのディレクトリを指定します
	 *		(初期値:FILE_URL[={@og.value SystemData#FILE_URL}])。
	 *
	 * @og.tag
	 * この属性で指定されるディレクトリのファイルを操作します。
	 * 指定方法は、通常の fileURL 属性と同様に、先頭が、'/' (UNIX) または、２文字目が、
	 * ":" (Windows)の場合は、指定のURLそのままのディレクトリに、そうでない場合は、
	 * (初期値:システム定数のFILE_URL[={@og.value SystemData#FILE_URL}])。
	 *
	 * @og.rev 6.4.2.1 (2016/02/05) URLの最後に、"/" を追加する処理を廃止。
	 *
	 * @param	url ファイルURL
	 * @see		org.opengion.hayabusa.common.SystemData#FILE_URL
	 */
	public void setFileURL( final String url ) {
		final String furl = nval( getRequestParameter( url ),null );
		if( furl != null ) {
			fileURL = StringUtil.urlAppend( fileURL,furl );
		}
	}

	/**
	 * 【TAG】インストールファイルを指定します。
	 *
	 * @og.tag
	 * インストールファイルを指定します。
	 * 各コンテキストのインストール用アーカイブを指定する必要があります。
	 *
	 * @param	fname 出力ファイル名
	 */
	public void setInsFile( final String fname ) {
		insFile = nval( getRequestParameter( fname ),insFile );
	}

	/**
	 * 【TAG】(通常は使いません)インストールを行うDB接続IDを指定します(初期値:null)。
	 *
	 * @og.tag Queryオブジェクトを作成する時のDB接続IDを指定します。
	 *
	 * @og.rev 5.5.4.5 (2012/07/27) dbid 新規追加
	 *
	 * @param	id データベース接続ID
	 */
	public void setDbid( final String id ) {
		dbid = nval( getRequestParameter( id ),dbid );
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION	)
				.println( "insFile"		,insFile	)
				.println( "fileURL"		,fileURL	)
				.println( "dbid"		,dbid		)				// 5.5.4.5 (2012/07/27) 新規追加
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
