/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import java.util.Locale;

import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.db.AbstractDBType;
import org.opengion.hayabusa.db.DBTypeCheckUtil;

/**
 * 今までのカラム属性は昔からのルールに従っているため、
 * 品番情報の文字列を扱う為のカラム属性を新たに定義します。
 *
 * 品番で使用可能な文字化どうか（例えばI等）は考慮しません。
 *
 * タイプチェックとして、以下の条件を判定します。
 * ・文字列長は、Byte換算での文字数との比較
 * ・文字列チェックはXUと同じ
 * ・文字パラメータの 正規表現チェック
 * ・クロスサイトスクリプティングチェック
 * 
 * ハイフンは、
 * ・入力文字数が13または14桁
 * ・4桁目＆10桁目が"-"
 * ・他にハイフンがない
 * を全て満たした場合に削除を行います。
 *
 * @og.group データ属性
 *
 * @version  5.0
 * @author   Takahashi Masakazu
 * @since    JDK5.0,
 */
public class DBType_PN2 extends AbstractDBType {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.4.3.2 (2016/02/19)" ;

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.3.2 (2016/02/19) PMD refactoring. Each class should declare at least one constructor.
	 */
	public DBType_PN2() { super(); }		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。

	/**
	 * エディターで編集されたデータを登録する場合に、データそのものを
	 * 変換して、実登録データを作成します。
	 * データの表示用文字列を返します。
	 * 特定の条件の場合、XXX-XXXXX-XXX 形式で入力された情報を、XXXXXXXXXXX 形式で表示します。
	 * カット＆ペースト対策です。
	 * ※入力文字数が12または13桁
	 * 　かつ、4桁目＆10桁目が"-"
	 * 　かつ、他にハイフンがない
	 * 場合にハイフンの削除を行います。
	 *
	 * @og.rev 6.4.3.1 (2016/02/12) newValの求め方が、おかしい。
	 * @og.rev 5.9.7.1 (2016/04/06) スペースは前後を取るようにする
	 *
	 * @param	value	(一般に編集データとして登録されたデータ)
	 *
	 * @return  修正後の文字列(一般にデータベースに登録するデータ)
	 */
	@Override
	public String valueSet( final String value ) {

		String newVal = value;
		if( newVal != null ) {
			newVal = newVal.trim().toUpperCase(Locale.JAPAN);
			if( ( newVal.length()==12 || newVal.length()==13 )
					&& newVal.indexOf( '-' ) == 3
					&& newVal.lastIndexOf( '-' ) == 9
					&& newVal.indexOf( '-' , 4 ) == 9 ) {
				newVal = StringUtil.replace( newVal,"-","" );
			}
		}

		return newVal;

	}

	/**
	 * データが登録可能かどうかをチェックします。
	 * データがエラーの場合は、そのエラー内容を返します。
	 *
	 * @param   key		キー
	 * @param   value	値
	 * @param   sizeX	整数部分の文字列の長さ
	 * @param   sizeY	少数部分の文字列の長さ
	 * @param   typeParam	dbType パラメータ(文字パラメータ)
	 * @param   isStrict	厳密にチェックするかどうか[true:する/false:標準的]
	 *
	 * @return  エラー内容
	 */
	@Override
	public ErrorMessage valueCheck( final String key ,final String value ,
									final int sizeX ,final int sizeY ,final String typeParam ,final boolean isStrict) {

		ErrorMessage msg = new ErrorMessage();
		if( value == null || value.length() == 0 ) { return msg; }

		final int len = (sizeY == 0) ? sizeX : sizeX + sizeY + 1;
		if( value.length() > len ) {
			// 文字列の長さが指定の長さよりも長いです。
			msg.addMessage( 0,ErrorMessage.NG,"ERR0006", key,value, String.valueOf( value.length() ), String.valueOf( len ) );
		}

		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );
		boolean isError = false;
		for( int i=0; i<value.length(); i++ ) {
			final char ch = value.charAt( i );
			if( ch < 0x20 || ch > 0x7e || 'a' <= ch && ch <= 'z' ) {			// 6.9.7.0 (2018/05/14) PMD Useless parentheses.
				buf.append( "<span class=\"NG\">" ).append( ch ).append( "</span>" );
				isError = true;
			}
			else {
				buf.append( ch );
			}
		}
		if( isError ) {
			// 指定の文字以外の文字が使われています。
			msg.addMessage( 0,ErrorMessage.NG,"ERR0009", key,buf.toString() );
		}

		// 3.6.0.0 (2004/09/22) dbType パラメータ(文字パラメータ)を使用したマッチチェック
		final String check = DBTypeCheckUtil.matcheCheck( value,typeParam );
		if( check != null ) {
			// 指定の文字以外の文字が使われています。
			msg.addMessage( 0,ErrorMessage.NG,"ERR0009", key,check );
		}

		// クロスサイトスクリプティング対策：'<', '>' は登録させない。
		msg = xssCheck( key ,value, msg );
		return msg;
	}
}
