/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.taglet;

import org.opengion.fukurou.system.HybsConst;	// 6.1.0.0 (2014/12/26) refactoring

import com.sun.javadoc.Tag;
import com.sun.tools.doclets.Taglet;

/**
 * ソースコメントから、Javadoc を作成する場合のカスタムタグ情報を作成する
 * Taglet インターフェースのAbstract 実装クラスを作成します。
 * デフォルトの タグレットとしての初期値を設定します。
 * つまり、isInlineTag() のみ false を返すメソッドを定義してあります。
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
@SuppressWarnings(value={"deprecation"})			// Ver7.0.0.0
public abstract class AbstractTaglet implements Taglet {
	/** StringBilderなどの初期値を設定します。	{@value} */
	protected static final int BUFFER_MIDDLE = HybsConst.BUFFER_MIDDLE;	// 6.1.0.0 (2014/12/26) refactoring

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	protected AbstractTaglet() { super(); }		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。

	/**
	 * このタグレットがフィールドドキュメンテーションで
	 * 使用されている場合、true を返します。
	 * インラインタグについては false が設定されます。
	 *
	 * @return フィールドドキュメンテーションの場合は true、そうでない場合は false を返します
	 */
	public boolean inField() {
		return true;
	}

	/**
	 * このタグレットがコンストラクタドキュメンテーションで
	 * 使用されている場合、true を返します。
	 * インラインタグについては false が設定されます。
	 *
	 * @return コンストラクタドキュメンテーションの場合は true、そうでない場合は false を返します
	 */
	public boolean inConstructor() {
		return true;
	}

	/**
	 * このタグレットがメソッドドキュメンテーションで
	 * 使用されている場合、true を返します。
	 * インラインタグについては false が設定されます。
	 *
	 * @return メソッドドキュメンテーションの場合は true、そうでない場合は false を返します
	 */
	public boolean inMethod() {
		return true;
	}

	/**
	 * このタグレットが概要ドキュメンテーションで
	 * 使用されている場合、true を返します。
	 * インラインタグについては false が設定されます。
	 *
	 * @return 概要ドキュメンテーションの場合は true、そうでない場合は false を返します
	 */
	public boolean inOverview() {
		return true;
	}

	/**
	 * このタグレットがパッケージドキュメンテーションで
	 * 使用されている場合、true を返します。
	 * インラインタグについては false が設定されます。
	 *
	 * @return パッケージドキュメンテーションの場合は true、そうでない場合は false を返します
	 */
	public boolean inPackage() {
		return true;
	}

	/**
	 * このタグレットがタイプドキュメンテーションで
	 * 使用されている場合、true を返します。
	 * インラインタグについては false が設定されます。
	 *
	 * @return タイプドキュメンテーションの場合は true、そうでない場合は false を返します
	 */
	public boolean inType() {
		return true;
	}

	/**
	 * このタグレットがインラインタグで
	 * 使用されている場合、true を返します。
	 * そうでない場合については false が設定されます。
	 *
	 * @return インラインタグの場合は true、そうでない場合は false を返します
	 */
	public boolean isInlineTag() {
		return false;
	}

	/**
	 * このタグレットが各種インラインタグを処理するように用意された、
	 * カスタムメソッドです。
	 * 現時点では、{&#064;link XXXX YYYY} 、{&#064;og.value package.class#field}、
	 * {&#064;og.doc03Link queryType Query_**** クラス} を処理できます。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規追加
	 *
	 * @param tag Tagオブジェクト
	 *
	 * @return	インラインタグ を処理した結果の文字列
	 * @og.rtnNotNull
	 */
	protected String text( final Tag tag ) {
		if( tag == null ) { return ""; }
		final String in = tag.text();

		int index = in.indexOf( "{@" );
		if( index < 0 ) { return in; }

		final StringBuilder rtn = new StringBuilder( BUFFER_MIDDLE );
		int start = 0;
		while( index >= 0 ) {
			int key = in.indexOf( ' ',index );			// {@key ････} のkey分離
			final int end = in.indexOf( '}',index );			// {@key ････} の終端分離
			if( end < 0 ) { break; }					// { と } の整合性不一致なので、即抜ける。

			if( key < 0 || key > end ) { key = end; }	// keyのみの場合。

			// {@ より前方の文字列を追加
			rtn.append( in.substring( start,index ) );

			// 					key 部分					それ以降の引数
			rtn.append( inline( in.substring( index+2,key ),in.substring( key+1,end ) , tag ) );

			start = end+1 ;
			index = in.indexOf( "{@",start );			// 次のインラインタグ
		}
		rtn.append( in.substring( start ) );			// 残り部分を登録

		return rtn.toString();
	}

	/**
	 * このタグレットが各種インラインタグを処理するように用意された、
	 * カスタムメソッドです。
	 * 現時点では、{&#064;link XXXX YYYY} 、{&#064;og.value package.class#field}、
	 * {&#064;og.doc03Link queryType Query_**** クラス} を処理できます。
	 *
	 * @og.rev 6.0.2.0 (2014/09/19) 新規追加
	 *
	 * @param	key		タグの名称
	 * @param	body	処理する文字列
	 * @param	tag		Tagオブジェクト (og.value の時のみ必要)
	 *
	 * @return	インラインタグ を処理した結果の文字列
	 */
	private String inline( final String key , final String body , final Tag tag ) {
		if( "link".equalsIgnoreCase( key ) ) {
			return DocletUtil.linkTag( body );
		}
		else if( "og.value".equalsIgnoreCase( key ) ) {
			return DocletUtil.valueTag( body , tag );
		}
		else if( "og.doc03Link".equalsIgnoreCase( key ) ) {
			return DocletUtil.doc03LinkTag( body );
		}
		else {
			// タグを再作成して戻す。
			return "{@" + key + " " + body + "}" ;
		}
	}

}
