/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.mail;

import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeMessage;

/**
 * MailCharset は、E-Mail 送信時のエンコードに応じた処理を行う為の、
 * インターフェースです。
 *
 * E-Mail で日本語を送信する場合、ISO-2022-JP(JISコード)化して、7bit で
 * エンコードして送信する必要がありますが、Windows系の特殊文字や、unicodeと
 * 文字のマッピングが異なる文字などが、文字化けします。
 * 対応方法としては、
 * 『１．Windows-31J/UTF-8 + 8bit 送信』
 * 『２．ISO-2022-JP に独自変換 + 7bit 送信』
 * の方法があります。
 * 今回、この２つの方法について、それぞれサブクラス化を行い、処理できるように
 * したのが、このインターフェース、および、サブクラスです。
 *
 * 『１．Windows-31J/UTF-8 + 8bit 送信』の方法は、通常の JavaMail API に準拠して
 * <del>6.3.8.0 (2015/09/11) 処理を行う、Mail_Windows31J_Charset サブクラスで実装しています。</del>
 * 処理を行う、Mail_8bit_Charset サブクラスで実装しています。
 * 古いメイラーおよび、古いメールサーバーではメール転送できない為、
 * この方式は、社内で使用する場合のみに、利用できますが、主としてWindows系の
 * 社内システムにおいては、こちらの方が、なにかとトラブルは少ないと思います。
 *
 * 『２．ISO-2022-JP に独自変換 + 7bit 送信』の実装は、
 * JAVA PRESS Vol.37 (http://www.gihyo.co.jp/magazines/javapress)の
 * 【特集1】 決定版！ サーバサイドJavaの日本語処理
 *  第3章：JavaMailの日本語処理プログラミング……木下信
 *“マルチプラットフォーム”な日本語メール送信術 完全解説
 * でのサンプルアプリケーション
 * http://www.gihyo.co.jp/book/2004/225371/download/toku1_3.zip
 * を、使用して、Mail_ISO2022JP_Charset サブクラスで実装しています。
 *
 * これらのサブクラスは、MailCharsetFactory ファクトリクラスより、作成されます。
 * その場合、引数のキャラクタセット名は、Windows-31J 、MS932 、 UTF-8 か、それ以外となっています。
 * それ以外が指定された場合は、ISO-2022-JP を使用します。
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public interface MailCharset {

	/**
	 * テキストをセットします。
	 * Part#setText() の代わりにこちらを使うようにします。
	 *
	 * ※ 内部で、MessagingException が発生した場合は、RuntimeException に変換されて throw されます。
	 *
	 * @param mimeMsg MimeMessage最大取り込み件数
	 * @param text    設定するテキスト
	 */
	void setTextContent( MimeMessage mimeMsg, String text ) ;

	/**
	 * 日本語を含むヘッダ用テキストを生成します。
	 * 変換結果は ASCII なので、これをそのまま setSubject や InternetAddress
	 * のパラメタとして使用してください。
	 *
	 * ※ 内部で、UnsupportedEncodingException が発生した場合は、RuntimeException に変換されて throw されます。
	 *
	 * @param text    設定するテキスト
	 *
	 * @return	日本語を含むヘッダ用テキスト
	 */
	String encodeWord( String text ) ;

	/**
	 * 日本語を含むアドレスを生成します。
	 * personal に、日本語が含まれると想定しています。
	 * サブクラスで、日本語処理を行う場合の方法は、それぞれ異なります。
	 *
	 * ※ 内部で、UnsupportedEncodingException が発生した場合は、RuntimeException に変換されて throw されます。
	 *
	 * @param address    アドレス部分
	 * @param personal   日本語の説明部分
	 *
	 * @return	日本語を含むアドレス
	 */
	InternetAddress getAddress( String address,String personal ) ;

	/**
	 * Content-Transfer-Encoding を指定する場合の ビット数を返します。
	 *
	 * Windows系は、8bit / ISO-2022-JP 系は、7bit になります。
	 *
	 * @return	ビット数
	 */
	String getBit() ;
}
