/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.table;

import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.db.AbstractTableFilter;
import org.opengion.hayabusa.db.DBTableModel;

import java.util.Map;

/**
 * TableFilter_BIKO2CODE は、TableFilter インターフェースを継承した、DBTableModel 処理用の
 * 実装クラスです。
 *
 * ここでは、DB定義書情報の備考欄を分解し、コードリソースとして登録可能な形に再構築します。
 * 入力カラムとしてBIKOが、出力カラムとして、CODE,CODENAME,SEQが必要です。
 * それぞれのカラム名は、keys,valsの引数としても指定可能です。
 *
 * 分解方法としては、まず備考欄を' '(スペース)区切りに分解します。
 * その上で、さらに取り出した値を':'で分解してコードとコード名称に分離します。
 * 順番(SEQ)については、備考欄に記載されている順番になります。
 * 「キー:ラベル キー:ラベル」で、ラベル にスペースを含ませる場合は、ダブルクォーテーションで
 * 囲ってください。
 *
 * また、BIKO,CODE,CODENAME,SEQ で指定したカラムが DBTableModel に存在しない場合は、
 * 処理そのものを無視します。その場合は、警告も出力されませんので、ご注意ください。
 *
 * パラメータは、tableFilterタグの keys, vals にそれぞれ記述するか、BODY 部にCSS形式で記述します。
 * 【パラメータ】
 *  {
 *       BIKO       : 入力カラム ;    備考情報を格納している入力カラム名。
 *       CODE       : 出力カラム ;    コードを格納する出力カラム名。
 *       CODENAME   : 出力カラム ;    コード名称を格納する出力カラム名。
 *       SEQ        : 出力カラム ;    並び順を格納する出力カラム名。備考欄に記載されている順番になります。
 *  }
 *
 * @og.formSample
 * ●形式：
 *      select TABLE_NAME,CLM,NAME_JA,EDITOR,RENDERER,DBTYPE,BIKO,0 SEQ,'' CODE,'' CODENAME,UNIQ,SYSTEM_ID from GF05
 *      ① &lt;og:tableFilter classId="BIKO2CODE" keys="BIKO,CODE," vals='"TABLE_NAME,CLM"' /&gt;
 *
 *      ② &lt;og:tableFilter classId="BIKO2CODE" &gt;
 *               {
 *                       BIKO       : 入力カラム ;    備考情報を格納している入力カラム名。
 *                       CODE       : 出力カラム ;    コードを格納する出力カラム名。
 *                       CODENAME   : 出力カラム ;    コード名称を格納する出力カラム名。
 *                       SEQ        : 出力カラム ;    並び順を格納する出力カラム名。備考欄に記載されている順番になります。
 *               }
 *         &lt;/og:tableFilter&gt;
 *
 * @og.rev 4.1.0.0(2008/01/18) 新規作成
 * @og.rev 5.6.6.0 (2013/07/05) keys の整合性チェックを追加
 *
 * @version  0.9.0  2000/10/17
 * @author   Hiroki Nakamura
 * @since    JDK1.1,
 */
public class TableFilter_BIKO2CODE extends AbstractTableFilter {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "5.6.6.1 (2013/07/12)" ;

	/**
	 * keys の整合性チェックを行うための初期設定を行います。
	 *
	 * @og.rev 5.6.6.1 (2013/07/12) keys の整合性チェック対応
	 *
	 * @param	keysMap keys の整合性チェックを行うための Map
	 */
	@Override
	protected void init( final Map<String,String> keysMap ) {
		keysMap.put( "BIKO"		, "備考情報を格納している入力カラム名"		);
		keysMap.put( "CODE" 	, "コードを格納する出力カラム名"			);
		keysMap.put( "CODENAME" , "コード名称を格納する出力カラム名"		);
		keysMap.put( "SEQ" 		, "並び順を格納する出力カラム名"			);
	}

	/**
	 * DBTableModel処理を実行します。
	 *
	 * @og.rev 5.5.2.6 (2012/05/25) protected変数を、private化したため、getterメソッドで取得するように変更
	 * @og.rev 5.5.8.5 (2012/11/27) スペースで分割し、":" が存在する箇所のみ、コードリソース化します。
	 *
	 * @return 処理結果のDBTableModel
	 */
	public DBTableModel execute() {
		DBTableModel table = getDBTableModel();		// 5.5.2.6 (2012/05/25) インターフェースにgetterメソッド追加

		int bikoNo = table.getColumnNo( StringUtil.nval( getValue( "BIKO" ), "BIKO" ), false );		// 存在しない場合は、-1 を返す。
		int codeNo = table.getColumnNo( StringUtil.nval( getValue( "CODE" ), "CODE" ), false );
		int nameNo = table.getColumnNo( StringUtil.nval( getValue( "CODENAME" ),"CODENAME" ), false );
		int seqNo  = table.getColumnNo( StringUtil.nval( getValue( "SEQ" ),  "SEQ" ), false );

		char sep = ' ';

		if( bikoNo >= 0 && codeNo >= 0 && nameNo >= 0 && seqNo >= 0 ) {
			String[] data  = null;
			String[] nData = null;
			String[] clmValArr = null;
			int addRows = 0;
			int rowCnt = table.getRowCount();
			for( int row=0; row<rowCnt; row++ ) {
				addRows = 0;
				data = table.getValues( row );
				// タグがあった場合は無視
				if( data[bikoNo].indexOf( "/>" ) < 0 && data[bikoNo].indexOf( "</" ) < 0 ) {
					clmValArr = StringUtil.csv2Array( data[bikoNo], sep );
//					for ( int i = 0; i < clmValArr.length; i++ ) {
//						addRows++;
//						nData = new String[data.length];
//						System.arraycopy( data, 0, nData, 0, data.length );
//						int clnIdx = clmValArr[i].indexOf( ':' );
//						if( clnIdx < 0 ) {
//							nData[codeNo] = clmValArr[i];
//							nData[nameNo] = "";
//						}
//						else {
//							nData[codeNo] = clmValArr[i].substring( 0, clmValArr[i].indexOf( ':' ) );
//							nData[nameNo] = clmValArr[i].substring( clmValArr[i].indexOf( ':' ) + 1 );
//						}
//						nData[seqNo] = String.valueOf( i );
//						table.addValues( nData, row+addRows, false );
//					}
	 				// 5.5.8.5 (2012/11/27) スペースで分割し、":" が存在する箇所のみ、コードリソース化します。
					int seq = 0;
					for ( int i = 0; i < clmValArr.length; i++ ) {
						String clmVal = clmValArr[i];
						int clnIdx = clmVal.indexOf( ':' );
						if( clnIdx >= 0 ) {
							nData = new String[data.length];
							System.arraycopy( data, 0, nData, 0, data.length );

							nData[codeNo] = clmVal.substring( 0, clnIdx );
							nData[nameNo] = clmVal.substring( clnIdx + 1 );
							nData[seqNo]  = String.valueOf( seq++ );

							addRows++;
							table.addValues( nData, row+addRows, false );
						}
					}

					if( addRows > 0 ) {
						table.removeValue( row );
						addRows--;
					}

					row += addRows;
					rowCnt += addRows;
				}
			}
		}

		return table;
	}
}
