package org.opengion.plugin.column;

import org.opengion.fukurou.util.Attributes;
import org.opengion.fukurou.util.TagBuffer;
import org.opengion.fukurou.util.XHTMLTag;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.db.AbstractEditor;
import org.opengion.hayabusa.db.CellEditor;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.hayabusa.db.Selection;

/**
 * DATALIST_R エディターは、コードリソースから取得した値より、datalistを作成して
 * 入力候補となるデータリストを定義する編集用エディタークラスです。
 * datalist は、HTML5 から採用されたタグです。
 * 
 * 入力フィールドとdatalistタグの関係付は、カラムIDに、"カラムID.sel" で結びつけます。
 * 
 * &lt;input name="カラムID" list="カラムID.sel" /&gt;
 * &lt;div style="display:none;"&gt;
 *   &lt;datalist id="カラムID.sel"&gt;
 *     &lt;option value="KEY1"&gt;LABEL1&lt;/option&gt;
 *     &lt;option value="KEY2"&gt;LABEL2&lt;/option&gt;
 *     &lt;option value="KEY3"&gt;LABEL3&lt;/option&gt;
 *   &lt;/datalist&gt;
 * &lt;/div&gt;
 * 
 * divタグは、HTML5 非対応ブラウザを使用した場合、datalist の option がそのまま
 * テキストとして見えてしまうのを避けるためです。
 * 
 * @og.rev 5.9.18.1 (2017/03/24)
 * @og.group データ編集
 * 
 * @version 5.9.18.1
 * @author T.OTA
 * @since JDK6.0
 *
 */
public class Editor_DATALIST_R extends AbstractEditor {
	private static final String	VERSION			= "5.9.18.1 (2017/03/24)";
	private static final String	DIV1			= "<div style=\"display:none;\">";
	private static final String	DIV2			= "</div>";

	/** セレクションオブジェクト */
	protected Selection			selection;
	private Selection			bkSel			= null;							// Selection オブジェクトのキャッシュ機能
	private final boolean		seqFlag;
	private final TagBuffer		selTagBuffer	= new TagBuffer();

	/**
	 * デフォルトコンストラクター。
	 * このコンストラクターで、基本オブジェクトを作成します。
	 * 
	 */
	public Editor_DATALIST_R() {
		selection = null;
		seqFlag = false;
	}

	/**
	 * コンストラクター。
	 * 
	 * @param clm DBColumnオブジェクト
	 */
	private Editor_DATALIST_R( final DBColumn clm ) {
		super( clm );
		tagBuffer.add( XHTMLTag.inputAttri( attributes ) );
		seqFlag = "SEQ".equals( clm.getEditorParam() );

		String disabled = clm.isWritable() ? null : "disabled";

		Attributes selAttri = new Attributes();
		selAttri.set( "disabled", disabled );

		selAttri.addAttributes( clm.getEditorAttributes() );
		selTagBuffer.add( XHTMLTag.selectAttri( selAttri ) );

		selection = clm.getSelection();
	}

	/**
	 * 各オブジェクトから自分のインスタンスを返します。
	 * 自分自身をキャッシュするのか、新たに作成するのかは、各サブクラスの実装に
	 * まかされます。
	 * 
	 * @param	clm DBColumnオブジェクト
	 * 
	 * @return CellEditorオブジェクト
	 */
	@Override
	public CellEditor newInstance( final DBColumn clm ) {
		return new Editor_DATALIST_R( clm );
	}

	/**
	 * データの編集用文字列を返します。
	 * 
	 * @param value 入力値
	 * 
	 * @return データの編集用文字列
	 */
	@Override
	public String getValue( String value ) {
		// input タグの作成
		TagBuffer intag = new TagBuffer( "input" );
		intag.add( "name", name );
		if( attributes.get( "id" ) == null || attributes.get( "id" ).length() == 0 ) {
			intag.add( "id", name );
		}
		intag.add( "list", name + ".sel" );
		intag.add( "value", value );
		intag.add( "size", size1 );
		intag.add( tagBuffer.makeTag() );
		intag.add( optAttr );

		// datalist タグの作成
		TagBuffer dltag = new TagBuffer( "datalist" );
		dltag.add( "id", name + ".sel" );
		dltag = getOption( dltag, value, false ); // キャッシュは使用しない

		// display:none は、datalist の optionのBODY部が、HTML5以外では表示されてしまうのを防ぐため。
		return intag.makeTag() + HybsSystem.CR + DIV1 + dltag.makeTag() + DIV2 + HybsSystem.CR;
	}

	/**
	 * name属性を変えた、データ表示/編集用のHTML文字列を作成します。
	 * テーブル上の name に 行番号を付加して、名前_行番号 で登録するキーを作成し、
	 * リクエスト情報を１つ毎のフィールドで処理できます。
	 *
	 * @param	row		行番号
	 * @param	value	入力値
	 * 
	 * @return	データ表示/編集用の文字列
	 */
	@Override
	public String getValue( int row, String value ) {
		String name2 = name + HybsSystem.JOINT_STRING + row;

		// Selection オブジェクトのキャッシュ機能 (true:使用可能)
		boolean useSelCache = value != null && value.indexOf( ':' ) < 0;

		String listId = useSelCache ? name : name2; // キャッシュを使用する場合は、共通の name を使う。

		// input タグの作成
		TagBuffer intag = new TagBuffer( "input" );
		intag.add( "name", name2 );
		if( attributes.get( "id" ) == null || attributes.get( "id" ).length() == 0 ) {
			intag.add( "id", name2 );
		}
		intag.add( "list", listId + ".sel" );
		intag.add( "value", value );
		intag.add( "size", size2 );
		intag.add( tagBuffer.makeTag() );
		intag.add( optAttr );

		// datalist タグの作成
		TagBuffer dltag = new TagBuffer( "datalist" );
		dltag.add( "id", listId + ".sel" );
		dltag = getOption( dltag, value, useSelCache );

		// キャッシュが効くと、getOptionの戻り値は、nullになる。
		if( dltag != null ) {
			return intag.makeTag( row, value ) + HybsSystem.CR + DIV1 + dltag.makeTag( row, value ) + DIV2 + HybsSystem.CR;
		}
		else {
			return intag.makeTag( row, value ) + HybsSystem.CR;
		}
	}

	/**
	 * コードリソースから取得した、選択肢(オプション)をTagBuffer に反映します。
	 * 
	 * 第３引数は、Selection オブジェクトのキャッシュ機能を使用するかどうかを指定します。
	 * true で、使用する事を前提に、チェックを行います。
	 * 
	 * @param	buf			タグ文字列のバッファー
	 * @param	value		選択されている値
	 * @param	useSelCache	Selection オブジェクトのキャッシュ
	 * 
	 * @return	buf	オプションタグ
	 */
	private TagBuffer getOption( final TagBuffer buf, final String value, final boolean useSelCache ) {

		if( useSelCache ) {
			if( selection == bkSel ) { return null; }
			bkSel = selection;
		}

		buf.setBody( selection.getOption( value, seqFlag ) );

		return buf;
	}
}
